/*
 *  Apple II emulator preferences file routines
 *  by Aaron Culliney - chernabog@baldmountain.bbn.com - (C) 1997
 *  This code has nothing to do with my employer, BBN.  It was written
 *  completely on my own time and on my own machine.
 *
 *  prefs.l - simple lexical parser for the preferences file.
 *
 *  $Id: prefs.l,v 1.3 1997/06/29 22:14:17 chernabog Exp $
 *
 *   v0.4 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Jun 1997.
 *
 *  ASSUMPTIONS:
 *	flex version 2.5.2
 *
 *	use the `-Pprefs' to compile this.
 *
 **/

%{

#include "defs.H"
#include <stdio.h>
#include <ctype.h>
#include <string.h>

int line_num = 1;
void lowercase_string(char *s);
void strip_percent(char *s);

extern unsigned char disk_path[1024];
extern unsigned char system_path[2048];
extern unsigned short apple_speed;
extern short color_mode;
extern short sound_mode;

/* in keys.c */
extern joystick_mode joy_mode;
extern short joy_range, half_joy_range;
extern short joy_step;
extern short joy_center_x;
extern short joy_center_y;

#ifdef PC_JOYSTICK
/* in joystick.c */
extern void c_calculate_joystick_parms();
extern void c_open_joystick();
extern long js_timelimit;
extern short js_center_x;
extern short js_center_y;
extern short js_max_x;
extern short js_max_y;
extern short js_min_x;
extern short js_min_y;
#endif

%}


DIGIT		[0-9]
HEXDIGIT	[0-9abcdefABCDEF]
WHITESPACE	[\n\r\t" "]
CHARS		[^\n\r\t" "]+
BOS		^{WHITESPACE}*
EOS		{WHITESPACE}*\0
BOL		^{WHITESPACE}*
EOL		{WHITESPACE}*$


%%

{BOL}speed{WHITESPACE}*\={WHITESPACE}*{DIGIT}+\%{EOL} {
    /* speed of emulation, expressed as a percent (though it isn't!) */
    int speed;
    while (!isdigit(*prefstext)) ++prefstext;		/* advance to parm */
    strip_percent(prefstext);
    speed = atoi(prefstext);
    if (speed < 0)
	speed = 1;
    else if (speed > 100)
	speed = 100;
    apple_speed = 101 - speed;
}

{BOL}("disk path"|disk_path|path){WHITESPACE}*\={WHITESPACE}*{CHARS}+{EOL} {
    /* disk path = toplevel path to disks directory */
    while (*prefstext != '=') ++prefstext; ++prefstext; /* advance past '=' */
    while (isspace(*prefstext)) ++prefstext;
    strcpy(disk_path, prefstext);
}

{BOL}color{WHITESPACE}*\={WHITESPACE}*{CHARS}+{EOL} {
    /* color settings */
    while (*prefstext != '=') ++prefstext; ++prefstext; /* advance past '=' */
    while (isspace(*prefstext)) ++prefstext;
    lowercase_string(prefstext);
    if (!strcmp(prefstext, "off"))
	color_mode = 0;
    else if (!strcmp(prefstext, "on"))
	color_mode = 1;
    else if (!strcmp(prefstext, "interpolated"))
	color_mode = 2;
    else printf(
	"Illegal value to color parameter at line %d in .apple2\n", line_num);
}

{BOL}sound{WHITESPACE}*\={WHITESPACE}*{CHARS}+{WHITESPACE}*{CHARS}*{EOL} {
    /* sound output settings */
    while (*prefstext != '=') ++prefstext; ++prefstext; /* advance past '=' */
    while (isspace(*prefstext)) ++prefstext;
    lowercase_string(prefstext);
    if (!strcmp(prefstext, "off" ))
	sound_mode = 0;
    else if (!strcmp(prefstext, "on" ))
	sound_mode = 1;
    else if (!strcmp(prefstext, "pc speaker" ))
	sound_mode = 1;
    else printf(
	"Illegal value to sound parameter at line %d in .apple2\n", line_num);
}

{BOL}joystick{WHITESPACE}*\={WHITESPACE}*{CHARS}+{WHITESPACE}*{CHARS}*{EOL} {
    /* joystick preference setting */
    while (*prefstext != '=') ++prefstext; ++prefstext; /* advance past '=' */
    while (isspace(*prefstext)) ++prefstext;
    lowercase_string(prefstext);
    if (!strcmp(prefstext, "linear" ) || !strcmp(prefstext,"keyboard"))
	joy_mode = JOY_KYBD;
    else if (!strcmp(prefstext, "pc joystick" ) ||
	     !strcmp(prefstext, "pc_joystick" ))
    {
#ifdef PC_JOYSTICK
	joy_mode = JOY_PCJOY;
#else
	joy_mode = JOY_KYBD;
#endif
    }
    else if (!strcmp(prefstext, "digital" ))
    {
#if 0
	joy_mode = JOY_DIGITAL;		/* currently unsupported */
#else
	joy_mode = JOY_KYBD;
#endif
    }
    else if (!strcmp(prefstext, "off" ))
	joy_mode = JOY_OFF;
    else printf(
	"Illegal value to joystick parameter at line %d in .apple2\n",
	line_num);
}

{BOL}("joy range"|"joystick range"|joy_range){WHITESPACE}*\={WHITESPACE}*{DIGIT}+{EOL} {
    /* range = range of x and y joystick values */
    while (!isdigit(*prefstext)) ++prefstext;	/* advance to origin parm */
    joy_range = atoi(prefstext);
    if (joy_range < 10)
	joy_range = 10;
    else if (joy_range > 256)
	joy_range = 256;
    half_joy_range = joy_range/2;

    if (joy_center_x > joy_range)
	joy_center_x = half_joy_range;
    if (joy_center_y > joy_range)
	joy_center_y = half_joy_range;
}

{BOL}origin_x{WHITESPACE}*\={WHITESPACE}*{DIGIT}+{EOL} {
    /* origin_x = x coordinate of joystick origin (byte value) */
    while (!isdigit(*prefstext)) ++prefstext;	/* advance to origin parm */
    joy_center_x = atoi(prefstext);
    if (joy_center_x < 0)
	joy_center_x = 0;
    else if (joy_center_x > 255)
	joy_center_x = 255;

    if (joy_center_x > joy_range)
	joy_center_x = half_joy_range;
}

{BOL}origin_y{WHITESPACE}*\={WHITESPACE}*{DIGIT}+{EOL} {
    /* origin_y = y coordinate of joystick origin (byte value) */
    while (!isdigit(*prefstext)) ++prefstext;	/* advance to origin parm */
    joy_center_y = atoi(prefstext);
    if (joy_center_y < 0)
	joy_center_y = 0;
    else if (joy_center_y > 255)
	joy_center_y = 255;

    if (joy_center_y > joy_range)
	joy_center_y = half_joy_range;
}

{BOL}("pc joystick parms"|pc_joystick_parms){WHITESPACE}*\={WHITESPACE}*{DIGIT}+{WHITESPACE}+{DIGIT}+{WHITESPACE}+{DIGIT}+{WHITESPACE}+{DIGIT}+{WHITESPACE}+{DIGIT}+{WHITESPACE}+{DIGIT}+{WHITESPACE}+{DIGIT}+{EOL} {
#ifdef PC_JOYSTICK
    /* pc joystick parms generated by the calibration routine
       (shouldn't need to hand tweak these) = origin_x origin_y max_x
       min_x max_y min_y js_timelimit */
    while (!isdigit(*prefstext)) ++prefstext;	/* advance to 1st parm */
    js_center_x = strtol(prefstext, &prefstext, 10);
    js_center_y = strtol(prefstext, &prefstext, 10);
    js_max_x = strtol(prefstext, &prefstext, 10);
    if (js_max_x < 0)
	js_max_x = 0;
    js_min_x = strtol(prefstext, &prefstext, 10);
    if (js_min_x < 0)
	js_min_x = 0;

    js_max_y = strtol(prefstext, &prefstext, 10);
    if (js_max_y < 0)
	js_max_y = 0;
    js_min_y = strtol(prefstext, &prefstext, 10);
    if (js_min_y < 0)
	js_min_y = 0;

    js_timelimit = strtol(prefstext, &prefstext, 10);
    if (js_timelimit < 2)
	js_timelimit = 2;

    c_open_joystick();
    c_calculate_joystick_parms();	/* calculate the associated parms */
#endif
}

{BOL}sensitivity{WHITESPACE}*\={WHITESPACE}*{DIGIT}+\%{EOL} {
    /* keyboard-emulated joystick sensitivity */
    while (!isdigit(*prefstext)) ++prefstext;		/* advance to parm */
    strip_percent(prefstext);
    joy_step = atoi(prefstext);
    if (joy_step < 1)
	joy_step = 1;
    else if (joy_step > 100)
	joy_step = 100;
}

{BOL}("system path"|system_path){WHITESPACE}*\={WHITESPACE}*{CHARS}+{EOL} {
    /* system path = path to rom files */
    while (*prefstext != '=') ++prefstext; ++prefstext; /* advance past '=' */
    while (isspace(*prefstext)) ++prefstext;
    strcpy(system_path, prefstext);
}


\n	{
    ++line_num;		/* count lines */
}

.	{		/* flag extraneous characters */
    printf("Random character `%c' at line %d in .apple2\n",
	   *prefstext, line_num);
}


%%


int prefswrap () {
    return 1;
}

void lowercase_string( char *s ) {
    char *p;

    for (p = s; *p != '\0'; p++)
	*p = tolower(*p);
}

void strip_percent( char *s ) {
    char *p = s + strlen(s) - 1;
    if (p >= s && *p == '%')
	*p = '\0';
}

void pad_string(char *s, char c, int len) {
    char *p;

    for (p = s; ((*p != '\0') && (p-s < len-1)); p++);
    while (p-s < len-1) {
	*p++ = c;
    }
    *p = '\0';
}

/* for testing */
/* void main( int argc, char **argv ) { */
/*     ++argv, --argc; */
/*     if ( argc > 0 ) */
/* 	prefsin = fopen( argv[0], "r" ); */
/*     else */
/* 	prefsin = stdin; */
    
/*     prefslex(); */
/* } */
