/***************************************************************************
**    xMultiLinWin.cpp  $Revision: 1.13 $ - $Name: V2-18 $ 
**    QWidget for manipulating Multiple lines of text in a window
**
**    Copyright (C) 1996 Joseph Croft <jcroft@unicomp.net>
**
**    This library is free software; you can redistribute it and/or
**    modify it under the terms of the GNU Library General Public
**    License as published by the Free Software Foundation; either
**    version 2 of the License, or (at your option) any later version.
**
**    This library is distributed in the hope that it will be useful,
**    but WITHOUT ANY WARRANTY; without even the implied warranty of
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
**    Library General Public License for more details.
**
**    You should have received a copy of the GNU Library General Public
**    License along with this library; if not, write to the Free
**    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
 ***************************************************************************/
#include <ctype.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <qpainter.h>
#include "xMisc.h"
#include "xResources.h"
#include "xMultiLineWin.h"

static int dbg = 0;

#define WIN_MARGIN   2

static const char *pInitialResources[] =
{
   "*MultiLine.Window.Font.Family: Fixed",
   "*MultiLine.Window.HighColor: white",
   "*MultiLine.Window.TextColor: grey72",
   "*MultiLine.Window.Background: #001bbd",
   NULL
};

xMultiLineWin::xMultiLineWin(xWidgetResInfo *pPRes, QWidget *pParent,
                             const char *pName,
                             int width, int height,
                             int bufHeight) :
                  QWidget(pParent, pName, (WFlags)0), xMultiLineBuf(bufHeight)
{
   botLineOfs = 0;
   windowHeight = height;
   windowWidth = width;
   bufferLen = bufHeight;
   char size[20];
   const char *pHighLight;

   wdtRes = new xWidgetResInfo(pPRes, QString("multiline.window"), QString("MultiLine.Window"));
   Resources->setWidgetInit(pInitialResources);
   setDefPallet(this, wdtRes);
   setDefFont(this, wdtRes);
   sprintf(size, "%d", font().pointSize());
   if ((pHighLight = Resources->get(wdtRes, "highcolor", "HighColor"))
       == NULL)
      pHighLight = "yellow";

   boldColor.setNamedColor(pHighLight);

   QFontMetrics fm = QWidget::fontMetrics();
   setGeometry(0, 0, fm.maxWidth() * width + (WIN_MARGIN * 2), 
                     fm.lineSpacing() * height);
   setSizeIncrement(fm.maxWidth(), fm.lineSpacing());
   setMinimumSize(fm.maxWidth() * 10, fm.lineSpacing() * 3);
}

xMultiLineWin::~xMultiLineWin()
{
}

const QString *xMultiLineWin::putString(const char *pStr)
{
   QString tmpStr, tmpWord;
   const QString *rv;
   const char *cp;
   char ch;
   int linesUsed;

   if (dbg) fprintf(stdout, "MultiLineWin::putString(\"%s\"):Enter\n", pStr);
   if (dbg) fflush(stdout);
   linesUsed = getLinesUsed();
   tmpStr = "";
   tmpWord = "";
   rv = NULL;
   for (cp = pStr;; cp++)
   {
      ch = *cp;
      if (ch != '\r' && ch != '\n')
      {
         if (isspace(ch) || ch == '\0')
         {
            if (dbg) fprintf(stdout, "MultiLineWin::putString():Reached end of a word or the string\n");
            if (dbg) fflush(stdout);
            if (ch == '\t')
               ch = ' ';
            if (ch != '\0')
               tmpWord += ch;
            if ((signed)(strlen(tmpStr) + strlen(tmpWord)) < windowWidth)
            {
               tmpStr += tmpWord;
               if (dbg) fprintf(stdout, "MultiLineWin::putString():added word to line now: |%s| to screen\n", (const char *)tmpStr);
               if (dbg) fflush(stdout);
            }
            else
            {
               if (dbg) fprintf(stdout, "MultiLineWin::putString():Line to long for one line\n");
               if (dbg) fprintf(stdout, "MultiLineWin::putString():adding line |%s| to screen\n", 
                                        (const char *)tmpStr);
               if (dbg) fflush(stdout);
               if (getLinesUsed() < windowHeight || botLineOfs == getLinesUsed())
                  botLineOfs = (botLineOfs == bufferLen - 1) ? bufferLen - 1 : botLineOfs + 1;
               if (dbg) fprintf(stdout, "MultiLineWin::putString():Lines used = %d, botLineOfs = %d\n",
                                        getLinesUsed(), botLineOfs);
               rv = xMultiLineBuf::putString(tmpStr);
               if (testForScroll())
               {
                  if (dbg) fprintf(stdout, "MultiLineWin::putString():Need to scroll 1 line\n");
                  if (dbg) fflush(stdout);
                  scrollWindowImage(1);
               }
               tmpStr = tmpWord;
            }
            tmpWord = "";
            if (ch == '\0')
            {
               if (strlen(tmpStr) > 0)
               {
                  if (dbg) fprintf(stdout, "MultiLineWin::putString():Reached the end of the string\n");
                  if (dbg) fprintf(stdout, "MultiLineWin::putString():adding line |%s| to screen\n", 
                                           (const char *)tmpStr);
                  if (getLinesUsed() < windowHeight || botLineOfs == getLinesUsed())
                     botLineOfs = (botLineOfs == bufferLen - 1) ? bufferLen - 1 : botLineOfs + 1;

                  if (dbg) fprintf(stdout, "MultiLineWin::putString():Lines used = %d, botLineOfs = %d\n",
                                           getLinesUsed(), botLineOfs);
                  rv = xMultiLineBuf::putString(tmpStr);
                  if (testForScroll())
                  {
                     if (dbg) fprintf(stdout, "MultiLineWin::putString():Need to scroll 1 line\n");
                     if (dbg) fflush(stdout);
                        scrollWindowImage(1);
                  }
               }
               break;
            }
         }
         else
            tmpWord += ch;
      }
   }
   if (getLinesUsed() != linesUsed)
      emit linesUsedChanged(getLinesUsed());
   if (getLinesUsed() <= windowHeight)
      updateImage(getLinesUsed() - linesUsed);
   if (dbg) fprintf(stdout, "MultiLineWin::putString():Exit!!\n");
   if (dbg) fflush(stdout);
   return(rv);
}

bool xMultiLineWin::testForScroll()
{
   if (dbg) fprintf(stdout, "MultiLineWin::testForScroll():Enter- lines used: %d, buffer len: %d\n", 
                            getLinesUsed(), bufferLen);
   if ((getLinesUsed() > windowHeight && botLineOfs == getLinesUsed()) || 
       (getLinesUsed() >= bufferLen - 1))
      return(TRUE);
   else
      return(FALSE);
}
     
void xMultiLineWin::scrollWindow(int lines)
{
   int x;
   
   x = botLineOfs;
   if (lines < 0)
   {
      lines = abs(lines);
      botLineOfs = (botLineOfs - lines < windowHeight) ? windowHeight : botLineOfs - lines;
   }
   else
      botLineOfs = (botLineOfs + lines > getLinesUsed()) ? 
                     getLinesUsed() : botLineOfs + lines;
   botLineOfs = (botLineOfs == bufferLen - 1) ? bufferLen - 1 : botLineOfs;
   scrollWindowImage(botLineOfs - x);
}

void xMultiLineWin::scrollToLine(int line)
{
   int x;

   if (dbg) fprintf(stdout, "MultiLineWin::scrollToLine():Enter, line = %d\n", line);
   if (dbg) fflush(stdout);
   if (dbg) fprintf(stdout, "MultiLineWin::scrollToLine():botLineOfs = %d\n", botLineOfs);
   if (dbg) fflush(stdout);
   x = line - (botLineOfs - windowHeight);
   if (dbg) fprintf(stdout, "MultiLineWin::scrollToLine():x = %d\n", x);
   if (dbg) fflush(stdout);
   scrollWindow(x);
   if (dbg) fprintf(stdout, "MultiLineWin::scrollToLine(%d):Exit\n", line);
   if (dbg) fflush(stdout);
}

void xMultiLineWin::updateImage(int lines)
{
   QFontMetrics fm = QWidget::fontMetrics();
   QRect textRect;
   int textLineSpacing, textWindowWidth;
   
   if (dbg) fprintf(stdout, "MultiLineWin::updateImage(%d):Enter\n", lines);
   if (dbg) fflush(stdout);
   textLineSpacing = fm.lineSpacing();
   textWindowWidth = windowWidth * fm.maxWidth();
   textRect.setRect(0, (botLineOfs - lines) * textLineSpacing,
                    textWindowWidth, lines * textLineSpacing);
   
   if (lines > 0)
   {
      if (dbg) fprintf(stdout, "MultiLineWin::updateImage():Repainting the screen!\n");
      if (dbg) fflush(stdout);
      repaint(textRect, TRUE);
   }
   if (dbg) fprintf(stdout, "MultiLineWin::updateImage(%d):Exit\n", lines);
   if (dbg) fflush(stdout);
}

void xMultiLineWin::scrollWindowImage(int lines)
{
   QFontMetrics fm = QWidget::fontMetrics();
   QRect textRect;
   int textLineSpacing, textWindowWidth;
   int srcX = 0, srcY = 0, dstX = 0, dstY = 0;
   
   if (dbg) fprintf(stdout, "MultiLineWin::scrollWindowImage(%d):Enter\n", lines);
   
// if (dbg) fprintf(stdout, "MultiLineWin::scrollWindow():Scrolling %d lines\n", x);
   textLineSpacing = fm.lineSpacing();
   textWindowWidth = windowWidth * fm.maxWidth();
   if (lines > 0)
   {
      dstX = dstY = srcX = 0;
      srcY = lines * textLineSpacing;
      textRect.setRect(0, (windowHeight - lines) * textLineSpacing, 
                        textWindowWidth, lines * textLineSpacing);
   }
   else if (lines < 0)
   {
      lines = abs(lines);
      srcX = dstX = srcY = 0;
      dstY = lines * textLineSpacing;
      textRect.setRect(0, 0, textWindowWidth, lines * textLineSpacing);
   }
   
   if (lines != 0)
   {
      if (dbg) fprintf(stdout, "MultiLineWin::scrollWindow():moving %d to %d\n", srcY, dstY);
      bitBlt(this, dstX, dstY, this, srcX, srcY,
             textWindowWidth, (windowHeight - lines) * textLineSpacing);
      repaint(textRect, TRUE);
   }
   if (dbg) fprintf(stdout, "MultiLineWin::scrollWindow(%d):Exit\n", lines);
}
   
void xMultiLineWin::resizeEvent(QResizeEvent *pEvt)
{
   QFontMetrics fm = QWidget::fontMetrics();
   QSize newSize, s;
   int width, height;
   
   if (dbg > 2) fprintf(stdout, "xMultiLineWin::resizeEvent():Enter\n");
   if (dbg > 2) fflush(stdout);
   newSize = pEvt->size();
   if (newSize.isValid())
   {
      if (dbg > 2) fprintf(stdout, "xMultiLineWin::resizeEvent():Getting new Height / width info\n");
      if (dbg > 2) fflush(stdout);
      height = (int)(newSize.height() / fm.lineSpacing());
      width  = (int)(newSize.width() / fm.maxWidth());
      if (dbg > 2) fprintf(stdout, "xMultiLineWin::resizeEvent():Setting Term buf\'s Height / width info\n");
      if (dbg > 2) fflush(stdout);
      windowHeight = height;
      windowWidth = width;
      s.setHeight(height * fm.lineSpacing());
      s.setWidth(width * fm.maxWidth() + WIN_MARGIN);
      if (s != newSize)
         resize(s);
      emit changePageSize(height);
   }
   if (dbg > 2) fprintf(stdout, "xMultiLineWin::resizeEvent():Calling original resizeEvent\n");
   QWidget::resizeEvent(pEvt);
   if (dbg > 2) fprintf(stdout, "xMultiLineWin::resizeEvent():Exit\n");
}

void xMultiLineWin::paintEvent(QPaintEvent *pEvt)
{
   QFontMetrics fm = QWidget::fontMetrics();
   QPainter p;
   QRect    r;
   QFont fnt;
   int      x, y, curLine, numLines = 0, textLineSpacing;
   int      topLine, botLine, textBaseLine;
   int      haveNewLine;
   const QString  *pStr;
   const char *cp;
   bool bold = FALSE, underline = FALSE;
   
   if (dbg > 2) fprintf(stdout, "MultiLineWin::repaint():Enter\n");
   r = pEvt->rect();
   textLineSpacing = fm.lineSpacing();
   textBaseLine = fm.ascent();
   topLine = r.top();
   botLine = r.bottom();
   if (dbg > 2) fprintf(stdout, "MultiLineWin::repaint():topLine = %d, botLine = %d\n", topLine, botLine);
   if (dbg > 2) fprintf(stdout, "MultiLineWin::repaint():textLineSpacing = %d, textBaseLine = %d\n", textLineSpacing, textBaseLine);
   
   p.begin(this);
   p.setPen(colorGroup().text());
   p.setFont(font());

   x = (((getLinesUsed() - 1) - botLineOfs) + windowHeight < 0) ? 0 : ((getLinesUsed() - 1) - botLineOfs) + windowHeight;
   pStr = pointLine(x, x, numLines);
   if (dbg > 2) fprintf(stdout, "MultiLineWin::repaint():offset = %d, Lines returned = %d\n", x, numLines);
// numLines = pTermBuf->getTopLines(pText, windowHeight);
   haveNewLine = TRUE;
   for (x = curLine = 0; haveNewLine && numLines && curLine < numLines; curLine++, pStr = getNext(pStr, haveNewLine))
   {
      if (!haveNewLine)
         continue;
      if (dbg > 2) fprintf(stdout, "MultiLineWin::repaint():Line offset = %d, textBaseLine = %d\n", curLine * textLineSpacing, textBaseLine);
      if (x + textLineSpacing >= topLine && x <= botLine)
      {
         if (dbg > 2) fprintf(stdout, "MultiLineWin::repaint():Drawing text: |%s|\n", (const char *)*pStr);
         fnt = font();
         p.setFont(fnt);
         p.setPen(colorGroup().text());
         for (y = WIN_MARGIN, cp = (const char *)*pStr; *cp; cp++)
         {
            if (*cp == 0x02)
            {
               bold = (bold == TRUE) ? FALSE : TRUE;
               if (bold)
                  p.setPen(boldColor);
               else
                  p.setPen(colorGroup().text());
            }
            else if (*cp == ('_' - 0x40))
            {
               underline = (underline == TRUE) ? FALSE : TRUE;
               fnt.setUnderline(underline);
               p.setFont(fnt);
            }
            else
               drawChar(&p, y, x + textBaseLine, *cp);
         }
      }
      else if (x > botLine)
         break;
      x += textLineSpacing;
   }
   p.end();
   
   if (dbg > 2) fprintf(stdout, "MultiLineWin::repaint():Exit\n");
}

void xMultiLineWin::drawChar(QPainter *p, int &y, int x, int ch)
{
   char buf[2];

   buf[0] = ch;
   buf[1] = '\0';
   p->drawText(y, x, buf);
   y += p->fontMetrics().width(ch);   
}

void xMultiLineWin::mouseDoubleClickEvent(QMouseEvent *pEvt)
{
   QFontMetrics   fm = QWidget::fontMetrics();
   QPainter       p;
   QRect          r;
   QFont          fnt;
   int            x, y, curLine, numLines = 0, textLineSpacing;
   int            topLine, botLine, textBaseLine;
   int            haveNewLine;
   const QString  *pStr;
   QString        strWord;
   const char     *cp, *pWord;
   bool bold = FALSE, underline = FALSE, inspace = TRUE;
   
   if (dbg) fprintf(stdout, "MultiLineWin::doubleClickEvent():Enter\n");
   if (dbg) fprintf(stdout, "MultiLineWin::doubleClickEvent():evt->x = %d, evt->y = %d\n",
                            pEvt->pos().x(), pEvt->pos().y());
   r = rect();
   textLineSpacing = fm.lineSpacing();
   textBaseLine = fm.ascent();
   topLine = r.top();
   botLine = r.bottom();
   if (dbg) fprintf(stdout, "MultiLineWin::doubleClickEvent():topLine = %d, botLine = %d\n", topLine, botLine);
   if (dbg) fprintf(stdout, "MultiLineWin::doubleClickEvent():textLineSpacing = %d, textBaseLine = %d\n", textLineSpacing, textBaseLine);
   
   p.begin(this);
   p.setFont(font());

   x = (((getLinesUsed() - 1) - botLineOfs) + windowHeight < 0) ? 
       0 : ((getLinesUsed() - 1) - botLineOfs) + windowHeight;
   pStr = pointLine(x, x, numLines);
   if (dbg) fprintf(stdout, "MultiLineWin::doubleClickEvent():offset = %d, Lines returned = %d\n", x, numLines);
   haveNewLine = TRUE;
   for (x = curLine = 0; haveNewLine && numLines && curLine < numLines; curLine++, pStr = getNext(pStr, haveNewLine))
   {
      if (!haveNewLine)
         continue;
      inspace = TRUE;
      if (dbg) fprintf(stdout, "MultiLineWin::doubleClickEvent():Line offset = %d, textBaseLine = %d\n", curLine * textLineSpacing, textBaseLine);
      if (x + textLineSpacing >= topLine && x <= botLine &&
          pEvt->pos().y() >= x && pEvt->pos().y() <= x + textLineSpacing)
      {
         if (dbg) fprintf(stdout, "MultiLineWin::doubleClickEvent():Searching text: |%s|\n", (const char *)*pStr);
         fnt = font();
         p.setFont(fnt);
         for (y = WIN_MARGIN, pWord = cp = (const char *)*pStr; *cp; cp++)
         {
            if (*cp == 0x02)
            {
               bold = (bold == TRUE) ? FALSE : TRUE;
               if (bold)
               {
                  fnt = fontBold;
               }
               else
                  p.setFont(font());
               continue;
            }
            else if (*cp == ('_' - 0x40))
            {
               underline = (underline == TRUE) ? FALSE : TRUE;
               fnt.setUnderline(underline);
               p.setFont(fnt);
               continue;
            }
            else if ((isalnum(*cp) || 
                     *cp == '~' ||
                     *cp == '-' ||
                     *cp == '^' ||
                     *cp == '`' ||
                     *cp == '_') && inspace == TRUE)
            {
               pWord = cp;
               if (dbg) fprintf(stdout, "MultiLineWin::doubleClickEvent():At Start of Word |%s|\n", pWord);
               inspace = FALSE;
            }
            else if (inspace == FALSE && !isalnum(*cp) && 
                                         *cp != '~' &&
                                         *cp != '-' &&
                                         *cp != '^' &&
                                         *cp != '`' &&
                                         *cp != '_')
            {
               if (dbg) fprintf(stdout, "MultiLineWin::doubleClickEvent():In Space |%s|\n", cp);
               inspace = TRUE;
            }

            if (!inspace && pEvt->pos().x() >= y && 
                pEvt->pos().x() <= y + p.fontMetrics().width(*cp))
            {
               if (dbg) fprintf(stdout, "MultiLineWin::doubleClickEvent():Found Word!!!\n");
               for (strWord = ""; isalnum(*pWord) || 
                                  *pWord == '~' ||
                                  *pWord == '-' ||
                                  *pWord == '^' ||
                                  *pWord == '`' ||
                                  *pWord == '_'; 
                                  pWord++)
                  strWord += *pWord;
               if (dbg) fprintf(stdout, "MultiLineWin::doubleClickEvent():The Selected Word is |%s|\n",
                                (const char *)strWord);
               emit textSelected(strWord);
               p.end();
               if (dbg) fprintf(stdout, "MultiLineWin::doubleClickEvent():Exit\n");
               return;
            }
            y += p.fontMetrics().width(*cp);
         }
      }
      else if (x > botLine)
         break;
      x += textLineSpacing;
   }
   p.end();
   if (dbg) fprintf(stdout, "MultiLineWin::doubleClickEvent():Exit\n");
}


#include "xMultiLineWin.moc"
