/* Copyright (C) 1994 
            Olav Woelfelschneider (wosch@rbg.informatik.th-darmstadt.de)

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"

#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <fcntl.h>
#include <signal.h>
#include <unistd.h>
#include <strings.h>
#include <malloc.h>
#include <errno.h>
#include <sys/soundcard.h>
#include "volume.h"

#include "McApp.h"
#include <X11/Xatom.h>
#include "McAlloc.h"
#include "McResource.h"
#include "McGadget.h"
#include "McSlider.h"
#include "McText.h"
#include "McBitmap.h"
#include "McOrder.h"
#include "McDigits.h"
#include "McMenu.h"
#include "McInfoRequest.h"
#ifdef DEBUG_CODE
#include "McDebug.h" /* Geesh, blows up code... */
#endif

#include "icons.h"
#include "cd.h"
#include "scsi.h"
#include "struct.h"
#include "hardware.h"
#include "discid.h"
#include "version.h"
#include "editor.h"
#include "../XMdb/xmdbmessage.h"

#ifdef USE_RC5
#include <rcon.h>
#include "remotekeys.h"
int remotefd = -1, use_remote = -1;
rc5_keysym keysym[NUM_KEYSYMS];
char *knames[] = RKEYS;
static int remote_cue = 0;
static void remote_action(remote_data *rd);
static void remote_toggle(int id);
static void remote_press(int id);
static void remote_fast(int id, int idoff);
static void remote_nofast(void);
static void remote_change_volume(int dl, int dr);
#endif

/* Private routines
 * ================
 */
static void button_proc(McGadget *gadget);
/*static void menu_proc(McGadget *gadget);*/
static void menu_item_proc(int id);
static void goto_proc(McGadget *gadget);
static void replay_proc(McGadget *gadget);
static void slider_proc(McGadget *gadget);
static  int event_proc(McWindow *window, XEvent *event);
static void AddButton(int indx);
static void AddMessage(int x, int y, int width, int txt_x, char *title);
static void AddBitmap(int indx, int x, int y);
static void myXNextEvent(Display *display, XEvent *event_return);
static void CreateMyWindow(void);

/*  Public routines
 * ===============
 */
static int update_volume(int left, int right);

/* Public variables
 * ================
 */

int use_mixer = -1; /* probe for mixer first */
int emptypoll = 0;  /* An empty drive gets polled for a new cd when set */

/* Don't change this, please use the resource database
 * /usr/lib/X11/app-defaults/XPlaycd
 */
char *mixer_name = "/dev/mixer";
char *cdrom_name = "/dev/cdrom";
char *scsi_name = NULL;


McGadget	*Gad[NUM_GADGETS];
McApp		*app = NULL;
McWindow	*mainWindow;
McMenuList	*mainMenu = NULL;

int             NeedToPoll = 0;
int             GadCnt;
char		*myname;
int		my_argc;
char 		**my_argv;

McMenuInit mainItems[] = {
  MENUTITLE("Main Menu"),
  MENUDOUBLELINE,
  MENUITEM("Edit",       'E',  ITEM_ENABLED | ITEM_H3D, MAIN_EDIT),
  MENUITEM("Reload",     'R',  ITEM_ENABLED | ITEM_H3D, MAIN_REREAD),
  MENUITEM("Save",       'W',  ITEM_ENABLED | ITEM_H3D, MAIN_WRITE),
#ifdef DEBUG_CODE
  MENULINE,
  MENUITEM("Diskinfo",     0,  ITEM_ENABLED | ITEM_H3D, MAIN_PRINT),
  MENUITEM("Trackinfo",    0,  ITEM_ENABLED | ITEM_H3D, MAIN_TRACK),
  MENUITEM("Memory usage", 0,  ITEM_ENABLED | ITEM_H3D, MAIN_DEBUG),
#endif
  MENULINE,
  MENUITEM("Exit",       'X',  ITEM_ENABLED | ITEM_H3D, MAIN_EXIT),
};

static XrmOptionDescRec userTable[] = {
  { "-mixer",     "*useMixer",     XrmoptionNoArg , (caddr_t)"on" },
  { "-nomixer",   "*useMixer",     XrmoptionNoArg , (caddr_t)"off" },
  { "-emptypoll", "*emptyPoll",    XrmoptionNoArg , (caddr_t)"off" },
#ifdef USE_RC5
  { "-rc5",       "*useRc5",       XrmoptionNoArg , (caddr_t)"on" },
  { "-norc5",     "*useRc5",       XrmoptionNoArg , (caddr_t)"off" },
#endif
};

static struct timeval tout = { 0L, 1L }; /* provocate first timeout */

/**************************************************************************
 *  main(int ac, char **argv)
 *
 */
void main(int argc, char **argv) {
  int	left, right;
  char *str;

  myname = argv[0];
  my_argc = argc; my_argv = argv;

  if ((argc==2) && (!strcmp(argv[1], "-v"))) {
    printf("%s\n", XPLAYCD_VERSION);
    cleanup(0);
  }

  /*
   * Build connection with X server
   */
  app = McCreateApp(&argc, argv, XPLAYCD_VERSION, "XPlaycd",
		    userTable, sizeof(userTable)/sizeof(userTable[0]));
  if (argc>1) {
    fprintf(stderr, "%s: Too many arguments\n", myname);
    cleanup(1);
  }

  if ((str=McGetResource(app, "mixerDevice"))) {
    mixer_name=str;
  }

  emptypoll=McGetSwitch(app, "emptyPoll");

  if ((str=McGetResource(app, "useMixer"))) {
    if (!strcasecmp(str,"auto"))
      use_mixer=-1;
    else
      use_mixer=McTestSwitch(str);
  }

  if ((str=McGetResource(app, "cdromDevice"))) {
    cdrom_name=str;
  }

  if ((str=McGetResource(app, "scsiDevice"))) {
    scsi_name=str;
  }

  /*
   * Connect to the cdrom
   */
  cd_init();

#ifdef USE_RC5
  /*
   * Connect to the remote control daemon
   */

  if ((str=McGetResource(app, "useRc5"))) {
    if (!strcasecmp(str,"auto"))
      use_remote=-1;
    else
      use_remote=McTestSwitch(str);
  }


  if (use_remote) {
    if ((remotefd=rc5_connect())<0) {
      if (use_remote==1) {
	fprintf(stderr, "%s: ", myname);
	perror("Can't connect to rc5d");
	cleanup(1);
      }
    }
  } else {
    remotefd=-1;
  }

  /* Get all those keysyms */
  if (remotefd>=0) {
    int i;
    for (i=0; i<NUM_KEYSYMS; i++) {
      keysym[i]=rc5_query_keysym(remotefd, knames[i]);
    }
  }
#endif

  CreateMyWindow();

  cd_update();

  if (use_mixer>0)
    if (mix_open(SOUND_MIXER_CD)<0) {
      fprintf(stderr,"%s: Can't open %s: ",myname, mixer_name);
      perror("");
      cleanup(1);
    }

  if (use_mixer<0)
    if (mix_open(SOUND_MIXER_CD)>=0) use_mixer=1; else use_mixer=0;

  if (use_mixer) {
    left=right=-1;
    mix_set_volume(SOUND_MIXER_CD, &left, &right);
  } else {
    left=right=50;
    cd_volume(left, right);
  }
  update_volume(left, right);
  /*
   * Now do the main loop
   */



  while(1) {
    do {
      XEvent report;
      myXNextEvent(app->display, &report);
      McAppDoEvent(app, &report);
    } while(!(app->flags & MCAPP_REINIT));
    McInitApp(app);
    CreateMyWindow();
    tout.tv_sec=0; tout.tv_usec=1; /* provocate polling */
  }
}

/**************************************************************************
 *
 * Create the window
 */
static void CreateMyWindow(void) {
  XWMHints	 wm_hints;
  XSizeHints	 sizehints;
  int i;

  if (mainWindow) McFreeWindow(app, mainWindow);

  if (!(app->h || app->w)) {
    app->h=162; app->w=366;
  }
  mainWindow=McCreateAppWindow(app,-1,-1,-1,-1,NULL, event_proc);

  /*
   * Create all bitmaps
   */
  for (i=NUM_BITMAPS; --i>=0;) McCreateBitmapFromData(mainWindow, &pmap[i]);

  memset(&sizehints, 0, sizeof(XSizeHints));
  memset(&wm_hints, 0, sizeof(XWMHints));
  wm_hints.icon_pixmap = pmap[PIX_ICON].pixmap;
  wm_hints.icon_mask = pmap[PIX_ICONMASK].pixmap;
  wm_hints.flags = IconPixmapHint | IconMaskHint;
  sizehints.flags = PMinSize | PMaxSize;
  sizehints.min_width   = 57; sizehints.max_width  = app->w;
  sizehints.min_height  = 37; sizehints.max_height = app->h;
  McSetHints(mainWindow, NULL, my_argc, my_argv, &sizehints, &wm_hints);

  GadCnt=1;
  Gad[0] = McCreateGadget(mainWindow,GAD_H3D|GAD_3D|GAD_ACTIVE, BOOLGADGET,
			  10, 11, 39, 20);
  Gad[0]->normalLabel = McCreateText(mainWindow, "Exit", app->gc[GC_NORMAL],
				     app->defaultFont, 8, 4);
  Gad[0]->callbackUp = button_proc;
  Gad[0]->id = 0;
  AddButton(PIX_EJECT);
  AddButton(PIX_REWIND);
  Gad[GadCnt-1]->callbackUp = Gad[GadCnt-1]->callbackDown;
  AddButton(PIX_PLAY);
  Gad[GadCnt-1]->width+=13;
  AddButton(PIX_FF);
  Gad[GadCnt-1]->callbackUp = Gad[GadCnt-1]->callbackDown;
  AddButton(PIX_PAUSE);
  Gad[GadCnt-1]->flags &= ~GAD_HMASK;
  Gad[GadCnt-1]->flags |= GAD_HNONE;
  AddButton(PIX_SHUFFLE);
  AddButton(PIX_REPEAT);
  Gad[GadCnt-1]->flags |= GAD_TOGGLE;
  AddButton(PIX_PREV);
  McMoveGadget(Gad[GadCnt-1], Gad[BTN_REWIND]->x, 40);
  AddButton(PIX_NEXT);
  McMoveGadget(Gad[GadCnt-1], Gad[BTN_FF]->x, 40);

  AddMessage( 34, 40, 57, 12, "00:00");
  Gad[GadCnt-1]->flags |= GAD_ACTIVE | GAD_HNONE;
  Gad[GadCnt-1]->callbackDown = button_proc;
  Gad[GadCnt-1]->normalBitmap = &pmap[PIX_T];

  AddMessage(143, 40, 45,  8, "");
  AddMessage(240, 40, 56,  8, "");

  AddBitmap(PIX_CDLOGO, 300, 48);
  AddBitmap(PIX_VOLUME,   9, 73);
  AddBitmap(PIX_CLOCK,    9, 43);

  Gad[GadCnt] = McCreateGadget(mainWindow,
			       GAD_3D | GAD_H3D | GAD_ACTIVE | GAD_SELECTED,
			       SLIDERGADGET, 34, 69, 262, 25);
  Gad[GadCnt]->specialInfo = McCreateSlider(SLIDER_STEREO, 100);
  ((McSlider *)(Gad[GadCnt]->specialInfo))->step = 3;
  Gad[GadCnt]->callbackDown = slider_proc;
  Gad[GadCnt]->id = GadCnt++;

  Gad[GadCnt] = McCreateGadget(mainWindow,
			       GAD_3D | GAD_H3D | GAD_ACTIVE | GAD_SELECTED,
			       ORDERGADGET, 10, 103, 346, 18);
  Gad[GadCnt]->specialInfo = McCreateOrder(mainWindow,
					   ORD_PRINTID | ORD_SCROLL);
  Gad[GadCnt]->callbackDown = goto_proc;
  Gad[GadCnt]->callbackUp = replay_proc;
  ORDER(Gad[GadCnt])->ItemHeight = ORDER(Gad[GadCnt])->ItemWidth = 10;
  Gad[GadCnt]->id = GadCnt++;

  AddMessage(10, 130, 346,  12, "");

  /*
   * Menu init
   */
  mainMenu=McCreateMenu(mainWindow, mainItems, MENU_ITEMS(mainItems),
			MENU_CENTER);
  mainMenu->callback = menu_item_proc;
  /*
   * Display the window
   */
  McInitGadgets(mainWindow);
  XMapWindow(app->display, mainWindow->window);
}

/**************************************************************************
 * static void button_proc(McGadget *gadget)
 *
 */
static void button_proc(McGadget *gadget) {
  if (gadget->id==BTN_EXIT) cleanup(0);
  if (gadget->id==MSG_TIME) gadget->flags|=GAD_SELECTED;

  cd_action(gadget->id, gadget->flags&GAD_SELECTED);
  cd_update(); /* update messages */
}

/**************************************************************************
 * static void menu_item_proc(McMenuItem *item)
 *
 */
static void menu_item_proc(int id) {
  unsigned char *tmp;

  switch(id) {
  case MAIN_EXIT:
    cleanup(0);
#ifdef DEBUG_CODE
  case MAIN_DEBUG:
    McDebug(app,0);
    break;
  case MAIN_TRACK:
    pr_trackinfo(app, cur_track);
    break;
  case MAIN_PRINT:
    if (!(tmp=currentFile)) tmp="<none>";
    McInfo(app, "&L:ID of current disc: 0x%08x\n"
	   "&L:Current file name: %s\n",
	   get_discid(), tmp);
#endif
  case MAIN_REREAD:
    update_db(1);
    break;

  case MAIN_EDIT:
    create_edit_window();
    break;

  case MAIN_WRITE:
    write_db(currentFile);
    break;

  }
}

/**************************************************************************
 * static void goto_proc(McGadget *gadget)
 *
 */
static void goto_proc(McGadget *gadget) {
  cd_action(gadget->id, (ORDER(gadget)->grip)->id);
  cd_update(); /* update messages */
}

static void replay_proc(McGadget *gadget) {
  /* Be sure that the stop mark is behind the start mark. If not, swap them */
  McOrderItem *st, *it = ORDER(Gad[ORD_SONGS])->first;
  while(it && (it->id != DIG_RIGHT_ARROW) && (it->id !=DIG_LEFT_ARROW))
    it=it->next;
  if (!it) return; /* Empty list? */
  if (it->id==DIG_LEFT_ARROW) {
    st=it;
    while(it && (it->id != DIG_RIGHT_ARROW)) it=it->next;
    if (!it) return; /* Geesh, bug! */
    st->id=DIG_RIGHT_ARROW;
    it->id=DIG_LEFT_ARROW;
    McGadgetUpdate(Gad[ORD_SONGS]);
    XFlush(app->display);
  }
  cd_action(gadget->id, -1);
  cd_update(); /* update messages */
}

/**************************************************************************
 * VOLUME VOLUME VOLUME VOLUME VOLUME VOLUME VOLUME VOLUME VOLUME VOLUME */

#define SLIDER(gad) ((McSlider *)((gad)->specialInfo))

static void slider_proc(McGadget *gadget) {
  int left, right;
  left = SLIDER(Gad[SLD_VOLUME])->leftValue;
  right = SLIDER(Gad[SLD_VOLUME])->rightValue;
  if (use_mixer) {
    mix_set_volume(SOUND_MIXER_CD, &left, &right);
    update_volume(left, right);
  } else {
    cd_volume(left, right);
  }
}

static int update_volume(int left, int right) {
  if ((left>=0) && (right>=0) &&
      ((left != (SLIDER(Gad[SLD_VOLUME])->leftValue)) ||
       (right != (SLIDER(Gad[SLD_VOLUME])->rightValue)))) {
    SLIDER(Gad[SLD_VOLUME])->leftValue = left;
    SLIDER(Gad[SLD_VOLUME])->rightValue = right;
    McGadgetUpdate(Gad[SLD_VOLUME]);
    return 1;
  }
  return 0;
}

/**************************************************************************
 * static int event_proc(McWindow *mcw, XEvent *event)
 *
 */
static int event_proc(McWindow *mcw, XEvent *event) {

  switch(event->type) {

  case ButtonPress:
    if (event->xbutton.button==3) {
      McShowMenu(app, NULL, mainMenu, 0, 0);
      return 1;
    } else {
      return 0;
    }
  
  case ClientMessage:
    if (mcw==mainWindow) {
      if ((event->xclient.format == 32) &&
	  (event->xclient.data.l[0] == app->wmDelWin)) cleanup(0);

      /* Check if media editor has something to say... */
      if ((event->xclient.format == 32) &&
	  (event->xclient.data.l[0] == XA_NOTICE) &&
	  (event->xclient.data.l[1] == editor_pid)) {
	switch(event->xclient.data.l[2]) {
	case XMDB_FILE_WRITTEN:
	  update_db(1);
	  break;
	default:
	  printf("Client message from %ld saying: %ld\n",
		 event->xclient.data.l[1], event->xclient.data.l[2]);
	  break;
	}
      }

      return 1;
    }
  }
  return 0;
}

/**************************************************************************
 * static void myXNextEvent(Display *display, XEvent *event_return)
 *
 */
static void myXNextEvent(Display *display, XEvent *event_return) {
  fd_set rdfds, usefds;
  int dfd, width;

  if (XPending(display)>0) {
    XNextEvent(display, event_return);
    return;
  }

  FD_ZERO(&rdfds);
  dfd=ConnectionNumber(display);
  width = dfd;
  FD_SET(dfd,&rdfds);
#ifdef USE_RC5
  if (remotefd>=0) {
    FD_SET(remotefd, &rdfds);
    if (remotefd>width) width=remotefd;
  }
#endif

  while(1) {

    /* Flush before waiting for events, just like XLib does... */
    XFlush(display);

    if ((!tout.tv_usec) && (!tout.tv_sec)) {
      if (cue)
	tout.tv_usec = 100000;  /* Poll faster while searching */
      else
	if ((mainWindow->window_visible) && (cur_cdmode==CDPLAY))
	  tout.tv_usec =  333333; /* three polls per second should be enough to
				 * have a smooth display */
	else
	  tout.tv_usec = 1000000; /* When the window is iconified, we only have
				 * to poll to ensure repeat and playlist
				 * functions. Therefore, once per second is
				 * enough then. */
      tout.tv_sec  = 0;
    }
    usefds=rdfds;
    if (((mainWindow->window_visible) || NeedToPoll) &&
	(emptypoll || (cur_cdmode!=CDEJECT))) {
      select(width+1, &usefds, (fd_set *) NULL, (fd_set *) NULL, &tout);
    } else {
      select(width+1, &usefds, (fd_set *) NULL, (fd_set *) NULL,
	     (struct timeval *) NULL);
    }    

    if ((!tout.tv_sec) && (!tout.tv_usec)) {
      cd_action(REQ_POLL, 0);
      if (use_mixer) {
	int l,r;
	mix_get_volume(SOUND_MIXER_CD, &l, &r);
	update_volume(l,r);
      }
      cd_update();
    }

#ifdef USE_RC5
    if (remotefd>=0) {
      if (FD_ISSET(remotefd, &usefds)) {
	remote_data rd;
	if (rc5_read_socket(remotefd, &rd)<0) {
	  McError(app, "Can't read from rc5d socket\nReason: %s\n",
		  strerror(errno));
	  close(remotefd);
	  remotefd=-1;
	} else {
	    remote_action(&rd);
	}
      }
    }
#endif

    if (FD_ISSET(dfd, &usefds)) {
      XNextEvent(display, event_return);
      return;
    }
  }
}

/**************************************************************************
 * PRIVATE PRIVATE PRIVATE PRIVATE PRIVATE PRIVATE PRIVATE PRIVATE PRIVATE
 */

#ifdef USE_RC5
static void remote_action(remote_data *rd) {
  int key, i;

  key = -1;
  for (i=0; i<NUM_KEYSYMS; i++) {
    if (keysym[i]==rd->keysym) { key=i; break; }
  }

  switch(key) {
  case RK_CUE:
    remote_fast(BTN_FF, BTN_REWIND);
    return;
  case RK_REW:
    remote_fast(BTN_REWIND, BTN_FF);
    return;
  }
  
  if (rd->usec>200000) {
    switch(key) {
    case RK_PLAY:
      remote_nofast();
      if ((cur_cdmode==CDSTOP) || (cur_cdmode==CDPAUSE)) {
	remote_press(BTN_PLAYSTOP);
      }
      return;
    case RK_STOP:
      remote_nofast();
      if ((cur_cdmode==CDPLAY) || (cur_cdmode==CDPAUSE))
	remote_press(BTN_PLAYSTOP);
      return;
    case RK_PAUSE:
      remote_nofast();
      remote_press(BTN_PAUSE);
      return;
    case RK_PREV:
      remote_nofast();
      remote_press(BTN_PREV);
      return;
    case RK_NEXT:
      remote_nofast();
      remote_press(BTN_NEXT);
      return;
    case RK_EJECT:
      remote_nofast();
      remote_press(BTN_EJECT);
      return;
    case RK_DISPLAY:
      cd_action(MSG_TIME, 1);
      cd_update();
      return;
    case RK_REPEAT:
      remote_toggle(BTN_REPEAT);
      return;
    }
  }

  switch(key) {
  case RK_VOLUP:
    remote_change_volume( 1,  1);
    return;
  case RK_VOLDOWN:
    remote_change_volume(-1, -1);
    return;
  }
}

static void remote_fast(int id, int idoff) {
  int f=0;

  if (Gad[idoff]->flags&GAD_SELECTED) {
    Gad[idoff]->flags&=~GAD_SELECTED;
    McGadgetRedraw(Gad[idoff]);
    f=1;
  }

  if (!(Gad[id]->flags&GAD_SELECTED)) {
    Gad[id]->flags|=GAD_SELECTED;
    McGadgetRedraw(Gad[id]);
    XFlush(app->display);
    cd_action(id, 1);
  } else if (f) XFlush(app->display);

  remote_cue = 1;
}

static void remote_nofast(void) {
  if (remote_cue) {
    remote_cue = 0;
    cue=0;
    if (Gad[BTN_FF]->flags&GAD_SELECTED) {
      Gad[BTN_FF]->flags&=~GAD_SELECTED;
      McGadgetRedraw(Gad[BTN_FF]);
    }
    if (Gad[BTN_REWIND]->flags&GAD_SELECTED) {
      Gad[BTN_REWIND]->flags&=~GAD_SELECTED;
      McGadgetRedraw(Gad[BTN_REWIND]);
    }
  }
}

static void remote_press(int id) {
  Gad[id]->flags|=GAD_SELECTED;
  McGadgetBusy(mainWindow, Gad[id]);
  XFlush(app->display);
  cd_action(id, 1);
  Gad[id]->flags&=~GAD_SELECTED;
  McGadgetRedraw(Gad[id]);
  XFlush(app->display);
}

static void remote_toggle(int id) {
  Gad[id]->flags^=GAD_SELECTED;
  McGadgetBusy(mainWindow, Gad[id]);
  XFlush(app->display);
  cd_action(id, Gad[id]->flags&GAD_SELECTED);
  McGadgetRedraw(Gad[id]);
  XFlush(app->display);
}

static void remote_change_volume(int dl, int dr) {
  int left = SLIDER(Gad[SLD_VOLUME])->leftValue + dl;
  int right = SLIDER(Gad[SLD_VOLUME])->rightValue + dr;
 
  if ( left  > 100 ) left  = 100;
  if ( right > 100 ) right = 100;
  if ( left  <   0 ) left  = 0;
  if ( right <   0 ) right = 0;

  if (use_mixer)
    mix_set_volume(SOUND_MIXER_CD, &left, &right);
  else
    cd_volume(left, right);
  update_volume(left, right);
}

#endif



/**************************************************************************
 * GADGETS  GADGETS  GADGETS  GADGETS  GADGETS  GADGETS  GADGETS  GADGETS */

static void AddButton(int indx) {
  Gad[GadCnt] = McCreateGadget(mainWindow, GAD_3D | GAD_H3D | GAD_ACTIVE, BOOLGADGET,
			       Gad[GadCnt-1]->x + Gad[GadCnt-1]->width + 10,
			       Gad[GadCnt-1]->y, 32, 20);
  Gad[GadCnt]->callbackDown = button_proc;
  Gad[GadCnt]->normalBitmap = &pmap[indx];
  Gad[GadCnt]->id = GadCnt++;
}


static void AddMessage(int x, int y, int width, int txt_x, char *title) {
  Gad[GadCnt] = McCreateGadget(mainWindow, GAD_3D | GAD_H3D | GAD_SELECTED,
			       BOOLGADGET, x, y, width, 20);
  Gad[GadCnt]->normalLabel = McCreateText(mainWindow, title, app->gc[GC_NORMAL],
					  app->defaultFont, txt_x, 3);
  Gad[GadCnt]->id = GadCnt++;
}


static void AddBitmap(int indx, int x, int y) {
  Gad[GadCnt] = McCreateGadget(mainWindow, GAD_HNONE, BOOLGADGET, x, y,
			       pmap[indx].width, pmap[indx].height);
  Gad[GadCnt]->normalBitmap = &pmap[indx];
  Gad[GadCnt]->id = GadCnt++;
}

/**************************************************************************
 * BE CLEAN  BE CLEAN  BE CLEAN  BE CLEAN  BE CLEAN  BE CLEAN  BE CLEAN   */

void cleanup(int r) {
  close_editor();
  McFreeApp(app);
  mix_close();
  if (currentFile) McFree(currentFile);
#ifdef USE_RC5
  if (remotefd>=0) close(remotefd);
#endif
  exit(r);
}

