/* Copyright (C) 1994 
            Olav Woelfelschneider (wosch@rbg.informatik.th-darmstadt.de)

 This library is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>

#include "McApp.h"
#include "McResource.h"
#include "McAlloc.h"
#include <strings.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <locale.h>

static XrmOptionDescRec opTable[] = {
  { "-geometry",     "*geometry",     XrmoptionSepArg, (caddr_t)NULL },
  { "-iconGeometry", "*iconGeometry", XrmoptionSepArg, (caddr_t)NULL },
  { "-iconic",       "*iconStartup",  XrmoptionNoArg,  (caddr_t)"on" },
  { "-ic",           "*iconStartup",  XrmoptionNoArg,  (caddr_t)"on" },
  { "-background",   "*background",   XrmoptionSepArg, (caddr_t)NULL },
  { "-bg",           "*background",   XrmoptionSepArg, (caddr_t)NULL },
  { "-foreground",   "*foreground",   XrmoptionSepArg, (caddr_t)NULL },
  { "-fg",           "*foreground",   XrmoptionSepArg, (caddr_t)NULL },
  { "-darkborder",   "*darkBorder",   XrmoptionSepArg, (caddr_t)NULL },
  { "-db",           "*darkBorder",   XrmoptionSepArg, (caddr_t)NULL },
  { "-brightborder", "*brightBorder", XrmoptionSepArg, (caddr_t)NULL },
  { "-bb",           "*brightBorder", XrmoptionSepArg, (caddr_t)NULL },
  { "-hilightcolor", "*hilightColor", XrmoptionSepArg, (caddr_t)NULL },
  { "-hc",           "*hilightColor", XrmoptionSepArg, (caddr_t)NULL },
  { "-xrm",           NULL,           XrmoptionResArg, (caddr_t)NULL },
  { "-display",      ".display",      XrmoptionSepArg, (caddr_t)NULL },
  { "-font",         "McTools.font",  XrmoptionSepArg, (caddr_t)NULL },
  { "-fn",           "McTools.font",  XrmoptionSepArg, (caddr_t)NULL },
  { "-ffn",          "*fixedFont",    XrmoptionSepArg, (caddr_t)NULL },
  { "-wfn",          "Workbench.font",XrmoptionSepArg, (caddr_t)NULL },
  { "-name",         "*name",         XrmoptionSepArg, (caddr_t)NULL },
  { "-title",        "*title",        XrmoptionSepArg, (caddr_t)NULL },
  { "-resdebug",     "*resdebug",     XrmoptionNoArg,  (caddr_t)"on" },
  { "-clip",         "*clipGadgets",  XrmoptionNoArg,  (caddr_t)"on" },
  { "-dotty",        "*dottyInfo",    XrmoptionNoArg,  (caddr_t)"on" },
  { "-rv",           "*reverseVideo", XrmoptionNoArg,  (caddr_t)"on" },
  { "+rv",           "*reverseVideo", XrmoptionNoArg,  (caddr_t)"off" },
  { "-sync",         "*synchronize",  XrmoptionNoArg,  (caddr_t)"on" },
  { "-synchronize",  "*synchronize",  XrmoptionNoArg,  (caddr_t)"on" },
};

static char *FindName(char *str);
static char *Name2Class(char *str, char *buf);

/****************************************************************************
 * Parse the command line and open the display connection
 */
Display *McParseOpenDisp(McApp *app, int *argc, char *argv[], char *class,
			 XrmOptionDescRec *DescTable, int DescSize) {
  char dsn[256], dsc[256], *str_type[20], *displayname=NULL;
  XrmValue value;
  XrmDatabase appDB;
  Display *display = NULL;

  XrmInitialize();

  strcpy(dsn,myname=FindName(argv[0]));
  strcat(dsn,".display");
  if (class) {
    app->class=class;
    strcpy(dsc,class);
  } else {
    app->class=strdup(myname);
    Name2Class(myname,app->class);
  }
  strcat(dsc,".Display");

  XrmParseCommand(&app->cmdlineDB, opTable, sizeof(opTable)/sizeof(opTable[0]),
		  myname , argc, argv);

  if (DescTable && DescSize) {
    XrmParseCommand(&appDB, DescTable, DescSize, myname , argc, argv);
    XrmMergeDatabases(appDB, &app->cmdlineDB);
  }

  if (XrmGetResource(app->cmdlineDB, dsn, dsc, str_type, &value)==True) {
    displayname=value.addr;
  }

  if (!(display=XOpenDisplay(displayname))) {
    fprintf(stderr,"%s: Can't open display '%s'.\n", myname,
	    XDisplayName(displayname));
    cleanup(1);
  }

#if 0
  if (!XSupportsLocale()) {
    fprintf(stderr,
	    "%s: Warning, locale `%s' not supported by Xlib; trying `C'.\n",
	    myname, setlocale(LC_ALL, ""));
    setlocale(LC_ALL, "C");
  } else {
    if (XSetLocaleModifiers("")==NULL) {
      fprintf(stderr, "%s: Warning: Can't set locale modifiers.\n", myname);
    }
  }
#endif

  return display;
}

/****************************************************************************
 * Read the defaults from the resource database
 */
void McReadStandards(McApp *app) {
  char *str;

  if ((str=McGetResource(app, "McTools.name"))) {
    myname=str;
  }

  if (McGetSwitch(app, "McTools.resdebug")) app->flags|=MCAPP_RESDEBUG;

  if ((str=McGetResource(app, "McTools.title")))      app->window_title=str;
  if ((str=McGetResource(app, "McTools.foreground"))) 
    app->color_names[COL_FOREGROUND]=str;

  if ((str=McGetResource(app, "McTools.background")))
    app->color_names[COL_BACKGROUND]=str;

  if ((str=McGetResource(app, "McTools.brightBorder")))
    app->color_names[COL_BRIGHT]=str;

  if ((str=McGetResource(app, "McTools.darkBorder")))
    app->color_names[COL_DARK]=str;

  if ((str=McGetResource(app, "McTools.hilightColor")))
    app->color_names[COL_SELECTED]=str;

  if ((str=McGetResource(app, "McTools.font")))     app->default_font_name=str;
  if ((str=McGetResource(app, "McTools.fixedFont")))  app->fixed_font_name=str;

  if (McGetSwitch(app, "McTools.iconStartup"))  app->flags|=MCAPP_ICONIC;
  if (McGetSwitch(app, "McTools.iconic"))       app->flags|=MCAPP_ICONIC;
  if (McGetSwitch(app, "McTools.reverseVideo")) app->flags|=MCAPP_REVERSE;
  if (McGetSwitch(app, "McTools.synchronize"))  app->flags|=MCAPP_SYNCED;
  if (McGetSwitch(app, "McTools.backingstore")) app->flags|=MCAPP_BSTORE;
  if (McGetSwitch(app, "McTools.clipGadgets"))  app->flags|=MCAPP_CLIP;

  if ((str=McGetResource(app, "McTools.geometry")))
    app->geometry_flags=XParseGeometry(str, &app->x, &app->y, &app->w,&app->h);
  
  if ((str=McGetResource(app, "McTools.iconGeometry")))
    app->icon_geometry_flags=
      XParseGeometry(str, &app->ico_x, &app->ico_y, &app->ico_w, &app->ico_h);

  if ((str=McGetResource(app, "Workbench.font"))) app->wb_font_name=str;
  if (McGetSwitch(app, "Workbench.dottyInfo"))  app->flags|=MCAPP_DOTTY_INFO;
}

/****************************************************************************
 * Read one entry from the resource database
 * This resource management is a pain!!
 */

char *McGetResource(McApp *app, char *name) {
  char bun[256], buc[256], *res=NULL;
  int l;
  char *str_type[20];
  XrmValue value;

  strcpy(bun,myname);
  l=strlen(bun);
  bun[l++]='.';
  strcpy(bun+l,name);
  bun[l]=tolower(bun[l]);

  strcpy(buc, app->class);
  l=strlen(buc);
  buc[l++]='.';
  strcpy(buc+l,name);
  buc[l]=toupper(buc[l]);

  if (XrmGetResource(app->resDB, bun, buc, str_type, &value)==True)
    res= value.addr;

  if (app->flags&MCAPP_RESDEBUG) {
    if (res)
      printf("%s='%s'\n", name, res);
    else
      printf("%s undefined.\n", name);
  }

  return res;
}

int McTestSwitch(char *str) {
  if ((!strcasecmp(str,"on")) || (!strcasecmp(str,"true")) ||
      (!strcasecmp(str,"yes")) || (!strcmp(str,"1"))) return 1;
  return 0;
}

int McGetSwitch(McApp *app, char *name) {
  char *str;
  if (!(str=McGetResource(app,name))) return 0;
  return McTestSwitch(str);
}

/****************************************************************************
 * Read lotsa databases scattered across the system (Yuck!)
 * (Have I already mentioned that this resource management is a pain?)
 */

static int McMergeDB(char *path, char *file, XrmDatabase *db) {
  char *add, *start, *p=path, *home=(char *)getenv("HOME");
  char buf[1024];
  XrmDatabase daba;

  /*
   * This is a simple approach, but I think its enuff,
   * I even think noone will ever use this... :)
   */
  if (!path || !strlen(path)) return 0;
  do {
    start=p;
    while((*p) && (*p!=':')) p++;
    if ((*start=='~') && ((start[1]=='/')||(start[1]==0)||(start[1]==':'))) {
      strcpy(buf,home);
      add=buf+strlen(home);
      *add++='/';			/* You could do fun things in C */
      start++;
      if (*start=='/') start++;
    } else {
      add=buf;
    }
    strncpy(add, start, p-start);
    if (add[p-start-1]!='/') {
      add[p - start]='/';
      strcpy(add+(p-start)+1,file);
    } else
      strcpy(add+(p-start),file);
    daba=XrmGetFileDatabase(buf);
    if (daba) {
      XrmMergeDatabases(daba, db);
      return 1; /* Should I continue searching now? */
    }
  } while(*p++);
  return 0;
}

void McMergeDatabases(McApp *app) {
  XrmDatabase homeDB, serverDB, appDB, userDB;

  char buf[1024];
  char *env;

  /* Get standards from /usr/lib/X11/app-defaults/<name> */
  strcpy(buf, RESOURCE_PATH);
  strcat(buf, app->class);
  appDB=XrmGetFileDatabase(buf);
  XrmMergeDatabases(appDB, &app->resDB);

  /* Now get an user specific application setup file, in case the user cannot
   * install to RESOURCE_PATH, which is unlikely under linux, though...
   */
  strcpy(env=buf, (char *)getenv("HOME"));
  strcat(env,"/.");
  strcat(env, app->class);
  strcat(env,"-defaults");
  userDB=XrmGetFileDatabase(env);
  XrmMergeDatabases(userDB, &app->resDB);

  /* Search some paths, partially implemented, but this is WAY enough! */
  if (!McMergeDB((char *)getenv("XUSERFILESEARCHPATH"),app->class,&app->resDB))
    McMergeDB((char *)getenv("XAPPLRESDIR"), app->class, &app->resDB);

  /* Get standards from the server, or, if not available, read ~/.Xdefaults */
  if (XResourceManagerString(app->display) != NULL) {
    serverDB = XrmGetStringDatabase(XResourceManagerString(app->display));
  } else {
    strcpy(buf, (char *)getenv("HOME"));
    strcat(buf,"/.Xdefaults");
    serverDB=XrmGetFileDatabase(buf);
  }
  XrmMergeDatabases(serverDB, &app->resDB);

  /* Now get server specific data */
  if ((env=(char *)getenv("XENVIRONMENT"))==NULL) {
    int len;
    strcpy(env=buf, (char *)getenv("HOME"));
    strcat(env,"/.Xdefaults-");
    len=strlen(env);
    gethostname(env+len, 1024-len);
  }
  homeDB=XrmGetFileDatabase(env);
  XrmMergeDatabases(homeDB, &app->resDB);

  /* Commandline has highest priority */
  XrmMergeDatabases(app->cmdlineDB, &app->resDB);
}

/****************************************************************************
 * Tell the windowmanager what it has to expect from us.
 */

static void nostruc(char *s);

void McSetHints(McWindow *mcw, unsigned char *title, int argc, char *argv[],
		XSizeHints *sizehints, XWMHints *wm_hints) {
  McApp *app=mcw->app;
  XClassHint	class_hints;
  XTextProperty windowName;
  XTextProperty iconName;
  char *list[2];
  XTextProperty tprop;

  if (title) {
    list[0]=title;
  } else {
    list[0]=app->window_title;
  }
  list[1]=NULL;

  if (XStringListToTextProperty(list, 1, &windowName) == 0)
    nostruc("windowName");
  if (XStringListToTextProperty(list, 1, &iconName) == 0)
    nostruc("iconName");

  wm_hints->input = True;
  wm_hints->initial_state = NormalState;
  wm_hints->flags |= StateHint | InputHint;
  class_hints.res_name = myname;
  class_hints.res_class = app->class;

  if (argv) {
    if (mcw->app->flags & MCAPP_ICONIC) {
      wm_hints->initial_state = IconicState;
    }
    if (app->geometry_flags & (WidthValue|HeightValue)) {
      sizehints->flags |= USSize;
    } else {
      sizehints->flags |= PSize;
    }

    if (app->geometry_flags & (XValue|YValue)) {
      sizehints->flags |= USPosition;
      sizehints->x=app->x;
      sizehints->y=app->y;
    }
  }

  sizehints->width = mcw->w;
  sizehints->height= mcw->h;

  if (!(sizehints->flags & PBaseSize)) {
    sizehints->base_height = sizehints->base_width = 0;
    sizehints->flags |= PBaseSize;
  }
  if (!(sizehints->flags & PResizeInc)) {
    sizehints->width_inc   = sizehints->height_inc = 1;
    sizehints->flags |= PResizeInc;
  }

  if (!app->mainWindow) {
    app->mainWindow = mcw->window;
  }
  wm_hints->window_group=app->mainWindow;
  wm_hints->flags|=WindowGroupHint;

  if (app->icon_geometry_flags & (XValue|YValue)) {
    wm_hints->icon_x=app->ico_x;
    wm_hints->icon_y=app->ico_y;
    wm_hints->flags|=IconPositionHint;
  }

  XSetWMProperties(app->display, mcw->window, &windowName, &iconName,
		   argv, argc, sizehints, wm_hints, &class_hints);
  XFree(windowName.value);
  XFree(iconName.value);

  if (mcw->eventCallback) {
    if (!app->wmDelWin)
      app->wmDelWin = XInternAtom(app->display,"WM_DELETE_WINDOW",False);
    XSetWMProtocols(app->display, mcw->window, &app->wmDelWin, True);
  }

  if (!app->McToolName) { /* This is the first window */
    app->McToolName = XInternAtom(app->display, "_MCTOOL_NAME", 0);
    *list=app->class;
    XStringListToTextProperty(list, 1, &tprop);
    XSetTextProperty(app->display, mcw->window, &tprop, app->McToolName);
    XFree(tprop.value);
  }
}

/****************************************************************************
 * Titles...
 */

void McSetTitle(McWindow *mcw, unsigned char *title) {
  XTextProperty tprop;
  char *list[2];

  list[0]=title; list[1]=NULL;
  if (!XStringListToTextProperty(list, 1, &tprop))
    nostruc("windowTitle");
  XSetWMName(mcw->app->display, mcw->window, &tprop);
  XFree(tprop.value);
}

/****************************************************************************
 * This and that...
 */

static char *FindName(char *str) {
  char *name = str + strlen(str); /* What happens if strlen=0? (-: */
  while(((--name)>str) && *(name-1)!='/');
  return name;
}

static char *Name2Class(char *str, char *buf) {
  strcpy(buf, str);
  *buf=toupper(*buf);
  if ((*buf=='X') && (strlen(buf)>1))
    buf[1]=toupper(buf[1]);
  return buf;
}

static void nostruc(char *s) {
  fprintf(stderr, "%s: structure allocation for %s failed.\n", myname,s);
  cleanup(1);
}
