/* filename: rlprd.c
 * project: rlpr
 * author: meem  --  meem@sherilyn.wustl.edu
 * version: $Id: rlprd.c,v 1.9 1996/10/12 09:17:22 meem Exp $
 * contents: daemon which does "reflecting" of messages from nonprivileged
 *           to priviledged ports
 *
 * Time-stamp: <1996/10/12 03:00 -- meem@sherilyn.wustl.edu>
 */

/* copyright (c) 1996 meem, meem@gnu.ai.mit.edu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 */

#include "config.h"

#include <sys/types.h>          /* for pid_t */
#include <signal.h>

#ifndef STDC_HEADERS
#error there is currently no support for compiling on machines without ANSI headers. \
       write meem@gnu.ai.mit.edu if you are interested in adding it.
#endif

#ifdef __GNU_LIBRARY__		          
#undef __GNU_LIBRARY__
#include "getopt.h"
#define __GNU_LIBRARY__
#else  /* not __GNU_LIBRARY__ */
#include "getopt.h"
#endif /* __GNU_LIBRARY__ */

#include "wait.h"

#include <stdlib.h>             /* for EXIT_* constants */
#include <netdb.h>              /* got gethostbyname() */
#include <sys/time.h>           /* select() and macros */
#include <sys/stat.h>           /* for umask() */
#include <unistd.h>             /* for setsid() */
#include <string.h>             /* for strerror() */
#include <syslog.h>             /* logging facilities */
#include <errno.h>
#include <stdio.h>              /* BUFSIZ */
#include <sys/socket.h>         /* for bind(), connect(), etc */
#include <netinet/in.h>         /* for sockaddr_in */
#include <stdarg.h>

#include "rlprd.h"
#include "rlpr-common.h"

static struct daemon_options d_opts_;

char * name;

int main(int argc, char *argv[]) {
  struct hostent *hp;
  struct sockaddr_in sin_in;      /* where we will listen */
  struct sockaddr_in sin_out;     /* our projected identity (to lpd) */
  struct sockaddr_in sin_lpd;     /* the connection we will send to lpd */  
  struct sockaddr_in sin_from;    /* the connection that came in */
  int inc_rlprd, out_lpd;         /* socket descriptors */
  int listen_fd;                  /* passive descriptor listening connxns */
  int sin_fromlen = sizeof sin_from;  /* the length of the incoming descriptor */
  char printhost[MAXHOSTNAMELEN]; /* name of printhost */
  char localhost[MAXHOSTNAMELEN]; /* for finding out our name */
  pid_t childpid;                 /* for fork() */
  int i;                          /* scrap variable */

  name = *argv;

  set_default_options();
  parse_args(argc, argv);

  ++argv; --argc;                         /* skip name */
  while (argc && **argv == '-') {         /* skip command-line options */
    argv++;
    argc--;
  }

  /* make sure euid == root */
  if (geteuid() != 0) {
    errno = 0;
    rlpr_fatal("must be run root or setuid root!");
  }

  if (d_opts_.daemon) daemon_init();      /* if daemon, do the magic */

  get_local_hostname(localhost, sizeof(localhost));
         
  if (!(hp = gethostbyname(localhost)))
    rlpr_fatal("gethostbyname on %s", localhost);
  
  if ((listen_fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) 
    rlpr_fatal("socket");
  
  /* zero out structs to get the unused part zeroed */
  memset(&sin_in,   0, sizeof(sin_in));
  memset(&sin_out,  0, sizeof(sin_out));
  memset(&sin_lpd,  0, sizeof(sin_lpd));
  memset(&sin_from, 0, sizeof(sin_from));

  sin_in.sin_family = sin_out.sin_family = sin_lpd.sin_family = AF_INET;

  sin_in.sin_addr.s_addr = htonl(INADDR_ANY);
  memcpy(&sin_out.sin_addr, hp->h_addr, hp->h_length); 
  sin_in.sin_port  = htons(d_opts_.port);
  sin_lpd.sin_port = htons(LPD_TO_PORT);
  
  if (bind(listen_fd,  (struct sockaddr *) &sin_in,  sizeof(sin_in)) < 0)
    rlpr_fatal("bind to port %hi failed", d_opts_.port);
  
  register_sigchld();               /* register the reaper */
  register_sigalrm();		    /* register the alarm */
  listen(listen_fd, 5);             /* don't rely on more than 5 */

  for (;;) {
    if ((inc_rlprd = accept(listen_fd, (struct sockaddr *) &sin_from, &sin_fromlen)) < 0)
      rlpr_warn("accept");
    
    if ((childpid = fork()) < 0) rlpr_fatal("fork");
    else if (childpid == 0) {
      /* CHILD */
      alarm(d_opts_.timeout);	/* set timeout */

      i = 0;
      do
        if (read(inc_rlprd, &printhost[i], 1) < 0) {
	  errno = 0;
          rlpr_fatal("unable to read proxy hostname");
	}
      while (printhost[i++] != '\n');
      printhost[--i] = '\0';
      
      alarm(0);			/* disarm the timer */

      syslog(LOG_INFO,"proxy from %s to %s", inc_host(&sin_from), printhost);

      /* bind our local socket so we come from a privileged port */

      if ((out_lpd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
        rlpr_fatal("socket");

      for (i = LO_LPD_FROM_PORT; i <= HI_LPD_FROM_PORT; i++) {
        sin_out.sin_port = htons(i);
        if (bind(out_lpd, (struct sockaddr *) &sin_out, sizeof(sin_out)) == 0)
          break;
        else if (i == HI_LPD_FROM_PORT)
          rlpr_fatal("bind to ports %hi-%hi", LO_LPD_FROM_PORT, HI_LPD_FROM_PORT);
      }

      /* fill in remaining part of sin_lpd struct with printhost */

      if (!(hp = gethostbyname(printhost)))
        rlpr_fatal("gethostbyname on printhost");
      memcpy(&sin_lpd.sin_addr, hp->h_addr, hp->h_length); 

      if (connect(out_lpd, (struct sockaddr *) &sin_lpd, sizeof(sin_lpd)) < 0)
        rlpr_fatal("connect");
      
      for (;;) {                                /* conversation loop */
        static int nc, nfds;                    /* data moved in read/write */
        static fd_set readfds;
        static char buf[BUFSIZ];                /* could be a more optimal size */
  
        FD_ZERO(&readfds);                      /* initialize fds structure */
        FD_SET(inc_rlprd, &readfds);
        FD_SET(out_lpd, &readfds);
    
        if ((nfds = select(16, &readfds, NULL, NULL, NULL)) < 0) /* is 16 right here!? */
          rlpr_fatal("select");
        
        if (nfds) {
          if (FD_ISSET(inc_rlprd, &readfds)) {         /* data from rlpr client */
            if ((nc = read(inc_rlprd, buf, sizeof(buf))) < 0)
              rlpr_fatal("read1");
            else if (!nc) break;
            else writen(out_lpd, buf, nc);
          }
          if (FD_ISSET(out_lpd, &readfds)) {     /* data from lpd server */
            if ((nc = read(out_lpd, buf, sizeof(buf))) < 0)
              rlpr_fatal("read2");
            else if (!nc) break;
            else writen(inc_rlprd, buf, nc);
          }
        }
      }
      exit(EXIT_SUCCESS);
  }

    /* PARENT */
    close(inc_rlprd);
  }
  exit(EXIT_SUCCESS);
}

/* daemon_init()
 *
 * purpose: initialize the server to run in daemon mode.
 *   input: void
 *  output: void
 */ 

static void daemon_init(void) {
  pid_t childpid;

  if (getppid() > 1) {  
    /* we weren't started by init.  let's fork ourselves silly */
    
    /* turn on the logger */
    openlog(name, LOG_PID, LOG_LPR);
    d_opts_.is_daemon = 1;

    if ((childpid = fork()) == -1)
      rlpr_fatal("fork");
    else if (childpid > 0)      /* parent */
      exit(EXIT_SUCCESS);

    /* child */

    setsid();                   /* now unique session, PGID, no ctty */

    /* fork again to make sure we don't reacquire a ctty */

    if ((childpid = fork()) == -1)
      rlpr_fatal("fork");
    else if (childpid > 0)       /* first child */
      exit(EXIT_SUCCESS);

    /* second child */
  }

  chdir("/");                   /* in case we were on a mounted partition */
  umask(0);                     /* clear file mode creation mask */
}

static const char * inc_host(struct sockaddr_in *sin) {
  struct hostent *hp;
  hp = gethostbyaddr((char *) &sin->sin_addr, sizeof(struct in_addr), AF_INET);
  if (!hp) rlpr_fatal("gethostbyaddr");
  return hp->h_name;
}
     

static RETSIGTYPE reapchild(int unused) {
  int olderrno = errno;
  while (waitpid( -1, NULL, WNOHANG) > 0) /* NULL BODY */;
  errno = olderrno;
}

static RETSIGTYPE sigalrm(int unused) {
  rlpr_fatal("connection timed out");
}

/* register_sigchld()
 *
 * purpose: to register the SIGCHLD signal with the reaper
 *   input: void
 *  output: void
 */ 

static void register_sigchld(void) {
  struct sigaction sa;
  sa.sa_handler = reapchild;
  sa.sa_flags = SA_RESTART;
  sigemptyset(&sa.sa_mask);

  if (sigaction(SIGCHLD, &sa, NULL) == -1)
    rlpr_fatal("sigaction");
}

static void register_sigalrm(void) {
  if (signal(SIGALRM, sigalrm) == SIG_ERR)
    rlpr_fatal("signal");
}


/* parse_args()
 *
 * purpose: to handle command line arguments
 *   input: argc  the number of elements in argv
 *          argv  an array of command line arguments
 *  output: void
 */ 

static void parse_args(int argc, char *argv[]) {
  int c;

  static struct option long_opts[] = {
    { "no-daemon",    0, NULL, 'n' },
    { "port",         1, NULL, 'p' },
    { "quiet",        0, NULL, 'q' },
    { "silent",       0, NULL, 'q' },
    { "timeout",      1, NULL, 't' },
    { "version",      0, NULL, 'V' },
    { 0, 0, 0, 0 }
  };

  while ((c = getopt_long(argc, argv, "np:qtvV", long_opts, NULL)) != EOF)
    switch(c) {
    case 'n':
      d_opts_.daemon  = 0;                   break;
    case 'p':
      d_opts_.port    = atoi(optarg);        break;
    case 'q':
      d_opts_.quiet   = 1;                   break;
    case 't':
      d_opts_.timeout = atoi(optarg);        break;
    case 'V':
      fprintf(stdout, "%s: version "VERSION" from "__DATE__" "__TIME__ 
              " -- meem@sherilyn.wustl.edu\n", name);
      exit(EXIT_SUCCESS);
    case '?':
      fprintf(stderr,"please see manpage for help\n");
      break;
    default:                                 break; /* just in case */
    }
}


/* set_default_options()
 *
 * purpose: to provide reasonable defaults for the d_opts_ struct
 *   input: void (uses evil global variables)
 *  output: void
 */ 

static void set_default_options(void) {
   d_opts_.port    = DEFAULT_RLPRD_TO_PORT;
   d_opts_.timeout = DEFAULT_TIMEOUT;
   d_opts_.daemon  = 1;		/* daemon on */
   d_opts_.quiet   = 0;		/* quiet off */
}

/* rlpr_fatal()  __attribute__((noreturn))
 *
 * purpose: to localize fatal errors. if server is a daemon, print
 *          to logger, else dump it to the screen
 *   input: char *fmt   a format string
 *          ...         endless pain and suffering
 */ 

void rlpr_fatal(char *fmt, ...) {
  va_list ap;
  va_start(ap, fmt);

  if (d_opts_.is_daemon) {      /* if we're a daemon, log it */
    char buf[BUFSIZ];           /* why isn't vsyslog() standard?! */
    vsprintf(buf, fmt, ap);
    if (errno)
      strcat(buf, ": %m");
    syslog(LOG_ERR, buf);

  } else {
    fprintf(stderr, "%s: fatal: ", name);
    vfprintf(stderr, fmt, ap);
    if (errno)
      fprintf(stderr, ": %s", ERRNO);
    fputc('\n', stderr);
  }

  va_end(ap);
  exit(EXIT_FAILURE);
}

/* rlpr_warn()
 *
 * purpose: to localize warning messages.  if server is a daemon, print
 *          to logger, else dump it to the screen
 *   input: char *fmt   a format string
 *          ...         endless pain and suffering
 *  output: void
 */ 

void rlpr_warn(char *fmt, ...) {
  va_list ap;
  if (d_opts_.quiet) return;    /* quiet suppresses warnings */
  va_start(ap, fmt);
  
  if (d_opts_.is_daemon) {      /* if we're a daemon, log it */
    static char buf[BUFSIZ];    /* why isn't vsyslog() standard?! */
    vsprintf(buf, fmt, ap);
    if (errno)
      strcat(buf, ": %m");
    syslog(LOG_WARNING, buf);

  } else {                      /* else write it to the screen */
    fprintf(stderr, "%s: warning: ", name);
    vfprintf(stderr, fmt, ap);
    if (errno)
      fprintf(stderr, ": %s", ERRNO);
    fputc('\n', stderr);
  }
  va_end(ap);
}

