/*======================================================================

    A driver for the Qlogic SCSI card

    Written by David Hinds, dhinds@allegro.stanford.edu
    
======================================================================*/

#include "config.h"
#include "k_compat.h"

#ifdef MODULE
#define init_qlogic_cs init_module
#endif

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/ioport.h>

#ifdef GET_SCSI_INFO
#if (LINUX_VERSION_CODE >= VERSION(1,3,98))
#include <scsi/scsi.h>
#else
#include <linux/scsi.h>
#endif
#include <linux/major.h>
#endif

#include BLK_DEV_HDR
#include "drivers/scsi/scsi.h"
#include "drivers/scsi/hosts.h"
#if (LINUX_VERSION_CODE >= VERSION(1,3,98))
#include <scsi/scsi_ioctl.h>
#else
#include "drivers/scsi/scsi_ioctl.h"
#endif
#ifdef NEW_QLOGIC
#include "drivers/scsi/qlogicfas.h"
#define QLOGIC QLOGICFAS
#define qlogic_preset qlogicfas_preset
#define qlogic_reset qlogicfas_reset
#else
#include "drivers/scsi/qlogic.h"
#endif

#include "version.h"
#include "cs_types.h"
#include "cs.h"
#include "cistpl.h"
#include "ds.h"

extern void qlogic_preset(int port, int irq);

#ifdef PCMCIA_DEBUG
static int pc_debug = PCMCIA_DEBUG;
static char *version =
"qlogic_cs.c 1.39 1996/08/06 03:14:00 (David Hinds)";
#endif

/*====================================================================*/

/* Parameters that can be set with 'insmod' */

/* Bit map of interrupts to choose from */
static u_long irq_mask = 0xdeb8;

/*====================================================================*/

typedef struct scsi_info_t {
    int		ndev;
    dev_node_t	node[8];
} scsi_info_t;

static void qlogic_release(u_long arg);
static int qlogic_event(event_t event, int priority,
			event_callback_args_t *args);

static dev_link_t *qlogic_attach(void);
static void qlogic_detach(dev_link_t *);

static Scsi_Host_Template driver_template = QLOGIC;

static dev_link_t *dev_list = NULL;

static dev_info_t dev_info = "qlogic_cs";

/*====================================================================*/

static void cs_error(int func, int ret)
{
    CardServices(ReportError, dev_info, (void *)func, (void *)ret);
}

/*====================================================================*/

static dev_link_t *qlogic_attach(void)
{
    client_reg_t client_reg;
    dev_link_t *link;
    int ret;
    
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_DEBUG "qlogic_attach()\n");
#endif

    /* Create new SCSI device */
    link = kmalloc(sizeof(struct dev_link_t), GFP_KERNEL);
    memset(link, 0, sizeof(struct dev_link_t));
    link->priv = kmalloc(sizeof(struct scsi_info_t), GFP_KERNEL);
    memset(link->priv, 0, sizeof(struct scsi_info_t));
    link->release.function = &qlogic_release;
    link->release.data = (u_long)link;

    link->io.NumPorts1 = 16;
    link->io.Attributes1 = IO_DATA_PATH_WIDTH_AUTO;
    link->io.IOAddrLines = 10;
    link->irq.Attributes = IRQ_TYPE_EXCLUSIVE;
    link->irq.IRQInfo1 = IRQ_INFO2_VALID|IRQ_LEVEL_ID;
    link->irq.IRQInfo2 = irq_mask;
    link->conf.Attributes = CONF_ENABLE_IRQ;
    link->conf.Vcc = 50;
    link->conf.IntType = INT_MEMORY_AND_IO;
    link->conf.Present = PRESENT_OPTION;

    /* Register with Card Services */
    link->next = dev_list;
    dev_list = link;
    client_reg.dev_info = &dev_info;
    client_reg.Attributes = INFO_IO_CLIENT | INFO_CARD_SHARE;
    client_reg.event_handler = &qlogic_event;
    client_reg.EventMask =
	CS_EVENT_RESET_REQUEST | CS_EVENT_CARD_RESET |
	CS_EVENT_CARD_INSERTION | CS_EVENT_CARD_REMOVAL |
	CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
    client_reg.Version = 0x0210;
    client_reg.event_callback_args.client_data = link;
    ret = CardServices(RegisterClient, &link->handle, &client_reg);
    if (ret != 0) {
	cs_error(RegisterClient, ret);
	qlogic_detach(link);
	return NULL;
    }
    
    return link;
} /* qlogic_attach */

/*====================================================================*/

static void qlogic_detach(dev_link_t *link)
{
    dev_link_t **linkp;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_DEBUG "qlogic_detach(0x%p)\n", link);
#endif
    
    /* Locate device structure */
    for (linkp = &dev_list; *linkp; linkp = &(*linkp)->next)
	if (*linkp == link) break;
    if (*linkp == NULL)
	return;

    if (link->state & DEV_CONFIG) {
	qlogic_release((u_long)link);
	if (link->state & DEV_STALE_CONFIG) {
	    link->state |= DEV_STALE_LINK;
	    return;
	}
    }

    if (link->handle)
	CardServices(DeregisterClient, link->handle);
    
    /* Unlink device structure, free bits */
    *linkp = link->next;
    if (link->priv) {
	kfree_s(link->priv, sizeof(struct scsi_info_t));
    }
    kfree_s(link, sizeof(struct dev_link_t));
    
} /* qlogic_detach */

/*====================================================================*/

static void qlogic_config(dev_link_t *link)
{
    client_handle_t handle;
    scsi_info_t *info;
    tuple_t tuple;
    cisparse_t parse;
    int i;
    u_char tuple_data[64];
#ifdef GET_SCSI_INFO
    Scsi_Device *dev;
    dev_node_t **tail, *node;
#endif
    
    handle = link->handle;
    info = link->priv;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_DEBUG "qlogic_config(0x%p)\n", link);
#endif

    tuple.TupleData = tuple_data;
    tuple.TupleDataMax = 64;
    tuple.TupleOffset = 0;
    do {
	tuple.DesiredTuple = CISTPL_CONFIG;
	i = CardServices(GetFirstTuple, handle, &tuple);
	if (i != CS_SUCCESS) break;
	i = CardServices(GetTupleData, handle, &tuple);
	if (i != CS_SUCCESS) break;
	i = CardServices(ParseTuple, handle, &tuple, &parse);
	if (i != CS_SUCCESS) break;
	link->conf.ConfigBase = parse.config.base;
    } while (0);
    if (i != CS_SUCCESS) {
	cs_error(ParseTuple, i);
	link->state &= ~DEV_CONFIG_PENDING;
	return;
    }

    /* Configure card */
    link->state |= DEV_CONFIG;
#ifdef MODULE
    driver_template.usage_count = &mod_use_count_;
#endif

    do {

	tuple.DesiredTuple = CISTPL_CFTABLE_ENTRY;
	i = CardServices(GetFirstTuple, handle, &tuple);
	while (i == CS_SUCCESS) {
	    i = CardServices(GetTupleData, handle, &tuple);
	    if (i != CS_SUCCESS) break;
	    i = CardServices(ParseTuple, handle, &tuple, &parse);
	    if (i != CS_SUCCESS) break;
	    link->conf.ConfigIndex = parse.cftable_entry.index;
	    link->io.BasePort1 = parse.cftable_entry.io.win[0].base;
	    if (link->io.BasePort1 != 0) {
		i = CardServices(RequestIO, link->handle, &link->io);
		if (i == CS_SUCCESS) break;
	    }
	    i = CardServices(GetNextTuple, handle, &tuple);
	}
	if (i != CS_SUCCESS) {
	    cs_error(RequestIO, i);
	    break;
	}
	
	i = CardServices(RequestIRQ, link->handle, &link->irq);
	if (i != CS_SUCCESS) {
	    cs_error(RequestIRQ, i);
	    break;
	}
	i = CardServices(RequestConfiguration, link->handle, &link->conf);
	if (i != CS_SUCCESS) {
	    cs_error(RequestConfiguration, i);
	    break;
	}

    } while (0);

    if (i != 0) {
	qlogic_release((u_long)link);
	return;
    }

    /* A bad hack... */
    release_region(link->io.BasePort1, link->io.NumPorts1);
    
    qlogic_preset(link->io.BasePort1, link->irq.AssignedIRQ);
    
    scsi_register_module(MODULE_SCSI_HA, &driver_template);

#ifdef GET_SCSI_INFO
    tail = &link->dev;
    info->ndev = 0;
    for (dev = scsi_devices; dev != NULL; dev = dev->next)
	if (dev->host->hostt == &driver_template) {
	    u_long arg[2], id;
	    kernel_scsi_ioctl(dev, SCSI_IOCTL_GET_IDLUN, arg);
	    id = (arg[0]&0x0f) + ((arg[0]>>4)&0xf0) +
		((arg[0]>>8)&0xf00) + ((arg[0]>>12)&0xf000);
	    node = &info->node[info->ndev];
	    node->minor = 0;
	    switch (dev->type) {
	    case TYPE_TAPE:
		node->major = SCSI_TAPE_MAJOR;
		sprintf(node->dev_name, "st#%04lx", id);
		break;
	    case TYPE_DISK:
	    case TYPE_MOD:
		node->major = SCSI_DISK_MAJOR;
		sprintf(node->dev_name, "sd#%04lx", id);
		break;
	    case TYPE_ROM:
	    case TYPE_WORM:
		node->major = SCSI_CDROM_MAJOR;
		sprintf(node->dev_name, "sr#%04lx", id);
		break;
	    default:
		node->major = SCSI_GENERIC_MAJOR;
		sprintf(node->dev_name, "sg#%04lx", id);
		break;
	    }
	    *tail = node; tail = &node->next;
	    info->ndev++;
	}
    *tail = NULL;
    if (info->ndev == 0)
	printk(KERN_INFO "qlogic_cs: no SCSI devices found\n");
#else
    strcpy(info->node[0].dev_name, "n/a");
    link->dev = &info->node[0];
#endif /* GET_SCSI_INFO */
    
    link->state &= ~DEV_CONFIG_PENDING;
    
} /* qlogic_config */

/*====================================================================*/

static void qlogic_release(u_long arg)
{
    dev_link_t *link = (dev_link_t *)arg;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_DEBUG "qlogic_release(0x%p)\n", link);
#endif

    if (*driver_template.usage_count != 0) {
#ifdef PCMCIA_DEBUG
	if (pc_debug)
	    printk(KERN_DEBUG "qlogic_cs: release postponed, "
		   "device still open\n");
#endif
	link->state |= DEV_STALE_CONFIG;
	return;
    }

    scsi_unregister_module(MODULE_SCSI_HA, &driver_template);
    link->dev = NULL;
    
    CardServices(ReleaseConfiguration, link->handle);
    CardServices(ReleaseIO, link->handle, &link->io);
    CardServices(ReleaseIRQ, link->handle, &link->irq);
    
    link->state &= ~DEV_CONFIG;
    if (link->state & DEV_STALE_LINK)
	qlogic_detach(link);
    
} /* qlogic_release */

/*====================================================================*/

static int qlogic_event(event_t event, int priority,
			event_callback_args_t *args)
{
    dev_link_t *link = args->client_data;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_DEBUG "qlogic_event()\n");
#endif
    
    switch (event) {
#ifdef PCMCIA_DEBUG
    case CS_EVENT_REGISTRATION_COMPLETE:
	if (pc_debug)
	    printk(KERN_DEBUG "qlogic_cs: registration complete\n");
	break;
#endif
    case CS_EVENT_CARD_REMOVAL:
	link->state &= ~DEV_PRESENT;
	if (link->state & DEV_CONFIG) {
	    link->release.expires = RUN_AT(HZ/20);
	    add_timer(&link->release);
	}
	break;
    case CS_EVENT_CARD_INSERTION:
	link->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
	qlogic_config(link);
	break;
    case CS_EVENT_PM_SUSPEND:
	link->state |= DEV_SUSPEND;
	/* Fall through... */
    case CS_EVENT_RESET_PHYSICAL:
	if (link->state & DEV_CONFIG)
	    CardServices(ReleaseConfiguration, link->handle);
	break;
    case CS_EVENT_PM_RESUME:
	link->state &= ~DEV_SUSPEND;
	/* Fall through... */
    case CS_EVENT_CARD_RESET:
	if (link->state & DEV_CONFIG) {
	    CardServices(RequestConfiguration, link->handle, &link->conf);
	    qlogic_reset(NULL);
	}
	break;
    }
    return 0;
} /* qlogic_event */

/*====================================================================*/

int init_qlogic_cs(void) {
    servinfo_t serv;
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_INFO "%s\n", version);
#endif
    CardServices(GetCardServicesInfo, &serv);
    if (serv.Revision != CS_RELEASE_CODE) {
	printk(KERN_NOTICE "qlogic_cs: Card Services release "
	       "does not match!\n");
	return -1;
    }
    register_pcmcia_driver(&dev_info, &qlogic_attach, &qlogic_detach);
    return 0;
}

#ifdef MODULE
void cleanup_module(void) {
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_DEBUG "qlogic_cs: unloading\n");
#endif
    unregister_pcmcia_driver(&dev_info);
    while (dev_list != NULL)
	qlogic_detach(dev_list);
}
#endif

