
/* Various filters etc for Physical models */

#include "cs.h"
#include <math.h>

#define AMP_SCALE	(32768.0f)
#define AMP_RSCALE	(.000030517578125f)

/*******************************************/
/*  Noise Generator Class,                 */
/*  by Perry R. Cook, 1995-96              */ 
/*  White noise as often as you like.      */
/*  Recoded by John ffitch 1997            */
/*******************************************/

#if !defined(__PhysUtil_h)
#define __PhysUtil_h

typedef float Noise;

#define make_Noise(n) n = 0.0f

float Noise_tick(Noise*);
#define Noise_lastOut(n) (n)

/*******************************************/
/*  Linearly Interpolating Delay Line      */
/*  Object by Perry R. Cook 1995-96        */
/*  Recoded by John ffitch 1997            */
/*******************************************/

typedef struct DLineL{
    AUXCH	inputs;
    float	lastOutput;
    long	inPoint;
    long	outPoint;
    long	length;
    float	alpha;
    float	omAlpha;
} DLineL;

#define DLineL_lastOut(d)	((d)->lastOutput)
void make_DLineL(DLineL *, long);  
void DLineL_setDelay(DLineL *, float);
float DLineL_tick(DLineL *, float);

/*******************************************/
/*  Envelope Class, Perry R. Cook, 1995-96 */ 
/*  This is the base class for envelopes.  */
/*  This one is capable of ramping state   */
/*  from where it is to a target value by  */
/*  a rate.  It also responds to simple    */
/*  KeyOn and KeyOff messages, ramping to  */         
/*  1.0 on keyon and to 0.0 on keyoff.     */
/*  There are two tick (update value)      */
/*  methods, one returns the value, and    */
/*  other returns 0 if the envelope is at  */
/*  the target value (the state bit).      */
/*******************************************/

#define RATE_NORM	(22050.0f/esr)

typedef struct Envelope {
    float	value;
    float	target;
    float	rate;
    int		state;
} Envelope;

void make_Envelope(Envelope*);
void Envelope_keyOn(Envelope*);
void Envelope_keyOff(Envelope*);
void Envelope_setRate(Envelope*, float);
void Envelope_setTarget(Envelope*, float);
void Envelope_setValue(Envelope*,float);
float Envelope_tick(Envelope*);
void Envelope_print(Envelope*);

/*******************************************/
/*  One Pole Filter Class,                 */
/*  by Perry R. Cook, 1995-96              */ 
/*  The parameter gain is an additional    */
/*  gain parameter applied to the filter   */  
/*  on top of the normalization that takes */
/*  place automatically.  So the net max   */
/*  gain through the system equals the     */
/*  value of gain.  sgain is the combina-  */
/*  tion of gain and the normalization     */
/*  parameter, so if you set the poleCoeff */
/*  to alpha, sgain is always set to       */
/*  gain * (1.0 - fabs(alpha)).            */
/*******************************************/

typedef struct OnePole {
    float gain;                 /* Start Filter subclass */
    float outputs;
    /*    float *inputs; */
/*     float lastOutput;  */          /* End */
    float poleCoeff;
    float sgain;
} OnePole;

void make_OnePole(OnePole*);
/* void OnePole_clear(OnePole*); */
void OnePole_setPole(OnePole*, float aValue);
void OnePole_setGain(OnePole*, float aValue);
float OnePole_tick(OnePole*, float sample);
void OnePole_print(OnePole*);

/*******************************************/
/*  DC Blocking Filter                     */ 
/*  by Perry R. Cook, 1995-96              */ 
/*  This guy is very helpful in, uh,       */
/*  blocking DC.  Needed because a simple  */
/*  low-pass reflection filter allows DC   */
/*  to build up inside recursive           */ 
/*  structures.                            */
/*******************************************/

typedef struct DCBlock {
    float	gain;
    float	outputs;
    float	inputs;
/*     float	lastOutput; */
} DCBlock;

void make_DCBlock(DCBlock*);
/* void DCBlock_clear(DCBlock*); */
float DCBlock_tick(DCBlock*, float);

/*******************************************/
/*  ADSR Subclass of the Envelope Class,   */
/*  by Perry R. Cook, 1995-96              */ 
/*  This is the traditional ADSR (Attack   */
/*  Decay, Sustain, Release) envelope.     */
/*  It responds to simple KeyOn and KeyOff */
/*  messages, keeping track of it's state. */         
/*  There are two tick (update value)      */
/*  methods, one returns the value, and    */
/*  other returns the state (0 = A, 1 = D, */
/*  2 = S, 3 = R)                          */
/*******************************************/



#define ATTACK	(0)
#define DECAY	(1)
#define SUSTAIN (2)
#define RELEASE (3)
#define CLEAR	(4)

typedef struct ADSR {
    float	value;                /* Envelope subclass */
    float	target;
    float	rate;
    int 	state;                  /* end */
    float	attackRate;
    float	decayRate;
    float	sustainLevel;
    float	releaseRate;
} ADSR;

void make_ADSR(ADSR*);
void dest_ADSR(ADSR*);
void ADSR_keyOn(ADSR*);
void ADSR_keyOff(ADSR*);
void ADSR_setAttackRate(ADSR*, float);
void ADSR_setDecayRate(ADSR*, float);
void ADSR_setSustainLevel(ADSR*, float);
void ADSR_setReleaseRate(ADSR*, float);
void ADSR_setAll(ADSR*, float, float, float, float);
void ADSR_setTarget(ADSR*, float);
void ADSR_setValue(ADSR*, float);
float ADSR_tick(ADSR*);
int ADSR_informTick(ADSR*);  
float ADSR_lastOut(ADSR*);

/*******************************************/
/*  BiQuad (2-pole, 2-zero) Filter Class,  */
/*  by Perry R. Cook, 1995-96              */ 
/*  See books on filters to understand     */
/*  more about how this works.  Nothing    */
/*  out of the ordinary in this version.   */
/*******************************************/


typedef struct BiQuad {
    float	gain;                 /* Start if filter subclass */
/*     float *outputs; */
    float	inputs[2];
    float	lastOutput;           /* End */
    float	poleCoeffs[2];
    float	zeroCoeffs[2];
} BiQuad;

void make_BiQuad(BiQuad*);
void dest_BiQuad(BiQuad*);
void BiQuad_clear(BiQuad*);
void BiQuad_setPoleCoeffs(BiQuad*, float *);
void BiQuad_setZeroCoeffs(BiQuad*, float *);
#define BiQuad_setGain(b,aValue)	((b).gain = aValue)
#define BiQuad_setEqualGainZeroes(b)	\
	{ (b).zeroCoeffs[1] = -1.0f; (b).zeroCoeffs[0] = 0.0f; }
#define BiQuad_setFreqAndReson(b,freq,reson)	\
	{ (b).poleCoeffs[1]= -((reson)*(reson)); \
          (b).poleCoeffs[0]= 2.0f*(reson)*(float)cos(TWOPI*(double)(freq)/(double)esr); }
float BiQuad_tick(BiQuad*, float);
#define BiQuad_lastOut(x)	(x)->lastOutput

#endif

