/* transform.c */
/************************************************************
 *                                                          *
 *  Permission is hereby granted  to  any  individual   or  *
 *  institution   for  use,  copying, or redistribution of  *
 *  the xgobi code and associated documentation,  provided  *
 *  that   such  code  and documentation are not sold  for  *
 *  profit and the  following copyright notice is retained  *
 *  in the code and documentation:                          *
 *        Copyright (c) 1990, ..., 1996 Bellcore            *
 *                                                          *
 *  We welcome your questions and comments, and request     *
 *  that you share any modifications with us.               *
 *                                                          *
 *    Deborah F. Swayne            Dianne Cook              *
 *   dfs@research.att.com       dicook@iastate.edu          *
 *      (973) 360-8423    www.public.iastate.edu/~dicook/   *
 *                                                          *
 *                    Andreas Buja                          *
 *                andreas@research.att.com                  *
 *              www.research.att.com/~andreas/              *
 *                                                          *
 ************************************************************/

#include <values.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include "xincludes.h"
#include "xgobitypes.h"
#include "xgobivars.h"
#include "xgobiexterns.h"

#define NTFORMS 22
char message[MSGLENGTH];
#define DOMAIN_ERROR sprintf(message, "Data outside the domain of function.\n")

#define RESTORE      0
#define PERMUTE      1
#define ABSVALUE     2
#define NEGATIVE     3
#define INVERSE      4
#define LN           5
#define LNPLUS1      6
#define LOG10        7
#define LOG10PLUS1   8
#define FOURTH_ROOT  9
#define CUBE_ROOT   10
#define SQUARE_ROOT 11
#define SQUARE      12
#define CUBE        13
#define FOURTH_POWER 14
#define SCALE       15
#define STANDARDIZE 16
#define DISCRETE2   17
#define LNMINPLUS   18
#define NEGLN       19
#define SORT        20
#define NORMSCORE   21

static char *tform_names[] = {
  "Restore Variable",
  "Permute",
  "Absolute Value",                   /*rpc_xgobi.c~*/  
  "Negative",
  "Inverse",
  "Natural Log",
  "Natural Log (x+1)",
  "Common Log",
  "Common Log (x+1)",
  "Fourth Root",
  "Cube Root",
  "Square Root",
  "Square",
  "Cube",
  "Fourth Power",
  "Scale to [0,1]",
  "Standardize",
  "Discretize: 2 levels",
  "Natural Log (x+min)",
  "Abs(Natural Log)",
  "Sort",
  "Normal Score"
};
int *tform_type;  /* the transformation applied to each variable */
static struct {
	Widget cascade;
	Widget btn[NTFORMS];
} tform_menu;

static int ntform_cols, *tform_cols = NULL;  /* set in which_cols */

int
which_cols(int *cols, int varno, xgobidata *xg) {
/*
 * Figure out which columns to transform.
*/
  int j, ncols = 0;
  int groupno = xg->vgroup_ids[varno];

  /* allocated before this is called */
  for (j=0; j<xg->ncols_used; j++) {
    if (xg->vgroup_ids[j] == groupno)
      cols[ncols++] = j;
  }
  return(ncols);
}

void
reset_tform(xgobidata *xg) {
  int j;

  for (j=0; j<xg->ncols_used; j++)
    tform_type[j] = RESTORE;
}

int 
sort_compare (val1, val2)
  float *val1, *val2;
{
  if (*val1 < *val2) 
    return (-1);
  else if (*val1 == *val2)
    return (0);
  else 
    return (1);
}

int
transform(xgobidata *xg, int *cols, int ncols, int tfnum)
{
  int i, j, n;
  float min, max, diff;
  float mean, stddev;
  float median, mad, ref;
  Boolean allequal;
  int tmpi, numperm, indx_flag;

  switch(tfnum)
  {
    case RESTORE:    /* Restore original values */
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
          xg->tform_data[i][j] = xg->raw_data[i][j];

        (void) strcpy(xg->collab_tform[j], xg->collab[j]);
      }
      break;

    case ABSVALUE:    /* Absolute value */
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          /* quicker to fabs everything, or do this test? */
          if (xg->raw_data[i][j] < 0)
            xg->tform_data[i][j] = (float) fabs((double)xg->raw_data[i][j]) ;
          else
            xg->tform_data[i][j] = xg->raw_data[i][j] ;
        }

        (void) sprintf(xg->collab_tform[j], "Abs(%s)", xg->collab[j]);
      }
      break;

    case NEGATIVE:    /* Multiply original values by -1 */
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
          xg->tform_data[i][j] = (float) -1.0 * xg->raw_data[i][j] ;

        (void) sprintf(xg->collab_tform[j], "-%s", xg->collab[j]);
      }
      break;

    case INVERSE:    /* 1/x */
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          if (xg->raw_data[i][j] == 0) {
            DOMAIN_ERROR;
            show_message(message, xg);
            return(0);
          }
        }
      }

      for (n=0; n<ncols; n++)
      {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
        {
          xg->tform_data[i][j] = (float)
            pow((double) xg->raw_data[i][j], (double) (-1.0));
        }

        (void) sprintf(xg->collab_tform[j], "1/%s", xg->collab[j]);
      }
      break;

    case LN:    /* Natural log */
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          if (xg->raw_data[i][j] <= 0) {
            DOMAIN_ERROR;
            show_message(message, xg);
            return(0);
          }
        }
      }
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          xg->tform_data[i][j] = (float)
            log((double) xg->raw_data[i][j]);
        }

        (void) sprintf(xg->collab_tform[j], "ln(%s)", xg->collab[j]);
      }
      break;

    case LNPLUS1:    /* Natural log of x+1 */
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          if (xg->raw_data[i][j] <= -1) {
            DOMAIN_ERROR;
            show_message(message, xg);
            return(0);
          }
        }
      }
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          xg->tform_data[i][j] = (float)
            log1p((double) xg->raw_data[i][j]);
        }

        (void) sprintf(xg->collab_tform[j], "ln(%s+1)", xg->collab[j]);
      }
      break;

      case LNMINPLUS:    
      for (n=0; n<ncols; n++) {
        j = cols[n];
      }
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          xg->tform_data[i][j] = (float)
            log((double) (xg->raw_data[i][j]+xg->lim0[j].min+1));
        }

        (void) sprintf(xg->collab_tform[j], "ln(%s+1)", xg->collab[j]);
      }
      break;

    case NEGLN:    
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          if (-1.0*xg->raw_data[i][j] <= 0) {
            DOMAIN_ERROR;
            show_message(message, xg);
            return(0);
          }
        }
      }
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          xg->tform_data[i][j] = (float)
            log((double) (-1.0*xg->raw_data[i][j]));
        }

        (void) sprintf(xg->collab_tform[j], "ln(%s)", xg->collab[j]);
      }
      break;

    case LOG10:    /* Common log */
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          if (xg->raw_data[i][j] <= 0) {
            DOMAIN_ERROR;
            show_message(message, xg);
            return(0);
          }
        }
      }
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          xg->tform_data[i][j] = (float)
            log10((double) xg->raw_data[i][j]);
        }

        (void) sprintf(xg->collab_tform[j], "log10(%s)", xg->collab[j]);
      }
      break;

    case LOG10PLUS1:    /* Common log of x+1 */
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          if (xg->raw_data[i][j] <= -1) {
            DOMAIN_ERROR;
            show_message(message, xg);
            return(0);
          }
        }
      }
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          xg->tform_data[i][j] = (float)
            log10((double) (xg->raw_data[i][j] + 1.0));
        }

        (void) sprintf(xg->collab_tform[j], "log10(%s+1)", xg->collab[j]);
      }
      break;

    case FOURTH_ROOT:    /* Fourth root */
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          if (xg->raw_data[i][j] < 0) {
            DOMAIN_ERROR;
            show_message(message, xg);
            return(0);
          }
        }
      }

      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          if (xg->raw_data[i][j] < 0) {
            xg->tform_data[i][j] = (float)
              - pow( - (double) xg->raw_data[i][j],
                   (double) (1.0/4.0));
          }
          else {
            xg->tform_data[i][j] = (float)
              pow((double) xg->raw_data[i][j],
                (double) (1.0/4.0));
          }
        }
        (void) sprintf(xg->collab_tform[j], "%s^(1/4)", xg->collab[j]);
      }
      break;

    case CUBE_ROOT:    /* Cube root */
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          if (xg->raw_data[i][j] < 0) {
            xg->tform_data[i][j] = (float)
              - pow( - (double) xg->raw_data[i][j],
                   (double) (1.0/3.0));
          }
          else {
            xg->tform_data[i][j] = (float)
              pow((double) xg->raw_data[i][j],
                (double) (1.0/3.0));
          }
        }
        (void) sprintf(xg->collab_tform[j], "%s^(1/3)", xg->collab[j]);
      }
      break;

    case SQUARE_ROOT:    /* Square root */
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          if (xg->raw_data[i][j] < 0) {
            DOMAIN_ERROR;
            show_message(message, xg);
            return(0);
          }
        }
      }
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          xg->tform_data[i][j] = (float)
            sqrt((double) xg->raw_data[i][j]);
        }

        (void) sprintf(xg->collab_tform[j], "%s^1/2", xg->collab[j]);
      }
      break;

    case SQUARE:    /* Square */
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
          xg->tform_data[i][j] = (float)
            pow((double) xg->raw_data[i][j], (double) 2);

        (void) sprintf(xg->collab_tform[j], "%s^2", xg->collab[j]);
      }
      break;

    case CUBE:    /* Cube */
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
          xg->tform_data[i][j] = (float)
            pow((double) xg->raw_data[i][j], (double) 3);

        (void) sprintf(xg->collab_tform[j], "%s^3", xg->collab[j]);
      }
      break;

    case FOURTH_POWER:    /* Fourth Power */
      for (n=0; n<ncols; n++)
      {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
          xg->tform_data[i][j] = (float)
            pow((double) xg->raw_data[i][j], (double) 4);

        (void) sprintf(xg->collab_tform[j], "%s^4", xg->collab[j]);
      }
      break;

    case SCALE:    /* Map onto [0,1] */
      /* First find min and max; they get updated after transformations */
      min_max(xg, xg->raw_data, cols, ncols, &min, &max);
      adjust_limits(&min, &max);
      diff = max - min;

      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
          xg->tform_data[i][j] = (float) (xg->raw_data[i][j] - min)/diff;

        (void) sprintf(xg->collab_tform[j], "%s [0,1]", xg->collab[j]);
      }
      break;

    case STANDARDIZE:    /* (x-mean)/sigma */
      /* First find min and max */
      mean_stddev(xg, xg->raw_data, cols, ncols, &min, &max, &mean, &stddev);

      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
          xg->tform_data[i][j] = (float) (xg->raw_data[i][j] - mean)/stddev;

        (void) sprintf(xg->collab_tform[j], "(%s-m)/s", xg->collab[j]);
      }
      break;

    case DISCRETE2:    /* x>median */
      /* refuse to discretize if all values are the same */
      allequal = True;
      ref = xg->raw_data[0][cols[0]] ;
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          if (xg->raw_data[i][j] != ref) {
            allequal = False;
            break;
          }
        }
      }
      if (allequal) {
        DOMAIN_ERROR;
        show_message(message, xg);
        return(0);
      }

      /* First find median */
      med_mad(xg, xg->raw_data, cols, ncols, &min, &max, &median, &mad);
      /* Then find the true min and max */
      min_max(xg, xg->raw_data, cols, ncols, &min, &max);
      /* This prevents the collapse of the data in a special case */
      if (max == median)
        median = (min + max)/2.0;

      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
          if( xg->raw_data[i][j] > median )
              xg->tform_data[i][j] = (float) 1.0;
            else
              xg->tform_data[i][j] = (float) 0.0;

        (void) sprintf(xg->collab_tform[j], "%s:0,1", xg->collab[j]);
      }
      break;

    case PERMUTE:    /* (x-mean)/sigma */
      { /* This curly brace allows me to define variables for this case */
      /* First generate a random vector */
      /* Allocate array for permutation indices */
      int *permindx;
      permindx = (int *) XtMalloc((Cardinal) xg->nrows * sizeof(int));

      numperm = 0;
      while (numperm < xg->nrows)
      {
#ifdef USE_DRAND48
        tmpi = (int) (drand48() * (double)xg->nrows);
#else
        tmpi =  (int) ((double) random()/ (double) MAXINT * xg->nrows);
#endif
        indx_flag = 0;
        for (i=0; i<numperm; i++) {
          if (tmpi == permindx[i]) {
            indx_flag = 1;
	      }
	    }
        if (!indx_flag) {
          permindx[numperm] = tmpi;
          numperm++;
        }
      }

      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
          xg->tform_data[i][j] = xg->raw_data[permindx[i]][j];

        (void) sprintf(xg->collab_tform[j], "perm(%s)", xg->collab[j]);
      }
      XtFree((XtPointer) permindx);

      }
      break;

    case SORT: 
      {
      /*  create sort index  here - mallika */
      float *sort_data; /* mallika */
      /* Allocate array for sorted columns - mallika */
      sort_data = (float *) XtMalloc((Cardinal) xg->nrows * sizeof(float));
    
      for (n=0; n<ncols; n++)
      {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
          sort_data[i]=xg->raw_data[i][j];

        qsort((char *) sort_data, xg->nrows, sizeof(float), sort_compare);
   
        for (i=0; i<xg->nrows; i++)
          xg->tform_data[i][j] = sort_data[i]; 

        (void) sprintf(xg->collab_tform[j], "sort(%s)", xg->collab[j]);
      }
      XtFree((XtPointer) sort_data);/* mallika */
      }
      break;

    case NORMSCORE:  /* mallika*/
      {
      float *norm_score_data;  /*mallika*/
      /* Allocate array for normalized scores  - mallika */
      norm_score_data = (float *)
        XtMalloc((Cardinal) xg->nrows * sizeof(float));

       for (n=0; n<ncols; n++) {
        float normmean=0, normvar=0;
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          norm_score_data[i]=xg->raw_data[i][j];
          normmean+=xg->raw_data[i][j];
          normvar+=(xg->raw_data[i][j]*xg->raw_data[i][j]);
        }
        normmean/=xg->nrows;
        normvar=(float)sqrt((float)(normvar/xg->nrows-normmean*normmean));
        for (i=0; i<xg->nrows; i++)
          norm_score_data[i]=(norm_score_data[i]-normmean)/normvar;
        for (i=0; i<xg->nrows; i++)
        {
          if (norm_score_data[i]>0)
            norm_score_data[i] = erf(norm_score_data[i]/sqrt(2.))/
              2.8284271+0.5;
          else if (norm_score_data[i]<0)
            norm_score_data[i] = 0.5-erf((float) fabs((double) 
              norm_score_data[i])/sqrt(2.))/2.8284271;
          else 
            norm_score_data[i]=0.5;
	    }
        
        for (i=0; i<xg->nrows; i++)
          xg->tform_data[i][j] = norm_score_data[i]; 

        (void) sprintf(xg->collab_tform[j], "normsc(%s)", xg->collab[j]);
      }
      XtFree((XtPointer) norm_score_data);/* mallika */
      }

  }
  return(1);
}

static void
update_transformed_data(xgobidata *xg, int *cols, int ncols, int tfno) {
  int j, n;

  if (xg->ncols_used > 2)
    update_sphered(xg, cols, ncols);
  update_lims(xg);
  update_world(xg);

  /* Set tform_type[] for transformed columns */
  for (n=0; n<ncols; n++) {
    tform_type[cols[n]] = tfno;
  }

  world_to_plane(xg);
  plane_to_screen(xg);
  /*
    This bit of init_axes() is needed.
  */
  for (n=0; n<ncols; n++) {
    j = cols[n];
    xg->nicelim[j].min = xg->lim0[j].min;
    xg->nicelim[j].max = xg->lim0[j].max;
    SetNiceRange(j, xg);
    xg->deci[j] = set_deci(xg->tickdelta[j]);
  }

  if (xg->is_xyplotting)
    init_ticks(&xg->xy_vars, xg);
  else if (xg->is_dotplotting)
    init_ticks(&xg->dotplot_vars, xg);

  if (xg->is_brushing) {
    assign_points_to_bins(xg);
    if (xg->brush_mode == transient)
      reinit_transient_brushing(xg);
  }

  plot_once(xg);

  if (xg->is_cprof_plotting)
    update_cprof_plot(xg);
}


/* ARGSUSED */
XtCallbackProc
tform_choose_cback(Widget w, xgobidata *xg, XtPointer cback_data)
{
  int j, n;
  int varno = -1, tfno;
  char *tf_name;
  Widget label = XtParent(XtParent(w));
  int *cols, ncols;
  int groupno;

  if (xg->is_touring && (xg->is_princ_comp || xg->is_pp)) {
    sprintf(message,
      "Sorry, it isn't possible to transform variables during\n");
    strcat(message,
      "projection pursuit or when principal components are used.\n");
    show_message(message, xg);
  }
  else
  {
    /*
     * Figure out which variable is to be transformed.
    */
    for (varno=0; varno<xg->ncols_used; varno++)
      if (xg->varlabw[varno] == label)
        break;

    if (varno < 0) {
      (void) fprintf(stderr, "error in tform_choose_cback\n");
      return ((XtCallbackProc) 0);
    }

    /*
     * Figure out which transformation to do.  32 is (so far)
     * longer than the longest member of tform_names[].
    */
    tf_name = XtMalloc(32 * sizeof(char));
    XtVaGetValues(w, XtNlabel, (String) &tf_name, NULL);

    for (tfno=0; tfno<NTFORMS; tfno++)
      if (strcmp(tf_name, tform_names[tfno]) == 0) {
        /*
         * If I free tf_name here, I get 'Freeing memory already
         * freed' from purify and the transformation menus get screwed up.
         * If I don't free it, purify reports memory leaks but at least
         * the transformation menus seem fine.  Easy choice:  don't free
         * the memory.  Check this again in a later version of X.
         * dfs   September, 1995
        */
        /*XtFree(tf_name);*/
        break;
      }

    /*
     * Figure out which columns to transform.
    */
    tform_cols = (int *) XtRealloc((XtPointer) tform_cols,
      (Cardinal) xg->ncols_used * sizeof(int));
    ntform_cols = which_cols(tform_cols, varno, xg);

    /*
     * Call the appropriate transformation program.
    */
    if ( transform(xg, tform_cols, ntform_cols, tfno) )
      update_transformed_data(xg, tform_cols, ntform_cols, tfno);

    /*XtFree((XtPointer) cols);*/
  }
}

XtCallbackProc
popdown_cback(w, id, cbdata) 
  Widget w;
  XtPointer id;
  XtPointer cbdata;
{
  XtDestroyWidget(w);
}

/* ARGSUSED */
XtEventHandler
add_tform_menu(w, xg, evnt)
/*
 * Add menus for choosing a data transformation.
*/
  Widget w;
  xgobidata *xg;
  XEvent *evnt;
{
  int k, varno;

  for (varno=0; varno<xg->ncols; varno++)
    if (xg->varlabw[varno] == w)
      break;

  tform_menu.cascade = XtVaCreatePopupShell("menu",
    simpleMenuWidgetClass, xg->varlabw[varno],
    NULL);
  if (mono) set_mono(tform_menu.cascade);
  XtAddCallback(tform_menu.cascade, XtNpopdownCallback, 
    (XtCallbackProc) popdown_cback, (XtPointer) NULL);

  for (k=0; k<NTFORMS; k++) {
    tform_menu.btn[k] = XtVaCreateManagedWidget("Command",
      smeBSBObjectClass, tform_menu.cascade,
      XtNlabel, (String) tform_names[k],
      XtNleftMargin, (Dimension) 24,
      NULL);
    if (mono) set_mono(tform_menu.btn[k]);
    XtAddCallback(tform_menu.btn[k], XtNcallback,
      (XtCallbackProc) tform_choose_cback, (XtPointer) xg);
  }

  XtVaSetValues(tform_menu.btn[tform_type[varno]],
    XtNleftBitmap, menu_mark, NULL);
}

void
alloc_transform_types(xg)
  xgobidata *xg;
{
  tform_type = (int *) XtRealloc((char *) tform_type,
    (Cardinal) xg->ncols * sizeof(int));
}

void
init_transform_types(xg)
/*
 * Set each tranformation type to 0;
*/
  xgobidata *xg;
{
  int j;

  for (j=0; j<xg->ncols; j++)
    tform_type[j] = 0;
}

/********** do the reverse transform; needed by move_points *******/

#define signum(x) (((x)<0.0)?(-1.0):(((x)>0.0)?(1.0):(0.0)))
#define INV_DOMAIN_ERROR sprintf(message, \
  "Data outside the domain of function; can\'t do the transformation.\n")
float
inv_transform(int icase, int jvar, xgobidata *xg) {
  double tx = xg->tform_data[icase][jvar];
  double rx = xg->raw_data[icase][jvar];
  double new_rx = tx;

  float min, max, diff;
  int *cols;
  int ncols = 0, i, j, n;
  float mean, stddev;

  switch (tform_type[jvar]) {
    case RESTORE:
      /* new_rx = tx; */
      break;
    case ABSVALUE:
      new_rx = tx * signum(rx);
      break;
    case NEGATIVE:
      new_rx = -tx;
      break;
    case INVERSE:
      if (tx == 0) {
        INV_DOMAIN_ERROR;
        show_message(message, xg);
      } else
        new_rx = 1.0/tx;
      break;
    case LN:    /* Natural log : take e^tx */
      new_rx = exp(tx);
      break;
    case LNPLUS1:    /* Natural log of x+1 : take e^(x+1) - x */
      new_rx = exp(tx+1) - 1.0;
      break;
      case LNMINPLUS:    
      new_rx = exp(tx) - xg->lim0[jvar].min;
      break;
    case NEGLN:    
      new_rx = exp(tx)*(-1.0);
      break;
    case LOG10:    /* Common log */
      new_rx = pow(10.0, tx);
      break;

    case LOG10PLUS1:    /* Common log of x+1 */
      new_rx = pow(10.0, tx) - 1.0;
      break;

    case FOURTH_ROOT:    /* Fourth root */
      new_rx = pow(tx, 4.0);
      break;

    case CUBE_ROOT:    /* Cube root */
      new_rx = pow(tx, 3.0);
      break;

    case SQUARE_ROOT:    /* Square root */
      new_rx = pow(tx, 2.0);
      break;

    case SQUARE:    /* Square */
      if (tx < 0) {
        INV_DOMAIN_ERROR;
        show_message(message, xg);
      } else
        new_rx = (float) sqrt((double) tx);
      break;

    case CUBE:    /* Cube */
      new_rx = pow(tx, (1.0/3.0));
      break;

    case FOURTH_POWER:    /* Fourth Power */
      if (tx < 0) {
        INV_DOMAIN_ERROR;
        show_message(message, xg);
      } else
        new_rx = signum(rx) * pow(tx, (1.0/4.0));
      break;

    case SCALE:    /* Map onto [0,1] */
      cols = (int *) XtMalloc((Cardinal) xg->ncols_used * sizeof(int));
      ncols = which_cols(cols, jvar, xg);

      min_max(xg, xg->raw_data, cols, ncols, &min, &max);
      adjust_limits(&min, &max);
      diff = max - min;

      new_rx = (tx * diff) + min;
      break;

    case STANDARDIZE:    /* (x-mean)/sigma */
      cols = (int *) XtMalloc((Cardinal) xg->ncols_used * sizeof(int));
      ncols = which_cols(cols, jvar, xg);

      mean_stddev(xg, xg->raw_data, cols, ncols, &min, &max, &mean, &stddev);
      new_rx = (tx * stddev) + mean;

      break;

    case DISCRETE2:    /* x>median */
      show_message(
        "Sorry, I\'m unable to perform the transformation for this point\n", 
        xg);
      break;

    case PERMUTE:    
      cols = (int *) XtMalloc((Cardinal) xg->ncols_used * sizeof(int));
      ncols = which_cols(cols, jvar, xg);

      for (n=0; n<ncols; n++)
      {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
          xg->tform_data[i][j] = xg->raw_data[i][j];
      }
      break;

    case SORT:    
      cols = (int *) XtMalloc((Cardinal) xg->ncols_used * sizeof(int));
      ncols = which_cols(cols, jvar, xg);

      for (n=0; n<ncols; n++)
      {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
          xg->tform_data[i][j] = xg->raw_data[i][j];
      }

      break;
    case NORMSCORE:    
      cols = (int *) XtMalloc((Cardinal) xg->ncols_used * sizeof(int));
      ncols = which_cols(cols, jvar, xg);

      for (n=0; n<ncols; n++)
      {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
          xg->tform_data[i][j] = xg->raw_data[i][j];
      }
      break;

    default:
      break;
  }
  return(new_rx);
}

void
permute_again(xgobidata *xg) {
  if (xg->is_touring && (xg->is_princ_comp || xg->is_pp))
    ;
  else {
    if (ntform_cols > 0 && tform_cols != NULL) {
      if ( transform(xg, tform_cols, ntform_cols, PERMUTE) )
        update_transformed_data(xg, tform_cols, ntform_cols, PERMUTE);
    }
  }
}

/* called by pressing the "," key */
/* It should work at it is, but leave the transformation menu
   screwed up.
*/
void
restore_variables(xgobidata *xg) {
  if (ntform_cols > 0 && tform_cols != NULL) {
    if ( transform(xg, tform_cols, ntform_cols, RESTORE) )
      update_transformed_data(xg, tform_cols, ntform_cols, RESTORE);
  }
}
