#
# Copyright (C) 1997, 1998, 1999  Motoyuki Kasahara
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

# Whether BookView connects to server1 at start.
# (used in the setup window only)
set setup_start_server1 1

#
# `Setup...'
#
proc setup {} {
    global start_server1
    global setup_start_server1
    global server_names

    setup_make_window

    if {[llength $server_names] == 0} {
	set setup_start_server1 1
    } else {
	set setup_start_server1 $start_server1
    }

    set oldfocus [focus]

    #
    # Popup a sub window.
    #
    wm title .setup "bookview setup"
    wm iconname .setup "bookview setup"
    wm withdraw .setup

    set x [expr [winfo rootx .] + 24]
    set y [expr [winfo rooty .] + 24]
    wm geometry .setup +$x+$y
    wm deiconify .setup

    focus .setup
    grab .setup
    tkwait window .setup

    if {![catch {winfo .setup}]} {
	destroy .setup
    }
    focus $oldfocus
}

#
# Make a setup window.
#
proc setup_make_window {} {
    global max_servers
    global start_server1
    global setup_start_server1
    global server_names
    global server_ports
    global server_starts
    global prefer_sizes
    global prefer_index

    #
    # Toplevel
    #
    toplevel .setup
    bind .setup <Return> setup_ok
    bind .setup <Control-m> setup_ok
    bind .setup <Escape> {destroy .setup}

    #
    # `Servers:' part
    #
    frame .setup.servers -relief groove -borderwidth 4
    pack .setup.servers -side top -fill x -padx 4 -pady 2

    label .setup.servers.label -text "Servers:" -anchor w
    pack .setup.servers.label -side top -fill x

    #
    # Header
    #
    frame .setup.servers.header
    pack .setup.servers.header -side top -padx 2

    label .setup.servers.header.id -width 8
    pack .setup.servers.header.id -side left

    label .setup.servers.header.hostname -text "hostname" -width 30
    pack .setup.servers.header.hostname -side left

    label .setup.servers.header.port -text "port" -width 6
    pack .setup.servers.header.port -side left

    label .setup.servers.header.start -text "start book ID" -width 18
    pack .setup.servers.header.start -side left

    #
    # Server 1, 2, ... n
    #
    set length [llength $server_names]
    for {set i 1} {$i <= $max_servers} {incr i} {
	frame .setup.servers.server$i
	pack .setup.servers.server$i -side top -padx 2

	label .setup.servers.server$i.id -text "server $i" -width 8
	pack .setup.servers.server$i.id -side left

	entry .setup.servers.server$i.hostname -relief sunken -borderwidth 2 \
		-width 30
	pack .setup.servers.server$i.hostname -side left

	entry .setup.servers.server$i.port -relief sunken -borderwidth 2 \
		-width 6
	pack .setup.servers.server$i.port -side left

	entry .setup.servers.server$i.start -relief sunken -borderwidth 2 \
		-width 18
	pack .setup.servers.server$i.start -side left

	#
	# Set default values.
	#
	if {$i <= $length} {
	    .setup.servers.server$i.hostname insert 0 \
		    [lindex $server_names [expr $i - 1]]
	    .setup.servers.server$i.port insert 0 \
		    [lindex $server_ports [expr $i - 1]]
	    .setup.servers.server$i.start insert 0 \
		    [lindex $server_starts [expr $i - 1]]
	} else {
	    .setup.servers.server$i.hostname insert 0 ""
	    .setup.servers.server$i.port insert 0 "ndtp"
	    .setup.servers.server$i.start insert 0 ""
	}
    }

    #
    # Whether connect to server 1 at start of BookView.
    #
    frame .setup.servers.at_start
    pack .setup.servers.at_start -side top
    checkbutton .setup.servers.at_start.check -variable setup_start_server1 \
	    -text "Connect to server 1 at start of BookView" -onvalue 1 \
	    -offvalue 0
    pack .setup.servers.at_start.check -side top -pady 2 -fill x

    #
    # `Bitmap:' part
    #
    frame .setup.bitmap -relief groove -borderwidth 4
    pack .setup.bitmap -side top -fill x -padx 4 -pady 2

    label .setup.bitmap.label -anchor w -text "Bitmap:"
    pack .setup.bitmap.label -side top -fill x

    frame .setup.bitmap.prefer
    pack .setup.bitmap.prefer -side top

    label .setup.bitmap.prefer.header -text "preference order of sizes"
    pack .setup.bitmap.prefer.header -side top
    listbox .setup.bitmap.prefer.listbox -relief sunken -borderwidth 2 \
	    -width 20 -height 3  -exportselection 0 -selectmode single \
	    -yscrollcommand ".setup.bitmap.prefer.scroll set" 
    pack .setup.bitmap.prefer.listbox -side left -fill y -pady 2
    scrollbar .setup.bitmap.prefer.scroll \
	    -command ".setup.bitmap.prefer.listbox yview"
    pack .setup.bitmap.prefer.scroll -side left -fill y

    foreach i $prefer_sizes {
	.setup.bitmap.prefer.listbox insert end $i
    }
    .setup.bitmap.prefer.listbox selection set $prefer_index
    .setup.bitmap.prefer.listbox see $prefer_index

    #
    # Ok and Cancel buttons.
    #
    frame .setup.ok-cancel
    pack .setup.ok-cancel -side top -pady 4 -fill x

    button .setup.ok-cancel.ok -text "Ok" -command setup_ok -default active
    pack .setup.ok-cancel.ok -side left -expand 1

    button .setup.ok-cancel.cancel -text "Cancel" -command {destroy .setup}
    pack .setup.ok-cancel.cancel -side left -expand 1
}

#
# `Ok' on the setup window.
#
proc setup_ok {} {
    global max_servers
    global start_server1
    global setup_start_server1
    global server_names
    global server_ports
    global server_starts
    global prefer_sizes
    global prefer_index

    #
    # Close the current connection.
    #
    select_server 0

    #
    # Set a server list.
    #
    set server_names {}
    set server_ports {}
    set server_starts {}
    for {set i 1} {$i <= $max_servers} {incr i} {
	set hostname [.setup.servers.server$i.hostname get]
	set port [.setup.servers.server$i.port get]
	set start [.setup.servers.server$i.start get]
	regsub {[ 	:].*} $hostname {} hostname
	regsub {[ 	:].*} $port {} port
	regsub {[ 	:].*} $start {} start

	if {![regexp {^[ 	]*$} $hostname]} {
	    if {[regexp {^[ 	]*$} $port]} {
		set $port "ndtp"
	    }
	    if {[regexp {^[ 	]*$} $start]} {
		set $start ""
	    }
	    lappend server_names $hostname
	    lappend server_ports $port
	    lappend server_starts $start
	}

    }

    #
    # Whether connect to server 1 at start of BookView.
    #
    if {![regexp {^[ 	]*$} [.setup.servers.server1.hostname get]] \
	    && $setup_start_server1} {
	set start_server1 1
    } else {
	set start_server1 0
    }

    #
    # Set bitmap size preferences.
    #
    set prefer_index [.setup.bitmap.prefer.listbox curselection]

    #
    # Update `Server'.
    #
    set_server_menu

    #
    # Save a setup file.
    #
    setup_save_file

    destroy .setup
}

#
# Load a setup file.
#
proc setup_load_file {} {
    global system_setup_file
    global user_setup_file
    global server_names
    global server_ports
    global server_starts
    global start_server1
    global prefer_index
    global prefer_sizes
    global bitmap_checkbutton
    global toolbar_checkbutton
    global max_servers
    global text_spacing

    set server_names {}
    set server_ports {}
    set server_starts {}
    set start_server1 0

    #
    # Open a setup-file.
    # Try user setup file first, and then system setup file.
    #
    set setup_file $user_setup_file
    output_information "Load $setup_file..."
    if {[catch {set file [open $setup_file "r"]}]} {
	set setup_file $system_setup_file
	output_information "Load $setup_file..."
	if {[catch {set file [open $setup_file "r"]}]} {
	    output_information "Cannot open a setup file."
	    return 0
	}
    }

    #
    # Read a setup-file.
    #
    set lineno 0
    while {1} {
	if {[catch {gets $file line}]} {
	    output_information "Cannot read $setup_file."
	    catch {close $file}
	    return 0
	}
	incr lineno
	if {[regexp {^server ([^:]+):([^:]+):(.*)} $line dummy \
		hostname port start]} {
	    lappend server_names $hostname
	    lappend server_ports $port
	    lappend server_starts $start
	} elseif {[regexp {^server ([^:]+):([^:]+)} $line dummy \
		hostname port]} {
		# For BookView 2.2 and earlier versions.
	    lappend server_names $hostname
	    lappend server_ports $port
	    lappend server_starts ""
	} elseif {[regexp {^start-server1 ([01])$} $line dummy arg]} {
	    set start_server1 $arg
	} elseif {[regexp {^prefer-sizes ([0-9]+)$} $line dummy arg]} {
	    set prefer_index $arg
	} elseif {[regexp {^display-bitmaps ([0-9]+)$} $line dummy arg]} {
	    set bitmap_checkbutton $arg
	} elseif {[regexp {^display-toolbar ([0-9]+)$} $line dummy arg]} {
	    set toolbar_checkbutton $arg
	} elseif {[regexp {^text-spacing ([0-9]+)$} $line dummy arg]} {
	    set text_spacing $arg
	} elseif {![regexp {^\#} $line] && $line != ""} {
	    puts "Syntex error at line $lineno in $setup_file"
	}
	if {[eof $file]} {
	    break
	}
    }

    #
    # Close a setup-file.
    #
    output_information "Load $setup_file...done."
    catch {close $file}

    #
    # Check for the setup.
    #
    set length [llength $server_names]
    if {$max_servers < $length} {
	set server_names [lrange $server_names 0 [expr $max_servers - 1]]
	set server_ports [lrange $server_ports 0 [expr $max_servers - 1]]
	set server_starts [lrange $server_starts 0 [expr $max_servers - 1]]
    } elseif {$length == 0} {
	set start_server1 0
    }

    set length [llength $prefer_sizes]
    if {$length <= $prefer_index} {
	set prefer_index [expr $length - 1]
    }

    if {$text_spacing < 0} {
	set text_spacing 0
    }

    return 1
}

#
# Save a setup file.
#
proc setup_save_file {} {
    global user_setup_file
    global server_names
    global server_ports
    global server_starts
    global start_server1
    global prefer_index
    global bitmap_checkbutton
    global toolbar_checkbutton
    global text_spacing

    #
    # Open a setup file.
    #
    output_information "Save $user_setup_file..."
    if {[catch {set file [open $user_setup_file "w"]}]} {
	output_information "Cannot open $user_setup_file"
	return
    }

    #
    # Write to a setup file.
    #
    set length [llength $server_names]
    for {set i 0} {$i < $length} {incr i} {
	set hostname [lindex $server_names $i]
	set port [lindex $server_ports $i]
	set start [lindex $server_starts $i]
	if {[catch {puts $file "server $hostname:$port:$start"}]} {
	    output_information "Cannot write to $user_setup_file."
	    catch {close $file}
	    return
	}
    }
    if {[catch {puts $file "start-server1 $start_server1"}]} {
	output_information "Cannot write to $user_setup_file."
	catch {close $file}
	return
    }
    if {[catch {puts $file "prefer-sizes $prefer_index"}]} {
	output_information "Cannot write to $user_setup_file."
	catch {close $file}
	return
    }
    if {[catch {puts $file "display-bitmaps $bitmap_checkbutton"}]} {
	output_information "Cannot write to $user_setup_file."
	catch {close $file}
	return
    }
    if {[catch {puts $file "display-toolbar $toolbar_checkbutton"}]} {
	output_information "Cannot write to $user_setup_file."
	catch {close $file}
	return
    }
    if {[catch {puts $file "text-spacing $text_spacing"}]} {
	output_information "Cannot write to $user_setup_file."
	catch {close $file}
	return
    }
    if {[catch {flush $file}]} {
	output_information "Cannot write to $user_setup_file."
	catch {close $file}
	return
    }

    #
    # Close a setup file.
    #
    catch {close $file}
    output_information "Save $user_setup_file....done."
}

