/* piping functions for data filters and cd writing */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <fcntl.h>
#include <gtk/gtk.h>
#include <stdlib.h>

#include "piping.h"
#include "varman.h"
#include "preferences.h"
#include "rterm.h"

#define FDREAD 0
#define FDWRITE 1

/* uncomment for debugging */
/* #define DEBUG */

int piping_pidwatch_callback(gpointer data)
{
	piping_watch *w;
	int callagain;

	w=(piping_watch*)data;
	if (waitpid(w->pid,&w->status,WNOHANG)!=0) 
	  {
		  callagain=w->finishcb(w->status,w->data);
	  }
	else 
	    callagain=1;
	if (callagain==0) piping_pidwatch_remove(w);
	
	return callagain;
}
;

piping_watch *piping_pidwatch_add(int pid,
				  piping_finishcb callback,
				  gpointer data)
{
	piping_watch *w;
	
	w=(piping_watch*)malloc(sizeof(piping_watch));
	w->pid=pid;
	w->status=0;
	w->finishcb=callback;
	w->data=data;
	w->handle=gtk_timeout_add(PIPING_PIDWATCH_TIMEOUT,
				  piping_pidwatch_callback,
				  (gpointer)w);
	
	return w;
}
;

void piping_pidwatch_remove(piping_watch *w)
{
	gtk_timeout_remove(w->handle);
	free(w);
}
;

/* callback called by piping_create_function() when invoked with 
 * piping_create() */
int piping_create_callback(int inpipe,int outpipe,int errpipe,gpointer data)
{
   char *command;
   char *args[PIPING_MAX_ARGS+1];
   char *argstring;
   int  currentpos;
   int  x;
   int  specialchar;
   int  quoted;
   int  sensitive;
   
   command=(char*)data;
   argstring=(char*)malloc(strlen(command)+1);
   strcpy(argstring,command);  // make a copy first,then modify it appropriately
   
#ifdef DEBUG
   printf ("commandstring: %s\n",argstring);	    
#endif
   
	for (x=0;x<PIPING_MAX_ARGS+1;x++) /* initialize parameters */
	    args[x]=NULL;
	x=0;currentpos=0;
	quoted=0;sensitive=1;specialchar=0;
	args[x]=(char*)malloc(256);
	args[x][0]=0;
	do
	  {
		  specialchar=0;
		  if (sensitive)
		    {				      
			    switch (argstring[currentpos])
			      {
			      case '\\':
				      specialchar=1;
				      sensitive=0;
				      break;
			      case '"':
				      specialchar=1;
				      quoted=1-quoted;
				      break;
			      case ' ':
				      if ((!quoted)&&(x<PIPING_MAX_ARGS-1))
					{
						specialchar=1;
						x++;
						args[x]=(char*)malloc(256);
						args[x][0]=0;
					}
				      ;
				      break;						
			      }
			    ;
		    }
		  else 
		      sensitive=1;
		  if (!specialchar)
		      strncat(args[x],&argstring[currentpos],1);
		  currentpos++;
	  }
	while (currentpos<strlen(argstring));
	/* last entry has to be a NULL pointer */
	/* due to preinitialization this is generally the case */
	
#ifdef DEBUG
	printf ("calling with args:\n");
	for (x=0;args[x]!=NULL;x++)
	  {
		  printf ("%s\n",args[x]);
	  }
	;
#endif

	if (inpipe!=-1)
	  {		  
		  close (0);
		  dup(inpipe);
		  close(inpipe);
	  }
	;
	
	if (outpipe!=-1)
	  {		  
		  close (1);
		  dup(outpipe);
		  close(outpipe);
	  }
	;
	
	if (errpipe!=-1)
	  {		  
		  close (2);
		  dup(errpipe);
		  close(errpipe);
	  }
	;
		  
	execvp(args[0],args);
	perror ("couldnt run client");
	_exit(-1);
}
;

/* The new "light" version of piping_create(). it utilizes piping_create_function,
 * effectively contracting all fork and piping calls to a single function 
 * piping_create_function. Thus future big in the piping routines have
 * to be fixed only once 
 *    command: program to start and parameters separated by spaces
 *    *in:     thats where the filedescriptor for sending data to the 
 *             client will be written to
 *             if *in is set to something other than -1 it is assumed that
 *             that filedescriptor represented by *in is already open and
 *             should be used to read data by the client
 *    *out:    the filedescriptor with which you can obtain data from the client
 *             will be written here
 *    *err:    the same for the error channel of the client
 * any of the pointers in,out and err set to NULL means that youre not
 * interested in communicating with the client over that pipe.
 * the corresponding pipe of the client will be connected to /dev/zero in that
 * case */
int piping_create(char *command,int *in,int *out,int *err) 
{	
	return piping_create_function(GTK_SIGNAL_FUNC(piping_create_callback),
				      (gpointer)command,
				      in,out,err);				      
}
;

/* create a pipe to part of the main program (essentially a fork with 
 * a few pipes for communication with the main process) 
 * if *in,*out,*err is -1 a new pipe will be created
 * if in,out,err is NULL the corresponding pipe is connected to /dev/zero
 * if *in,*out,*err !=-1 we take that pipe as input,output or stderror 
 * if *in,*out,*err !=-1 we return -1 as there is no way of controlling
 * a direct link to a client. 
 * WARNING! the filedescriptors in *in,*out,*err
 * are closed on the callers side. */
int piping_create_function(GtkSignalFunc func,gpointer data,int *in,int *out,int *err)
{
	int inpipe[2];
	int outpipe[2];
	int errpipe[2];
	
	int childpid;
					
	if (in!=NULL)
	  {
		  /* create new input pipe only if value of in is -1 */
		  if (*in==-1)
		    {			    
			    if (pipe(inpipe)==-1)
			      {
				      perror("piping_create_func couldnt create inpipe");
				      inpipe[FDREAD]=-1;
				      inpipe[FDWRITE]=-1;
			      }
			    ;
		    }
		  else
		    {
			    inpipe[FDREAD]=*in;
			    /* if inpipe was open theres no way of
			     * controlling the data flow to the child */
			    inpipe[FDWRITE]=-1;
		    }
		  ;
	  }	
	else
	  {		  
		  inpipe[FDREAD]=open("/dev/zero",O_RDONLY);
		  inpipe[FDWRITE]=-1;
	  }
	;
	if (out!=NULL)
	  {		  
		  /* create new output pipe only if value of out is -1 */
		  if (*out==-1)
		    {			    
		       if (pipe(outpipe)==-1)
			 {
			    perror("piping_create_func couldnt create outpipe");
			    outpipe[FDREAD]=-1;
			    outpipe[FDWRITE]=-1;
			 }
		       ;
		    }
		  else
		    {
			    outpipe[FDREAD]=-1;
			    outpipe[FDWRITE]=*out;
		    }
		  ;
	     
	  }
	else
	  {		  
		  outpipe[FDREAD]=-1;
		  outpipe[FDWRITE]=open("/dev/zero",O_WRONLY);
	  }
	;
	if (err!=NULL)
	  {		  
	     /* create new error pipe only if value of out is -1 */	
	     if (*err==-1)
	       {			    
		  if (pipe(errpipe)==-1)
		    {
			    perror("piping_create_func couldnt create errpipe");
			    errpipe[FDREAD]=-1;
			    errpipe[FDWRITE]=-1;
		    }
		  ;
	       }
	     else
	       {
		  errpipe[FDREAD]=-1;		
		  errpipe[FDWRITE]=*err;
	       }	
	     ;       
	  }	
	else
	  {		  
		  errpipe[FDREAD]=-1;
		  errpipe[FDWRITE]=open("/dev/zero",O_WRONLY);
	  }
	;
	childpid=fork();
	switch (childpid)
	  {
	  case 0:
		  /* this section is a bit different to the implementation
		   * of the normal pipe function. It doesnt redirect stdin
		   * stdout and stderr but merely gives the descriptors used
		   * for communication with the same names and meanings
		   * as arguments to the function.
		   * Thus,the client function may still use the stdin,stdout
		   * and stderr of the main program while at the same time
		   * having access to the pipes created for communication
		   * with the main process. */
		  
		  if (inpipe[FDWRITE]!=-1)      /* only client side pipes are needed here */
		      close (inpipe[FDWRITE]);  
		  if (outpipe[FDREAD]!=-1)
		      close (outpipe[FDREAD]);
		  if (errpipe[FDREAD]!=-1)
		      close(errpipe[FDREAD]);

		  func(inpipe[FDREAD],
		       outpipe[FDWRITE],
		       errpipe[FDWRITE],
		       data);
		  _exit(0);
	  case -1:
		  perror ("forking error");
		  exit(-1);
	  default:
		  if (inpipe[FDREAD]!=-1)
		      close(inpipe[FDREAD]);
		  if (outpipe[FDWRITE]!=-1)
		      close(outpipe[FDWRITE]);
		  if (errpipe[FDWRITE]!=-1)
		      close(errpipe[FDWRITE]);
		  if (in!=NULL)  
		      *in=inpipe[FDWRITE];
		  if (out!=NULL) 
		      *out=outpipe[FDREAD];
		  if (err!=NULL) 
		      *err=errpipe[FDREAD];
		  return childpid;
	  }
	;
}
;

/* write the output of a command into *buf */
void piping_create_getoutput(char *command,char *buf,int maxoutput,char watchmask)
{		  
	int outp=-1,errp=-1;
	int *outpp,*errpp;
	int pid;
	int count;
	int status;
	int readresult_err;
	int readresult_out;
	
	if (watchmask&PIPING_WATCHSTDOUT)
	    outpp=&outp;
	else
	    outpp=NULL;
	
	if (watchmask&PIPING_WATCHSTDERR)
	    errpp=&errp;
	else
	    errpp=NULL;

#ifdef DEBUG
	printf ("outpp:%p errpp:%p\n",outpp,errpp);
#endif
	pid=piping_create(command,NULL,outpp,errpp);
#ifdef DEBUG
	printf ("outpp:%p errpp:%p\n",outpp,errpp);
	printf ("outp:%d errp:%d\n",outp,errp);
#endif
	count=0;
	
	if (watchmask&PIPING_WATCHSTDOUT)
	    fcntl(outp,F_SETFL,O_NONBLOCK);
	if (watchmask&PIPING_WATCHSTDERR)
	    fcntl(errp,F_SETFL,O_NONBLOCK);
	
	do
	  {
		  readresult_out=0;
		  if  (watchmask&PIPING_WATCHSTDOUT)
		    {			    
			    if ((readresult_out=read(outp,&buf[count],1))>0)
			      {			    
				      if (count<maxoutput-1)
					{			    
#ifdef DEBUG
						printf ("%c",buf[count]);
#endif
						count++;
					}
				      ;
			      }
			    ;
		    }
		  ;
#ifdef DEBUG
		  if ((readresult_out==-1) && (errno!=EAGAIN))
		      perror ("piping_create_getoutput: error while reading from stdoutpipe");
#endif
		  readresult_err=0;
		  if   (watchmask&PIPING_WATCHSTDERR)
		    {
			    if ((readresult_err=read(errp,&buf[count],1))>0)
			      {			    
				      if (count<maxoutput-1)
					{				      
#ifdef DEBUG
						printf ("%c",buf[count]);
#endif
						count++;
					}
				      ;
			      }
			    ;
		    }
		  ;
#ifdef DEBUG
		  if ((readresult_err==-1) && (errno!=EAGAIN))
		      perror ("piping_create_getoutput: error while reading from stderrpipe");
#endif
	  }
	while ((readresult_out!=0)||(readresult_err!=0));
	
	if (watchmask&PIPING_WATCHSTDOUT)
	    close(outp);
	if (watchmask&PIPING_WATCHSTDERR)
	    close(errp);

	waitpid(pid,&status,0); /* wait for the client to finish */
 
	buf[count]=0; /* terminate the output with 0 to create a valid string */
}
;
		  
			
	

		  
		  
	
	
