package Peruser;

# Simple interface to News Peruser's spool files.

use Exporter;
use Socket;

@ISA = qw( Exporter );

@EXPORT = qw( get_servers get_groups get_summary search_list search_groups
              xpat get_list renew_list sort_list safe_open safe_close
              add_server subscribe_group add_folder remove_folder
              file_article select_article get_header_line transfer_groups
              delete_article undelete_article pack_expire_groups 
              write_read_file write_request_file export_current_article 
              open_server disconnect feedback %current_article $folders
              $current_server $home $connected $newsrc $spooldir $X );

@EXPORT_OK = ();

$X = 0;
$home = $ENV{ HOME };
$spooldir = "$home/.peruser_spool";
$newsrc = "$home/.peruser4-newsrc";
$folders = "$home/.peruser4-folders";
$connected = 0;
$current_server = "";
%current_article = ();
%locks = ();

# Checks for the existence of/creates Peruser-style lockfiles. 
#
# Arguments: reference to the typeglob to contain the filehandle, filename
# Returns: 1 on success, 0 on failure

sub safe_open
{
   my ( $handle_ref, $path ) = @_;

   my $lockpath;

   $path =~ /^[>+<]*(.+)/;
   $lockpath = "$1:lock";

   local $lock_handle = \*LOCK;
   $locks{ $handle_ref } = [ $lock_handle, $lockpath ];

   if ( -e $lockpath )
   {
      my $file;
      if ( defined $1 )
      {
         $file = $1;
      }
      else
      {
         $file = $path;
      }

      feedback( "Cannot open $file: file is locked\n" );
      return 0;
   }

   unless ( open $lock_handle, ">$lockpath" )
   {
      $locks{ $handle_ref } = undef;
      feedback( "Cannot create lockfile: $path: $!\n" );
      return 0;
   }

   unless ( open $handle_ref, $path )
   {
      close LOCK;
      unlink $lockpath;
      $locks{ $handle_ref } = undef;
      return 0;
   }

   1;
}

sub safe_close
{
   my $handle_ref = shift;
   my @lock_array = @{ $locks{ $handle_ref }};

   if ( not defined @lock_array )
   {
      feedback( "No associated reference in locks hash\n" );
      return 0;
   }

   my $lock_ref = shift @lock_array;
   my $lockpath = shift @lock_array;

   close $handle_ref;
   close $lock_ref;
   unlink $lockpath;

   1;
}

# Arguments: none
# Returns: array containing configured news server hostnames

sub get_servers
{
   unless ( safe_open( \*NEWSRC, $newsrc ))
   {
      return undef;
   }

   my @servers;

   while( <NEWSRC> )
   {
      next if ( substr( $_, 0, 1 ) eq "\t" );
      chop;
      push @servers, $_;
   }

   safe_close( \*NEWSRC );

   unless ( safe_open( \*FOLDERS, "$home/.peruser4-folders" ))
   {
      return @servers;
   }

   if ( <FOLDERS> )
   {
      push @servers, 'Folders';
   }

   safe_close( \*FOLDERS );

   @servers;
}

# Arguments: server name
# Returns: array of group names subscribed from server

sub get_groups
{
   my $server = shift;
   my $len = length $server;

   my $found = 0;
   my @groups;


   if ( $server eq 'Folders' )
   {
      unless( safe_open( \*FOLDERS, $folders ))
      {
         return undef;
      }

      while( <FOLDERS> )
      {
         chop;
         push @groups, $_;
      }

      safe_close( \*FOLDERS );
   }
   else
   {
      unless ( safe_open( \*NEWSRC, $newsrc ))
      {
         return undef;
      }

      while ( <NEWSRC> )
      {
         chop;
         next if ( not ( substr( $_, 0, $len ) eq $server ));
         $found = 1;
         last;
      }

      if ( not $found )
      {
         safe_close( \*NEWSRC );
         feedback( "No such server in ~/.peruser4-newsrc: $server\n" );
         return ();
      }

      while( <NEWSRC> )
      {
         last if ( not( substr( $_, 0, 1 ) eq "\t" ));
         if ( not /^\t(.*):/ )
         {
            feedback( "malformed line in ~/.peruser4-newsrc: $_\n" );
         }

         push @groups, $1;
      }

      safe_close( \*NEWSRC );
   }

   @groups;
}

# Arguments: server name, group name, 
#            boolean indicating whether to exclude seen articles, 
#            boolean indicating whether to exclude header-only articles.
#
# Returns: reference to first element of linked list of hashes of the form:
#
#          {
#              SPOOL_NEXT => Reference to next node in spool file order. 
#              SPOOL_PREV => Reference to previous node in spool file order.
#            
#              NEXT => Reference to next node in sorted order.
#              PREV => Reference to previous node in sorted order.
#             
#              CHILD_HEAD => Reference to first node of list of follow-ups.
#              CHILD_PREV => Reference to last node of list of follow-ups.
#
#              PARENT => Reference to precursor node.
#
#              CHILD_COUNT => Number of follow-ups.
#              DESCENDENTS => Number of all descedents in thread.
#              UNSEEN_DESCENDENTS => Number of unseen descendents in thread.  
#              HEADER_DESCENDENTS => Number of header-only descendents in
#                                    thread.
#              REQUESTED_DESCENDENTS => Number of requested header-only
#                                       nodes in thread.
#
#              OFFSET => Offset, in bytes, of beginning of article in spool
#                        file.
#              SIZE => Size, in bytes, of article in spool file.
#              ORDINAL => Ordinal position of article in spool file. First
#                         article has ORDINAL zero.
#
#              SUBJECT => Contents of Subject header line of article.
#              DATE => Contents of Date header line of article.
#              FROM => Contents of From header line of article.
#              MESSAGE_ID => Contents of Message-ID: header line of
#                            article.
#              REFERENCES => Reference to array of Message-Ids of precursor
#                            articles corresponding to those listed in the
#                            article's References header.
#
#              IS_ARTICLE => Boolean indicating whether article is full
#                            article or header-only article.
#              IS_CHILD => Boolean indicator of existence of precursor node.
#              IS_UNSEEN => Boolean indicator of whether article is
#                           considered to be unseen or seen. 
#          }

sub get_summary
{
   my ( $server, $group, $hide_seen, $hide_headers ) = @_;

   my $spool = "$spooldir/$server-$group";

   my $passed_header, $subject_found, $date_found, $from_found,
      $message_id_found, $references_found, $total, $unseen, $headers,
      $requested, $old_offset, $offset, $position, $first_pass, $read,
      $line, $nodes, $threads;

   unless ( safe_open( \*SPOOL, $spool ))
   { 
      return undef; 
   }

   $read = (( -e "$spool:read" ) ? 1 : 0 );
   if ( $read )
   {
      unless ( safe_open( \*READ, "$spool:read" ))
      {
         safe_close( \*SPOOL );
         feedback( "Cannot open $spool:read: $!\n" );
         return undef;
      }
   }

   $requests = (( -e "$spool:requests" ) ? 1 : 0 );
   if ( $requests )
   {
      unless ( safe_open( \*REQUESTS, "$spool:requests" ))
      {
         safe_close( \*SPOOL );
         safe_close( \*READ );
         feedback( "Cannot open $spool:requests: $!\n" );
         return undef;
      }
   }

   $offset = 0;

ARTICLE:
   for( ; ; )
   {
      $old_offset = $offset;
      $offset = tell SPOOL;

      last if ( eof SPOOL );

      $line = <SPOOL>;
      $is_seen = ( $read ? <READ> : 'r' );

      if (( $hide_headers && substr( $line, 0, 1 ) eq '@' ) ||
          ( $hide_seen && substr( $is_seen, 0, 1 ) eq 'r' ))
      {
         while( <SPOOL> )
         {
            last if ( substr( $_, 0, 3 ) eq ".\r\n" );
         }

         next ARTICLE
      }

      if ( not defined $nodes )
      {
         $nodes->{ PREV } = $nodes->{ SPOOL_PREV } = undef;
         $nodes->{ OFFSET } = $offset;
         $threads = $nodes;
      }
      else
      {
         $nodes->{ NEXT } = {};

         $nodes->{ SPOOL_NEXT } = $nodes->{ NEXT };
         $nodes->{ NEXT }->{ PREV } = $nodes->{ NEXT }->{ SPOOL_PREV } =
            $nodes;
         $nodes->{ SIZE } = $offset - $old_offset;
         $nodes = $nodes->{ NEXT };
         $nodes->{ OFFSET } = $offset;
      }

      $nodes->{ SERVER } = $server;
      $nodes->{ GROUP } = $group;

      $nodes->{ CHILD_NEXT } = $nodes->{ CHILD_PREV } = 
         $nodes->{ CHILD_HEAD } = $nodes->{ CHILD_TAIL } = 
         $nodes->{ PARENT } = undef;

      $nodes->{ CHILD_COUNT } = $nodes->{ DESCENDENTS } = 
         $nodes->{ UNSEEN_DESCENDENTS } = $nodes->{ REQUESTED_DESCENDENTS } =
         $nodes->{ HEADER_DESCENDENTS } = $nodes->{ GAP } = 0;

      $nodes->{ ORDINAL } = $position++;
      $nodes->{ REFERENCES } = undef;

      $nodes->{ IS_UNSEEN } = 
         ( $read ? (( substr( $is_unseen, 0, 1 ) eq 'u' ) ? 1 : 0 ) : 0 );

      $nodes->{ IS_ARTICLE } = 1;
      $nodes->{ IS_CHILD } = 0;

      $first_pass = 1;

      $passed_header = $subject_found = $date_found = $from_found =
         $message_id_found = $references_found = 0;

      do
      {
LINE: 
         ( not $first_pass ) && ( $line = <SPOOL> );

         $first_pass = 0;

         if ( substr( $line, 0, 2 ) eq "\r\n" )
         {
            $passed_header = 1;
            goto LINE;
         }

         if ( not $passed_header )
         {
            if ( substr( $line, 0, 1 ) eq '@' )
            {
               $nodes->{ IS_ARTICLE } = 0;
               goto LINE;
            }

            if (( not $from_found ) && $line =~ /^From:\s*(.*)\r\n/i )
            {
               $nodes->{ FROM } = $1;
               $from_found = 1;
               goto LINE;
            }

            if (( not $subject_found ) && $line =~ /^Subject:\s*(.*)\r\n/i )
            {
               $nodes->{ SUBJECT } = $1;
               $subject_found = 1;
               goto LINE;
            }

            if (( not $date_found ) && $line =~ /^Date:\s*(.*)\r\n/i )
            {
               $nodes->{ DATE } = $1;
               $date_found = 1;
               goto LINE;
            }

            if (( not $references_found ) && $line =~ /^References:.+/i )
            {
               $references_found = 1;
               $line = substr( $line, 11 );
               $nodes->{ REFERENCES } = [ split( /\s+/, $line ) ];

               my $position = tell SPOOL;
               my $found = 0;

               while( <SPOOL> )
               {
                  if ( not( substr( $_, 0, 1 ) =~ /\s/ ))
                  {
                     seek SPOOL, $position, SEEK_SET;
                     last;
                  }

                  $nodes->{ REFERENCES } =
                     [ @{ $nodes->{ REFERENCES } }, split /\s+/, $_ ];

                  $position = tell SPOOL;
               }

               goto LINE;
            }

            if (( not $message_id_found ) && 
                  $line =~ /^Message-ID:\s*(.*)\r\n/i )
            {
               $nodes->{ MESSAGE_ID } = $1;
               $message_id_found = 1;

               if ( not $requests )
               {
                  $nodes->{ IS_REQUESTED } = 0;
                  goto LINE;
               }

               seek REQUESTS, 0, SEEK_SET;
              
               while( <REQUESTS> )
               {
                  chop;
                  if ( $_ eq $1 )
                  {
                     $nodes->{ IS_REQUESTED } = 1;
                     last;
                  }
               }

               goto LINE;
            }
         }

      }
      while( not ( substr( $line, 0, 3 ) eq ".\r\n" ));
   }

   safe_close( \*SPOOL );
   safe_close( \*READ ) if ( $read );
   safe_close( \*REQUESTS ) if ( $requests );

   $total = $unseen = $headers = $requested = 0;

   if ( defined $threads )
   {
      $nodes->{ SIZE } = $offset - $old_offset;
      $nodes->{ NEXT } = $nodes->{ SPOOL_NEXT } = undef;

      for( $nodes = $threads;
            defined $nodes; 
            $nodes = $nodes->{ NEXT } )
      {
         ++$total;

         $unseen += $nodes->{ IS_UNSEEN };
         $headers += (( $nodes->{ IS_ARTICLE } ) ? 0 : 1 );
         $requested += $nodes->{ IS_REQUESTED };
      }
   }

   my $beginning = $threads;

   if ( $total > 2 )
   {
      $threads = thread_summary( sort_summary( $threads ));
   }

   ( $total, $unseen, $headers, $requested, $beginning, $threads );
}

# Comparator used internally by module sort functions. Converts arguments
# to lower case, and removes leading "'`([ characters, and Re: from them
# before feeding them to cmp.

sub compare
{
   my $a = lc $a->{ SUBJECT };
   my $b = lc $b->{ SUBJECT };

   my $pat = '^\s*(?:["\x27\x60([]\s*)?(?:Re:\s*)?(.*)';

   $a =~ s/$pat/\1/i;
   $b =~ s/$pat/\1/i;

   $a cmp $b;
}

# Used internally by get_summary to sort completed list by Subject header
# lines.
#
# Arguments: reference to linked list create by get_summary
# Returns: reference to linked list

sub sort_summary
{
   my $threads = shift;
   my @array, $node;

   for( $node = $threads; defined $node; $node = $node->{ NEXT } )
   {
      push @array, $node;
   }

   @array = sort compare @array;

   $threads = $array[ 0 ];
   $threads->{ PREV } = undef;
   $threads->{ NEXT } = $array[ 1 ];

   my $idx;
   my $total = @array;

   for( $idx = 1; $idx < $total; ++$idx )
   {
      $array[ $idx ]->{ PREV } = $array[ $idx - 1 ];
      $array[ $idx ]->{ NEXT } = $array[ $idx + 1 ];
   }

   $array[ $total - 1 ]->{ NEXT } = undef;

   $threads;
}

# Used internally by get_summary to thread completed list by References.
#
# Arguments: reference to linked list returned by sort_summary
# Returns: reference to linked list

sub thread_summary
{
   my $threads = shift;
   my $reference, $current, $other, $gap;
   
   for( $current = $threads; 
         defined $current; 
         $current = $current->{ NEXT } )
   {
      next if ( not exists $current->{ REFERENCES } );

      $gap = 0;

      for( $reference = pop @{ $current->{ REFERENCES } };
            defined $reference;
            $reference = pop @{ $current->{ REFERENCES } } )
      {
         for( $other = $threads;
               defined $other;
               $other = $other->{ NEXT } )
         {
            next if ( $other == $current );

            if ( $other->{ MESSAGE_ID } eq $reference )
            {
               if ( not defined $other->{ CHILD_HEAD } )
               {
                  $other->{ CHILD_HEAD } = $other->{ CHILD_TAIL } =
                     $current;
               }
               else
               {
                  $current->{ CHILD_PREV } = $other->{ CHILD_TAIL };
                  $other->{ CHILD_TAIL }->{ CHILD_NEXT } = $current;

                  $current->{ CHILD_NEXT } = undef;
                  $other->{ CHILD_TAIL } = $current;
               }

               $current->{ IS_CHILD } = 1;
               $current->{ PARENT } = $other;
               $current->{ GAP } = $gap;
               ++( $other->{ CHILD_COUNT } );
               last;
            }

            last if ( $current->{ IS_CHILD } );

            ++$gap;
         }
      }
   }

   $threads;
}

# Connects filehandle SOCKET to specified host.
#
# Arguments: the news server hostname
# Returns: 1 on success, 0 on failure

sub open_server
{
   if ( $connected )
   {
      feedback( "SOCKET is already connected to a news server. " .
                "Disconnect first.\n" );
      return 0;
   }

   my $hostname = shift;
   if (( not defined $hostname ) || $hostname eq "" )
   {
      feedback( "Peruser::open_server: No hostname passed as argument\n" );
      return 0;
   }

   my $port = getservbyname( "nntp", "tcp" );
   unless( $port )
   {
      feedback( "cannot get port for nntp over tcp: $!\n" );
      return 0;
   }

   my $address = gethostbyname( $hostname );
   unless ( $address )
   {
      feedback( "Cannot get IP address for $hostname: $!\n" );
      return 0;
   }

   unless ( socket( SOCKET, AF_INET, SOCK_STREAM, 0 ))
   {
      feedback( "Cannot create socket: $!\n" );
      return 0;
   }

   my $addr_in = sockaddr_in( $port, $address ); 
   unless ( $addr_in )
   {
      feedback( "Cannot create socket address structure: $!\n" );
      return 0;
   }

   unless( connect( SOCKET, $addr_in ))
   {
      feedback( "Cannot open connection to $hostname on port $port: $!\n" );
      return 0;
   }

   select SOCKET; $| = 1; select STDOUT;

   my $response = <SOCKET>;
   if ( not substr( $response, 0, 2 ) eq '20' )
   {
      feedback( "$hostname says: $response" );
      disconnect();
      return 0;
   }
   else
   {
      if ( substr( $response, 2, 1 ))
      {
         feedback( "$hostname does not allow posting.\n" );
      }
   }

   $current_server = $hostname;
   $connected = 1;
}

# Arguments: none.
# Returns: 0 always

sub disconnect
{
   return 0 if ( not $connected );

   print SOCKET "quit\r\n";
   close SOCKET;

   $connected = 0;
   $current_server = undef;

   0;
}

# Arguments: news server hostname Returns: array of active groups for news
# server, from local storage.

sub get_list
{
   my $server = shift;

   unless ( safe_open( \*LIST, "$spooldir/$server-LIST" ))
   {
      feedback( "Cannot open $spooldir/$server-LIST for reading: $!\n" );
      return undef;
   }

   my @list;

   while( <LIST> )
   {
      chop;
      push @list, $_;
   }

   safe_close( \*LIST );

   @list;
}

# Sorts the local list of active groups for the specified news server.
#
# Arguments: new server hostname
# Returns: 1 on success, 0 on failure.

sub sort_list
{
   my $server = shift;
   my @list = get_list( $server );

   @list = sort ( @list );

   unless ( safe_open( \*LIST, ">$spooldir/$server-LIST" ))
   {
      feedback( "Cannot open $spooldir/$server-LIST for writing: $!\n" );
      return 0;
   }

   foreach( @list )
   {
      print LIST $_;
   }

   safe_close( \*LIST );

   1;
}

# Retrieves or updates, as necessary, the local list of active groups for
# a news server.
#
# Arguments: news server hostname, boolean to indicate whether to update
#            current list or to replace current list with a completely new 
#            list from the server.
# Returns: 1 on success, 0 on failure.

sub renew_list
{
   if ( not $connect )
   {
      feedback( "Not current connected to a news server\n" );
      return 0;
   }

   my ( $server, $update ) = @_;
   my $how = ( $update ? '>>' : '>' );

   unless ( safe_open( \*LIST, "$how$spooldir/$server-LIST" ))
   {
      feedback( "Cannot open $spooldir/$server-LIST: $!\n" );
      safe_close( \*LIST );
      return 0;
   }

   my $timestamp, @gmt;
   if ( $update )
   {
      if ( open( STAMP, "<$spooldir/$server-TIMESTAMP" ))
      {
         $timestamp = <STAMP>;
         safe_close( \*STAMP );
      }
      else
      {
         feedback( "Cannot open $spooldir/$server-TIMESTAMP for reading:" .
               " $!\n" );
         return 0;
      }
   }

   my $command = ( $update ? "LIST\r\n" : "NEWGROUPS $timestamp GMT\r\n" );
   my $line;

   print SOCKET $command;
   read SOCKET, $line, 1024;

   if ( $update )
   {
      if ( not( substr( $line, 0, 3 ) eq '231' ))
      {
         feedback( "$server responded to newgroups command with: $line" );
         safe_close( \*LIST );
         return 0;
      }
   }
   else
   {
      if ( not( substr( $line, 0, 3 ) eq '215' ))
      {
         feedback( "$server responded to list command with: $line" );
         safe_close( \*LIST );
         return 0;
      }
   }

   while( not ( <SOCKET> eq ".\r\n" ))
   {
      next if ( $_ eq 'junk' || $_ eq 'control' );

      if ( not( /(\w+)\s+\w+\s+(\d+)/ ))
      {
         feedback( "malformed line from server: $_" );
         safe_close( \*LIST );
         return 0;
      }

      printf LIST "%s: %ld\n", $1, $2;
   }

   safe_close( \*LIST );

   unless ( safe_open( \*TIMESTAMP, ">$spooldir/$server-TIMESTAMP" ))
   {
      return 0;
   }

   @gmt = gmtime();
   printf TIMESTAMP "%2.2u%2.2u%2.2u %2.2u%2.2u%2.2u", 
            $gmt[ 5 ], $gmt[ 4 ] + 1, $gmt[ 3 ], 
            $gmt[ 2 ], $gmt[ 1 ], $gmt[ 0 ];

   safe_close( \*TIMESTAMP );

   1;
}

# Searches the local list of active groups for a news server for matches on a
# specified regular expression.
#
# Arguments: news server hostname, regular expression
# Returns: array of matching newsgroup names

sub search_list
{
   my ( $server, $pattern ) = @_;

   unless( safe_open( \*LIST, "$spooldir/$server-LIST" ))
   {
      feedback( "Cannot open $spooldir/$server-LIST for reading: $!\n" );
      return undef;
   }

   my @results;

   while( <LIST> )
   {
      $_ =~ s/(.+):/\1/;
      push @results, $_ if ( /$pattern/ );
   }

   safe_close( \*LIST );

   @results;
}

# Searches spool file for matches on a specified regular expression.
#
# Arguments: news server hostname, reference to array of group names, regular
#            expression. 
#            If server is undefined affected scope is broadened
#            to include all groups of all servers. If server is defined 
#            but groups array reference isn't, search scope is broadened 
#            to include all groups of specified server.
# Returns: an array of references to arrays describing matches, each 
#          containing: the news server hostname, the group name,
#          ordinal position of the article containing the match,
#          the line number of the matching line, and the matching line 
#          itself.

sub search_groups
{
   my ( $server, $groups, $pattern ) = @_;
   my @results;
   my @groups;

   if ( not defined $server )
   {
      my @servers = get_servers();

      foreach $server ( @servers )
      {
         @groups = get_groups( $server );

         foreach $group ( @groups )
         {
            push @results,
               [ $server, $group, search_spool( $server, $group, $pattern ) ];
         }
      }
   }
   else
   {
      if ( not defined $groups )
      {
         @groups = get_groups( $server );
      }
      else
      {
         @groups = @$groups;
      }

      foreach $group ( @groups )
      {
         push @results,
            [ $server, $group, search_spool( $server, $group, $pattern ) ];
      }
   }

   @results;
}

# Used internally by search groups to search individual spool files.
#
# Arguments: news server hostname, group name, regular expression
# Returns: array of references to arrays of match info of the form:
#          [ ( ordinal posisition of article containing match ), 
#            ( line number of matching line ), ( matching line ) ]

sub search_spool
{
   my ( $server, $group, $pattern ) = @_;

   unless ( safe_open( \*SPOOL, "$spooldir/$server-$group" ))
   {
      feedback( "Cannot open $spooldir/$server-$group for reading: $!\n" );
      return undef;
   }

   my ( $article, $line ) = ( 0, 0 );

   while( <SPOOL> )
   {
      if ( $_ eq ".\r\n" )
      {
         $line = 0;
         ++$article;
         next;
      }

      if ( /$pattern/ )
      {
         push @results, [ $article, $line, $_ ];
      }

      ++$line;
   }

   safe_close( \*SPOOL );

   @results;
}

# Gives xpat command to SOCKET with specified arguments.
#
# Arguments: group name, header to be searched, wildmat pattern
# Returns: hash of server responses. Keys are article numbers, values
#          are the corresponding matching header lines.

sub xpat
{
   if ( not $connected )
   {
      feedback( "SOCKET is not currently connected to a news server\n" );
      return undef;
   }

   my ( $group, $header, $pattern ) = @_;

   print SOCKET "group $group\r\n";
   my $response = <SOCKET>;

   if ( not( substr( $response, 0, 3 ) eq '211' ))
   {
      feedback( "$current_server responded to \" )group $group\\r\\n\" with " . 
         "$response\n" );
      return undef;
   }

   if ( not( $response =~ /\d\d\d +\d+ +(\d+) +(\d+)/ ))
   {
      feedback( "malformed server response: $response" );
      return undef;
   }

   print SOCKET "xpat $header $1-$2 $pattern\r\n";
   $response = <SOCKET>;

   if ( not( substr( $response, 0, 3 ) eq '221' ))
   {
      feedback( "$current_server responded to \"xpat $header $1-$2 " . 
         "$pattern\\r\\n\" with $response\n" );
      return undef;
   }

   my %results;

   while( <SOCKET> )
   {
      last if ( $_ eq ".\r\n" );
      chop; chop;
      my @array = split;
      my $key = shift @array;
      $results{ $key } = "@array";
   }

   %results;
}

# Adds a news server hostname to ~/.peruser4-newsrc
#
# Arguments: news server hostname
# Returns: 1 on sucess; 0 on failure

sub add_server
{
   my $server = shift;

   unless ( safe_open( \*NEWSRC, "+<$newsrc" ))
   {
      feedback( "Cannot open $newsrc for reading and writing: $!\n" );
      return 0;
   }

   while( <NEWSRC> )
   {
      chop;
      if ( $server eq $_ )
      {
         feedback( "$server is already listed in $newsrc\n" );
         safe_close( \*NEWSRC );
         return 0;
      }
   }

   print NEWSRC "$server\n";
   safe_close( \*NEWSRC );

   1;
}

# Adds group to list of subscribed groups for specified server, in
# ~/.peruser4-newsrc
#
# Arguments: news server hostname, group name
# returns: 1 on success, 0 on error

sub subscribe
{
   my ( $server, $group ) = @_;

   unless( safe_open( \*NEWSRC, "$newsrc" ))
   {
      feedback( "Cannot open $newsrc for reading: $!\n" );
      return 0;
   }

   my $found = 0;

   while( <NEWSRC> )
   {
      chop;
      if ( $_ eq $server )
      {
         $found = 1;
         last;
      }
   }

   if ( not $found )
   {
      feedback( "$server is not listed in $newsrc\n" );
      safe_close( \*NEWSRC );
      return 0;
   }

   unless ( seek NEWSRC, 0, SEEK_SET )
   {
      feedback( "Cannot seek to beginning of $newsrc: $!\n" );
      safe_close( \*NEWSRC );
      return 0;
   }

   unless ( safe_open( \*TEMP, ">$newsrc:tmp" ))
   {
      feedback( "Cannot open temporary file for writing: $!\n" );
      safe_close( \*NEWSRC );
      return 0;
   }

   while( <NEWSRC> )
   {
      print TEMP $_;
      chop;
      last if ( $_ eq $server );
   }

   print TEMP "\t$group:\n";

   while( <NEWSRC> )
   {
      print TEMP $_;
   }

   safe_close( \*TEMP );
   safe_close( \*NEWSRC );

   unless ( rename "$newsrc:tmp", $newsrc )
   {
      feedback( "Cannot rename $newsrc:tmp to $newsrc: $!\n" );
      return 0;
   }

   1;
}

# Adds a folder name to ~/.peruser4-folders
#
# Arguments: folder name
# Returns: 1 on success, 0 on failure

sub add_folder
{
   my $folder = shift;

   if ( $folder !~ /(?:\w+|[-_.]+)+/ )
   {
      feedback( "Folder names can only contain alphanumerics " .
            "and '-', '_', and '.' characters.\n" );
      return 0;
   }

   unless ( safe_open( \*FOLDERS, "+<$folders" ))
   {
      feedback( "Cannot open $folders for reading and writing: $!\n" );
      return 0;
   }

   while( <FOLDERS> )
   {
      chop;
      if ( $_ eq $folder )
      {
         feedback( "$folder is already listed in $folders\n" );
         safe_close( \*FOLDERS );
         return 0;
      }
   }

   print FOLDERS "$folder\n";
   safe_close( \*FOLDERS );

   1;
}

# Removes folder name from ~/.peruser4-folders.
#
# Arguments: folder name
# Returns: 1 on success, 0 on failure

sub remove_folder 
{ 
   my $folder = shift;

   unless ( safe_open( \*FOLDERS, "$folders" ))
   {
      feedback( "Cannot open $folders for reading: $!\n" );
      return 0;
   }

   unless ( safe_open( \*TEMP, ">$folders:tmp" ))
   {
      safe_close( \*FOLDERS );
      feedback( "Cannot open $folders:tmp for writing: $!\n" );
      return 0;
   }

   while( <FOLDERS> )
   {
      chop;
      next if ( $_ eq $folder );
      print TEMP "$_\n";
   }

   safe_close( \*FOLDERS );
   safe_close( \*TEMP );

   rename "$folders:tmp", $folders;

   1;
}

# Copies specified article to specified folders.
#
# Arguments: news server hostname, group name, 
#            offset of article, size of article, folder name
# Returns: 1 on success, 0 on failure

sub file_article
{
   my ( $server, $group, $offset, $size, $folder ) = @_;

   unless ( safe_open( \*SPOOL, "$spooldir/$server-$group" ))
   {
      feedback( "Cannot open $spooldir/$server for reading: $!\n" );
      return 0;
   }

   unless ( seek SPOOL, $offset, SEEK_SET )
   {
      feedback( "Cannot seek to $offset in $spooldir/$server-$group: $!\n" );
      return 0;
   }

   my $text;
   unless ( read SPOOL, $text, $size )
   {
      feedback( "Cannot read article at $offset in $spooldir/" .
            "$server-$group: $!\n" );
      return 0;
   }

   safe_close( \*SPOOL );

   unless ( safe_open( \*SPOOL, ">>$spooldir/Folders-$folder" ))
   {
      feedback( "Cannot open $spooldir/Folders-$folder for appending: $!\n" );
      return 0;
   }

   print SPOOL $text;
   safe_close( \*SPOOL );

   1;
}

# Loads article into %current_article. Two keys are defined:
#     HEADER => complete header text
#     BODY => complete body text
# 
# Arguments: news server hostname, group name, 
#            ordinal position of article, offset of article, size of article
#
#            If ordinal is defined, then offset and size are ignored.
#            If ordinal is undefined, then offset and size are used.
#
# Returns: 1 on success, 0 on failure

sub select_article
{
   my ( $server, $group, $ordinal, $offset, $size, ) = @_;

   my $text;

   unless ( safe_open( \*SPOOL, "$spooldir/$server-$group" ))
   {
      return 0;
   }

   if ( defined $ordinal )
   {
      if ( $ordinal )
      {
         while( <SPOOL> )
         {
            --$ordinal if ( $_ eq ".\r\n" );
            last if ( not $ordinal );
         }
      }

      while( <SPOOL> )
      {
         last if ( $_ eq ".\r\n" );
         $text .= $_;
      }
   }
   else
   {
      unless ( seek SPOOL, $offset, SEEK_SET )
      {
         feedback( "Cannot seek to $offset in $spooldir/$server-$group:" .
               " $!\n" );
         return 0;
      }

      unless ( read SPOOL, $text, $size )
      {
         feedback( "Cannot read $size bytes from $spooldir/$server-$group:" .
               "$!\n" );
         safe_close( \*SPOOL );
         return 0;
      }
   }

   safe_close( \*SPOOL );

   @parts = split /\r\n\r\n/, $text, 2;
   $current_article{ HEADER } = $parts[ 0 ] . "\r\n";
   
   if ( defined $parts[ 1 ] )
   {
      $parts[ 1 ] =~ s/\.\r\n//;
      $current_article{ BODY } = $parts[ 1 ];
   }

   1;
}

# Retrieves contents of specified header line from %current_article. Only
# returns first line of multiple line headers. 
#
# Arguments: header keyword, capitalized, without trailing colon
# Returns: contents of header line without keyword, colon, or trailing \r\n.

sub get_header_line
{
   my $keyword = shift;

   if ( not exists $current_article{ HEADER } )
   {
      feedback( "\%current_article is empty\n" );
      return undef;
   }

   return $1 if ( $current_article{ HEADER } =~ /.*keyword:(.*)\r\n/ );

   undef;
}

# Transfers articles to and from news servers using nptransfer.
#
# Arguments: what to transfer: headers, articles, or requests,
#            quantity (zero means all new), news server hostname, 
#            reference to array of group names.
#            If server is undefined affected scope is broadened
#            to include all groups of all servers. If server is defined 
#            but groups array reference isn't, search scope is broadened 
#            to include all groups of specified server.
# Returns: exit status of nptransfer

sub transfer_groups
{
   my ( $what, $quantity, $server, $groups ) = @_;

   unless ( open PIPE, "| nptransfer -t $what -q $quantity 2>&1 >/dev/null" )
   {
      feedback( "Cannot open subprocess: $!\n" );
      return undef;
   }

   my @groups;

   if ( not defined $server )
   {
      my @servers = get_servers();

      foreach $server ( @servers )
      {
         last if ( $server eq 'Folders' );

         my @groups = get_groups( $server );

         foreach $group ( @groups )
         {
            print PIPE "$server:$group\n";
         }
      }
   }
   else
   {
      if ( not defined $groups )
      {
         @groups = get_groups( $server );
      }
      else
      {
         @groups = @$groups;
      }

      foreach ( @groups )
      {
         print PIPE "$server:$_\n";
      }
   }

   close PIPE;
   $?
}

# Deletes an article from a spool file and places it into the DELETE 
# file.
#
# Arguments: news server hostname, group name, 
#            article offset, article size, contents of Subject header,
#            contents of From header, contents of Date header.
#            This information can be supplied by a node in
#            the linked list of hashes created by get_summary,
#            corresponding to the article to be deleted.
# Returns: 1 on success, 0 on failure

sub delete_article
{
   my ( $server, $group, $offset, $size, $subject, $from, $date ) = @_;

   my $spool = "$spooldir/$server-$group";
   unless ( safe_open( \*SPOOL, $spool ))
   {
      feedback( "Cannot open $spool for reading: $!\n" );
      return 0;
   }

   my $temp = "$spool:tmp";
   unless ( safe_open( \*TEMP, "+>$temp" ))
   {
      safe_close( \*SPOOL );
      feedback( "Cannot open $temp for writing: $!\n" );
      return 0;
   }

   $how = (( -e "$spooldir/DELETED" ) ? '>>' : '>' );

   unless ( safe_open( \*DELETED, "$how$spooldir/DELETED" ))
   {
      safe_close( \*SPOOL );
      safe_close( \*TEMP );
      feedback( "Cannot open $spooldir/DELETED: $!\n" );
      return 0;
   }

   unless ( seek SPOOL, $offset, SEEK_SET )
   {
      safe_close( \*SPOOL );
      safe_close( \*TEMP );
      safe_close( \*DELETED );
      feedback( "Cannot seek to $offset in $spooldir/$server-$group: $!\n" );
      return 0;
   }

   my $text;
   my $count = read SPOOL, $text, $size;
   unless ( $count )
   {
      feedback( "Cannot read $size bytes from $offset in" .
         " $spooldir/$server-$group: $!\n" );
      safe_close( \*SPOOL );
      safe_close( \*TEMP );
      safe_close( \*DELETED );
      return 0;
   }

   print DELETED "$server\n$group\n$subject\n$from\n$date\n";
   print DELETED $text;
   $text = undef;
   safe_close( \*DELETED );

   while( <SPOOL> )
   {
      print TEMP $_;
   }

   safe_close( \*SPOOL );

   unless ( truncate $spool, $offset )
   {
      safe_close( \*TEMP );
      safe_close( \*DELETED );
      feedback( "Cannot truncate $spooldir/$server-$group at $offset: $!\n" );
      return 0;
   }

   unless ( safe_open( \*SPOOL, ">>$spooldir/$server-$group" ))
   {
      safe_close( \*TEMP );
      safe_close( \*DELETED );
      feedback( "Cannot open $spooldir/$server-$group for appending: $!\n" );
      return 0;
   }

   unless ( seek TEMP, 0, SEEK_SET )
   {
      safe_close( \*TEMP );
      safe_close( \*SPOOL );
      feedback( "Cannot seek to beginning of $spooldir/$server-$group:tmp" .
            ": $!\n" );
      return 0;
   }

   while( <TEMP> )
   {
      print SPOOL $_;
   }

   safe_close( \*TEMP );
   safe_close( \*SPOOL );

   rename $temp, $spool;
   unlink $temp;

   1;
}

# Removes specified article from the DELETED file and puts it back into
# the spool file from whence it was deleted.
#
# Arguments: the ordinal position of the article in the DELETED file
# Returns: 1 on success, 0 on failure

sub undelete_article
{
   my $ordinal = shift;

   unless ( safe_open( \*DELETED, "$spooldir/DELETED" ))
   {
      feedback( "Cannot open $spooldir/DELETED for reading and writing: $!\n" );
      return 0;
   }

   my $line;

   while( $ordinal-- )
   {
      do
      {
         $line = <DELETED>;
      }
      while( not( $line eq ".\r\n" ));
   }

   my $position = tell DELETED;

   my $server = <DELETED>;
   my $group = <DELETED>;

   <DELETED>; <DELETED>; <DELETED>;

   $line = undef;
   my $text;

   do
   {
      $line = <DELETED>;
      $text .= $line;
   }
   while( not( $line eq ".\r\n" ));

   unless ( safe_open( \*TEMP, ">$spooldir/DELETED:tmp" ))
   {
      safe_close( \*DELETED );
      feedback( "Cannot open $spooldir/DELETED:tmp for writing: $!\n" );
      return 0;
   }

   while( <DELETED> )
   {
      print TEMP $_;
   }

   safe_close( \*TEMP );
   safe_close( \*DELETED );

   unless ( truncate DELETED, $position )
   {
      feedback( "Cannot truncate $spooldir/DELETED to $postion: $!\n" );
      return 0;
   }

   unless ( safe_open( \*DELETED, ">>$spooldir/DELETED" ))
   {
      feedback( "Cannot open $spooldir/DELETED for appending: $!\n" );
      return 0;
   }

   unless ( safe_open( \*TEMP, "$spooldir/DELETED:tmp" ))
   {
      safe_close( \*DELETED );
      feedback( "Cannot open $spooldir/DELETED:tmp for reading: $!\n" );
      return 0;
   }

   while( <TEMP> )
   {
      print DELETED $_;
   }
   safe_close( \*TEMP );
   safe_close( \*DELETED );
   unlink "$spooldir/DELETED:tmp";

   unless ( safe_open( \*SPOOL, ">>$spooldir/$server-$group" ))
   {
      feedback( "Cannot open $spooldir/$server-$group for appending: $!\n" );
      return 0;
   }

   unless ( safe_open( \*READ, ">>$spooldir/$server-$group:read" ))
   {
      safe_close( \*SPOOL );
      feedback( "Cannot open $spooldir/$server-$group:read for appending: " . 
            "$!\n" );
      return 0;
   }

   print SPOOL $text;
   print READ "r\n";

   safe_close( \*SPOOL );
   safe_close( \*READ );

   1;
}

# Packs or Expires selected groups using npsepax.
#
# Arguments: what to do: pack or expire, news server hostname,
#            reference to array of groups to act on.
#            If server is undefined affected scope is broadened
#            to include all groups of all servers. If server is defined 
#            but groups array reference isn't, search scope is broadened 
#            to include all groups of specified server.
# Returns: exit status of npsepax

sub pack_expire_groups
{
   my $what = shift;
   my $server = shift;
   my @groups = @_;

   unless ( open CHILD, "| npsepax --$what 2>&1 >/dev/null" )
   {
      feedback( "Cannot open npsepax process for piping: $!\n" );
      return 0;
   }

   if ( defined $server )
   {
      if ( not defined @groups )
      {
         @groups = get_groups( $server );
      }

      foreach ( @groups )
      {
         print CHILD "$server:$_\n";
      }
   }
   else
   {
      my @servers = get_servers();

      foreach $server ( @servers )
      {
         @groups = get_groups( $server );

         foreach ( @groups )
         {
            print CHILD "$server:$group\n";
         }
      }
   }

   $?;
}

# Traverses linked list of hashes returned by get_summary, and writes
# a new :read file for the specified group, based on the IS_UNSEEN hash
# values.
#
# Arguments: news server hostname, group name, reference to list
# Returns: 1 on success, 0 on failure

sub write_read_file
{
   my ( $server, $group, $ref ) = @_;

   unless ( safe_open( \*READ, ">$spooldir/$server-$group:read" ))
   {
      feedback( "Cannot open $spooldir/$server-$group:read for " .
            "writing: $!\n" );
      return 0;
   }

   for( ; defined $ref; $ref = $ref->{ SPOOL_NEXT } )
   {
      print READ ( $ref->{ IS_UNSEEN } ? "u\n" : "r\n" );
   }

   safe_close( \*READ );

   1;
}

# Traverses linked list of hashes returned by get_summary, and writes
# a new :requests file for the specified group, based on the 
# IS_REQUESTED hash values.
#
# Arguments: news server hostname, group name, reference to list
# Returns: 1 on success, 0 on failure

sub write_requests_file
{
   my ( $server, $group, $ref ) = @_;

   unless ( safe_open( \*REQUESTS, ">$spooldir/$server-$group:requests" ))
   {
      feedback( "Cannot open $spooldir/$server-$group:requests for writing: " .
            "$!\n" );
      return 0;
   }

   for( ; defined $ref; $ref = $ref->{ SPOOL_NEXT } )
   {
      if ( $ref->{ IS_REQUESTED } )
      {
         print REQUESTS "$ref->{ MESSAGE_ID }\n";
      }
   }

   safe_close( \*REQUESTS );

   1;
}

sub export_current_article
{
   my $append = shift;

   if ( not defined $current_article{ HEADER } )
   {
      feedback( "No article is currently selected\n." );
      return 0;
   }

   my $how = ( $append ? ">>" : ">" );

   unless ( safe_open( \*FILE, "$how$spooldir/.current_article" ))
   {
      return 0;
   }

   print FILE $current_article{ HEADER };
   if ( defined $current_article{ BODY } )
   {
      print FILE "\r\n", $current_article{ BODY };
   }

   safe_close( \*FILE );

   1;
}

# Arguments: text to display
# Returns: 1 always.

sub feedback
{
   my $text = shift;

   $text =~ s/('|")//g;
   $text =~ s/\n/\\n/g;

   $X ? system( "npfeedback '$text'" ) : warn "$text\n";

   1;
}

sub show
{
   my $text = shift;

   $text =~ s/('|")//g;
   $text =~ s/\n/\\n/g;

   $X ? system( "npshow '$text'" ) : warn "$text\n";

   1;
}

unless ( open CHILD, "ps aux |" )
{
   feedback( "Cannot fork ps\n" );
}
else
{
   while( <CHILD> )
   {
      if ( $_ =~ /.*xinit.*/ )
      {
         $X = 1;
         last;
      }
   }

   close CHILD;
}

1;
