/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include "article.h"
#include "article-db.h"
#include "date.h"
#include "debug.h"
#include "log.h"
#include "queue-item-headers.h"

#include "articlelist.h"
#include "grouplist.h"
#include "util.h"

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

static gint queue_item_headers_run (QueueItem* item);

static gchar* queue_item_headers_describe (const StatusItem *item);

static int nntp_articlelist_download (QueueItemHeaders* item, int *count, GSList** setme);

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PUBLIC ROUTINES
************/

/*****
******
*****/

PanObject*
queue_item_headers_new (server_data *sdata,
	                gboolean high_priority,
			group_data *gdata,
			HeaderDownloadType dl_type)
{
	QueueItemHeaders *item = g_new0 (QueueItemHeaders, 1);
	gboolean need_socket = dl_type != HEADERS_LOCAL;

	debug (DEBUG_PAN_OBJECT, "queue_item_headers_new: %p", item);

	queue_item_constructor (QUEUE_ITEM(item),
		queue_item_destructor,
		queue_item_headers_describe,
		queue_item_headers_run, sdata, high_priority, need_socket);

	item->gdata = gdata;
	item->download_type = dl_type;

	return PAN_OBJECT(item);
}


/*****
******
*****/

static int
pp_adata_message_id_comp (const void* va, const void* vb )
{
	const article_data* a = *(const article_data**)va;
	const article_data* b = *(const article_data**)vb;
	return strcmp ( a->message_id, b->message_id );
}

static gint
queue_item_headers_run (QueueItem* item)
{
	QueueItemHeaders *item_h = QUEUE_ITEM_HEADERS(item);
	int count = 0;
	GSList *alist = NULL;
	GSList *l = NULL;
	GSList *old_alist = NULL;
	article_db db = NULL;
	group_data *gdata = NULL;
	server_data *sdata = NULL;
	article_data **old_articles = NULL;
	int old_article_qty = 0;
	int new_article_qty = 0;
	gboolean active;
	int i;

	if (item_h->download_type == HEADERS_LOCAL) {
		articlelist_load (
			item->sdata, item_h->gdata, STATUS_ITEM(item));
		return 0;
	}

	sdata = item->sdata;
	gdata = item_h->gdata;

	/* get the article headers... */
	i = nntp_articlelist_download (
		QUEUE_ITEM_HEADERS(item), &count, &alist);
	if (i)
		return i;

	/* no articles! */
	if (!alist) {
		status_item_emit_status (STATUS_ITEM(item),
			_("No articles retrieved for group %s"),
			gdata->name);
		return 0;
	}

	log_add_va (
		_("Merging %d article headers with old headers from %s"),
		count,
		gdata->name);

	active = sdata==articlelist_get_current_server() &&
	         gdata==articlelist_get_current_group();


	/* merge with existing articles (1) calculate # of steps */
	db = ahdb_ref (sdata, gdata);
	old_article_qty = ahdb_length (db);
	new_article_qty = g_slist_length (alist);
	i = new_article_qty; /* i is max number of articles when all done */
	if (QUEUE_ITEM_HEADERS(item)->download_type != HEADERS_ALL)
		i += old_article_qty;
	status_item_emit_init_steps (STATUS_ITEM(item),
		old_article_qty       /* load old headers */
		+ new_article_qty     /* merge states */
		+ (i)                 /* worst-case on save */
		+ (active?(i*3):0));  /* worse-case on alist_set_contents */

	/* merge with existing articles (2) load old articles.
	   We put them in an array sorted by message-id so that
	   merging time will be o(n log n) instead of o(n^2) */
	status_item_emit_status (STATUS_ITEM(item), _("Loading Old Headers"));
	old_articles = g_new0 (article_data*, old_article_qty);
	old_alist = ahdb_get_all (db, status_item_next_step_gfunc, item);
	for (i=0, l=old_alist; l!=NULL; l=l->next)
		old_articles[i++] = ARTICLE_DATA(l->data);
	g_assert (i<=old_article_qty);
	old_article_qty = i; // FIXME: != because get_all does expiration
	qsort (old_articles, old_article_qty,
		sizeof(article_data*),
		pp_adata_message_id_comp);
	g_slist_free (old_alist);
	i = 0;
	old_alist = NULL;

	/* merge with existing articles (3) merge states.
	   This is so that read messages don't suddenly become unread,
	   and so forth. */
	status_item_emit_status (STATUS_ITEM(item), _("Merging Headers"));
	for (l=alist; l!=NULL; l=l->next) {
		gboolean exact = FALSE;
		article_data* adata = ARTICLE_DATA(l->data);
		int index = lower_bound (&adata,
			old_articles, old_article_qty,
			sizeof(article_data*),
			pp_adata_message_id_comp,
			&exact);
		if (exact) { /* got a match! */
			adata->state = old_articles[index]->state;
			article_free (old_articles[index]);
			old_articles[index] = NULL; //ccc
			array_shrink (old_articles, index,
				sizeof(article_data*), old_article_qty--);
		}
		status_item_emit_next_step (STATUS_ITEM(item));
	}

	/* merge with existing articles (4) get list of all existing articles */
	for (i=0; i<old_article_qty; i++) {
		if (QUEUE_ITEM_HEADERS(item)->download_type == HEADERS_ALL) {
			article_free (old_articles[i]);
		} else {
			alist = g_slist_prepend (alist, old_articles[i]);
		}
	}
	g_free (old_articles);
	old_articles = NULL;
	old_article_qty = 0;

	/* merge with existing articles (5) update the database */
	for (new_article_qty=i=0, l=alist; l!=NULL; l=l->next) {
		++new_article_qty;
		if (article_flag_on (ARTICLE_DATA(l->data), STATE_READ))
			++i;
	}
	ahdb_erase_all (db);
	group_set_attrib_i (sdata, gdata, "Read", i);
	group_set_attrib_i (sdata, gdata, "Total", new_article_qty);
	status_item_emit_status (STATUS_ITEM(item), _("Saving Headers"));
	ahdb_save_all (db, alist, status_item_next_step_gfunc, item);

	/* merge with existing articles (5) update UI */
	grouplist_update_row (gdata);
	if (active) { /* articlelist assumes ownership, so don't free */
		articlelist_set_contents (
			sdata, gdata,
			alist, new_article_qty,
			STATUS_ITEM(item));
	}
	else {
		g_slist_foreach (alist, article_free_gfunc, NULL);
		g_slist_free (alist);
	}

	/* cleanup */	
	status_item_emit_progress (STATUS_ITEM(item), 0);
	ahdb_sync (db);
	ahdb_unref (db);

	return 0;
}

/*****
******
*****/

static int
nntp_articlelist_download (QueueItemHeaders* item, int *count, GSList **setme)
{
	PanSocket *sock = QUEUE_ITEM(item)->sock;
	server_data *sdata = QUEUE_ITEM(item)->sdata;
	group_data *gdata = QUEUE_ITEM_HEADERS(item)->gdata;
	HeaderDownloadType dl_type = QUEUE_ITEM_HEADERS(item)->download_type;

	const char *buffer = NULL;
	article_data *adata = NULL;
	gint total, total_new;
	int steps=0;
	int first=0, last=0, p_last=0;
	GSList *alist = NULL;
	gboolean read_status = 0;

	/* sanity checks */
	g_return_val_if_fail (sock!=NULL, -1);
	g_return_val_if_fail (sdata!=NULL, -1);
	g_return_val_if_fail (gdata!=NULL, -1);
	g_return_val_if_fail (count!=NULL, -1);

	/* change to the right group */
	if (pan_socket_putline_va (sock, "GROUP %s\r\n", gdata->name)) {
		queue_item_emit_sockwrite_err (QUEUE_ITEM(item));
		return QUEUE_ITEM_STATUS_ESOCKET;
	}
	if (pan_socket_getline (sock, &buffer)) {
		queue_item_emit_sockread_err (QUEUE_ITEM(item));
		return QUEUE_ITEM_STATUS_ESOCKET;
	}
	if (strtol(buffer, NULL, 10) != 211) {
		status_item_emit_error (STATUS_ITEM(item),
			_("(Couldn't select group '%s'"), gdata->name);
		return -1;
	}
	sscanf (buffer, "%*d %d %d %d", &total, &first, &last);

	p_last = group_get_attrib_i (sdata, gdata, "Last");

	total_new = last - p_last;

	if (total == '0') {
		status_item_emit_status (STATUS_ITEM(item),
			_("No articles found for group '%s'"), gdata->name);
		return -1;
	}

	if ((dl_type == HEADERS_NEW) && p_last)
	{
		if (p_last == last)
		{
			status_item_emit_status (STATUS_ITEM(item),
				_("No new articles found for group '%s'"),
				gdata->name);
			*setme = NULL;
			return 0;
		}
		else
		{
			status_item_emit_status (STATUS_ITEM(item),
				_("Getting new article headers..."));

			if (pan_socket_putline_va (sock,
					"XOVER %d-%d\r\n", p_last, last))
			{
				queue_item_emit_sockwrite_err(QUEUE_ITEM(item));
				return QUEUE_ITEM_STATUS_ESOCKET;
			}
		}
	}
	else
	{
		if ((total > 150) && ((dl_type==HEADERS_SAMPLE)))
		{
			status_item_emit_status (STATUS_ITEM(item),
				_("Getting sample"));
			last = first + 150;
		}
		else
		{
			status_item_emit_status (STATUS_ITEM(item),
				_("Getting all article headers"));
		}

		if (pan_socket_putline_va (sock,"XOVER %d-%d\r\n",first,last))
		{
			queue_item_emit_sockwrite_err (QUEUE_ITEM(item));
			return QUEUE_ITEM_STATUS_ESOCKET;
		}
	}

	status_item_emit_status (STATUS_ITEM(item),
		_("Waiting for Response from Server..."));

	if (pan_socket_getline (sock, &buffer)) {
		queue_item_emit_sockread_err (QUEUE_ITEM(item));
		return QUEUE_ITEM_STATUS_ESOCKET;
	}

	if (strtol(buffer, (char **) NULL, 10) != 224) {
		status_item_emit_error (STATUS_ITEM(item),
			_("Couldn't retrieve headers\nServer said: %s"), buffer);
		return -1;
	}

	steps = ((dl_type==HEADERS_NEW) && (group_get_attrib_i (sdata, gdata,"Total")>0))
		? total_new
		: total;
	status_item_emit_init_steps (STATUS_ITEM(item), steps);

	while (!QUEUE_ITEM(item)->abort && !((read_status = pan_socket_getline (sock, &buffer))))
	{
		int i = 0;
		int line_qty = 0;
		int part = 0;
	        int parts = 0;
		gchar **sd = NULL;
		gchar *s = NULL;

		/* handle end-of-list */
		if (!strncmp(buffer, ".\r\n", 3))
			break;

		/* split the header */
		sd = g_strsplit (buffer, "\t", 8);

		/* check for incomplete header */
		line_qty = 0;
		while ( sd[line_qty]!=NULL )
			++line_qty;
		if (line_qty<8) {
			g_message ( "Corrupt Header");
			status_item_emit_error (STATUS_ITEM(item),
				_("Corrupt header ``%s'' skipped"),buffer);
			g_strfreev (sd);
			continue;
		}

		/* create the article data */
		adata = article_new ();
		adata->number = g_strdup(sd[0]);
		adata->subject = g_strdup(sd[1]);
		adata->author = g_strdup(sd[2]);
		adata->date = parse_date (sd[3]);
		adata->message_id = g_strdup(sd[4]);
		adata->references = g_strdup(sd[5]);
		adata->linecount = strtol(sd[7], NULL, 10);
		adata->state = STATE_NEW;

		/* cleanup */
		g_strfreev (sd);

		/* Look for the (n/N) or [n/N] construct in subject lines,
		 * starting at the end of the string and working backwards */
		part = 0;
		parts = 0;
		s = adata->subject;
		for (i = strlen(s) - 1; i > 0; --i) {
			if ((s[i] == ')') || (s[i] == ']')) {
				for (--i; isdigit((int)s[i]); --i);
				if (s[i] == '/') {
					parts = atoi(s + i + 1);
					for (--i; isdigit((int)s[i]); --i);
					if ((s[i] == '(') || (s[i] == '[')) {
						part = atoi(s + i + 1);
						break;
					}
				}
			}
		}

		/* if not a multipart yet, AND if it's a big message, AND
		   it's either in one of the pictures/fan/sex groups or it
		   has commonly-used image names in the subject, guess it's
		   a single-part binary */
		if (!parts
			&& adata->linecount>400
			&& (((strstr(gdata->name,"pictures")
					|| strstr(gdata->name,"fan")
					|| strstr(gdata->name,"sex")))
				|| ((strstr(s,".jpg") || strstr(s,".JPG")
					|| strstr(s,".jpeg") || strstr(s,".JPEG")
					|| strstr(s,".gif") || strstr(s,".GIF")
					|| strstr(s,".tiff") || strstr(s,".TIFF")))))
			part = parts = 1;

		/* Verify Multipart info */
		if ((parts>=1) && (part<=parts)) {
			/* We found one! */
			adata->parts = parts;
			adata->part = part;
		}
		else {
			adata->parts = 0;
			adata->part = 0;
		}

		/* Add the article to the article list */
		++*count;
		alist = g_slist_prepend (alist, adata);

		/* update the count & progress ui */
		status_item_emit_next_step (STATUS_ITEM(item));
		if (!(*count%29)) {
			status_item_emit_status (STATUS_ITEM(item),
				((dl_type==HEADERS_NEW)
				 ? _("Getting new article headers %d of %d")
				 : _("Getting article headers %d of %d")),
				*count, steps);
		}
	}
	if (read_status!=0 || QUEUE_ITEM(item)->abort)
	{
		if (read_status!=0)
			queue_item_emit_sockread_err (QUEUE_ITEM(item));
		g_slist_foreach (alist, article_free_gfunc, NULL);
		g_slist_free (alist);
		return QUEUE_ITEM_STATUS_ESOCKET;
	}
	
	group_set_attrib_i (sdata, gdata, "First", first);
	group_set_attrib_i (sdata, gdata, "Last", last);

	*setme = alist;
	return 0;
}

static gchar*
queue_item_headers_describe (const StatusItem* item)
{
	const gchar* action = NULL;

	/* sanity checks */
	g_return_val_if_fail (item!=NULL, NULL);

	switch (QUEUE_ITEM_HEADERS(item)->download_type)
	{
		case HEADERS_LOCAL: action=_("cached headers"); break;
		case HEADERS_ALL: action=_("all headers"); break;
		case HEADERS_NEW: action=_("new headers"); break;
		case HEADERS_SAMPLE: action=_("a sample of headers"); break;
		default: pan_warn_if_reached(); action=_("BUG IN CODE"); break;
	}

	return g_strdup_printf ( _("Getting %s for newsgroup `%s'"),
		action,
		QUEUE_ITEM_HEADERS(item)->gdata->name );
}
