/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <stdlib.h>
#include <string.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include "acache.h"
#include "article.h"
#include "article-db.h"
#include "debug.h"
#include "decode.h"
#include "nntp.h"
#include "queue-item-decode.h"

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

static gint queue_item_decode_run (QueueItem* item);

static gchar* queue_item_decode_describe (const StatusItem* item);

static int nntp_article_decode (QueueItem *item);

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PUBLIC ROUTINES
************/

/*****
******
*****/

static void
queue_item_decode_destructor (PanObject* object)
{
	GSList* l = NULL;

        debug (DEBUG_PAN_OBJECT, "queue_item_decode destructor: %p", object);

	/* free user-readable name */
	g_free (QUEUE_ITEM_DECODE(object)->readable_name);

	/* free message_ids  */
	l = QUEUE_ITEM_DECODE(object)->message_ids;
	g_slist_foreach (l,(GFunc)g_free,NULL);
	g_slist_free(l);

	/* unref the database */
	ahdb_unref (QUEUE_ITEM_DECODE(object)->article_header_db);

	g_free (QUEUE_ITEM_DECODE(object)->filename);

	queue_item_destructor (PAN_OBJECT(object));
}

PanObject*
queue_item_decode_new (server_data *sdata,
                       group_data *gdata,
		       const gchar *readable_name,
		       GSList *message_ids,
		       gboolean open)
{
	GSList *l;
	QueueItemDecode *item = g_new0 (QueueItemDecode, 1);

        debug (DEBUG_PAN_OBJECT, "queue_item_decode_new: %p", item);

	queue_item_constructor (QUEUE_ITEM(item),
		queue_item_decode_destructor,
		queue_item_decode_describe,
		queue_item_decode_run, sdata, FALSE, TRUE);

	QUEUE_ITEM(item)->is_leech = TRUE;

	/* construct this class... */
	item->gdata = gdata;
	item->article_header_db = ahdb_ref (sdata, gdata);
	item->open = open;
	item->readable_name = g_strdup(readable_name);
	item->message_ids = g_slist_copy (message_ids);
	item->filename = NULL;
	for (l=item->message_ids; l!=NULL; l=l->next)
		l->data = g_strdup ((gchar*)l->data);

	return PAN_OBJECT(item);
}

/*****
******
*****/

static gchar*
queue_item_decode_describe (const StatusItem* item)
{
	QueueItemDecode *decode = QUEUE_ITEM_DECODE(item);

	/* sanity checks */
	g_return_val_if_fail (item!=NULL, NULL);

        return decode->open
		? g_strdup_printf (_("Opening %s"), decode->readable_name)
		: g_strdup_printf (_("Saving %s"), decode->readable_name);
}

/*****
******
*****/

static gint
queue_item_decode_run (QueueItem* item)
{
	GSList *l;
	GSList *message_ids = QUEUE_ITEM_DECODE(item)->message_ids;
	server_data *sdata = item->sdata;
	group_data *gdata = QUEUE_ITEM_DECODE(item)->gdata;
	decode_data *dd = NULL;
	article_db db = QUEUE_ITEM_DECODE(item)->article_header_db;

	/* Make sure the first one is in the db */
	article_data* adata = ahdb_get (db, (const char*)message_ids->data);
	if ( !adata ) {
		status_item_emit_error (STATUS_ITEM(item), 
			_("Couldn't find article header for message_id '%s'"),
			(char*)message_ids->data );
		return -1;
	}

	/* Checkout the article bodies so they don't get
	 * expired while we're using them */
	for (l = message_ids; l; l = l->next)
		acache_file_checkout (l->data);

	/* get the articles bodies */
	if ( nntp_article_decode (item) )
		return -1;

	/* update group attribs */
	if ((adata->state==STATE_NONE) || (adata->state==STATE_NEW))
		group_add_attrib_i (sdata, gdata, "unread", -1);

	/* pass it off to article-decode */
	dd = g_malloc (sizeof(decode_data));
	dd->sdata = sdata;
	dd->gdata = gdata;
	dd->adata = adata;
	dd->open = QUEUE_ITEM_DECODE(item)->open;
	dd->item = STATUS_ITEM(item);
	dd->list = message_ids;
	dd->filename = QUEUE_ITEM_DECODE(item)->filename;

	decode_article (dd);

	/* Check the bodies back in -- make them
	 * eligible for expire again */
	for (l = message_ids; l; l = l->next)
		acache_file_checkin (l->data);

	article_free (dd->adata);
	g_free (dd);

	return 0;
}

/*****
******
*****/

static int
nntp_article_decode (QueueItem *item)
{
	PanSocket *sock = item->sock;
	server_data *sdata = item->sdata;
	group_data *gdata = QUEUE_ITEM_DECODE(item)->gdata;
	GSList* list_of_msg_id = QUEUE_ITEM_DECODE(item)->message_ids;
	article_db db = QUEUE_ITEM_DECODE(item)->article_header_db;
	GSList* adata_list = NULL;
	const gchar *buffer = NULL;
	float count = 0;
	GSList *l = NULL;
	gint total_line_count = 0;
	gint lines_so_far = 0;

	/* sanity check */
	g_return_val_if_fail (sock!=NULL, 0);
	g_return_val_if_fail (sdata!=NULL, 0);
	g_return_val_if_fail (gdata!=NULL, 0);
	g_return_val_if_fail (list_of_msg_id!=NULL, 0);

	/* set the group */
	if (nntp_set_group (sock, gdata)) {
		status_item_emit_error (STATUS_ITEM(item),
			_("Couldn't select group '%s'"), gdata->name );
		return -1;
	} 

	/* get the article headers */
	adata_list = NULL;
	for ( l=list_of_msg_id; l!=NULL; l=l->next) {
		const gchar* id = (const gchar*) l->data;
		article_data *adata = ahdb_get (db,id);
		if (!adata) {
			status_item_emit_error (STATUS_ITEM(item),
				_("Couldn't find article header for msg: %s"),
				(char*)l->data );
			g_slist_foreach (adata_list, article_free_gfunc, NULL);
			g_slist_free (adata_list);
			return -1;
		}
		adata_list = g_slist_prepend (adata_list, adata);
	}
	adata_list = g_slist_reverse (adata_list);

	/* get the total line count */
	total_line_count = 0;
	for (l=adata_list; l!=NULL; l=l->next)
		total_line_count += ((article_data*)l->data)->linecount;

	/* Download bodies for all of the parts */
	lines_so_far = 0;
	for ( l=adata_list; l!=NULL; l=l->next)
	{
		FILE *f = NULL;
		article_data* adata0 = (article_data*) l->data;

		/* see if we already have this article/part cached */
		if (acache_body_exists (adata0->message_id))
		{
			status_item_emit_status (STATUS_ITEM(item),
				_("already have %s"),
				adata0->subject);

			lines_so_far += adata0->linecount;

			status_item_emit_progress (STATUS_ITEM(item),
				(100.0*lines_so_far)/total_line_count);

			continue;
		}

		/* let listeners know what we're doing */
		if (adata0->parts > 1)
			status_item_emit_status (STATUS_ITEM(item),
				_("Downloading %d of %d: %s"),
				adata0->part, adata0->parts, adata0->subject);
		else
			status_item_emit_status (STATUS_ITEM(item),
				_("Downloading: %s"), adata0->subject);

		/* download the article */
		if (pan_socket_putline_va(sock, "BODY %s\r\n",adata0->number))
		{
			queue_item_emit_sockwrite_err(QUEUE_ITEM(item));
			g_slist_foreach (adata_list, article_free_gfunc, NULL);
			g_slist_free (adata_list);
			return QUEUE_ITEM_STATUS_ESOCKET;
		}
		if (pan_socket_getline (sock, &buffer))
		{
			queue_item_emit_sockwrite_err(QUEUE_ITEM(item));
			g_slist_foreach (adata_list, article_free_gfunc, NULL);
			g_slist_free (adata_list);
			return QUEUE_ITEM_STATUS_ESOCKET;
		}
		if (strtol(buffer, (char **) NULL, 10) != 222) {
			article_add_flag (
				sdata, gdata, adata0, STATE_ERROR, TRUE);
			g_slist_foreach (adata_list, article_free_gfunc, NULL);
			g_slist_free (adata_list);
			return -1;
		}

		count = 1;
		if (pan_socket_getline (sock, &buffer))
		{
			queue_item_emit_sockread_err(QUEUE_ITEM(item));
			g_slist_foreach (adata_list, article_free_gfunc, NULL);
			g_slist_free (adata_list);
			return QUEUE_ITEM_STATUS_ESOCKET;
		}
	
		f = acache_open_body (adata0);
		while (strncmp(buffer, ".\r\n", 3))
		{
			if (must_exit)
			{
				acache_close_body (f);
				acache_delete (adata0->message_id);
				g_slist_foreach (
					adata_list, article_free_gfunc, NULL);
				g_slist_free (adata_list);
				return -1;
			}
			acache_putline_body (f, buffer);
			if (pan_socket_getline (sock, &buffer))
			{
				queue_item_emit_sockread_err(QUEUE_ITEM(item));
				acache_close_body (f);
				acache_delete (adata0->message_id);
				g_slist_foreach (
					adata_list, article_free_gfunc, NULL);
				g_slist_free (adata_list);
				return QUEUE_ITEM_STATUS_ESOCKET;
			}

			/* emit progress every once in awhile */
			if (++lines_so_far % 47 == 0)
				status_item_emit_progress (STATUS_ITEM(item),
					(100.0*lines_so_far)/total_line_count);
		}
		acache_close_body (f);
	}

	/* no errors received; mark as read */
	if (adata_list != NULL)
		article_mark_read (sdata, gdata, ARTICLE_DATA(adata_list->data), TRUE);

	g_slist_foreach (adata_list, article_free_gfunc, NULL);
	g_slist_free (adata_list);
	return 0;
}
