/*
 *  Copyright (C) 1998-99 Luca Deri <deri@unipi.it>
 *                      
 *  			  Centro SERRA, University of Pisa
 *  			  http://www-serra.unipi.it/
 *  					
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "ntop.h"

#ifdef SLACKWARE
#include "./linux-include/netinet/ip_icmp.h"
#else
#include <netinet/ip_icmp.h>
#endif /* SLACKWARE */

typedef struct icmpPktInfo {
  time_t         pktTime;
  struct in_addr sourceHost;
  struct in_addr destHost;
  struct icmp icmpPkt;
} IcmpPktInfo;

typedef struct icmpHostInfo {
  unsigned long icmpMsgSent[ICMP_MAXTYPE+1];
  unsigned long icmpMsgRcvd[ICMP_MAXTYPE+1];
  struct in_addr hostIpAddress;
  time_t         lastUpdated;
  HostTraffic* host;
} IcmpHostInfo;

IcmpHostInfo *icmpHostsTable[HASHNAMESIZE];

#define ICMP_LIST_MAX_LEN  128
static IcmpPktInfo  icmpHostsList[ICMP_LIST_MAX_LEN+2];
static unsigned int icmpListEntries;

/* ****************************** */

extern char* intoa(struct in_addr addr);
extern char* etheraddr_string(const u_char *ep);
extern void sendStringLen(char *theString, int len);
extern void sendString(char *theString);
extern void quicksort(void*, size_t, size_t, int(*cmp)());
extern char* makeHostLink(HostTraffic *el, short mode, 
			  short cutName, short addCountryFlag);
extern void printHTTPheader();
extern void printHTTPtrailer();
extern char* getRowColor();
extern HostTraffic* findHostByNumIP(char* numIPaddr);
extern char* makeHoostLink(HostTraffic *el, short mode, 
			  short cutName, short addCountryFlag);
extern char* ipaddr2str(struct in_addr hostIpAddress);
extern int findHostInfo(struct in_addr *hostIpAddress);
extern u_int getHostInfo(struct in_addr *hostIpAddress, 
			 u_char *ether_addr) ;
extern char* formatPkts(TrafficCounter pktNr);
extern int webMode;
static GDBM_FILE icmpDB;
static int columnSort = 0, numIcmpEntries=0;

extern time_t actTime;

extern HostTraffic* hash_hostTraffic[HASHNAMESIZE];

#ifdef MULTITHREADED
extern pthread_mutex_t gdbmMutex;
#endif

/* ****************************** 

   NOTE:

   Most of the code below has been 
   borrowed from tcpdump.
   
   ****************************** */




/* rfc1700 */
#ifndef ICMP_UNREACH_NET_UNKNOWN
#define ICMP_UNREACH_NET_UNKNOWN	6	/* destination net unknown */
#endif
#ifndef ICMP_UNREACH_HOST_UNKNOWN
#define ICMP_UNREACH_HOST_UNKNOWN	7	/* destination host unknown */
#endif
#ifndef ICMP_UNREACH_ISOLATED
#define ICMP_UNREACH_ISOLATED		8	/* source host isolated */
#endif
#ifndef ICMP_UNREACH_NET_PROHIB
#define ICMP_UNREACH_NET_PROHIB		9	/* admin prohibited net */
#endif
#ifndef ICMP_UNREACH_HOST_PROHIB
#define ICMP_UNREACH_HOST_PROHIB	10	/* admin prohibited host */
#endif
#ifndef ICMP_UNREACH_TOSNET
#define ICMP_UNREACH_TOSNET		11	/* tos prohibited net */
#endif
#ifndef ICMP_UNREACH_TOSHOST
#define ICMP_UNREACH_TOSHOST		12	/* tos prohibited host */
#endif

/* rfc1716 */
#ifndef ICMP_UNREACH_FILTER_PROHIB
#define ICMP_UNREACH_FILTER_PROHIB	13	/* admin prohibited filter */
#endif
#ifndef ICMP_UNREACH_HOST_PRECEDENCE
#define ICMP_UNREACH_HOST_PRECEDENCE	14	/* host precedence violation */
#endif
#ifndef ICMP_UNREACH_PRECEDENCE_CUTOFF
#define ICMP_UNREACH_PRECEDENCE_CUTOFF	15	/* precedence cutoff */
#endif

/* rfc1256 */
#ifndef ICMP_ROUTERADVERT
#define ICMP_ROUTERADVERT		9	/* router advertisement */
#endif
#ifndef ICMP_ROUTERSOLICIT
#define ICMP_ROUTERSOLICIT		10	/* router solicitation */
#endif

#ifndef ICMP_INFO_REQUEST
#define ICMP_INFO_REQUEST               15      /* Information Request          */
#endif

#ifndef ICMP_INFO_REPLY
#define ICMP_INFO_REPLY                 16      /* Information Reply            */
#endif

#ifndef ICMP_SOURCE_QUENCH
#define ICMP_SOURCE_QUENCH               4      /* Source Quench: packet lost, slow down */
#endif

#ifndef ICMP_TIMESTAMP
#define ICMP_TIMESTAMP                   13      /* Timestamp Request            */
#endif

#ifndef ICMP_TIMESTAMPREPLY
#define ICMP_TIMESTAMPREPLY              14      /* Timestamp Reply            */
#endif


struct tok {
  int v;			/* value */
  char *s;		/* string */
};

/* Formats for most of the ICMP_UNREACH codes */
static struct tok unreach2str[] = {
	{ ICMP_UNREACH_NET,		"net %s unreachable" },
	{ ICMP_UNREACH_HOST,		"host %s unreachable" },
	{ ICMP_UNREACH_SRCFAIL,
	    "%s unreachable - source route failed" },
	{ ICMP_UNREACH_NET_UNKNOWN,	"net %s unreachable - unknown" },
	{ ICMP_UNREACH_HOST_UNKNOWN,	"host %s unreachable - unknown" },
	{ ICMP_UNREACH_ISOLATED,
	    "%s unreachable - source host isolated" },
	{ ICMP_UNREACH_NET_PROHIB,
	    "net %s unreachable - admin prohibited" },
	{ ICMP_UNREACH_HOST_PROHIB,
	    "host %s unreachable - admin prohibited" },
	{ ICMP_UNREACH_TOSNET,
	    "net %s unreachable - tos prohibited" },
	{ ICMP_UNREACH_TOSHOST,
	    "host %s unreachable - tos prohibited" },
	{ ICMP_UNREACH_FILTER_PROHIB,
	    "host %s unreachable - admin prohibited filter" },
	{ ICMP_UNREACH_HOST_PRECEDENCE,
	    "host %s unreachable - host precedence violation" },
	{ ICMP_UNREACH_PRECEDENCE_CUTOFF,
	    "host %s unreachable - precedence cutoff" },
	{ 0,				NULL }
};

/* Formats for the ICMP_REDIRECT codes */
static struct tok type2str[] = {
	{ ICMP_REDIRECT_NET,		"redirect %s to net %s" },
	{ ICMP_REDIRECT_HOST,		"redirect %s to host %s" },
	{ ICMP_REDIRECT_TOSNET,		"redirect-tos %s to net %s" },
	{ ICMP_REDIRECT_TOSHOST,	"redirect-tos %s to net %s" },
	{ 0,				NULL }
};

/* rfc1191 */
struct mtu_discovery {
	short unused;
	short nexthopmtu;
};

/* rfc1256 */
struct ih_rdiscovery {
	u_char ird_addrnum;
	u_char ird_addrsiz;
	u_short ird_lifetime;
};

struct id_rdiscovery {
	u_int32_t ird_addr;
	u_int32_t ird_pref;
};

static char *tok2str(register const struct tok *lp, register const char *fmt,
	register int v)
{
	static char buf[128];

	while (lp->s != NULL) {
		if (lp->v == v)
			return (lp->s);
		++lp;
	}
	if (fmt == NULL)
		fmt = "#%d";
	(void)sprintf(buf, fmt, v);
	return (buf);
}

/* *********************************** */

static int getIcmpHostInfo(struct in_addr hostIpAddress) {
  int i, idx, found=0;

  idx = hostIpAddress.s_addr%HASHNAMESIZE;
  for(i=0; i<HASHNAMESIZE; i++) {
    if(icmpHostsTable[idx] != NULL) {
      if(icmpHostsTable[idx]->hostIpAddress.s_addr == hostIpAddress.s_addr) {
	found = 1;
	break;
      } else {
	idx = (idx+1)%HASHNAMESIZE;
      }
    } else {
      found = 1;
      break;
    }
  }
  
  if(!found) {
    /* Garbage collector */
    idx = 0;

    for(i=0; i<HASHNAMESIZE; i++)
      if(icmpHostsTable[i]->lastUpdated <  icmpHostsTable[idx]->lastUpdated) {
	idx = i;	
      }

    memset(icmpHostsTable[idx], 0, sizeof(IcmpHostInfo));
    icmpHostsTable[idx]->lastUpdated = actTime;
    icmpHostsTable[idx]->hostIpAddress = hostIpAddress;
  } else {
    if(icmpHostsTable[idx] == NULL) {
      /* New entry */
      numIcmpEntries++;
      icmpHostsTable[idx] = malloc(sizeof(IcmpHostInfo));
      memset(icmpHostsTable[idx], 0, sizeof(IcmpHostInfo));
      icmpHostsTable[idx]->lastUpdated = actTime;
      icmpHostsTable[idx]->hostIpAddress = hostIpAddress;
    }
  }


  return(idx);
}

/* *********************************** */
#ifdef TO_RECICLE

#endif /* TO_RECICLE */

#ifdef TO_RECICLE
  HostTraffic *srcHost=NULL, *dstHost=NULL;

  ether_src = ESRC(ep), ether_dst = EDST(ep);
  NTOHL(ip.ip_src.s_addr); srcHostIdx = getHostInfo(&ip.ip_src, ether_src);
  NTOHL(ip.ip_dst.s_addr); dstHostIdx = getHostInfo(&ip.ip_dst, ether_dst);  
#endif /* TO_RECICLE */


/* ***************************** */

static void handleIcmpPacket(const struct pcap_pkthdr *h, const u_char *p) {
  struct icmp icmpPkt;
  struct ip ip;
  u_int hlen;
  struct ether_header *ep;
  int src_idx, dst_idx;
  datum key_data, data_data;
  char tmpStr[32];
  IcmpPktInfo pktInfo;

  ep = (struct ether_header *)p;
  memcpy(&ip, (p+sizeof(struct ether_header)), sizeof(struct ip));
  hlen = (u_int)ip.ip_hl * 4;
  memcpy(&icmpPkt, (p+sizeof(struct ether_header)+hlen), sizeof(struct icmp));  

  NTOHL(ip.ip_src.s_addr);
  NTOHL(ip.ip_dst.s_addr);

  src_idx = getIcmpHostInfo(ip.ip_src);     
  dst_idx = getIcmpHostInfo(ip.ip_dst);

#ifdef DEBUG
  printf("%d [%d-%d]\n", icmpPkt.icmp_type,
	 icmpHostsTable[src_idx]->icmpMsgSent[icmpPkt.icmp_type],
	 icmpHostsTable[dst_idx]->icmpMsgRcvd[icmpPkt.icmp_type]);
#endif
  (icmpHostsTable[src_idx]->icmpMsgSent[icmpPkt.icmp_type])++;
  (icmpHostsTable[dst_idx]->icmpMsgRcvd[icmpPkt.icmp_type])++;      

  switch (icmpPkt.icmp_type) 
    {
    case ICMP_ECHOREPLY:
    case ICMP_ECHO:
      /* Do not log anything */
      break;
      
    case ICMP_UNREACH:
    case ICMP_REDIRECT:
    case ICMP_ROUTERADVERT:
    case ICMP_TIMXCEED:
    case ICMP_PARAMPROB:
    case ICMP_MASKREPLY:
    case ICMP_MASKREQ:
    case ICMP_INFO_REQUEST:
    case ICMP_INFO_REPLY:
    case ICMP_TIMESTAMP:
    case ICMP_TIMESTAMPREPLY:
    case ICMP_SOURCE_QUENCH:
      sprintf(tmpStr, "%u/%u", 
	      (unsigned long)h->ts.tv_sec,
	      (unsigned long)h->ts.tv_usec);
      key_data.dptr = tmpStr; key_data.dsize = strlen(key_data.dptr)+1;
      pktInfo.pktTime = h->ts.tv_sec;
      pktInfo.sourceHost.s_addr = ip.ip_src.s_addr;
      pktInfo.destHost.s_addr = ip.ip_dst.s_addr;
      memcpy(&pktInfo.icmpPkt, &icmpPkt, sizeof(struct icmp));
      data_data.dptr = (char*)&pktInfo; data_data.dsize = sizeof(IcmpPktInfo)+1;

#ifdef DEBUG
      if(icmpPkt.icmp_type == 5)
	printf("Adding %u/%u [%d] @ %s\n", 
	       pktInfo.sourceHost.s_addr, pktInfo.destHost.s_addr,
	       pktInfo.icmpPkt.icmp_type, tmpStr); 
#endif

#ifdef MULTITHREADED
      accessMutex(&gdbmMutex);
#endif 
      gdbm_store(icmpDB, key_data, data_data, GDBM_REPLACE);	
#ifdef MULTITHREADED
      releaseMutex(&gdbmMutex);
#endif 
      break;
    }
 

  /* Note: ipaddr2str uses the gdbm semaphore !!! */
}


/* ****************************** */

static int sortICMPhosts(const void *_a, const void *_b) {
  IcmpHostInfo **a = (IcmpHostInfo **)_a;
  IcmpHostInfo **b = (IcmpHostInfo **)_b;
  unsigned long n1, n2;

  if(((*a) == NULL) && ((*b) != NULL)) {
    printf("WARNING (1)\n");
    return(1);
  } else if(((*a) != NULL) && ((*b) == NULL)) {
    printf("WARNING (2)\n");
    return(-1);
  } else if(((*a) == NULL) && ((*b) == NULL)) {
    printf("WARNING (3)\n");
    return(0);
  }
  
  switch(columnSort) {
  case 2:
    n1 = (*a)->icmpMsgSent[ICMP_ECHO] + (*a)->icmpMsgSent[ICMP_ECHOREPLY] +
      (*a)->icmpMsgRcvd[ICMP_ECHO]+(*a)->icmpMsgRcvd[ICMP_ECHOREPLY];
    n2 = (*b)->icmpMsgSent[ICMP_ECHO] + (*b)->icmpMsgSent[ICMP_ECHOREPLY] + 
      (*b)->icmpMsgRcvd[ICMP_ECHO]+ (*b)->icmpMsgRcvd[ICMP_ECHOREPLY];
    if(n1 > n2) return(1); else if(n1 < n2) return(-1); else return(0);
    break;

  case 3:
    n1 = (*a)->icmpMsgSent[ICMP_UNREACH] + (*a)->icmpMsgRcvd[ICMP_UNREACH];
    n2 = (*b)->icmpMsgSent[ICMP_UNREACH] + (*b)->icmpMsgRcvd[ICMP_UNREACH];
    if(n1 > n2) return(1); else if(n1 < n2) return(-1); else return(0);
    break;

  case 4:
    n1 = (*a)->icmpMsgSent[ICMP_REDIRECT] + (*a)->icmpMsgRcvd[ICMP_REDIRECT];
    n2 = (*b)->icmpMsgSent[ICMP_REDIRECT] + (*b)->icmpMsgRcvd[ICMP_REDIRECT];
    if(n1 > n2) return(1); else if(n1 < n2) return(-1); else return(0);
    break;

  case 5:
    n1 = (*a)->icmpMsgSent[ICMP_ROUTERADVERT] + (*a)->icmpMsgRcvd[ICMP_ROUTERADVERT];
    n2 = (*b)->icmpMsgSent[ICMP_ROUTERADVERT] + (*b)->icmpMsgRcvd[ICMP_ROUTERADVERT];
    if(n1 > n2) return(1); else if(n1 < n2) return(-1); else return(0);
    break;

  case 6:
    n1 = (*a)->icmpMsgSent[ICMP_TIMXCEED] + (*a)->icmpMsgRcvd[ICMP_TIMXCEED];
    n2 = (*b)->icmpMsgSent[ICMP_TIMXCEED] + (*b)->icmpMsgRcvd[ICMP_TIMXCEED];
    if(n1 > n2) return(1); else if(n1 < n2) return(-1); else return(0);
    break;

  case 7:
    n1 = (*a)->icmpMsgSent[ICMP_PARAMPROB] + (*a)->icmpMsgRcvd[ICMP_PARAMPROB];
    n2 = (*b)->icmpMsgSent[ICMP_PARAMPROB] + (*b)->icmpMsgRcvd[ICMP_PARAMPROB];
    if(n1 > n2) return(1); else if(n1 < n2) return(-1); else return(0);
    break;

  case 8:
    n1 = (*a)->icmpMsgSent[ICMP_MASKREQ] + (*a)->icmpMsgSent[ICMP_MASKREPLY] + 
      (*a)->icmpMsgRcvd[ICMP_MASKREQ]+ (*a)->icmpMsgRcvd[ICMP_MASKREPLY];
    n2 = (*b)->icmpMsgSent[ICMP_MASKREQ] + (*b)->icmpMsgSent[ICMP_MASKREPLY] + 
      (*b)->icmpMsgRcvd[ICMP_MASKREQ]+ (*b)->icmpMsgRcvd[ICMP_MASKREPLY];
    if(n1 > n2) return(1); else if(n1 < n2) return(-1); else return(0);
    break;


  case 9:
    n1 = (*a)->icmpMsgSent[ICMP_SOURCE_QUENCH] + (*a)->icmpMsgRcvd[ICMP_SOURCE_QUENCH];
    n2 = (*b)->icmpMsgSent[ICMP_SOURCE_QUENCH] + (*b)->icmpMsgRcvd[ICMP_SOURCE_QUENCH];
    if(n1 > n2) return(1); else if(n1 < n2) return(-1); else return(0);
    break;

  case 10:
    n1 = (*a)->icmpMsgSent[ICMP_TIMESTAMP] + (*a)->icmpMsgSent[ICMP_TIMESTAMPREPLY] + 
      (*a)->icmpMsgRcvd[ICMP_TIMESTAMP]+ (*a)->icmpMsgRcvd[ICMP_TIMESTAMPREPLY];
    n2 = (*b)->icmpMsgSent[ICMP_TIMESTAMP] + (*b)->icmpMsgSent[ICMP_TIMESTAMPREPLY] +
      (*b)->icmpMsgRcvd[ICMP_TIMESTAMP]+ (*b)->icmpMsgRcvd[ICMP_TIMESTAMPREPLY];
    if(n1 > n2) return(1); else if(n1 < n2) return(-1); else return(0);
    break;

  case 11:
    n1 = (*a)->icmpMsgSent[ICMP_INFO_REQUEST] + (*a)->icmpMsgSent[ICMP_INFO_REPLY] +
      (*a)->icmpMsgRcvd[ICMP_INFO_REQUEST]+ (*a)->icmpMsgRcvd[ICMP_INFO_REPLY];
    n2 = (*b)->icmpMsgSent[ICMP_INFO_REQUEST] + (*b)->icmpMsgSent[ICMP_INFO_REPLY] +
      (*b)->icmpMsgRcvd[ICMP_INFO_REQUEST]+ (*b)->icmpMsgRcvd[ICMP_INFO_REPLY];
    if(n1 > n2) return(1); else if(n1 < n2) return(-1); else return(0);
    break;

  default:
    if((*a)->host->hostSymIpAddress == NULL) 
      (*a)->host->hostSymIpAddress = (*a)->host->hostNumIpAddress;
    if((*b)->host->hostSymIpAddress == NULL) 
      (*b)->host->hostSymIpAddress = (*b)->host->hostNumIpAddress;		

    return(strcasecmp((*a)->host->hostSymIpAddress, (*b)->host->hostSymIpAddress));
    break;
  }
}

/* ******************************* */

static void insertICMPPkt(IcmpPktInfo *icmpPktInfo) {
  if(icmpListEntries < ICMP_LIST_MAX_LEN)
    memcpy(&icmpHostsList[icmpListEntries++], 
	   icmpPktInfo, sizeof(IcmpHostInfo));
}

/* ******************************* */

static void printIcmpPkt(IcmpPktInfo *icmpPktInfo) {
  int s_idx, d_idx;
  char icmpBuf[512];
  struct ip *oip;
  u_int hlen;
  u_int dport, mtu;
  char *str, *fmt, *cp;
  const struct udphdr *ouh;
  unsigned long theTime = icmpPktInfo->pktTime;

  s_idx = findHostInfo(&icmpPktInfo->sourceHost);
  d_idx = findHostInfo(&icmpPktInfo->destHost);
  
  if((s_idx == -1) || (d_idx == -1))
    return;

  sendString("<TR ");
  sendString(getRowColor());
  sendString("><TD>");
  sendString(ctime((time_t*)&theTime));
  sendString("</TD>");
  sendString(makeHostLink(hash_hostTraffic[checkSessionIdx(s_idx)], LONG_FORMAT, 0, 0));
  sendString(makeHostLink(hash_hostTraffic[checkSessionIdx(d_idx)], LONG_FORMAT, 0, 0));
  sendString("<TD>");

  switch (icmpPktInfo->icmpPkt.icmp_type) {

  case ICMP_ECHOREPLY:
    strcpy(icmpBuf, "ECHO reply");
    break;

  case ICMP_ECHO:
    strcpy(icmpBuf, "ECHO request");
    break;

  case ICMP_UNREACH:
    switch (icmpPktInfo->icmpPkt.icmp_code) {

    case ICMP_UNREACH_PROTOCOL:
       NTOHL(icmpPktInfo->icmpPkt.icmp_ip.ip_dst.s_addr);
      (void)sprintf(icmpBuf, "%s protocol 0x%X unreachable",
		    ipaddr2str(icmpPktInfo->icmpPkt.icmp_ip.ip_dst),
		    icmpPktInfo->icmpPkt.icmp_ip.ip_p);
      break;

    case ICMP_UNREACH_PORT:
      oip = &icmpPktInfo->icmpPkt.icmp_ip;
      hlen = oip->ip_hl * 4;
      ouh = (struct udphdr *)(((u_char *)oip) + hlen);
#ifdef SLACKWARE
      dport = ntohs(ouh->dest);
#else
      dport = ntohs(ouh->uh_dport);
#endif
      NTOHL(oip->ip_dst.s_addr);
      switch (oip->ip_p) {

      case IPPROTO_TCP:	
	(void)sprintf(icmpBuf,
		      "%s tcp port %d unreachable",
		      ipaddr2str(oip->ip_dst),
		      dport);
	break;

      case IPPROTO_UDP:
	(void)sprintf(icmpBuf,
		      "%s udp port %d unreachable",
		      ipaddr2str(oip->ip_dst),
		      dport);
	break;

      default:
	(void)sprintf(icmpBuf,
		      "%s protocol 0x%X port %d unreachable",
		      ipaddr2str(oip->ip_dst),
		      oip->ip_p, dport);
	break;
      }
      break;

    case ICMP_UNREACH_NEEDFRAG:
      {
	register const struct mtu_discovery *mp;

	mp = (struct mtu_discovery *)&icmpPktInfo->icmpPkt.icmp_void;
	mtu = EXTRACT_16BITS(&mp->nexthopmtu);
	NTOHL(icmpPktInfo->icmpPkt.icmp_ip.ip_dst.s_addr);
	if (mtu)
	  (void)sprintf(icmpBuf,
			"%s unreachable - need to frag (mtu %d)",
			ipaddr2str(icmpPktInfo->icmpPkt.icmp_ip.ip_dst), mtu);
	else
	  (void)sprintf(icmpBuf,
			"%s unreachable - need to frag",
			ipaddr2str(icmpPktInfo->icmpPkt.icmp_ip.ip_dst));
      }
      break;

    default:
      NTOHL(icmpPktInfo->icmpPkt.icmp_ip.ip_dst.s_addr);
      fmt = tok2str(unreach2str, "#%d %%s unreachable",
		    icmpPktInfo->icmpPkt.icmp_code);
      (void)sprintf(icmpBuf, fmt,
		    ipaddr2str(icmpPktInfo->icmpPkt.icmp_ip.ip_dst));
      break;
    }
    break;

  case ICMP_REDIRECT:
    NTOHL(icmpPktInfo->icmpPkt.icmp_ip.ip_dst.s_addr);
    NTOHL(icmpPktInfo->icmpPkt.icmp_gwaddr.s_addr);
    fmt = tok2str(type2str, "redirect-#%d %%s to net %%s",
		  icmpPktInfo->icmpPkt.icmp_code);
    (void)sprintf(icmpBuf, fmt,
		  ipaddr2str(icmpPktInfo->icmpPkt.icmp_ip.ip_dst),
		  ipaddr2str(icmpPktInfo->icmpPkt.icmp_gwaddr));
    break;

  case ICMP_ROUTERADVERT:
    {
      register const struct ih_rdiscovery *ihp;
      register const struct id_rdiscovery *idp;
      u_int lifetime, num, size;

      (void)strcpy(icmpBuf, "router advertisement");
      cp = icmpBuf + strlen(icmpBuf);

      ihp = (struct ih_rdiscovery *)&icmpPktInfo->icmpPkt.icmp_void;
      (void)strcpy(cp, " lifetime ");
      cp = icmpBuf + strlen(icmpBuf);
      lifetime = EXTRACT_16BITS(&ihp->ird_lifetime);
      if (lifetime < 60)
	(void)sprintf(cp, "%u", lifetime);
      else if (lifetime < 60 * 60)
	(void)sprintf(cp, "%u:%02u",
		      lifetime / 60, lifetime % 60);
      else
	(void)sprintf(cp, "%u:%02u:%02u",
		      lifetime / 3600,
		      (lifetime % 3600) / 60,
		      lifetime % 60);
      cp = icmpBuf + strlen(icmpBuf);

      num = ihp->ird_addrnum;
      (void)sprintf(cp, " %d:", num);
      cp = icmpBuf + strlen(icmpBuf);

      size = ihp->ird_addrsiz;
      if (size != 2) {
	(void)sprintf(cp, " [size %d]", size);
	break;
      }
      idp = (struct id_rdiscovery *)&icmpPktInfo->icmpPkt.icmp_data;
      while (num-- > 0) {
	struct in_addr theAddr;

	theAddr.s_addr = idp->ird_addr;
	NTOHL(theAddr.s_addr);
	(void)sprintf(cp, " {%s %u}",
		      ipaddr2str(theAddr),
		      EXTRACT_32BITS(&idp->ird_pref));
	cp = icmpBuf + strlen(icmpBuf);
      }
    }
    break;

  case ICMP_TIMXCEED:
    switch (icmpPktInfo->icmpPkt.icmp_code) {

    case ICMP_TIMXCEED_INTRANS:
      str = "time exceeded in-transit";
      break;

    case ICMP_TIMXCEED_REASS:
      str = "ip reassembly time exceeded";
      break;

    default:
      (void)sprintf(icmpBuf, "time exceeded-#%d", icmpPktInfo->icmpPkt.icmp_code);
      break;
    }
    break;

  case ICMP_PARAMPROB:
    if (icmpPktInfo->icmpPkt.icmp_code)
      (void)sprintf(icmpBuf, "parameter problem - code %d",
		    icmpPktInfo->icmpPkt.icmp_code);
    else {
      (void)sprintf(icmpBuf, "parameter problem - octet %d",
		    icmpPktInfo->icmpPkt.icmp_pptr);
    }
    break;

  case ICMP_MASKREQ:
    strcpy(icmpBuf, "ICMP network mask request");
    break;

  case ICMP_MASKREPLY:
    (void)sprintf(icmpBuf, "address mask is 0x%08x",
		  (u_int32_t)ntohl(icmpPktInfo->icmpPkt.icmp_mask));
    break;

  default:
    (void)sprintf(icmpBuf, "type-#%d", icmpPktInfo->icmpPkt.icmp_type);
    break;
  }

  sendString(icmpBuf);
  sendString("</TR>\n");
}

/* ******************************* */

static int sortICMPHostsInfo(const void *_a, const void *_b) {
  IcmpPktInfo *a = (IcmpPktInfo *)_a;
  IcmpPktInfo *b = (IcmpPktInfo *)_b;

    if(a->pktTime > b->pktTime)
      return(1);
    else if(a->pktTime < b->pktTime)
      return(-1);
    else
      return(0);
}

/* ******************************* */

static void printIcmpHostPkts(struct in_addr hostIpAddress, 
			      int icmpId) {
  datum key_data, data_data;
  datum return_data;
  unsigned long theTime, garbageTime = actTime-2*(60*60); /* 2 Hours */
  IcmpPktInfo icmpPktInfo;
  int num=0;
  char tmpTime[32];

  icmpListEntries=0;
  memset(icmpHostsList, 0, sizeof(icmpHostsList));

#ifdef DEBUG
    printf("Searching for '%u' [%d]...\n", hostIpAddress.s_addr, icmpId); 
#endif    

#ifdef MULTITHREADED
  accessMutex(&gdbmMutex);
#endif 
  return_data = gdbm_firstkey (icmpDB);
#ifdef MULTITHREADED
  releaseMutex(&gdbmMutex);
#endif 

  while (return_data.dptr != NULL) {
    key_data = return_data;
#ifdef MULTITHREADED
    accessMutex(&gdbmMutex);
#endif 
    return_data = gdbm_nextkey(icmpDB, key_data);    
#ifdef MULTITHREADED
    releaseMutex(&gdbmMutex);
#endif 
    strcpy(tmpTime, key_data.dptr);    
    theTime = atol(strtok(tmpTime, "/"));
    if(theTime < garbageTime) {
#ifdef DEBUG
      printf("%d) Purging entry %u/%u\n", num++,
	     theTime,garbageTime);
#endif
#ifdef MULTITHREADED
      accessMutex(&gdbmMutex);
#endif 
      gdbm_delete(icmpDB, key_data);
#ifdef MULTITHREADED
      releaseMutex(&gdbmMutex);
#endif 
    } else {
#ifdef MULTITHREADED
      accessMutex(&gdbmMutex);
#endif 
      data_data = gdbm_fetch(icmpDB, key_data);
#ifdef MULTITHREADED
      releaseMutex(&gdbmMutex);
#endif 

      if(data_data.dptr != NULL) {
	memcpy(&icmpPktInfo, data_data.dptr, sizeof(IcmpPktInfo));
#ifdef DEBUG
	printf("%d) %u/%u [%d]\n", ++num,
	       icmpPktInfo.sourceHost.s_addr,
	       icmpPktInfo.destHost.s_addr,
	       icmpPktInfo.icmpPkt.icmp_type); 
#endif
	if((icmpPktInfo.sourceHost.s_addr == hostIpAddress.s_addr)
	   || (icmpPktInfo.destHost.s_addr == hostIpAddress.s_addr)) {
	  switch(icmpId) {
	  case ICMP_ECHO:
	    if((icmpPktInfo.icmpPkt.icmp_type == ICMP_ECHO)
	       || (icmpPktInfo.icmpPkt.icmp_type == ICMP_ECHOREPLY))
	      insertICMPPkt(&icmpPktInfo);
	    break;
	  case ICMP_UNREACH:
	    if(icmpPktInfo.icmpPkt.icmp_type == ICMP_UNREACH)
	      insertICMPPkt(&icmpPktInfo);
	    break;
	  case ICMP_REDIRECT:
	    if(icmpPktInfo.icmpPkt.icmp_type == ICMP_REDIRECT)
	      insertICMPPkt(&icmpPktInfo);
	    break;
	  case ICMP_ROUTERADVERT:
	    if(icmpPktInfo.icmpPkt.icmp_type == ICMP_ROUTERADVERT)
	      insertICMPPkt(&icmpPktInfo);
	    break;
	  case ICMP_TIMXCEED:
	    if(icmpPktInfo.icmpPkt.icmp_type == ICMP_TIMXCEED)
	      insertICMPPkt(&icmpPktInfo);
	    break;
	  case ICMP_PARAMPROB:
	    if(icmpPktInfo.icmpPkt.icmp_type == ICMP_PARAMPROB)
	      insertICMPPkt(&icmpPktInfo);
	    break;
	  case ICMP_MASKREQ:
	    if((icmpPktInfo.icmpPkt.icmp_type == ICMP_MASKREQ)
	       || (icmpPktInfo.icmpPkt.icmp_type == ICMP_MASKREPLY))
	      insertICMPPkt(&icmpPktInfo);
	    break;
	    break;
	  case ICMP_SOURCE_QUENCH:
	    if(icmpPktInfo.icmpPkt.icmp_type == ICMP_SOURCE_QUENCH)
	      insertICMPPkt(&icmpPktInfo);
	    break;
	  case ICMP_TIMESTAMP:
	    if((icmpPktInfo.icmpPkt.icmp_type == ICMP_TIMESTAMP)
	       || (icmpPktInfo.icmpPkt.icmp_type == ICMP_TIMESTAMPREPLY))
	      insertICMPPkt(&icmpPktInfo);
	    break;
	  case ICMP_INFO_REQUEST:
	    if((icmpPktInfo.icmpPkt.icmp_type == ICMP_INFO_REQUEST)
	       || (icmpPktInfo.icmpPkt.icmp_type == ICMP_INFO_REPLY))
	      insertICMPPkt(&icmpPktInfo);
	    break;
	  }
	}
      }

      if(data_data.dptr != NULL) free(data_data.dptr);
      free(key_data.dptr);
    }
  } /* while */

  /* ************************ */

  sendString("<CENTER><TABLE BORDER>\n");
  sendString("<TH>Time</TH><TH>Source</TH><TH>Dest</TH><TH>Packet</TH></TR>\n");

  quicksort(icmpHostsList, icmpListEntries, sizeof(IcmpPktInfo), sortICMPHostsInfo);
  
  for(num=0; num<icmpListEntries; num++)
    printIcmpPkt(&icmpHostsList[num]);

  sendString("</TABLE></CENTER>\n");
  printHTTPtrailer();
}

/* ******************************* */

static void handleIcmpWatchHTTPrequest(char* url) {
  char buf[1024], anchor[256];
  char *sign = "-";
  char *pluginName = "<A HREF=/plugins/icmpWatch";
  int i, revertOrder=0, num, idx, icmpId=-1;
  IcmpHostInfo *localIcmpHostsTable[HASHNAMESIZE];
  struct in_addr hostIpAddress;

  memset(localIcmpHostsTable, 0, sizeof(localIcmpHostsTable));

  for(i=0, num=0; i<HASHNAMESIZE; i++) 
    if(icmpHostsTable[i] != NULL) {
      localIcmpHostsTable[num++] = icmpHostsTable[i];
      idx = findHostInfo(&icmpHostsTable[i]->hostIpAddress);
      if(idx == -1)
	num--; /* Discard the entry */
      else {
	icmpHostsTable[i]->host = hash_hostTraffic[checkSessionIdx(idx)];
	if(icmpHostsTable[i]->host == NULL)
	  num--; /* Discard the entry */
	else {
#ifdef DEBUG
	  printf("%d) %s\n", num, icmpHostsTable[i]->host->hostSymIpAddress);
#endif
	}
      }
    }

  if(url[0] == '\0')
    columnSort = 0;
  else if((url[0] == '-') || isdigit(url[0])) {
    if(url[0] == '-') {
      sign = "";
      revertOrder = 1;
      columnSort = atoi(&url[1]);
    } else
      columnSort = atoi(url);
  } else /* host=3240847503&icmp=3 */ {   
    char *tmpStr;
    strtok(url, "=");
    
    tmpStr = strtok(NULL, "&");
    hostIpAddress.s_addr = strtoul(tmpStr, (char **)NULL, 10); 
#ifdef DEBUG
    printf("-> %s [%u]\n", tmpStr, hostIpAddress.s_addr );
#endif
    strtok(NULL, "=");
    icmpId = atoi(strtok(NULL, "&"));
  }

  printHTTPheader();

  sendString("<CENTER><FONT FACE=Helvetica><H1>ICMP Statistics</H1><p>\n");

  if(numIcmpEntries == 0) {
    sendString("<P><CENTER><H1>"
	       "<i>No Data To Display (yet)</i></H1>"
	       "</CENTER></FONT>\n");
    printHTTPtrailer();
    return;
  }

  if(icmpId != -1) {  
    printIcmpHostPkts(hostIpAddress, icmpId);
    return;
  }

  sendString("<TABLE BORDER>\n");
  sprintf(buf, "<TR><TH>%s?%s1>Host</A><br>[Pkt&nbsp;Sent/Rcvd]</TH>"
	 "<TH>%s?%s2>Echo</A></TH>"
	 "<TH>%s?%s3>Unreach</A></TH>"
	 "<TH>%s?%s4>Redirect</A></TH>"
	 "<TH>%s?%s5>Router<br>Advert.</A></TH>"
	 "<TH>%s?%s6>Time<br>Exceeded</A></TH>"
	 "<TH>%s?%s7>Param.<br>Problem</A></TH>"
	 "<TH>%s?%s8>Network<br>Mask</A></TH>"
	 "<TH>%s?%s9>Source<br>Quench</A></TH>"
	 "<TH>%s?%s10>Timestamp</A></TH>"
	 "<TH>%s?%s11>Info</A></TH>"
	 "</TR>\n",
	  pluginName, sign,
	  pluginName, sign,
	  pluginName, sign,
	  pluginName, sign,
	  pluginName, sign,
	  pluginName, sign,
	  pluginName, sign,
	  pluginName, sign,
	  pluginName, sign,
	  pluginName, sign,
	  pluginName, sign);
  sendString(buf);

  quicksort(localIcmpHostsTable, num, sizeof(struct localIcmpHostsTable*), sortICMPhosts);

  for(i=0; i<HASHNAMESIZE; i++) 
    if(localIcmpHostsTable[i] != NULL) {
      unsigned long tot;
      char *postAnchor;
      int idx;
      HostTraffic *host;

      if(revertOrder)
	idx = num-i-1;
      else
	idx = i;

      host = localIcmpHostsTable[idx]->host;

      sprintf(buf, "<TR %s> %s", 	  
	      getRowColor(),
	      makeHostLink(host, LONG_FORMAT, 0, 0));
      sendString(buf);
      
      sprintf(buf, "<TD ALIGN=right>%s/%s</TD>",  
	      formatPkts(localIcmpHostsTable[idx]->icmpMsgSent[ICMP_ECHO]+
			 localIcmpHostsTable[idx]->icmpMsgSent[ICMP_ECHOREPLY]),
	      formatPkts(localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_ECHO]+
			 localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_ECHOREPLY]));
      sendString(buf);


      tot=localIcmpHostsTable[idx]->icmpMsgSent[ICMP_UNREACH]+
	localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_UNREACH];      
      if(tot > 0) {
	sprintf(anchor, "%s?host=%lu&icmp=%d>", 
		pluginName, 
		localIcmpHostsTable[idx]->hostIpAddress.s_addr, 
		ICMP_UNREACH);
	postAnchor = "</A>";
      } else {
	anchor[0] = '\0';
	postAnchor = "";
      }
      sprintf(buf, "<TD ALIGN=center>%s%s/%s%s</TD>",
	      anchor,
	      formatPkts(localIcmpHostsTable[idx]->icmpMsgSent[ICMP_UNREACH]),
	      formatPkts(localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_UNREACH]),
	      postAnchor);
      sendString(buf);

      
      tot=localIcmpHostsTable[idx]->icmpMsgSent[ICMP_REDIRECT]+
	localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_REDIRECT];
      if(tot > 0) {
	sprintf(anchor, "%s?host=%lu&icmp=%d>", 
		pluginName, 
		localIcmpHostsTable[idx]->hostIpAddress.s_addr, 
		ICMP_REDIRECT);
	postAnchor = "</A>";
      } else {
	anchor[0] = '\0';
	postAnchor = "";
      }
      sprintf(buf, "<TD ALIGN=center>%s%s/%s%s</TD>", anchor,
	      formatPkts(localIcmpHostsTable[idx]->icmpMsgSent[ICMP_REDIRECT]),
	      formatPkts(localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_REDIRECT]),
	      postAnchor);
      sendString(buf);

      tot=localIcmpHostsTable[idx]->icmpMsgSent[ICMP_ROUTERADVERT]+
	localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_ROUTERADVERT];
      if(tot > 0) {
	sprintf(anchor, "%s?host=%lu&icmp=%d>", 
		pluginName, 
		localIcmpHostsTable[idx]->hostIpAddress.s_addr, 
		ICMP_ROUTERADVERT);
	postAnchor = "</A>";
      } else {
	anchor[0] = '\0';
	postAnchor = "";
      }
      sprintf(buf, "<TD ALIGN=center>%s%s/%s%s</TD>", anchor,
	      formatPkts(localIcmpHostsTable[idx]->icmpMsgSent[ICMP_ROUTERADVERT]),
	      formatPkts(localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_ROUTERADVERT]),
	      postAnchor);
      sendString(buf);
      
      tot=localIcmpHostsTable[idx]->icmpMsgSent[ICMP_TIMXCEED]+
	localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_TIMXCEED];
      if(tot > 0) {
	sprintf(anchor, "%s?host=%lu&icmp=%d>", 
		pluginName, 
		localIcmpHostsTable[idx]->hostIpAddress.s_addr, 
		ICMP_TIMXCEED);
	postAnchor = "</A>";
      } else {
	anchor[0] = '\0';
	postAnchor = "";
      }
      sprintf(buf, "<TD ALIGN=center>%s%s/%s%s</TD>", anchor,
	      formatPkts(localIcmpHostsTable[idx]->icmpMsgSent[ICMP_TIMXCEED]),
	     formatPkts(localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_TIMXCEED]),
	      postAnchor);
      sendString(buf);

      tot=localIcmpHostsTable[idx]->icmpMsgSent[ICMP_PARAMPROB]+
	localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_PARAMPROB];
      if(tot > 0) {
	sprintf(anchor, "%s?host=%lu&icmp=%d>", 
		pluginName, 
		localIcmpHostsTable[idx]->hostIpAddress.s_addr, 
		ICMP_PARAMPROB);
	postAnchor = "</A>";
      } else {
	anchor[0] = '\0';
	postAnchor = "";
      }
      sprintf(buf, "<TD ALIGN=center>%s%s/%s%s</TD>", anchor,
	      formatPkts(localIcmpHostsTable[idx]->icmpMsgSent[ICMP_PARAMPROB]),
	      formatPkts(localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_PARAMPROB]),
	      postAnchor);
      sendString(buf);

      tot=localIcmpHostsTable[idx]->icmpMsgSent[ICMP_MASKREQ]+
	localIcmpHostsTable[idx]->icmpMsgSent[ICMP_MASKREPLY]+
	localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_MASKREQ]+
	localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_MASKREPLY];
      if(tot > 0) {
	sprintf(anchor, "%s?host=%lu&icmp=%d>", 
		pluginName, 
		localIcmpHostsTable[idx]->hostIpAddress.s_addr, 
		ICMP_MASKREQ);
	postAnchor = "</A>";
      } else {
	anchor[0] = '\0';
	postAnchor = "";
      }
      sprintf(buf, "<TD ALIGN=center>%s%s/%s%s</TD>", anchor,
	      formatPkts(localIcmpHostsTable[idx]->icmpMsgSent[ICMP_MASKREQ]+
			 localIcmpHostsTable[idx]->icmpMsgSent[ICMP_MASKREPLY]),
	      formatPkts(localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_MASKREQ]+
			 localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_MASKREPLY]),
	      postAnchor);
      sendString(buf);

      tot=localIcmpHostsTable[idx]->icmpMsgSent[ICMP_SOURCE_QUENCH]+
	localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_SOURCE_QUENCH];
      if(tot > 0) {
	sprintf(anchor, "%s?host=%lu&icmp=%d>", 
		pluginName, 
		localIcmpHostsTable[idx]->hostIpAddress.s_addr, 
		ICMP_SOURCE_QUENCH);
	postAnchor = "</A>";
      } else {
	anchor[0] = '\0';
	postAnchor = "";
      }
      sprintf(buf, "<TD ALIGN=center>%s%s/%s%s</TD>", anchor,
	      formatPkts(localIcmpHostsTable[idx]->icmpMsgSent[ICMP_SOURCE_QUENCH]),
	      formatPkts(localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_SOURCE_QUENCH]),
	      postAnchor);
      sendString(buf);

      tot=localIcmpHostsTable[idx]->icmpMsgSent[ICMP_TIMESTAMP]+
	localIcmpHostsTable[idx]->icmpMsgSent[ICMP_TIMESTAMPREPLY]+
	localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_TIMESTAMP]+
	localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_TIMESTAMPREPLY];
      if(tot > 0) {
	sprintf(anchor, "%s?host=%lu&icmp=%d>", 
		pluginName, 
		localIcmpHostsTable[idx]->hostIpAddress.s_addr, 
		ICMP_TIMESTAMP);
	postAnchor = "</A>";
      } else {
	anchor[0] = '\0';
	postAnchor = "";
      }
      sprintf(buf, "<TD ALIGN=center>%s%s/%s%s</TD>", anchor,
	      formatPkts(localIcmpHostsTable[idx]->icmpMsgSent[ICMP_TIMESTAMP]+
			 localIcmpHostsTable[idx]->icmpMsgSent[ICMP_TIMESTAMPREPLY]),
	      formatPkts(localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_TIMESTAMP]+
			 localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_TIMESTAMPREPLY]),
	      postAnchor);
      sendString(buf);

      tot=localIcmpHostsTable[idx]->icmpMsgSent[ICMP_INFO_REQUEST]+
	localIcmpHostsTable[idx]->icmpMsgSent[ICMP_INFO_REPLY]+
	localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_INFO_REQUEST]+
	localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_INFO_REPLY];
      if(tot > 0) {
	sprintf(anchor, "%s?host=%lu&icmp=%d>", 
		pluginName, 
		localIcmpHostsTable[idx]->hostIpAddress.s_addr, 
		ICMP_INFO_REQUEST);
	postAnchor = "</A>";
      } else {
	anchor[0] = '\0';
	postAnchor = "";
      }
      sprintf(buf, "<TD ALIGN=center>%s%s/%s%s</TD>", anchor,
	      formatPkts(localIcmpHostsTable[idx]->icmpMsgSent[ICMP_INFO_REQUEST]+
			 localIcmpHostsTable[idx]->icmpMsgSent[ICMP_INFO_REPLY]),
	      formatPkts(localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_INFO_REQUEST]+
			 localIcmpHostsTable[idx]->icmpMsgRcvd[ICMP_INFO_REPLY]),
	      postAnchor);
      sendString(buf);

      sendString("</TR>\n");
  }


  sendString("</TABLE>\n");
  printHTTPtrailer();
}

/* ****************************** */

static void termIcmpFunct() {
  if(webMode)
    printf("Thanks for using icmpWatch.\n");
  
  if(icmpDB != NULL) {
    gdbm_close(icmpDB);
    icmpDB = NULL;
  }
}

/* ****************************** */

static PluginInfo icmpPluginInfo[] = {
  "icmpWatchPlugin",
  "This plugin handles ICMP packets",
  "1.0", /* version */
  "<A HREF=http://jake.unipi.it/~deri/>Luca Deri</A>", 
  "icmpWatch", /* http://<host>:<port>/plugins/icmpWatch */
  termIcmpFunct, /* TermFunc   */
  handleIcmpPacket, /* PluginFunc */
  handleIcmpWatchHTTPrequest,
  "icmp" /* BPF filter: filter all the ICMP packets */
};

/* Plugin entry fctn */
PluginInfo* PluginEntryFctn() {

  if(webMode)
    printf("Welcome to %s. (C) 1999 by Luca Deri.\n", icmpPluginInfo->pluginName);

  icmpDB = gdbm_open ("icmpWatch.db", 0, GDBM_NEWDB, 00664, NULL);

  if(icmpDB == NULL) 
    printf("Unable to open icmpWatch database. This plugin will be disabled.\n");
  else {
    memset(icmpHostsTable, 0, sizeof(icmpHostsTable));
    numIcmpEntries = 0;
  }

  return(icmpPluginInfo);
}
