#include "links.h"

void get_system_name()
{
	FILE *f;
	unsigned char *p;
	memset(system_name, 0, MAX_STR_LEN);
	if (!(f = popen("uname -srm", "r"))) goto fail;
	if (fread(system_name, 1, MAX_STR_LEN - 1, f) <= 0) {
		pclose(f);
		goto fail;
	}
	pclose(f);
	for (p = system_name; *p; p++) if (*p < ' ') {
		*p = 0;
		break;
	}
	return;
	fail:
	system_name[0] = 0;
}

struct option {
	int p;
	unsigned char *(*rd_cmd)(struct option *, unsigned char ***, int *);
	unsigned char *(*rd_cfg)(struct option *, unsigned char *);
	void (*wr_cfg)(struct option *, unsigned char **, int *);
	int min, max;
	void *ptr;
	unsigned char *cfg_name;
	unsigned char *cmd_name;
};

struct option options[];

unsigned char *parse_options(int argc, unsigned char *argv[])
{
	unsigned char *e, *u = NULL;
	while (argc) {
		int i;
		argv++, argc--;
		if (argv[-1][0] == '-') {
			for (i = 0; options[i].p; i++)
				if (options[i].rd_cmd && options[i].cmd_name &&
				    !strcasecmp(options[i].cmd_name, &argv[-1][1])) {
					if ((e = options[i].rd_cmd(&options[i], &argv, &argc))) {
						if (e[0]) fprintf(stderr, "Error parsing option %s: %s\n", argv[-1], e);
						return NULL;
					}
					goto found;
				}
			uu:
			fprintf(stderr, "Unknown option %s\n", argv[-1]);
			return NULL;
		} else if (!u) u = argv[-1];
		else goto uu;
		found:;
	}
	if (u) return u;
	return "";
}

void parse_config_file(unsigned char *name, unsigned char *file)
{
	int err = 0;
	int line = 0;
	unsigned char *e;
	int i;
	unsigned char *n, *p;
	int nl, pl;
	while (file[0]) {
		line++;
		while (file[0] && (file[0] == ' ' || file[0] == 9)) file++;
		n = file;
		while (file[0] && file[0] > ' ') file++;
		if (file == n) {
			if (file[0]) file++;
			continue;
		}
		nl = file - n;
		while (file[0] == 9 || file[0] == ' ') file++;
		p = file;
		while (file[0] && file[0] != 10 && file[0] != 13) file++;
		pl = file - p;
		if (file[0]) {
			if ((file[1] == 10 || file[1] == 13) && file[0] != file[1]) file++;
			file++;
		}
		if (n[0] != '#') for (i = 0; options[i].p; i++) if (options[i].cfg_name && nl == strlen(options[i].cfg_name) && !casecmp(n, options[i].cfg_name, nl)) {
			unsigned char *o = memacpy(p, pl);
			if ((e = options[i].rd_cfg(&options[i], o))) {
				if (e[0]) fprintf(stderr, "Error parsing config file %s, line %d: %s\n", name, line, e), err = 1;
			}
			mem_free(o);
			goto f;
		}
		fprintf(stderr, "Unknown option in config file %s, line %d\n", name, line);
		err = 1;
		f:;
	}
	if (err) fprintf(stderr, "\007"), sleep(3);
}

unsigned char *create_config_string()
{
	unsigned char *s = init_str();
	int l = 0;
	int i;
	for (i = 0; options[i].p; i++) if (options[i].wr_cfg)
		options[i].wr_cfg(&options[i], &s, &l);
	add_to_str(&s, &l, NEWLINE);
	return s;
}

#define FILE_BUF	1024

unsigned char cfg_buffer[FILE_BUF];

unsigned char *read_config_file(unsigned char *name, int *n)
{
	int h, r;
	int l = 0;
	unsigned char *s;
	if (n) *n = 0;
	if ((h = open(name, O_RDONLY | O_NOCTTY)) == -1) {
		if (n) *n = 1;
		if ((h = open(name, O_RDWR | O_NOCTTY | O_CREAT, 0666)) == -1) {
			return NULL;
		}
	}
	set_bin(h);
	s = init_str();
	while ((r = read(h, cfg_buffer, FILE_BUF)) > 0) {
		int i;
		for (i = 0; i < r; i++) if (!cfg_buffer[i]) cfg_buffer[i] = ' ';
		add_bytes_to_str(&s, &l, cfg_buffer, r);
	}
	if (r == -1) mem_free(s), s = NULL;
	close(h);
	return s;
}

int write_to_config_file(unsigned char *name, unsigned char *c)
{
	int rr = strlen(c);
	int r = rr;
	int h, w;
	if ((h = open(name, O_WRONLY | O_NOCTTY | O_CREAT | O_TRUNC, 0666)) == -1) return -1;
	set_bin(h);
	while (r > 0) {
		if ((w = write(h, c + rr - r, r)) <= 0) {
			close(h);
			return -1;
		}
		r -= w;
	}
	close(h);
	return 0;
}

unsigned char *get_home(int *n)
{
	struct stat st;
	unsigned char *home = stracpy(getenv("HOME"));
	unsigned char *links_home;
	if (n) *n = 1;
	if (!home) {
		int i;
		home = stracpy(path_to_exe);
		if (!home) return NULL;
		for (i = strlen(home) - 1; i >= 0; i--) if (dir_sep(home[i])) {
			home[i + 1] = 0;
			break;
		}
	}
	while (home[0] && dir_sep(home[strlen(home) - 1])) home[strlen(home) - 1] = 0;
	add_to_strn(&home, "/");
	links_home = stracpy(home);
	add_to_strn(&links_home, ".links");
	if (stat(links_home, &st)) {
		if (!mkdir(links_home, 0777)) goto home_creat;
		goto first_failed;
	}
	if (S_ISDIR(st.st_mode)) goto home_ok;
	first_failed:
	mem_free(links_home);
	links_home = stracpy(home);
	add_to_strn(&links_home, "links");
	if (stat(links_home, &st)) {
		if (!mkdir(links_home, 0777)) goto home_creat;
		goto failed;
	}
	if (S_ISDIR(st.st_mode)) goto home_ok;
	failed:
	mem_free(links_home);
	mem_free(home);
	return NULL;
	home_ok:
	if (n) *n = 0;
	home_creat:
	add_to_strn(&links_home, "/");
	mem_free(home);
	return links_home;
}

void init_home()
{
	get_system_name();
	links_home = get_home(&first_use);
	if (!links_home) {
		fprintf(stderr, "%s\n\007", get_text("Unable to find or create links config directory. Please check, that you have $HOME variable set correctly and that you have write permission to your home directory"));
		sleep(3);
		return;
	}
}

void load_config()
{
	unsigned char *c;
	if (!config_file) config_file = stracpy(links_home);
	if (!config_file) return;
	add_to_strn(&config_file, ".links.cfg");
	if ((c = read_config_file(config_file, &created_cfg))) goto ok;
	mem_free(config_file);
	config_file = stracpy(links_home);
	if (!config_file) return;
	add_to_strn(&config_file, "links.cfg");
	if ((c = read_config_file(config_file, &created_cfg))) goto ok;
	mem_free(config_file);
	fprintf(stderr, "%s\n\007", get_text("Unable to find or create config file in your links directory. Please check, that you have write permission to links home directory"));
	sleep(3);
	config_file = NULL;
	return;
	ok:
	parse_config_file(config_file, c);
	mem_free(c);
	if (created_cfg) write_config(NULL);
}

void write_config(struct terminal *term)
{
	unsigned char *c;
	if (!config_file) {
		if (term) msg_box(term, NULL, get_text("Config error"), AL_CENTER, get_text("Links was not able to create config file"), NULL, 1, get_text("Cancel"), NULL, B_ENTER | B_ESC);
		return;
	}
	c = create_config_string();
	if (write_to_config_file(config_file, c)) {
		if (term) msg_box(term, NULL, get_text("Config error"), AL_CENTER, get_text("Unable to write to config file"), NULL, 1, get_text("Cancel"), NULL, B_ENTER | B_ESC);
	}
	mem_free(c);
}

void add_nm(struct option *o, unsigned char **s, int *l)
{
	if (*l) add_to_str(s, l, NEWLINE);
	add_to_str(s, l, o->cfg_name);
	add_to_str(s, l, " ");
}

unsigned char *num_rd(struct option *o, unsigned char *c)
{
	unsigned char *end;
	long l = strtolx(c, &end);
	if (*end) return get_text("Number expected");
	if (l < o->min || l > o->max) return get_text("Out of range");
	*(int *)o->ptr = l;
	return NULL;
}

void num_wr(struct option *o, unsigned char **s, int *l)
{
	add_nm(o, s, l);
	add_knum_to_str(s, l, *(int *)o->ptr);
}

unsigned char *str_rd(struct option *o, unsigned char *c)
{
	if (strlen(c) + 1 > o->max) return get_text("String too long");
	strcpy(o->ptr, c);
	return NULL;
}

void str_wr(struct option *o, unsigned char **s, int *l)
{
	add_nm(o, s, l);
	if (strlen(o->ptr) > o->max - 1) add_bytes_to_str(s, l, o->ptr, o->max - 1);
	else add_to_str(s, l, o->ptr);
}

unsigned char *cp_rd(struct option *o, unsigned char *c)
{
	int i;
	if (!strcasecmp(c, "none")) i = -1;
	else if ((i = get_cp_index(c)) == -1) return get_text("Unknown codepage");
	*(int *)o->ptr = i;
	return NULL;
}

void cp_wr(struct option *o, unsigned char **s, int *l)
{
	unsigned char *n = get_cp_mime_name(*(int *)o->ptr);
	add_nm(o, s, l);
	add_to_str(s, l, n);
}

unsigned char *get_word(unsigned char **s)
{
	unsigned char *w = *s;
	unsigned char *ww;
	if (!**s) return NULL;
	while (**s > ' ') (*s)++;
	ww = memacpy(w, *s - w);
	while (**s && **s <= ' ') (*s)++;
	return ww;
}

unsigned char *get_quoted(unsigned char **s)
{
	unsigned char *st;
	int l;
	if (!**s || **s != '"') return NULL;
	(*s)++;
	st = init_str();
	l = 0;
	while (**s && **s != '"') quak:add_chr_to_str(&st, &l, *((*s)++));
	if (!**s) {
		mem_free(st);
		return NULL;
	}
	(*s)++;
	if (**s == '"') goto quak;
	while (**s && **s <= ' ') (*s)++;
	return st;
}

void add_quoted_to_str(unsigned char **s, int *l, unsigned char *q)
{
	add_chr_to_str(s, l, '"');
	while (*q) {
		if (*q == '"') add_to_str(s, l, "\"\"");
		else add_chr_to_str(s, l, *q);
		q++;
	}
	add_chr_to_str(s, l, '"');
}

unsigned char *type_rd(struct option *o, unsigned char *c)
{
	unsigned char *err = get_text("Error reading association specification");
	struct assoc new;
	unsigned char *w;
	int n;
	memset(&new, 0, sizeof(struct assoc));
	if (!(new.label = get_quoted(&c))) goto err;
	if (!(new.ct = get_quoted(&c))) goto err;
	if (!(new.prog = get_quoted(&c))) goto err;
	if (!(w = get_word(&c))) goto err;
	if (strlen(w) != 1 || w[0] < '0' || w[0] > '8') goto err_f;
	n = w[0] - '0';
	mem_free(w);
	new.cons = !!(n & 1);
	new.xwin = !!(n & 2);
	new.ask = !!(n & 4);
	if (!(w = get_word(&c))) goto err;
	if (strlen(w) != 1 || w[0] < '0' || w[0] > '9') goto err_f;
	new.system = w[0] - '0';
	mem_free(w);
	update_assoc(&new);
	err = NULL;
	err:
	if (new.label) mem_free(new.label);
	if (new.ct) mem_free(new.ct);
	if (new.prog) mem_free(new.prog);
	return err;
	err_f:
	mem_free(w);
	goto err;
}


void type_wr(struct option *o, unsigned char **s, int *l)
{
	struct assoc *a;
	foreachback(a, assoc) {
		add_nm(o, s, l);
		add_quoted_to_str(s, l, a->label);
		add_to_str(s, l, " ");
		add_quoted_to_str(s, l, a->ct);
		add_to_str(s, l, " ");
		add_quoted_to_str(s, l, a->prog);
		add_to_str(s, l, " ");
		add_num_to_str(s, l, (!!a->cons) + (!!a->xwin) * 2 + (!!a->ask) * 4);
		add_to_str(s, l, " ");
		add_num_to_str(s, l, a->system);
	}
}

unsigned char *ext_rd(struct option *o, unsigned char *c)
{
	unsigned char *err = get_text("Error reading extension specification");
	struct extension new;
	unsigned char *w;
	int n;
	memset(&new, 0, sizeof(struct extension));
	if (!(new.ext = get_quoted(&c))) goto err;
	if (!(new.ct = get_quoted(&c))) goto err;
	update_ext(&new);
	err = NULL;
	err:
	if (new.ext) mem_free(new.ext);
	if (new.ct) mem_free(new.ct);
	return err;
}


void ext_wr(struct option *o, unsigned char **s, int *l)
{
	struct extension *a;
	foreachback(a, extensions) {
		add_nm(o, s, l);
		add_quoted_to_str(s, l, a->ext);
		add_to_str(s, l, " ");
		add_quoted_to_str(s, l, a->ct);
	}
}

unsigned char *term_rd(struct option *o, unsigned char *c)
{
	struct term_spec *ts;
	unsigned char *w;
	int i;
	if (!(w = get_quoted(&c))) goto err;
	if (!(ts = new_term_spec(w))) {
		mem_free(w);
		goto end;
	}
	mem_free(w);
	if (!(w = get_word(&c))) goto err;
	if (strlen(w) != 1 || w[0] < '0' || w[0] > '2') goto err_f;
	ts->mode = w[0] - '0';
	mem_free(w);
	if (!(w = get_word(&c))) goto err;
	if (strlen(w) != 1 || w[0] < '0' || w[0] > '1') goto err_f;
	ts->m11_hack = w[0] - '0';
	mem_free(w);
	if (!(w = get_word(&c))) goto err;
	if (strlen(w) != 1 || w[0] < '0' || w[0] > '1') goto err_f;
	ts->col = w[0] - '0';
	mem_free(w);
	if (!(w = get_word(&c))) goto err;
	if ((i = get_cp_index(w)) == -1) goto err_f;
	ts->charset = i;
	mem_free(w);
	end:
	return NULL;
	err_f:
	mem_free(w);
	err:
	return get_text("Error reading terminal specification");
}

unsigned char *term2_rd(struct option *o, unsigned char *c)
{
	struct term_spec *ts;
	unsigned char *w;
	int i;
	if (!(w = get_quoted(&c))) goto err;
	if (!(ts = new_term_spec(w))) {
		mem_free(w);
		goto end;
	}
	mem_free(w);
	if (!(w = get_word(&c))) goto err;
	if (strlen(w) != 1 || w[0] < '0' || w[0] > '2') goto err_f;
	ts->mode = w[0] - '0';
	mem_free(w);
	if (!(w = get_word(&c))) goto err;
	if (strlen(w) != 1 || w[0] < '0' || w[0] > '1') goto err_f;
	ts->m11_hack = w[0] - '0';
	mem_free(w);
	if (!(w = get_word(&c))) goto err;
	if (strlen(w) != 1 || w[0] < '0' || w[0] > '1') goto err_f;
	ts->restrict_852 = w[0] - '0';
	mem_free(w);
	if (!(w = get_word(&c))) goto err;
	if (strlen(w) != 1 || w[0] < '0' || w[0] > '1') goto err_f;
	ts->col = w[0] - '0';
	mem_free(w);
	if (!(w = get_word(&c))) goto err;
	if ((i = get_cp_index(w)) == -1) goto err_f;
	ts->charset = i;
	mem_free(w);
	end:
	return NULL;
	err_f:
	mem_free(w);
	err:
	return get_text("Error reading terminal specification");
}

void term2_wr(struct option *o, unsigned char **s, int *l)
{
	struct term_spec *ts;
	foreachback(ts, term_specs) {
		add_nm(o, s, l);
		add_quoted_to_str(s, l, ts->term);
		add_to_str(s, l, " ");
		add_num_to_str(s, l, ts->mode);
		add_to_str(s, l, " ");
		add_num_to_str(s, l, ts->m11_hack);
		add_to_str(s, l, " ");
		add_num_to_str(s, l, ts->restrict_852);
		add_to_str(s, l, " ");
		add_num_to_str(s, l, ts->col);
		add_to_str(s, l, " ");
		add_to_str(s, l, get_cp_mime_name(ts->charset));
	}
}

unsigned char *gen_cmd(struct option *o, unsigned char ***argv, int *argc)
{
	if (!*argc) return get_text("Parameter expected");
	(*argv)++; (*argc)--;
	return o->rd_cfg(o, *(*argv - 1));
}

unsigned char *lookup_cmd(struct option *o, unsigned char ***argv, int *argc)
{
	ip addr;
	unsigned char *p = (unsigned char *)&addr;
	if (!*argc) return get_text("Parameter expected");
	if (*argc >= 2) return get_text("Too many parameters");
	(*argv)++; (*argc)--;
	if (do_real_lookup(*(*argv - 1), &addr)) {
#ifdef HAVE_HERROR
		herror("error");
#else
		fprintf(stderr, "error: host not found\n");
#endif
		return "";
	}
	printf("%d.%d.%d.%d\n", (int)p[0], (int)p[1], (int)p[2], (int)p[3]);
	fflush(stdout);
	return "";
}

void end_config()
{
	if (links_home) mem_free(links_home);
	if (config_file) mem_free(config_file);
}

unsigned char system_name[MAX_STR_LEN];

unsigned char *links_home = NULL;
int first_use = 0;
unsigned char *config_file = NULL;
int created_cfg;

int async_lookup = 1;
int max_connections = 10;
int max_connections_to_host = 2;
int max_tries = 3;
int receive_timeout = 120;
int unrestartable_receive_timeout = 600;

int max_format_cache_entries = 5;
long memory_cache_size = 1048576;

int enable_html_tables = 1;
int enable_html_frames = 0;
int display_images = 0;

int assume_cp = 0;

struct rgb default_fg = { 191, 191, 191 };
struct rgb default_bg = { 0, 0, 0 };
struct rgb default_link = { 255, 255, 255 };
struct rgb default_vlink = { 255, 255, 0 };

int default_left_margin = HTML_LEFT_MARGIN;

 /* !!! WARNING: terminate all connections before change! */
unsigned char http_proxy[MAX_STR_LEN] = "";
unsigned char ftp_proxy[MAX_STR_LEN] = "";

unsigned char download_dir[MAX_STR_LEN] = "";

/* These are workarounds for some CGI script bugs */
int bug_302_redirect = 0;
	/* When got 301 or 302 from POST request, change it to GET
	   - this violates RFC2068, but some buggy message board scripts rely on it */
int bug_post_no_keepalive = 0;
	/* No keepalive connection after POST request. Some buggy PHP databases report bad
	   results if GET wants to retreive data POSTed in the same connection */

struct option options[] = {
	1, lookup_cmd, NULL, NULL, 0, 0, NULL, NULL, "lookup",
	1, gen_cmd, num_rd, num_wr, 0, 1, &async_lookup, "async_dns", "async-dns",
	1, gen_cmd, num_rd, num_wr, 1, 16, &max_connections, "max_connections", "max-connections",
	1, gen_cmd, num_rd, num_wr, 1, 8, &max_connections_to_host, "max_connections_to_host", "max-connections-to-host",
	1, gen_cmd, num_rd, num_wr, 1, 16, &max_tries, "retries", "retries",
	1, gen_cmd, num_rd, num_wr, 1, 1800, &receive_timeout, "receive_timeout", "receive-timeout",
	1, gen_cmd, num_rd, num_wr, 1, 1800, &unrestartable_receive_timeout, "unrestartable_receive_timeout", "unrestartable-receive-timeout",
	1, gen_cmd, num_rd, num_wr, 0, 256, &max_format_cache_entries, "format_cache_size", "format-cache-size",
	1, gen_cmd, num_rd, num_wr, 0, MAXINT, &memory_cache_size, "memory_cache_size", "memory-cache-size",
	1, gen_cmd, str_rd, str_wr, 0, MAX_STR_LEN, http_proxy, "http_proxy", "http-proxy",
	1, gen_cmd, str_rd, str_wr, 0, MAX_STR_LEN, ftp_proxy, "ftp_proxy", "ftp-proxy",
	1, gen_cmd, str_rd, str_wr, 0, MAX_STR_LEN, download_dir, "download_dir", "download-dir",
	1, gen_cmd, cp_rd, cp_wr, 0, 0, &assume_cp, "assume_codepage", "assume-codepage",
	1, NULL, term_rd, NULL, 0, 0, NULL, "terminal", NULL,
	1, NULL, term2_rd, term2_wr, 0, 0, NULL, "terminal2", NULL,
	1, NULL, type_rd, type_wr, 0, 0, NULL, "association", NULL,
	1, NULL, ext_rd, ext_wr, 0, 0, NULL, "extension", NULL,
	0, NULL, NULL, NULL, 0, 0, NULL, NULL, NULL,
};

