/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)psxy.c	2.139  11/24/99
 *
 *	Copyright (c) 1991-1999 by P. Wessel and W. H. F. Smith
 *	See COPYING file for copying and redistribution conditions.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; version 2 of the License.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	Contact info: www.soest.hawaii.edu/gmt
 *--------------------------------------------------------------------*/
/*
 * psxy will read <x,y> pairs (or <lon,lat>) from GMT_stdin and plot
 * the lines, polygons, or symbols on a map. A variety of symbols
 * may be specified, some of which require additional columns of
 * data, like symbol-size etc.  Only one symbol may be plotted
 * at the time.  PostScript code is written to GMT_stdout.
 *
 * Author:	Paul Wessel
 * Date:	24-NOV-1999
 * Version:	3.3.3
 *
 */

#include "gmt.h"

#define NONE		99
#define UNSPECIFIED	999
#define LINE		0
#define BAR		1
#define CROSS		2
#define POINT		3
#define CIRCLE		4
#define SQUARE		5
#define TRIANGLE	6
#define DIAMOND		7
#define STAR		8
#define HEXAGON		9
#define ITRIANGLE	10
#define ELLIPSE		11
#define VECTOR		12	
#define VECTOR2		13
#define TEXT		14
#define FAULT		-15

#define POINTSIZE 0.005

struct PSXY_SYMBOL {
	int symbol;		/* Symbol id */
	int n_required;		/* Number of additional columns necessary to decode chosen symbol */
	int u;			/* Measure unit id (0 = cm, 1 = inch, 2 = m, 3 = point */
	double size;		/* Current symbol size */
	double given_size;	/* Symbol size read from file or command line */
	BOOLEAN read_size;	/* TRUE when we must read symbol size from file */
	int font_no;		/* font to use for the -Sl symbol */

	/* These apply to bar symbols */

	double base;		/* From what level to draw the bar */
	BOOLEAN user_unit;	/* if TRUE */

	/* These apply to vectors */

	BOOLEAN convert_angles;	/* If TRUE, convert azimuth to angle on map */
	BOOLEAN read_vector;	/* if TRUE must read vector attributes */
	BOOLEAN shrink;		/* If TRUE, shrink vector attributes for small lengths */
	double v_norm;		/* shrink when lengths are smaller than this */
	double v_shrink;	/* Required scale factor */
	double v_width;		/* Width of vector stem in inches */
	double h_length;	/* Length of vector head in inches */
	double h_width;		/* Width of vector head in inches */

	char string[64];	/* Character code to plot (could be octal) */

	double f_gap;		/* Gap between fault ticks in inches */
	double f_len;		/* Length of fault ticks in inches */
	int f_sense;		/* Draw tick to left (+1), centered (0), or right (-1) of line */
	BOOLEAN f_triangle;	/* TRUE if teeth should be drawn, default is ticks */
} S;

double *xx, *yy, *xp, *yp;
int *plot_pen;

int decode_symbol_option (char *text, struct PSXY_SYMBOL *p);
void GMT_plot_ellipse (double lon, double lat, double major, double minor, double azimuth, struct GMT_FILL fill, int outline);
void GMT_draw_fence (double x[], double y[], int n, double gap, double len, int sense, int triangle, int rgb[], BOOLEAN outline);

main (int argc, char **argv)
{

	int 	i, j, n, j0, ix = 0, iy = 1, n_files = 0, fno, xy_errors[2], two, three, four;
	int	n_alloc = GMT_CHUNK, plot_n, n_use, start, n_args, n_fields, n_cols_start = 2;
	int 	font_size, n_required, n_total_read = 0, n_expected = 0;
	
	BOOLEAN	error = FALSE, nofile = TRUE, polygon = FALSE, read_symbol = FALSE;
	BOOLEAN	outline = FALSE, default_outline = FALSE, draw_arc = TRUE, error_bars = FALSE;
	BOOLEAN done, get_rgb = FALSE, greenwich, default_polygon = FALSE;
	BOOLEAN old_GMT_world_map, skip_if_outside = TRUE, close_polygon = FALSE, cpt_given = FALSE;
	BOOLEAN error_x = FALSE, error_y = FALSE, fill_set = FALSE, def_err_xy = FALSE;
	
	double west = 0.0, east = 0.0, south = 0.0, north = 0.0, delta_x, delta_y;
	double plot_x, plot_y, x_1, x_2, y_1, y_2, y0, error_width = 0.1, error_width2;
	double direction, length, major, minor, x2, y2, z;
	double v_w, h_l, h_w, tmp, step = 1.0, *in;
	
	char line[BUFSIZ], *cpt, *p, *symb_arg = CNULL, txt_a[32];
	
	FILE *fp = NULL;
	
	struct GMT_PEN pen, epen, default_pen;
	struct GMT_FILL fill, default_fill;
	
	xy_errors[0] = xy_errors[1] = 0;	/* These will be col # of where to find this info in data */
	
	argc = GMT_begin (argc, argv);
	
	GMT_init_pen (&pen, GMT_PENWIDTH);
	GMT_init_pen (&epen, GMT_PENWIDTH);
	GMT_init_fill (&fill, -1, -1, -1);	/* Default is no fill */
	
	/* Initialize PSXY_SYMBOL structure */

	S.symbol = S.n_required = 0;
	S.size = S.base = S.v_norm = S.v_shrink = 0.0;
	S.read_size = S.user_unit = S.convert_angles = S.shrink = S.read_vector = FALSE;
	S.v_width = 0.03;
	memset ((void *)S.string, 0, (size_t)(16 * sizeof (char)));
	S.h_length = 0.12;
	S.h_width = 0.1;
	S.font_no = gmtdefs.anot_font;

	if ((S.u = gmtdefs.measure_unit) == GMT_CM) {
		S.v_width = 0.075 / 2.54;	S.h_length = 0.3 / 2.54;
		S.h_width = 0.25 / 2.54; error_width = 0.25 / 2.54;
	}
	
	/* Check and interpret the command line arguments */
	
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch(argv[i][1]) {
		
				/* Common parameters */
			
				case 'B':
				case 'H':
				case 'J':
				case 'K':
				case 'O':
				case 'P':
				case 'R':
				case 'U':
				case 'V':
				case 'X':
				case 'x':
				case 'Y':
				case 'y':
				case 'c':
				case ':':
				case '\0':
					error += GMT_get_common_args (argv[i], &west, &east, &south, &north);
					break;
				
				/* Supplemental parameters */
			
				case 'b':
					error += GMT_io_selection (&argv[i][2]);
					break;
					
				case 'A':	/* Turn off draw_arc mode */
					if (argv[i][2])
						step = atof (&argv[i][2]);	/* Undocumented test feature */
					else
						draw_arc = FALSE;
					break;
					
				case 'C':	/* Vary symbol color with z */
					cpt = &argv[i][2];
					cpt_given = TRUE;
					break;
				case 'E':	/* Get info for error bars */
					for (j = 2; argv[i][j] && argv[i][j] != '/'; j++) {
						if (argv[i][j] == 'x')
							xy_errors[0] = j;
						else if (argv[i][j] == 'y')
							xy_errors[1] = j;
						else {	/* Get error 'cap' width */
							strcpy (txt_a, &argv[i][j]);
							j0 = 0;
							while (txt_a[j0] && txt_a[j0] != '/') j0++;
							txt_a[j0] = 0;
							error_width = GMT_convert_units (txt_a, GMT_INCH);
							while (argv[i][j] && argv[i][j] != '/') j++;
							j--;
						}
					}
					error_bars = TRUE;
					if (!(xy_errors[0] || xy_errors[1])) {	/* Default is both */
						def_err_xy = TRUE;
						xy_errors[0] = 2;	/* Assumes xy input, later check for -: */
						xy_errors[1] = 3;
					}
					if (argv[i][j] == '/') GMT_getpen (&argv[i][j+1], &epen);
					break;
				case 'G':		/* Set fill for symbols or polygon */
					if (!argv[i][2] || (argv[i][2] && GMT_getfill (&argv[i][2], &fill))) {
						GMT_fill_syntax ('G');
						error++;
					}
					fill_set = TRUE;
					break;
				case 'L':		/* Close line segments */
					close_polygon = TRUE;
					break;
				case 'M':		/* Multiple line segments */
					GMT_multisegment (&argv[i][2]);
					break;
				case 'N':		/* Do not skip points outside border */
					skip_if_outside = FALSE;
					break;
				case 'S':		/* Get symbol [and size] */
					symb_arg = &argv[i][2];
					error += decode_symbol_option (&argv[i][2], &S);
					break;
				case 'W':		/* Set line attributes */
					if (argv[i][2] && GMT_getpen (&argv[i][2], &pen)) {
						GMT_pen_syntax ('W');
						error++;
					}
					default_outline = outline = TRUE;
					break;
					
				/* Illegal options */
			
				default:		/* Options not recognized */
					error = TRUE;
					GMT_default_error (argv[i][1]);
					break;
			}
		}
		else
			n_files++;
	}
	
	if (argc == 1 || GMT_quick) {	/* Display usage */
		fprintf (stderr,"psxy %s - Plot lines, polygons, and symbols on maps\n\n", GMT_VERSION);
		fprintf (stderr,"usage: psxy <infiles> -J<params> -R<west/east/south/north> [-A] [-B<tickinfo>]\n");
		fprintf (stderr, "	[-C<cpt>] [-E[x|y][cap][/<pen>]] [-G<fill>] [-H[<nrec>]] [-K] [-L] [-M<flag>] [-N]\n");
		fprintf (stderr, "	[-O] [-P] [-S[<symbol>][<size>]] [-U[<label>]] [-V] [-W[<pen>]] [-c<ncopies>]\n");
		fprintf (stderr, "	[-X<x_shift>] [-Y<y_shift>] [-:] [-bi[s][<n>]]\n\n");
		
		if (GMT_quick) exit (EXIT_FAILURE);
		
		fprintf (stderr, "	<infiles> is one or more files.  If no, read standard input\n");
		GMT_explain_option ('j');
		GMT_explain_option ('R');
		fprintf (stderr, "\n\tOPTIONS:\n");
		GMT_explain_option ('b');
		fprintf (stderr, "	-A Suppress drawing line segments as great circle arcs\n");
		fprintf (stderr, "	-C Use cpt-file to assign symbol colors based on z-value in 3rd column (requires -S)\n");
		fprintf (stderr, "	   Without -S, psxy excepts -M polygons and looks for -Z<val> options in each multiheader\n");
		fprintf (stderr, "	-E means draw error bars for x, y, or both.  Add cap-width [%lg]. Append pen attributes.\n", error_width);
		fprintf (stderr, "	-G Specify color (for symbols/polygons) or pattern (for polygons). fill can be either\n");
		fprintf (stderr, "	   1) <r/g/b> (each 0-255) for color or <gray> (0-255) for gray-shade [0].\n");
		fprintf (stderr, "	   2) p[or P]<dpi>/<pattern> for predefined patterns (1-90).\n");
		fprintf (stderr, "	   3) not given; -G option is present in all subheader (requires -M and no -S)\n");
		fprintf (stderr, "	   Default is no fill (transparent symbols or polygons)\n");
		GMT_explain_option ('H');
		GMT_explain_option ('K');
		fprintf (stderr, "	-L Force closed polygons.\n");
		GMT_explain_option ('M');
		fprintf (stderr, "	-N Do Not skip/clip symbols that fall outside map border [Default will ignore those outside]\n");
		GMT_explain_option ('O');
		GMT_explain_option ('P');
		fprintf (stderr, "	-S to select symbol type and symbol size (in %s).  Choose between\n", GMT_unit_names[gmtdefs.measure_unit]);
		fprintf (stderr, "	   st(a)r, (b)ar, (c)ircle, (d)iamond, (e)llipse, (f)ault, (h)exagon,\n");
		fprintf (stderr, "	   (i)nvtriangle, (l)etter, (p)oint, (s)quare, (t)riangle, (v)ector, (x)cross\n");
		fprintf (stderr, "	   If no size is specified, psxy expects the 3rd column to have sizes.\n");
		fprintf (stderr, "	   If no symbol is specified, psxy expects the last column to have symbol code.\n");
		fprintf (stderr, "	   [Note: if -C is selected then 3rd means 4th column, etc.]\n");
		fprintf (stderr, "	   Symbols A, C, D, H, I, S, T are adjusted to have same area as a circle of given size\n");
		fprintf (stderr, "	   Bars: Append b<base> to give the y-value of the base [Default = 0]\n");
		fprintf (stderr, "	      Append u if width is in x-input units [Default is %s]\n", GMT_unit_names[gmtdefs.measure_unit]);
		fprintf (stderr, "	   Ellipses: the direction, major, and minor axis must be in input columns 3, 4 and 5.\n");
		fprintf (stderr, "	     If -SE rather than -Se is selected, psxy will expect azimuth, and axes in km\n");
		fprintf (stderr, "	     and convert azimuths based on the chosen map projection\n");
		fprintf (stderr, "	   Faults: Give tickgap/ticklen[type], where type is:\n");
		fprintf (stderr, "	     l - draw tick to left, r - draw tick to right [Default is centered]\n");
		fprintf (stderr, "	     Upper case L or R gives filled triangles instead of tick\n");
		fprintf (stderr, "	   Letter: append /<string> after symbol size, and optionally %%<font>\n");
		fprintf (stderr, "	   Vectors: the direction and length must be in input columns 3 and 4.\n");
		fprintf (stderr, "	     Furthermore, <size> means arrowwidth/headlength/headwith [Default is %lg/%lg/%lg]\n", S.v_width, S.h_length, S.h_width);
		fprintf (stderr, "	     If -SV rather than -Sv is selected, psxy will expect azimuth and length\n");
		fprintf (stderr, "	     and convert azimuths based on the chosen map projection\n");
		GMT_explain_option ('U');
		GMT_explain_option ('V');
		fprintf (stderr, "	-W sets pen attributes [width = %lgp, color = (%d/%d/%d), texture = solid line].\n", 
			pen.width, pen.rgb[0], pen.rgb[1], pen.rgb[2]);
		fprintf (stderr, "	   Implicitly draws symbol outline with this pen.\n");
		GMT_explain_option ('X');
		GMT_explain_option ('c');
		GMT_explain_option (':');
		GMT_explain_option ('i');
		GMT_explain_option ('n');
		fprintf (stderr, "	   Default is the required number of columns\n");
		GMT_explain_option ('.');
		exit (EXIT_FAILURE);
	}

	/* Check that the options selected are mutually consistant */
	
	if (S.symbol > 0 && cpt_given) get_rgb = TRUE;	/* Need to read z-vales from input data file */
	
	if (error_bars && (S.read_vector || S.symbol == ELLIPSE || S.symbol == FAULT)) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -E option: Incompatible with -Se, -Sf, -Sv\n", GMT_program);
		error++;
	}
	if (fill.use_pattern && S.symbol != LINE) {	/* fill-pattern only for polygons */
		error++;
		fprintf (stderr, "%s: GMT SYNTAX ERROR -G option: Fill-pattern only used with polygons\n", GMT_program);
	}
	if (cpt_given && S.symbol == 0 && !GMT_io.multi_segments) {
		error++;
		fprintf (stderr, "%s: GMT SYNTAX ERROR -C option: Must also specify a symbol (see -S) or give polygon file with -M\n", GMT_program);
	}
	if (!project_info.region_supplied) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Must specify -R option\n", GMT_program);
		error++;
	}
	if (GMT_io.binary[0] && gmtdefs.io_header) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR.  Binary input data cannot have header -H\n", GMT_program);
		error++;
	}
	if (GMT_io.binary[0] && S.symbol == UNSPECIFIED) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR.  Binary input data cannot have symbol information\n", GMT_program);
		error++;
	}

	if (error) exit (EXIT_FAILURE);

	read_symbol = (S.symbol == UNSPECIFIED);	/* Only for ascii input */

	if (S.symbol == 0 && (fill_set || close_polygon)) polygon = TRUE;
	if (S.symbol == FAULT) polygon = FALSE;
	
	default_pen = pen;
	default_fill = fill;
	default_outline = outline;
	default_polygon = polygon;

	if (error_bars && S.symbol == 0)	/* Assume user only wants error bars */
		S.symbol = NONE;
		
	if (cpt_given) {
		GMT_read_cpt (cpt);
		if (get_rgb) n_cols_start++;
	}
		
	if (n_files > 0)
		nofile = FALSE;
	else
		n_files = 1;
	n_args = (argc > 1) ? argc : 2;
	
	if (error_bars) {
		if (def_err_xy && gmtdefs.xy_toggle) i_swap (xy_errors[0], xy_errors[1]);	/* -E should be -Eyx */
		if (xy_errors[0]) n_cols_start++, error_x = TRUE;
		if (xy_errors[1]) n_cols_start++, error_y = TRUE;
		xy_errors[0] += (S.read_size + get_rgb);	/* Move 0-2 columns over */
		xy_errors[1] += (S.read_size + get_rgb);
		error_width2 = 0.5 * error_width;
	}
	two   = (get_rgb) ? 3 : 2;
	three = (get_rgb) ? 4 : 3;
	four  = (get_rgb) ? 5 : 4;
	n_required = n_cols_start + S.n_required;

	if (GMT_io.binary[0] && GMT_io.ncol[0] == 0) GMT_io.ncol[0] = n_required;
	n_expected = (GMT_io.binary[0]) ? GMT_io.ncol[0] : n_required;
	if (n_expected < n_required) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR.  Binary input data must have at least %d columns\n", GMT_program, n_required);
		error++;
	}

	GMT_put_history (argc, argv);	/* Update .gmtcommands */

	if (GMT_io.binary[0] && gmtdefs.verbose) {
		char *type[2] = {"double", "single"};
		fprintf (stderr, "%s: Expects %d-column %s-precision binary data\n", GMT_program, GMT_io.ncol[0], type[GMT_io.single_precision[0]]);
	}

	GMT_map_setup (west, east, south, north);

	greenwich = (project_info.w <= 0.0 || project_info.e <= 0.0);
	
	ps_plotinit (CNULL, gmtdefs.overlay, gmtdefs.page_orientation, gmtdefs.x_origin, gmtdefs.y_origin,
		gmtdefs.global_x_scale, gmtdefs.global_y_scale, gmtdefs.n_copies,
		gmtdefs.dpi, GMT_INCH, gmtdefs.paper_width, gmtdefs.page_rgb, GMT_epsinfo (argv[0]));
	GMT_echo_command (argc, argv);
	if (gmtdefs.unix_time) GMT_timestamp (argc, argv);
	
	GMT_setpen (&pen);
	if (S.symbol == TEXT && fill_set && !outline) ps_setpaint (fill.rgb);
	
	if (S.symbol == TEXT) {
		font_size = irint (S.size * 72.0);	/* To get points */
		ps_setfont (S.font_no);			/* Set the required font */
	}
	if (S.symbol == FAULT || (S.symbol > 0 && skip_if_outside)) GMT_map_clip_on (GMT_no_rgb, 3);
	
	GMT_geo_to_xy (west, S.base, &plot_x, &y0);	/* Zero level for bars */
	old_GMT_world_map = GMT_world_map;
	
	ix = (gmtdefs.xy_toggle);	iy = 1 - ix;
	if (S.symbol <= 0) {
		xx = (double *) GMT_memory (VNULL, (size_t)n_alloc, sizeof (double), GMT_program);
		yy = (double *) GMT_memory (VNULL, (size_t)n_alloc, sizeof (double), GMT_program);
	}
	done = FALSE;
	for (fno = 1; !done && fno < n_args; fno++) {	/* Loop over all input files */
		if (!nofile && argv[fno][0] == '-') continue;
		if (nofile) {
			fp = GMT_stdin;
			done = TRUE;
#ifdef SET_IO_MODE
			GMT_setmode (0);
#endif
		}
		else if ((fp = GMT_fopen (argv[fno], GMT_io.r_mode)) == NULL) {
			fprintf (stderr, "%s: Cannot open file %s\n", GMT_program, argv[fno]);
			continue;
		}

		if (!nofile && gmtdefs.verbose) {
			fprintf (stderr, "%s: Working on file %s\n", GMT_program, argv[fno]);
			sprintf (line, "File: %s\0", argv[fno]);
			ps_comment (line);
		}
		if (gmtdefs.io_header) for (i = 0; i < gmtdefs.n_header_recs; i++) fgets (line, BUFSIZ, fp);
		
		if (S.symbol > 0) {	/* symbol part */
		
			GMT_world_map = (S.symbol == ELLIPSE && S.convert_angles) ? FALSE : TRUE;
			if (read_symbol) n_expected = BUFSIZ;
			while ((n_fields = GMT_input (fp, &n_expected, &in)) >= 0 && !(GMT_io.status & GMT_IO_EOF)) {

				while ((GMT_io.status & GMT_IO_SEGMENT_HEADER) && !(GMT_io.status & GMT_IO_EOF)) {	/* Skip segment headers */
					if (gmtdefs.verbose) ps_comment (GMT_io.segment_header);
					if ((p = strstr (GMT_io.segment_header, " -G"))) {
						strcpy (line, &p[3]);
						for (i = 0; line[i]; i++) if (line[i] == ' ') line[i] = '\0';
						if (GMT_getfill (line, &fill)) {
							fprintf (stderr, "%s: Trouble decoding fill info from header near line %d\n", GMT_program,  n_total_read);
						}
					}
					else
						fill = default_fill;
					if ((p = strstr (GMT_io.segment_header, " -W"))) {
						strcpy (line, &p[3]);
						for (i = 0; line[i]; i++) if (line[i] == ' ') line[i] = '\0';
						if (GMT_getpen (line, &pen)) {
							fprintf (stderr, "%s: Trouble decoding pen info from header near line %d\n", GMT_program,  n_total_read);
						}
						else {
							GMT_setpen (&pen);
							outline = TRUE;
						}
					}
					else {
						GMT_setpen (&default_pen);
						outline = default_outline;
					}

					if (read_symbol) n_expected = BUFSIZ;
					n_fields = GMT_input (fp, &n_expected, &in);
				}
				if ((GMT_io.status & GMT_IO_EOF)) continue;	/* At EOF */

				if (GMT_io.status & GMT_IO_MISMATCH && !read_symbol) {
					fprintf (stderr, "%s: Mismatch between actual (%d) and expected (%d) fields near line %d\n", GMT_program, n_fields, n_expected, n_total_read);
					exit (EXIT_FAILURE);
				}
				n_total_read++;
			
				S.size = 0.0;
				if (S.read_size) {
					S.size = in[two] * GMT_u2u[S.u][GMT_INCH];
					if (S.symbol == TEXT) font_size = irint (S.size * 72.0);
				}
				if (read_symbol) {
					i = strlen (GMT_io.current_record) - 1;
					while (GMT_io.current_record[i] && !strchr (" ,\t", (int)GMT_io.current_record[i])) i--;
					(void ) decode_symbol_option (&GMT_io.current_record[i+1], &S);
					n_required = n_cols_start + S.n_required;
				}
				else {
					(void ) decode_symbol_option (symb_arg, &S);
					n_required = n_cols_start + S.n_required;
				}

				if (get_rgb) {
					GMT_get_rgb24 (in[2], fill.rgb);
					if (S.symbol == CROSS || S.symbol == POINT || S.symbol == TEXT) ps_setpaint (fill.rgb);
				}
				if (S.read_vector) {
					direction = in[two];
					length = in[three] * GMT_u2u[S.u][GMT_INCH];
				}
				else if (S.symbol == ELLIPSE && S.convert_angles) {	/* Leave axes in km */
					direction = in[two];
					major = in[three];
					minor = in[four];
				}
				else if (S.symbol == ELLIPSE) {
					direction = in[two];
					major = in[three] * GMT_u2u[S.u][GMT_INCH];
					minor = in[four] * GMT_u2u[S.u][GMT_INCH];
				}
				
				/* Skip zero-size symbols */
				
				if (S.symbol != POINT && S.symbol < ELLIPSE && S.size <= 0.0) continue;
				if (S.read_vector && length <= 0.0) continue;
				if (S.symbol == ELLIPSE && (major <= 0.0 || minor <= 0.0)) continue;
				
				if (xy_errors[0]) delta_x = in[xy_errors[0]];
				if (xy_errors[1]) delta_y = in[xy_errors[1]];
				
				if (skip_if_outside) {
					GMT_map_outside (in[ix], in[iy]);
					if ( abs (GMT_x_status_new) > 1 || abs (GMT_y_status_new) > 1) continue;
				}

				GMT_geo_to_xy (in[ix], in[iy], &plot_x, &plot_y);
				if (GMT_is_dnan (plot_x) || GMT_is_dnan (plot_y)) {
					fprintf (stderr, "%s: Data point exceeded domain near line %d\n", GMT_program, n_total_read);
					continue;
				}
			
				if (error_bars) {
					GMT_setpen (&epen);
					if (error_x) {
						GMT_geo_to_xy (in[ix] - delta_x, in[iy], &x_1, &y_1);
						GMT_geo_to_xy (in[ix] + delta_x, in[iy], &x_2, &y_2);
						if (GMT_is_dnan (x_1)) {
							fprintf (stderr, "%s: X error bar exceeded domain near line %d\n", GMT_program, n_total_read);
							x_1 = project_info.xmin;
						}
						if (GMT_is_dnan (x_2)) {
							fprintf (stderr, "%s: X error bar exceeded domain near line %d\n", GMT_program, n_total_read);
							x_2 = project_info.xmax;
						}
						ps_plot (x_1, y_1, 3);
						ps_plot (x_2, y_2, -2);
						if (error_width > 0.0) {
							ps_plot (x_1, y_1 - error_width2, 3);
							ps_plotr (0.0, error_width, 2);
							ps_plot (x_2, y_2 - error_width2, 3);
							ps_plotr (0.0, error_width, -2);
						}
					}
					if (error_y) {
						GMT_geo_to_xy (in[ix], in[iy] - delta_y, &x_1, &y_1);
						GMT_geo_to_xy (in[ix], in[iy] + delta_y, &x_2, &y_2);
						if (GMT_is_dnan (y_1)) {
							fprintf (stderr, "%s: Y error bar exceeded domain near line %d\n", GMT_program, n_total_read);
							y_1 = project_info.ymin;
						}
						if (GMT_is_dnan (y_2)) {
							fprintf (stderr, "%s: Y error bar exceeded domain near line %d\n", GMT_program, n_total_read);
							y_2 = project_info.ymax;
						}
						ps_plot (x_1, y_1, 3);
						ps_plot (x_2, y_2, -2);
						if (error_width > 0.0) {
							ps_plot (x_1 - error_width2, y_1, 3);
							ps_plotr (error_width, 0.0, 2);
							ps_plot (x_2 - error_width2, y_2, 3);
							ps_plotr (error_width, 0.0, -2);
						}
					}
					GMT_setpen (&pen);
				}

				switch (S.symbol) {
					case NONE:
						break;
					case STAR:
						ps_star (plot_x, plot_y, S.size, fill.rgb, outline);
						break;
					case BAR:
						if (S.user_unit) {	/* Width measured in x units */
							GMT_geo_to_xy (in[ix]-S.size, S.base, &x_1, &y_1);
							GMT_geo_to_xy (in[ix]+S.size, in[iy], &x_2, &y_2);
					
							ps_rect (x_1, y_1, x_2, y_2, fill.rgb, outline);
						}
						else
							ps_rect (plot_x-S.size, y0, plot_x+S.size, plot_y, fill.rgb, outline);
						break;
					case CROSS:
						ps_cross (plot_x, plot_y, S.size);
						break;
					case POINT:
						ps_cross (plot_x, plot_y, POINTSIZE);
						break;
					case CIRCLE:
						ps_circle (plot_x, plot_y, S.size, fill.rgb, outline);
						break;
					case SQUARE:
						ps_square (plot_x, plot_y, S.size, fill.rgb, outline);
						break;
					case HEXAGON:
						ps_hexagon (plot_x, plot_y, S.size, fill.rgb, outline);
						break;
					case TRIANGLE:
						ps_triangle (plot_x, plot_y, S.size, fill.rgb, outline);
						break;
					case ITRIANGLE:
						ps_itriangle (plot_x, plot_y, S.size, fill.rgb, outline);
						break;
					case DIAMOND:
						ps_diamond (plot_x, plot_y, S.size, fill.rgb, outline);
						break;
					case TEXT:
						if (outline && fill_set) {
							ps_setpaint (fill.rgb);
							ps_text (plot_x, plot_y, font_size, S.string, 0.0, 6, FALSE);
							ps_setpaint (pen.rgb);
							ps_text (plot_x, plot_y, font_size, S.string, 0.0, 6, TRUE);
						}
						else if (fill_set)
							ps_text (plot_x, plot_y, font_size, S.string, 0.0, 6, FALSE);
						else
							ps_text (plot_x, plot_y, font_size, S.string, 0.0, 6, TRUE);
						break;
					case ELLIPSE:
						if (S.convert_angles)
							GMT_plot_ellipse (in[ix], in[iy], major, minor, direction, fill, outline);
						else
							ps_ellipse (plot_x, plot_y, direction, major, minor, fill.rgb, outline);
						break;
					case VECTOR:
						if (S.convert_angles) {
							GMT_azim_to_angle (in[ix], in[iy], 0.1, direction, &tmp);
							direction = tmp;
						}
						x2 = plot_x + length * cos (direction * D2R);
						y2 = plot_y + length * sin (direction * D2R);
						ps_vector (plot_x, plot_y, x2, y2, S.v_width, S.h_length, S.h_width, gmtdefs.vector_shape, fill.rgb, outline);
						break;
					case VECTOR2:
						if (S.convert_angles) {
							GMT_azim_to_angle (in[ix], in[iy], 1.0, direction, &tmp);
							direction = tmp;
						}
						x2 = plot_x + length * cos (direction * D2R);
						y2 = plot_y + length * sin (direction * D2R);
						if (length < S.v_norm) {	/* Scale arrow attributes down with length */
							v_w = S.v_width * length * S.v_shrink;
							h_l = S.h_length * length * S.v_shrink;
							h_w = S.h_width * length * S.v_shrink;
							ps_vector (plot_x, plot_y, x2, y2, v_w, h_l, h_w, gmtdefs.vector_shape, fill.rgb, outline);
						}
						else	/* Leave as specified */
							ps_vector (plot_x, plot_y, x2, y2, S.v_width, S.h_length, S.h_width, gmtdefs.vector_shape, fill.rgb, outline);
						break;
				}
				if (read_symbol) n_expected = BUFSIZ;
			}
		}
		else {	/* Line/polygon part */

			n_required = 2;
			n_fields = GMT_input (fp, &n_expected, &in);
			while (! (GMT_io.status & GMT_IO_EOF)) {	/* Not yet EOF */

				while (GMT_io.status & GMT_IO_SEGMENT_HEADER) {
					if (gmtdefs.verbose) ps_comment (GMT_io.segment_header);
					if ((p = strstr (GMT_io.segment_header, " -G"))) {	/* Set polygon fill */
						strcpy (line, &p[3]);
						for (i = 0; line[i]; i++) if (line[i] == ' ') line[i] = '\0';
						if (GMT_getfill (line, &fill)) {
							fprintf (stderr, "%s: Trouble decoding fill info from header near line %d\n", GMT_program,  n_total_read);
						}
						polygon = TRUE;
					}
					else if ((p = strstr (GMT_io.segment_header, " -Z"))) {	/* Set polygon r/g/b via cpt-lookup */
						z = atof (&p[3]);
						GMT_get_rgb24 (z, fill.rgb);
						fill.use_pattern = FALSE;
						polygon = TRUE;
					}
					else {
						polygon = default_polygon;
						fill = default_fill;
					}
					if ((p = strstr (GMT_io.segment_header, " -W"))) {
						strcpy (line, &p[3]);
						for (i = 0; line[i]; i++) if (line[i] == ' ') line[i] = '\0';
						if (GMT_getpen (line, &pen)) {
							fprintf (stderr, "%s: Trouble decoding pen info from header near line %d\n", GMT_program,  n_total_read);
						}
						else {
							GMT_setpen (&pen);
							outline = TRUE;
						}
					}
					else {
						GMT_setpen (&default_pen);
						outline = default_outline;
					}
					
					n_fields = GMT_input (fp, &n_expected, &in);
					n_total_read++;
				}
				if ((GMT_io.status & GMT_IO_EOF)) continue;	/* At EOF */

				n = 0;
				while (! (GMT_io.status & (GMT_IO_SEGMENT_HEADER | GMT_IO_EOF))) {	/* Keep going until FALSE or = 2 segment header */
					if (GMT_io.status & GMT_IO_MISMATCH) {
						fprintf (stderr, "%s: Mismatch between actual (%d) and expected (%d) fields near line %d\n", GMT_program, n_fields, n_expected, n_total_read);
						exit (EXIT_FAILURE);
					}

					xx[n] = in[ix];	yy[n] = in[iy];
					n++;
					if (n == n_alloc) {
						n_alloc += GMT_CHUNK;
						xx = (double *) GMT_memory ((void *)xx, (size_t)n_alloc, sizeof (double), GMT_program);
						yy = (double *) GMT_memory ((void *)yy, (size_t)n_alloc, sizeof (double), GMT_program);
					}
					n_fields = GMT_input (fp, &n_expected, &in);
				}
				if (polygon && !(xx[0] == xx[n-1] && yy[0] == yy[n-1])) {	/* Explicitly close polygon so that arc will work */
					xx[n] = xx[0];
					yy[n] = yy[0];
					n++;
					if (n == n_alloc) {
						n_alloc += GMT_CHUNK;
						xx = (double *) GMT_memory ((void *)xx, (size_t)n_alloc, sizeof (double), GMT_program);
						yy = (double *) GMT_memory ((void *)yy, (size_t)n_alloc, sizeof (double), GMT_program);
					}
				}
		
				xx = (double *) GMT_memory ((void *)xx, (size_t)n, sizeof (double), GMT_program);
				yy = (double *) GMT_memory ((void *)yy, (size_t)n, sizeof (double), GMT_program);
				n_alloc = n;
				
				if (draw_arc && project_info.projection > 5) n = GMT_fix_up_path (&xx, &yy, n, greenwich, step);

				if (polygon) {
					if ((plot_n = GMT_clip_to_map (xx, yy, n, &xp, &yp)) == 0) continue;
					while (plot_n > GMT_n_alloc) GMT_get_plot_array ();
					memcpy ((void *)GMT_x_plot, (void *)xp, (size_t)(plot_n * sizeof (double)));
					memcpy ((void *)GMT_y_plot, (void *)yp, (size_t)(plot_n * sizeof (double)));
				}
				else {
					if ((plot_n = GMT_geo_to_xy_line (xx, yy, n)) == 0) continue;
					xp = (double *) GMT_memory (VNULL, (size_t)plot_n, sizeof (double), GMT_program);
					yp = (double *) GMT_memory (VNULL, (size_t)plot_n, sizeof (double), GMT_program);
					plot_pen = (int *) GMT_memory (VNULL, (size_t)plot_n, sizeof (int), GMT_program);
					memcpy ((void *)xp, (void *)GMT_x_plot, (size_t)(plot_n * sizeof (double)));
					memcpy ((void *)yp, (void *)GMT_y_plot, (size_t)(plot_n * sizeof (double)));
					memcpy ((void *)plot_pen, (void *)GMT_pen, (size_t)(plot_n * sizeof (int)));
				}
		
				if (polygon) {
					
					if ((*GMT_will_it_wrap) (xp, yp, plot_n, &start)) {	/* Polygon wraps */
					
						/* First truncate agains left border */
						
						GMT_n_plot = (*GMT_truncate) (xp, yp, plot_n, start, -1);
						n_use = GMT_compact_line (GMT_x_plot, GMT_y_plot, GMT_n_plot, FALSE, 0);
						GMT_fill (GMT_x_plot, GMT_y_plot, n_use, &fill, outline);
								
						/* Then truncate agains right border */
						
						GMT_n_plot = (*GMT_truncate) (xp, yp, plot_n, start, +1);
						n_use = GMT_compact_line (GMT_x_plot, GMT_y_plot, GMT_n_plot, FALSE, 0);
						GMT_fill (GMT_x_plot, GMT_y_plot, n_use, &fill, outline);
					
					}
					else
						GMT_fill (xp, yp, plot_n, &fill, outline);
				}
				else
					GMT_plot_line (xp, yp, plot_pen, plot_n);
										
				if (S.symbol == FAULT) GMT_draw_fence (xp, yp, plot_n, S.f_gap, S.f_len, S.f_sense, S.f_triangle, fill.rgb, outline); 	/* Must draw fault crossbars */
				GMT_free ((void *)xp);
				GMT_free ((void *)yp);
				if (!polygon) GMT_free ((void *)plot_pen);
			}
		}
		if (fp != GMT_stdin) GMT_fclose (fp);
	}
	
	if (S.symbol == FAULT || (S.symbol > 0 && skip_if_outside)) GMT_map_clip_off ();

	GMT_world_map = old_GMT_world_map;
	
	if (frame_info.plot) GMT_map_basemap ();

	ps_plotend (gmtdefs.last_page);
	
	if (S.symbol == 0) {
		GMT_free ((void *)xx);
		GMT_free ((void *)yy);
	}
	
	GMT_end (argc, argv);
}

int decode_symbol_option (char *text, struct PSXY_SYMBOL *p)
{
	int decode_error = 0, slash, bset = 0, k, j, n, len;
	BOOLEAN check = TRUE;
	char symbol_type, txt_a[32], txt_b[32], txt_c[32], text_cp[80], *c;
	p->n_required = 0;

	if (!text) return -1;

	if (!text[0]) {	/* No symbol or size given */
		p->size = 0.0;
		symbol_type = '+';
	}
	else if (isdigit ((int)text[0]) || text[0] == '.') {	/* Size, but no symbol given */
		p->size = p->given_size = GMT_convert_units (text, GMT_INCH);
		symbol_type = '+';
	}
	else if (text[0] == 'l') {	/* Letter symbol is special case */
		strcpy (text_cp, text);
		if ((c = strchr (text_cp, '%'))) {	/* Gave font name or number, too */
			*c = ' ';	/* Make the % a space */
			c++;		/* Go to next character */
			if (c[0] >= '0' && c[0] <= '9')	/* Gave a font # */
				p->font_no = atoi (c);
			else
				p->font_no = GMT_key_lookup (c, GMT_font_name, N_FONTS);
			if (p->font_no >= N_FONTS) {
				fprintf (stderr, "%s: -Sl contains bad font (set to %s (0))\n", GMT_program, GMT_font_name[gmtdefs.anot_font]);
				p->font_no = gmtdefs.anot_font;
			}
		}
		if (text[1] == '/') {	/* No size given */
			symbol_type = 'l';
			if (p->size == 0.0) p->size = p->given_size;
		}
		else {
			n = sscanf (text_cp, "%c%[^/]/%s", &symbol_type, txt_a, &p->string);
			p->given_size = p->size = GMT_convert_units (txt_a, GMT_INCH);
		}
	}
	else if (strchr ("aAbCcDdeEfHhIipSsTtVvx", (int) text[0]) && strchr ("CcIiMmPp", (int) text[1])) {	/* Symbol, but no size given (size assumed given as input data), only unit information provided here */
		n = sscanf (text, "%c", &symbol_type);
		if (p->size == 0.0) p->size = p->given_size;
		if (text[1] && (p->u = GMT_get_unit (text[1])) < 0) decode_error = TRUE;
	}
	else if (strchr ("aAbCcDdeEfHhIipSsTtVvx", (int) text[0]) && (text[1] == '\n' || text[1] == '\0')) {	/* Symbol, but no size given (size assumed given as input data) */
		n = sscanf (text, "%c", &symbol_type);
		if (p->size == 0.0) p->size = p->given_size;
	}
	else if (text[0] == 'b') {	/* Bar with size */
		for (j = 1, bset = 0; text[j] && !bset; j++) if (text[j] == 'b') bset = j;
		strcpy (text_cp, text);
		if (bset) text_cp[bset] = 0;	/* Chop off the b<base> from copy */
		if ((bset && text_cp[bset-1] == 'u') || (j && text[j-1] == 'u')) p->user_unit = TRUE;
		n = sscanf (text_cp, "%c%s", &symbol_type, txt_a);
		if (n == 2) {
			if (p->user_unit)
				p->size = p->given_size = atof (txt_a);
			else
				p->size = p->given_size = GMT_convert_units (txt_a, GMT_INCH);
		}
		else	/* No size given */
			if (p->size == 0.0) p->size = p->given_size;
	}
	else {
		n = sscanf (text, "%c%s", &symbol_type, txt_a);
		p->given_size = p->size = GMT_convert_units (txt_a, GMT_INCH);
	}

	check = TRUE;
	switch (symbol_type) {
		case '+':
			p->symbol = UNSPECIFIED;
			break;
		case 'A':
			p->size *= 1.6728932614;	/* To equal area of circle with same size */
		case 'a':
			p->symbol = STAR;
			break;
		case 'b':
			p->symbol = BAR;
			p->size *= 0.5;		/* We will use +- to get full width */
			if (bset) p->base = atof (&text[bset+1]);
			break;
		case 'C':
		case 'c':
			p->symbol = CIRCLE;
			break;
		case 'D':
			p->size *= 1.2533141373;	/* To equal area of circle with same size */
		case 'd':
			p->symbol = DIAMOND;
			break;
		case 'E':	/* Expect axis in km to be scaled based on -J */
			p->convert_angles = TRUE;
		case 'e':
			p->symbol = ELLIPSE;
			p->n_required = 3;
			check = FALSE;
			break;
		case 'f':
			p->symbol = FAULT;
			len = strlen(text);
			switch (text[len-1]) {
				case 'L':
					p->f_triangle = TRUE;
				case 'l':
					p->f_sense = 1;
					len--;
					break;
				case 'R':
					p->f_triangle = TRUE;
				case 'r':
					p->f_sense = -1;
					len--;
					break;
				default:
					p->f_sense = 0;
					break;
			}
			text[len] = 0;	/* Gets rid of the Ll/Rr flag, if present */
			sscanf (&text[1], "%[^/]/%s", txt_a, txt_b);
			p->f_gap = GMT_convert_units (txt_a, GMT_INCH);
			p->f_len = GMT_convert_units (txt_b, GMT_INCH);
			break;
		case 'H':
			p->size *= 1.09963611078;	/* To equal area of circle with same size */
		case 'h':
			p->symbol = HEXAGON;
			break;
		case 'I':
			p->size *= 1.55512030155;	/* To equal area of circle with same size */
		case 'i':
			p->symbol = ITRIANGLE;
			break;
		case 'l':
			p->symbol = TEXT;
			/* Look for a slash that separates size and string: */
			for (j = 1, slash = 0; text_cp[j] && !slash; j++) if (text_cp[j] == '/') slash = j;
			/* Set j to the first char in the string: */
			j = slash + 1;
			/* Copy string characters */
			k = 0;
			while (text_cp[j] && text_cp[j] != ' ' && k < 63) p->string[k++] = text_cp[j++];
			if (!k) {
				fprintf (stderr, "%s: GMT SYNTAX ERROR -Sl option:  No string given\n", GMT_program);
				decode_error++;
			}
			p->string[k] = 0;
			break;
		case 'p':
			p->symbol = POINT;
			check = FALSE;
			break;
		case 'S':
			p->size *= 1.25331413731;	/* To equal area of circle with same size */
		case 's':
			p->symbol = SQUARE;
			break;
		case 'T':
			p->size *= 1.55512030155;	/* To equal area of circle with same size */
		case 't':
			p->symbol = TRIANGLE;
			break;
		case 'V':
			p->convert_angles = TRUE;
		case 'v':
			p->symbol = VECTOR;
			for (j = 1; text[j] && text[j] != 'n'; j++);
			len = strlen(text) - 1;
			if (text[j]) {	/* Normalize option used */
				k = GMT_get_unit (text[len]);
				if (k >= 0) p->u = k;
				p->v_norm = GMT_convert_units (&text[j+1], GMT_INCH);
				if (p->v_norm > 0.0) {
					p->v_shrink = 1.0 / p->v_norm;
					p->symbol = VECTOR2;
				}
				text[j] = 0;	/* Chop of the n<shrink> part */
			}
			if (text[1]) {
				/* It is possible that the user have appended a unit modifier after
				 * the <size> argument (which here are vector attributes).  We use that
				 * to set the unit, but only if the vector attributes themselves have
				 * units. (If not we would override MEASURE_UNIT without cause).
				 * So, -SV0.1i/0.2i/0.3ic will expect 4th colum to have length in cm
				 * while SV0.1i/0.2i/0.3i expects data units in MEASURE_UNIT
				 */

				if (isalpha ((int)text[len]) && isalpha ((int)text[len-1])) {
					p->u = GMT_get_unit (text[len]);
					text[len] = 0;
				}
				sscanf (&text[1], "%[^/]/%[^/]/%s", txt_a, txt_b, txt_c);
				p->v_width  = GMT_convert_units (txt_a, GMT_INCH);
				p->h_length = GMT_convert_units (txt_b, GMT_INCH);
				p->h_width  = GMT_convert_units (txt_c, GMT_INCH);
			}
			if (p->symbol == VECTOR2) text[j] = 'n';	/* Put back the n<shrink> part */
			p->read_vector = TRUE;
			p->n_required = 2;
			check = FALSE;
			break;
		case 'x':
			p->symbol = CROSS;
			break;
		default:
			decode_error = TRUE;
			fprintf (stderr, "%s: GMT SYNTAX ERROR -S option:  Unrecognized symbol type %c\n", GMT_program, symbol_type);
			break;
	}
	if (p->given_size == 0.0 && check) {
		p->read_size = TRUE;
		p->n_required++;
	}
	else
		p->read_size = FALSE;

	return (decode_error);
}

#define GMT_ELLIPSE_APPROX 72

void GMT_plot_ellipse (double lon, double lat, double major, double minor, double azimuth, struct GMT_FILL fill, int outline)
{
	/* GMT_plot_ellipse takes the location, axes (in km), and azimuth of an ellipse
	   and draws the ellipse using the chosen map projection */

	int i, plot_n, start, n_use;
	double delta_azimuth, sin_azimuth, cos_azimuth, sinp, cosp, angle, x, y, x_prime, y_prime, rho, c;
	double sin_c, cos_c; 
	double *px, *py, *xp, *yp;

	px = (double *) GMT_memory (VNULL, (size_t)GMT_ELLIPSE_APPROX, sizeof (double), GMT_program);
	py = (double *) GMT_memory (VNULL, (size_t)GMT_ELLIPSE_APPROX, sizeof (double), GMT_program);

	delta_azimuth = 2.0 * M_PI / GMT_ELLIPSE_APPROX;
	major *= 1000.0;	minor *= 1000.0;	/* Convert to meters */
	azimuth = 90.0 - azimuth;	/* Because the code below originally used directions instead */
	azimuth *= D2R;
	sincos (azimuth, &sin_azimuth, &cos_azimuth);
	sincos (lat * D2R, &sinp, &cosp);	/* Set up azimuthal equidistant projection */

	/* Approximate ellipse by a GMT_ELLIPSE_APPROX-sided polygon */

	for (i = 0; i < GMT_ELLIPSE_APPROX; i++) {

		angle = i * delta_azimuth;

		sincos (angle, &y, &x);
		x *= major;
		y *= minor;

		/* Get rotated coordinates in km */

		x_prime = x * cos_azimuth - y * sin_azimuth;
		y_prime = x * sin_azimuth + y * cos_azimuth;
		
		/* Convert km back to lon lat */

		rho = hypot (x_prime, y_prime);
	
		c = rho / project_info.EQ_RAD;
		sincos (c, &sin_c, &cos_c);
		py[i] = d_asin (cos_c * sinp + (y_prime * sin_c * cosp / rho)) * R2D;
		if ((lat - 90.0) > -GMT_CONV_LIMIT)
			px[i] = lon + R2D * d_atan2 (x_prime, -y_prime);
		else if ((lat + 90.0) < GMT_CONV_LIMIT)
			px[i] = lon + R2D * d_atan2 (x_prime, y_prime);
		else
			px[i] = lon +
				R2D * d_atan2 (x_prime * sin_c, (rho * cosp * cos_c - y_prime * sinp * sin_c));
		if ((px[i]) <= -180) (px[i]) += 360.0;
	}

	if ((plot_n = GMT_clip_to_map (px, py, GMT_ELLIPSE_APPROX, &xp, &yp)) == 0) return;

	while (plot_n > GMT_n_alloc) GMT_get_plot_array ();
	memcpy ((void *)GMT_x_plot, (void *)xp, (size_t)(plot_n * sizeof (double)));
	memcpy ((void *)GMT_y_plot, (void *)yp, (size_t)(plot_n * sizeof (double)));

	if ((*GMT_will_it_wrap) (xp, yp, plot_n, &start)) {	/* Polygon wraps */
					
		/* First truncate agains left border */
						
		GMT_n_plot = (*GMT_truncate) (xp, yp, plot_n, start, -1);
		n_use = GMT_compact_line (GMT_x_plot, GMT_y_plot, GMT_n_plot, FALSE, 0);
		GMT_fill (GMT_x_plot, GMT_y_plot, n_use, &fill, outline);
								
		/* Then truncate agains right border */
						
		GMT_n_plot = (*GMT_truncate) (xp, yp, plot_n, start, +1);
		n_use = GMT_compact_line (GMT_x_plot, GMT_y_plot, GMT_n_plot, FALSE, 0);
		GMT_fill (GMT_x_plot, GMT_y_plot, n_use, &fill, outline);
						
	}
	else
		GMT_fill (xp, yp, plot_n, &fill, outline);

	GMT_free ((void *)xp);
	GMT_free ((void *)yp);
	GMT_free ((void *)px);
	GMT_free ((void *)py);
}

void GMT_draw_fence (double x[], double y[], int n, double gap, double len, int sense, int triangle, int rgb[], BOOLEAN outline)
{
	int i, ngap;
	double *s, xx[3], yy[3], dist = 0.0, frac, dx, dy, angle;
	double x0, y0, len2, len3, cosa, sina;
	
	if (n < 2) return;

	s = (double *) GMT_memory (VNULL, (size_t)n, sizeof (double), GMT_program);
	for (i = 1, s[0] = 0.0; i < n; i++) s[i] = s[i-1] + hypot (x[i] - x[i-1], y[i] - y[i-1]);
		
	if (gap > 0.0) {
		ngap = irint (s[n-1] / gap);
		gap = s[n-1] / (double)ngap;
		ngap++;
	}
	else {
		ngap = (int) fabs (gap);
		gap = s[n-1] / (ngap - 1);
		if (ngap == 1) dist = 0.5 * s[n-1];
	}
	
	len2 = 0.5 * len;
	len3 = 0.866025404 * len;			
	i = 0;
	while (i < n) {
		while ((s[i] - dist) > -SMALL) {	/* Time for tick */
			if (i > 0) {
				dx = x[i] - x[i-1];
				dy = y[i] - y[i-1];
			}
			else {
				dx = x[1] - x[0];
				dy = y[1] - y[0];
			}
			if (fabs (dist - s[i]) < SMALL) {
				x0 = x[i];
				y0 = y[i];
			}
			else {
				frac = (s[i] - dist) / (s[i] - s[i-1]);
				x0 = x[i] - dx * frac;
				y0 = y[i] - dy * frac;
			}
			angle = d_atan2 (dy, dx);
						
			if (triangle) {
				if (sense == -1) angle += M_PI;
				sincos (angle, &sina, &cosa);
				xx[0] = x0 + len2 * cosa;
				yy[0] = y0 + len2 * sina;
				xx[1] = x0 - len3 * sina;
				yy[1] = y0 + len3 * cosa;
				xx[2] = x0 - len2 * cosa;
				yy[2] = y0 - len2 * sina;
				if (project_info.three_D) GMT_2D_to_3D (xx, yy, 3);
				ps_patch (xx, yy, 3, rgb, outline);
			}
			else {
				xx[0] = xx[1] = x0;	yy[0] = yy[1] = y0;
				if (sense == 0) {
					angle -= M_PI_2;
					sincos (angle, &sina, &cosa);
					xx[0] += len2 * cosa;
					yy[0] += len2 * sina;
					xx[1] -= len2 * cosa;
					yy[1] -= len2 * sina;
				}
				else {
					angle += (sense * M_PI_2);
					sincos (angle, &sina, &cosa);
					xx[1] += len * cosa;
					yy[1] += len * sina;
				}
				if (project_info.three_D) GMT_2D_to_3D (xx, yy, 2);
				ps_plot (xx[0], yy[0], 3);
				ps_plot (xx[1], yy[1], 2);
			}
			dist += gap;
		}
		i++;
	}
	GMT_free ((void *)s);
}
