/*
 * Copyright 1995,96 Thierry Bousch
 * Licensed under the Gnu Public License, Version 2
 *
 * $Id: rules.y,v 1.16 1996/09/30 08:30:28 bousch Exp $
 *
 * The yacc file of Induce, to parse "rules" files.
 */

%{
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "saml.h"
#include "induce.h"

#define yyerror(s) fprintf(stderr, "induce: %s\n", s)

static char* bcode_three (char* bc1, char* bc2, char* bc3, const char* bc4)
{
	bc1 = realloc(bc1, strlen(bc1) + strlen(bc2) + strlen(bc3)
		+ strlen(bc4) + 4);
	assert(bc1 != NULL);
	strcat(bc1, bc2); free(bc2);
	strcat(bc1, bc3); free(bc3);
	return strcat(bc1, bc4);
}

static char* bcode_join (char* bc1, char* bc2, const char* bc3)
{
	bc1 = realloc(bc1, strlen(bc1)+strlen(bc2)+strlen(bc3)+3);
	assert(bc1 != NULL);
	strcat(bc1, bc2);
	free(bc2);
	return strcat(bc1, bc3);
}

static char* bcode_append (char* bc1, const char* bc2)
{
	bc1 = realloc(bc1, strlen(bc1)+strlen(bc2)+2);
	assert(bc1 != NULL);
	return strcat(bc1, bc2);
}

static char* bcode_exponent (char* bc1, int exp)
{
	char buff[24];
	
	if (exp < -1)
	  	sprintf(buff, "i%d^", -exp);
	else if (exp == -1)
		strcpy(buff, "i");
	else /* if (exp >= 0) */
		sprintf(buff, "%d^", exp);

	return bcode_append(bc1, buff);
}

static char* bcode_function (char *bc1, int fn)
{
	char buff[24];
	sprintf(buff, "%df", fn);
	return bcode_append(bc1, buff);
}

static char* bcode_index (int index)
{
	char buff[24];
	sprintf(buff, "%dp", index);
	return strdup(buff);
}

static char* bcode_integer (int i)
{
	char buff[24];
	sprintf(buff, "%d.", i);
	return strdup(buff);
}

static void restricted_index (char* bcode)
{
	int i = new_index(strdup("$__restricted__$"));
	char *cond = bcode_join(bcode, bcode_index(i), "-!");
	collect_condition(cond);
}

%}

%union {
	int	integer;
	char*	string;
	alist	arglist;
}

%token TOK_IF TOK_PRECIOUS TOK_MODULO
%token TOK_GEQ TOK_LEQ TOK_NEQ TOK_ARROW TOK_SHIFT
%token <integer> INDEX INTEGER
%token <string> IDENTIFIER BIG_INTEGER QSTRING
%type <string> expr nested_expr iexpr number idx_var ident
%type <string> ext_function iext_function
%type <arglist> opt_exfn_args exfn_args opt_iexfn_args iexfn_args

%left '=' '>' '<' TOK_GEQ TOK_LEQ TOK_NEQ
%left TOK_MODULO
%left '+' '-'
%left '*' '.' '/' '%'
%left '!' UNARY_PLUS UNARY_MINUS
%left '^' ':'

%%

list_of_rules:	/* empty */
	| list_of_rules rule ';'
	| list_of_rules precious_decl ';'
	;

precious_decl: TOK_PRECIOUS opt_memo_name list_precious_things ;

opt_memo_name: /* empty */	{ register_memo(memo_file); }
	| QSTRING		{ register_memo($1); }
	;

list_precious_things: precious_thing
	| list_precious_things ',' precious_thing
	;

precious_thing: IDENTIFIER '(' INTEGER ')'
	    { declare_precious($1,$3); }
	;

rule:	/* empty */
	| rule_lhs '=' expr optional_condition
		{ add_this_rule($3); }
	;

rule_lhs: IDENTIFIER
	    { start_new_rule($1); }
	optional_indices
	;

optional_indices: /* empty */
	| optional_indices '[' ']'
	| optional_indices '[' list_of_indices ']'
	;

list_of_indices: ident_or_iexpr
	| list_of_indices ',' ident_or_iexpr
	;

ident_or_iexpr: IDENTIFIER
	    { new_index($1); }
	| iexpr
	    { restricted_index($1); }
	;

optional_condition: /* empty */
	| TOK_IF list_of_conditions
	;

list_of_conditions: iexpr
	    { collect_condition($1); }
	| list_of_conditions ',' iexpr
	    { collect_condition($3); }
	;

ext_function: IDENTIFIER '(' opt_exfn_args ')'
	    { int fn = register_function($1, $3.arity);
	    $$ = bcode_function($3.bytecode, fn); }
	;

opt_exfn_args: /* empty */
	    { $$.arity = 0; $$.bytecode = strdup(""); }
	| exfn_args
	;

exfn_args: expr
	    { $$.arity = 1; $$.bytecode = $1; }
	| exfn_args ',' expr
	    { $$.arity = $1.arity + 1;
	      $$.bytecode = bcode_join($1.bytecode, $3, ""); }
	;

expr: idx_var
	| ext_function
	| expr TOK_MODULO '(' expr ',' expr ')'
	    {
	    	int fn = register_function("modulo", 3);
	    	char buff[24];
	    	sprintf(buff, "%df", fn);
	    	$$ = bcode_three($1,$4,$6,buff);
	    }
	| '(' nested_expr ')'
	    { $$ = $2; }
	| expr '+' expr
	    { $$ = bcode_join($1,$3,"+"); }
	| expr '-' expr
	    { $$ = bcode_join($1,$3,"-"); }
	| expr '*' expr
	    { $$ = bcode_join($1,$3,"*"); }
	| expr '.' expr
	    { $$ = bcode_join($1,$3,"*"); }
	| expr '/' expr
	    { $$ = bcode_join($1,$3,"/"); }
	| '-' expr		%prec UNARY_MINUS
	    { $$ = bcode_append($2,"n"); }
	| '+' expr		%prec UNARY_PLUS
	    { $$ = $2; }
	| '!' expr		%prec '!'
	    { $$ = bcode_append($2,"!"); }
	| expr '^' INTEGER
	    { $$ = bcode_exponent($1,$3); }
	| expr ':' ident
	    { $$ = bcode_join($1,$3,"d"); }
	| '~' ident '(' expr ',' expr ')'
	    { $$ = bcode_three($2,$4,$6,"r"); }
	;

nested_expr: expr
	    { $$ = $1; }
	| nested_expr ',' expr TOK_ARROW expr
	    { $$ = bcode_three($1,$3,$5,"s"); }
	| nested_expr ',' expr TOK_SHIFT expr
	    { $$ = bcode_three($1,$3,$5,"S"); }
	;

ident: IDENTIFIER
	    { start_idxvar($1); }
	optional_list_iexprs
	    { $$ = collect_idxvar(); }
	;

idx_var: ident
	    { $$ = $1; }
	| number
	    { start_idxvar($1); $$ = collect_idxvar(); }
	| INDEX
	    { start_idxvar("__value__");
	      collect_iexpr(bcode_index($1));
	      $$ = collect_idxvar(); }
	;

optional_list_iexprs: /* empty */
	| optional_list_iexprs '[' ']'
	| optional_list_iexprs '[' list_iexprs ']'
	;

list_iexprs: iexpr
	    { collect_iexpr($1); }
	| list_iexprs ',' iexpr
	    { collect_iexpr($3); }
	;

iext_function: IDENTIFIER '(' opt_iexfn_args ')'
	    { int fn = register_function($1, $3.arity);
	    $$ = bcode_function($3.bytecode, fn); }
	;

opt_iexfn_args: /* empty */
	    { $$.arity = 0; $$.bytecode = strdup(""); }
	| iexfn_args
	;

iexfn_args: iexpr
	    { $$.arity = 1; $$.bytecode = $1; }
	| iexfn_args ',' iexpr
	    { $$.arity = $1.arity + 1;
	      $$.bytecode = bcode_join($1.bytecode, $3, ""); }
	;

iexpr: iext_function
	| INDEX
	    { $$ = bcode_index($1); }
	| INTEGER
	    { $$ = bcode_integer($1); }
	| '(' iexpr ')'
	    { $$ = $2; }
	| iexpr '+' iexpr
	    { $$ = bcode_join($1,$3,"+"); }
	| iexpr '-' iexpr
	    { $$ = bcode_join($1,$3,"-"); }
	| iexpr '*' iexpr
	    { $$ = bcode_join($1,$3,"*"); }
	| iexpr '.' iexpr
	    { $$ = bcode_join($1,$3,"*"); }
	| iexpr '/' iexpr
	    { $$ = bcode_join($1,$3,"/"); }
	| iexpr '%' iexpr
	    { $$ = bcode_join($1,$3,"%"); }
	| iexpr '^' INTEGER
	    { $$ = bcode_exponent($1,$3); }
	| '-' iexpr		%prec UNARY_MINUS
	    { $$ = bcode_append($2,"n"); }
	| '+' iexpr		%prec UNARY_PLUS
	    { $$ = $2; }
	| '!' iexpr
	    { $$ = bcode_append($2,"!"); }
	| iexpr '=' iexpr
	    { $$ = bcode_join($1,$3,"-!"); }
	| iexpr TOK_NEQ iexpr
	    { $$ = bcode_join($1,$3,"-!!"); }
	| iexpr '<' iexpr
	    { $$ = bcode_join($1,$3,"-<"); }
	| iexpr '>' iexpr
	    { $$ = bcode_join($1,$3,"->"); }
	| iexpr TOK_GEQ iexpr
	    { $$ = bcode_join($1,$3,"-<!"); }
	| iexpr TOK_LEQ iexpr
	    { $$ = bcode_join($1,$3,"->!"); }
	;

number: INTEGER
	    { char buff[24]; sprintf(buff, "%d", $1);
	      $$ = strdup(buff); }
	| BIG_INTEGER
	    { $$ = $1; }
	;
