# Version 2.1 for FWF V4.0
#
# $Id: Label.w,v 1.1 1996-09-25 09:23:49+02 mho Exp $

@class XfwfLabel (XfwfBoard) @file=Label

@ The Label class has the capability to display one or more lines of
text in a single font. Otherwise it is the same as the Board class.
The text can be left, right or center justified and it can be centered
vertically or put against the top or the bottom of the widget. There
is also a resource to set tab stops.

The text is `grayed out' when the widget becomes insensitive
(resource: |sensitive|), even though a Label widget has no actions of
its own.

There are two ways of highlighting portions of the label: reversing
the colors or changing the foreground. Both methods can be combined,
though the result when both highlighting methods are applied to the
same part of the text is undefined.


@public

@ The text is a single string, which may contain embedded newlines.
There is no provision for changing fonts in the middle of a text.

	@var String label = NULL

@ A tablist can be provided for tabbing to particular columns
within the label.

	@var String tablist = NULL

@ The text is drawn in the font which is given as the |font| resource.

	@var <FontStruct> XFontStruct *font = <String> XtDefaultFont

@ Instead of a label a |pixmap| may be displayed if |label| is NULL.
|mask| is used to clip the pixmap, if parts should be drawn in background
colour.

	@var Pixmap pixmap = 0
	@var Pixmap mask = 0

@ The foreground color is the color used to draw the
text. |hlForeground| is the foreground for highlighted text.

	@var Pixel foreground = <String> XtDefaultForeground
	@var Pixel hlForeground = <String> XtDefaultForeground

@ The text can be aligned in the widget in nine ways: left, right or
center, combined with top, center or bottom. Symbolic constants
|XfwfTop|, |XfwfBottom|, |XfwfLeft| and |XfwfRight| can be added together to
get the desired alignment.  The alignment is actually a four-bit
number made up of two parts of 2 bits added together: 1 is left, 2 is
right, 0 is center, 4 is top, 8 is bottom, 0 is vertical center. Thus
5 (= 1 + 4) means top left and 2 (= 2 + 0) means center right. For
easier specification, there is also a converter from strings, that
accepts string like `top left' or `center right'.

	@var Alignment alignment = 0

@ The |topmargin| is only used when the text is not centered. It gives
the number of pixels between the frame and the top of the text.

	@var Dimension topMargin = 2

@ The |bottomMargin| is only used to compute the preferred size of the
button in case |shrinkToFit = True|.

	@var Dimension bottomMargin = 2

@ The |leftMargin| is only used when the text is not centered. It
gives the number of pixels between the frame and the left edge of the
text, and if possible also between the frame and the right edge of the
text.

	@var Dimension leftMargin = 2

@ The |rightMargin| is only used to compute the preferred size of the
button in case |shrinkToFit = True|.

	@var Dimension rightMargin = 2

@ Buttons will normally not occupy the full area of their parents.
Most likely they will be a fixed size or a size depending on the
label. By setting the |shrinkToFit| resource to True, the width and
height are recomputed with every new label.

	@var Boolean shrinkToFit = False

@ It is possible to set a part of the label apart by drawing it in
reverse. The |rvStart| resource gives the index of the first
character to draw in reverse video.

	@var int rvStart = 0

@ The |rvLength| resource contains the number of characters to
draw in reverse video.

	@var int rvLength = 0

@ A label normally needs no keyboard interface, therefore traversal is
turned off.

	@var traversalOn = False

@ The start and length of the highlighted portion are set with the
resources |hlStart| and |hlLength|.

	@var int hlStart = 0
	@var int hlLength = 0

@exports

@ |XfwfSetLabel| allows to set the label only. It is easier than a call to
XtVaSetValues and does not cause a complete redrawing of the widget, only the
label area, not the border, frame, etc.

@proc XfwfSetLabel($, String new_label)
{
    if (! XtIsSubclass($, xfwfLabelWidgetClass))
	XtError("XfwfSetLabel called with incorrect widget type");
    $set_label($, new_label);
}

@private

@ For faster drawing, the number of lines in the text is stored in a
private variable by the |set_values| and |initialize| methods.

	@var int nlines

@ The tablist is converted from string format to a list of int's for speed.

	@var int *tabs

@ For drawing the text, this GC is used.

	@var GC gc

@ This GC is for the text that is drawn in reverse video.

	@var GC rv_gc

@ The GC for the highlighted portion of the text

	@var GC hl_gc

@ For graying out the text, another GC is used.

	@var GC graygc

@ When the |shrinkToFit| resource is set, we need the minimum area
necessary for the complete label to be visible. |label_width| and
|label_height| include the size of |margin|.

	@var Dimension label_width
	@var Dimension label_height

@ For a pixmap label I need the |label_depth|.

	@var unsigned int label_depth


@methods

@ The new method |set_label| makes a copy of the string that is passed
in, counts the number of lines and also draws the new label. This
could have been done in |set_values|, but it is expected that
subclasses will redraw the label frequently, so a more efficient way
is provided.

Note that this method does not resize the widget in case |shrinkToFit|
is set.

@proc set_label($, String newlabel)
{
    Position x, y;
    Dimension w, h;

    XtFree($label);
    $label = XtNewString(newlabel);
    $count_lines($);
    if (XtIsRealized($)) {
	$compute_inside($, &x, &y, &w, &h);
	XClearArea(XtDisplay($), XtWindow($), x, y, w, h, True);
	/* $_expose($, NULL, NULL); */
    }
}

@ The funtion |count_lines| computes the correct values for the
private variables |nlines|, |label_width| and |label_height|.

@proc count_lines($)
{
    String p, s;
    int w;

    $nlines = 0;
    $label_width = 0; $label_height = 0; $label_depth = 0;
    if ($label) {
	for (p = $label, $nlines = 1, s = $label; *s; s++) {
	    if (*s == '\n') {
		$nlines++;
		w = XfwfTextWidth($font, p, s - p, $tabs);
		p = s + 1;
		if (w > $label_width) $label_width = w;
	    }
	}
	w = XfwfTextWidth($font, p, s - p, $tabs);
	if (w > $label_width) $label_width = w;
	$label_height = $nlines * ($font->ascent + $font->descent);
    } else if ($pixmap) {
	Window        root;
	int           x, y;
	unsigned int  width, height, bw, depth;
	XGetGeometry(XtDisplay($), $pixmap, &root,
	             &x, &y, &width, &height, &bw, &depth);
	$label_width  = (Dimension)width;
	$label_height = (Dimension)height;
	$label_depth  = depth;
    }
    /* add border */
    $label_width += $leftMargin + $rightMargin;
    $label_height += $topMargin + $bottomMargin;
}

@ The |set_values| method checks the |background| resource, because is
is used in the GC |graygc|. When the text or the font change, the
private variables |nlines|, |label_height| and |label_width| are
updated.

|need_count| is set to |True| if the size of the label changes.
|need_count| implies |need_redisplay|.

@proc set_values
{
    Boolean need_redisplay = False, need_count = False;
    Position x, y;
    Dimension w, h, wd, ht;

    /* label or pixmap has changed */
    if ($label != $old$label || $pixmap != $old$pixmap || $mask != $old$mask) {
	XtFree($old$label);
	$label = XtNewString($label);
	need_count = True;
	if ($label != NULL || $pixmap != 0) need_redisplay = True;
    }
    /* tablist has changed */
    if ($tablist != $old$tablist) {
	XtFree((String) $old$tabs);
	$tabs = XfwfTablist2Tabs($tablist);
	if ($label != NULL) need_count = True;
    }
    /* colour or font has changed */
    if ($foreground       != $old$foreground
    ||  $hlForeground     != $hlForeground
    ||  $background_pixel != $old$background_pixel
    ||  $font             != $old$font             ) {
	make_gc($);
	if ($label != NULL || $pixmap != 0) need_redisplay = True;
    }
    /* a margin has changed */
    if ($topMargin    != $old$topMargin
    ||  $bottomMargin != $old$bottomMargin
    ||  $leftMargin   != $old$leftMargin
    ||  $rightMargin  != $old$rightMargin
    ||  $alignment    != $old$alignment    )
	need_count = True;
    /* sensitivity has changed */
    if ($sensitive != $old$sensitive)
	if ($label != NULL || $pixmap != 0) need_redisplay = True;
    /* reverse position has changed */
    if ($rvStart != $old$rvStart || $rvLength != $old$rvLength
    ||  $hlStart != $old$hlStart || $hlLength != $old$hlLength)
	if ($label != NULL) need_redisplay = True;
    /* compute size of label */
    if (need_count) {
	$count_lines($);
	need_redisplay = True;
    }
    /* check if label should be resized */
    if (need_count && $shrinkToFit) {
	$compute_inside($, &x, &y, &w, &h);
	wd = $label_width  + $width  - w;
	ht = $label_height + $height - h;
	if (wd != $width || ht != $height)
	    $set_abs_location($, CWWidth | CWHeight, 0, 0, wd, ht);
    }
    /* return if redisplay is desired */
    return need_redisplay;
}

@ The |initialize| methods creates the first GC's and initializes the
private variables. It sets the GC's to |NULL| and calls two utility
routines to actually create them.

@proc initialize
{
    Position x, y;
    Dimension w, h, wd, ht;

    /* allocate string */
    if ($label) $label = XtNewString($label);
    $tabs = XfwfTablist2Tabs($tablist);
    $count_lines($);
    /* create GCs */
    $gc = $rv_gc = $graygc = $hl_gc = NULL;
    make_gc($);
    /* resize if wanted */
    if ($shrinkToFit) {
	$compute_inside($, &x, &y, &w, &h);
	wd = $label_width + $width - w;
	ht = $label_height + $height - h;
	$set_abs_location($, CWWidth | CWHeight, 0, 0, wd, ht);
    }
}

@ The |destroy| method should free the label and the tabs. The
pixmap and the font are owned by the used.

@proc destroy
{
    /* free the graphic contexts */
    XtReleaseGC($, $gc);
    XtReleaseGC($, $rv_gc);
    XtReleaseGC($, $hl_gc);
    XtReleaseGC($, $graygc);
    /* free label and tabs */
    if ($label) XtFree($label);
    if ($tabs)  XtFree((String)$tabs);
}

@ The |expose| method is responsible for drawing the text. The text is
put in the position given in |alignment|. The text is always kept
within the frame. If necessary, the text is clipped. The routine ends
by calling the |expose| method from the superclass, which is
responsible for drawing the frame.

The part of the text that is to appear in reverse video is drawn with
the |rv_gc| GC.

@def draw_line(dpy, win, from, to) =
    do {
	if ($hlStart >= to) hstart = to;
 	else hstart = max($hlStart, from);
	if ($hlStart + $hlLength <= from) hend = hstart;
 	else hend = min($hlStart + $hlLength, to);
        if ($rvStart >= to) rstart = to;
	else rstart = max($rvStart, from);
	if ($rvStart + $rvLength <= from) rend = rstart;
	else rend = min($rvStart + $rvLength, to);
	w1 = XfwfTextWidth($font, $label + from, rstart - from, $tabs);
	w2 = XfwfTextWidth($font, $label + rstart, rend - rstart, $tabs);
	w3 = XfwfTextWidth($font, $label + rend, to - rend, $tabs);
 	w4 = XfwfTextWidth($font, $label + hstart, hend - hstart, $tabs);
 	w5 = XfwfTextWidth($font, $label + from, hstart - from, $tabs);
	if ($alignment & XfwfLeft)
	    x = rect.x;
	else if ($alignment & XfwfRight)
	    x = rect.x + rect.width - w1 - w2 - w3;
	else
	    x = rect.x + (rect.width - w1 - w2 - w3)/2;
	if (w1)
	    XfwfDrawString(dpy, win, $gc, $font, x, y, $label + from,
			   rstart - from, $tabs, NULL);
	if (w2)
	    XfwfDrawImageString(dpy, win, $rv_gc, $font, x + w1, y, $label
				+ rstart, rend - rstart, $tabs, NULL);
	if (w3)
	    XfwfDrawString(dpy, win, $gc, $font, x + w1 + w2, y, $label +
			   rend, to - rend, $tabs, NULL);
 	if (w4)
 	    XfwfDrawString(dpy, win, $hl_gc, $font, x + w5, y, $label
			   + hstart, hend - hstart, $tabs, NULL);
    } while (0)

@proc _expose
{
    Region reg;
    XRectangle rect;
    int baseline;
    int w1, w2, w3, w4, w5;
    int x, y, i, j, rstart, rend, hstart, hend;

    if (! XtIsRealized($)) return;
    #_expose($, event, region);
    if ($label != NULL || $pixmap != 0) {
	$compute_inside($, &rect.x, &rect.y, &rect.width, &rect.height);
	rect.x += $leftMargin;  rect.width -= $leftMargin + $rightMargin;
	rect.y += $topMargin;  rect.height -= $topMargin + $bottomMargin;
	reg = XCreateRegion();
	XUnionRectWithRegion(&rect, reg, reg);
	if (region != NULL) XIntersectRegion(region, reg, reg);
	XSetRegion(XtDisplay($), $gc, reg);
	XSetRegion(XtDisplay($), $rv_gc, reg);
 	XSetRegion(XtDisplay($), $hl_gc, reg);
	XDestroyRegion(reg);
    }
    if ($label != NULL) {
	baseline = $font->ascent + $font->descent;
	if ($alignment & XfwfTop)
	    y = rect.y + $font->ascent;
	else if ($alignment & XfwfBottom)
	    y = rect.y + rect.height - $nlines * baseline + $font->ascent;
	else
	    y = rect.y + (rect.height - $nlines * baseline)/2 + $font->ascent;
	for (i = 0, j = 0; $label[i]; i++) {
	    if ($label[i] == '\n') {
		draw_line(XtDisplay($), XtWindow($), j, i);
		j = i + 1;
		y += baseline;
	    }
	}
	draw_line(XtDisplay($), XtWindow($), j, i);
    } else if ($pixmap != 0) {
	Dimension width = $label_width - $leftMargin - $rightMargin;
	Dimension height = $label_height - $topMargin - $bottomMargin;
	if ($alignment & XfwfTop)
	    y = rect.y;
	else if ($alignment & XfwfBottom)
	    y = rect.y + rect.height - height;
	else
	    y = rect.y + (rect.height - height)/2;
	if ($alignment & XfwfLeft)
	    x = rect.x;
	else if ($alignment & XfwfRight)
	    x = rect.x + rect.width - width;
	else
	    x = rect.x + (rect.width - width)/2;
	if ($mask) {
	    XSetClipMask(XtDisplay($), $gc, $mask);
	    XSetClipOrigin(XtDisplay($), $gc, x, y);
	}
	if ($label_depth == 1) { /* depth */
	    XCopyPlane(XtDisplay($), $pixmap, XtWindow($), $gc,
		       0, 0, width, height, x, y, 1L);
	} else {
	    XCopyArea(XtDisplay($), $pixmap, XtWindow($), $gc,
		      0, 0, width, height, x, y);
	}
	if ($mask) {
	    XSetClipOrigin(XtDisplay($), $gc, 0, 0);
	}
    }
    if ($label != NULL || $pixmap != 0) {
	/* Gray out if not sensitive */
	if (! $sensitive) {
	    /* XSetRegion(XtDisplay($), $graygc, reg); */
	    XFillRectangle(XtDisplay($), XtWindow($), $graygc, rect.x,
			   rect.y, rect.width, rect.height);
	    /* XSetClipMask(XtDisplay($), $graygc, None); */
	}
	XSetClipMask(XtDisplay($), $gc, None);
	XSetClipMask(XtDisplay($), $rv_gc, None);
 	XSetClipMask(XtDisplay($), $hl_gc, None);
    }
}



@utilities

@ The |make_gc| routine creates the GCs for the normal and highlighted
text.

@proc make_gc($)
{
    XtGCMask  mask = GCFont | GCBackground | GCForeground;
    XGCValues values;

    /* the normal GC uses font and foreground */
    if ($gc != NULL) XtReleaseGC($, $gc);
    values.background = $background_pixel;
    values.foreground = $foreground;
    values.font       = $font->fid;
    $gc               = XtGetGC($, mask, &values);

    /* the highlight GC uses hlForeground instead */
    if ($hl_gc != NULL) XtReleaseGC($, $hl_gc);
    values.foreground = $hlForeground;
    $hl_gc            = XtGetGC($, mask, &values);

    /* the reverse GC needs the background colour too */
    if ($rv_gc != NULL) XtReleaseGC($, $rv_gc);
    values.foreground = $background_pixel;
    values.background = $foreground;
    $rv_gc            = XtGetGC($, mask, &values);

    /* the gray GC needs the background pixel and the gray stipple */
    if ($graygc != NULL) XtReleaseGC($, $graygc);
    values.foreground = $background_pixel;
    values.stipple    = $gray;
    values.fill_style = FillStippled;
    mask              = GCForeground | GCStipple | GCFillStyle;
    $graygc           = XtGetGC($, mask, &values);
}

@imports

@incl <stdio.h>
@incl <TabString.h>
