/*
 * GTKSee -- a image viewer based on gtk+
 * Copyright (C) 1998 Hotaru Lee <hotaru@163.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
//#include <sys/dir.h>
// POSIX compliant
#include <dirent.h>
#define direct dirent
#include <sys/stat.h>
#include <unistd.h>
#include "vdk/dirtree.h"
#include "folder_collapsed.xpm"
#include "folder_expanded.xpm"

typedef struct dir_node_struct
{
	Dirtree *tree;
	char dir_name[256];
	GtkWidget *subtree, *parent;
} dir_node;

typedef struct sel_node_struct
{
	Dirtree *tree;
	char new_dir[256];
	GtkWidget *parent;
} sel_node;

static void	dirtree_class_init	(DirtreeClass *class);
static void	dirtree_init		(Dirtree *dt);
int		dir_can_expand		(char *dir);
void		item_expanded		(GtkTreeItem *treeitem, dir_node *node);
void		item_collapsed		(GtkTreeItem *treeitem, dir_node *node);
void		item_selected		(GtkTreeItem *treeitem, sel_node *node);
void		item_deselected		(GtkTreeItem *treeitem, sel_node *node);
void		dirtree_expand_subtree	(Dirtree *dt, GtkTree *tree, char *dir, char *e_target);
void		dirtree_item_set_pixmap	(GtkWidget *parent, GtkTreeItem *treeitem, gint expanded);
GtkWidget*	dirtree_item_new	(GtkWidget *parent, gchar *dir);
gchar*		dirtree_item_get_label	(GtkTreeItem *treeitem);
gint		dirtree_get_height	(Dirtree *dt);
void		dirtree_create_root_item(Dirtree *dt, gchar *dir);

guint
dirtree_get_type ()
{
	static guint dt_type = 0;

	if (!dt_type)
	{
		GtkTypeInfo dt_info =
		{
			"Dirtree",
			sizeof (Dirtree),
			sizeof (DirtreeClass),
			(GtkClassInitFunc) dirtree_class_init,
			(GtkObjectInitFunc) dirtree_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		dt_type = gtk_type_unique (gtk_tree_get_type (), &dt_info);
	}
	return dt_type;
}

static void
dirtree_class_init (DirtreeClass *class)
{
	/*
	GtkObjectClass *object_class;
	object_class = (GtkObjectClass*) class;
	dirtree_signals[TICTACTOE_SIGNAL] = gtk_signal_new ("tictactoe",
		GTK_RUN_FIRST,
		object_class->type,
		GTK_SIGNAL_OFFSET (DirtreeClass, dirtree),
		gtk_signal_default_marshaller, GTK_TYPE_NONE, 0);
	gtk_object_class_add_signals (object_class, dirtree_signals, LAST_SIGNAL);
	class->dirtree = NULL;
	*/
}

static void
dirtree_init (Dirtree *dt)
{
	dt -> selected_dir [0] = '\0';
	dt -> item_height = 1;
	dt -> parent = NULL;
}

int
dir_can_expand(char *dir)
{
	DIR *dp;
	struct direct *item;
	char fullname[256];
	struct stat st;
	int guess;
	
	if ((dp = opendir(dir)) == NULL) return FALSE;
	guess = FALSE;
	while ((item = readdir(dp)) != NULL)
	{
		if (item -> d_ino == 0) continue;
		if (strcmp(item -> d_name, ".") == 0) continue;
		if (strcmp(item -> d_name, "..") == 0) continue;
		strcpy(fullname, dir);
		if (fullname[strlen(fullname) - 1] != '/') strcat(fullname,"/");
		strcat(fullname, item -> d_name);
		stat(fullname, &st);
		if ((st.st_mode & S_IFMT) == S_IFDIR)
		{
			guess = TRUE;
			break;
		}
	}
	closedir(dp);
	return guess;
}

void
dirtree_expand_subtree(Dirtree *dt, GtkTree *tree, char *dir, char *e_target)
{
	DIR *dp;
	struct direct *item;
	GtkWidget *treeitem;//, *subtreeitem;
	GtkWidget *subtree;
	char fullname[256], *idx;
	struct stat st;
	dir_node *node;
	sel_node *snode;
	GList *list, *t_pos;
	
	if ((dp = opendir(dir)) == NULL) return;
	list = NULL;
	/*
	 * Read dir items and sort them.
	 */
	while ((item = readdir(dp)) != NULL)
	{
		if (item -> d_ino == 0) continue;
		if (strcmp(item -> d_name, ".") == 0) continue;
		if (strcmp(item -> d_name, "..") == 0) continue;
		strcpy(fullname, dir);
		if (fullname[strlen(fullname) - 1] != '/') strcat(fullname,"/");
		strcat(fullname, item -> d_name);
		stat(fullname, &st);
		if ((st.st_mode & S_IFMT) != S_IFDIR) continue;
		list = g_list_insert_sorted(list, (char*)strdup(item -> d_name), (GCompareFunc)strcmp);
	}
	/*
	 * Now, we can add all items to the sub-tree.
	 */
	t_pos = g_list_first(list);
	while (t_pos != NULL)
	{
		treeitem = dirtree_item_new(dt -> parent, t_pos -> data);
		gtk_tree_append(GTK_TREE(tree), treeitem);
		strcpy(fullname, dir);
		if (fullname[strlen(fullname) - 1] != '/') strcat(fullname,"/");
		strcat(fullname, t_pos -> data);
		snode = g_malloc(sizeof(sel_node));
		snode -> tree = dt;
		strcpy(snode -> new_dir, fullname);
		snode -> parent = dt -> parent;
		gtk_signal_connect(GTK_OBJECT(treeitem), "select",
			GTK_SIGNAL_FUNC(item_selected), snode);
		gtk_signal_connect(GTK_OBJECT(treeitem), "deselect",
			GTK_SIGNAL_FUNC(item_deselected), snode);
		if (dir_can_expand(fullname))
		{
			subtree = gtk_tree_new();
			gtk_tree_item_set_subtree(GTK_TREE_ITEM(treeitem), subtree);
			node = g_malloc(sizeof(dir_node));
			strcpy(node -> dir_name, fullname);
			node -> subtree = subtree;
			node -> tree = dt;
			node -> parent = dt -> parent;
			gtk_signal_connect(GTK_OBJECT(treeitem), "expand",
				GTK_SIGNAL_FUNC(item_expanded), node);
			gtk_signal_connect(GTK_OBJECT(treeitem), "collapse",
				GTK_SIGNAL_FUNC(item_collapsed), node);
		}
		if (e_target != NULL && strlen(e_target) > 0 &&
			strncmp(t_pos -> data, e_target, strlen(t_pos -> data)) == 0)
		{
			if ((idx = (char *)strchr(e_target, '/')) != NULL)
			{
				(*idx) = '\0';
				dirtree_expand_subtree(dt, GTK_TREE(subtree), fullname, idx + 1);
				gtk_tree_item_expand(GTK_TREE_ITEM(treeitem));
			}
		}
		gtk_widget_show(treeitem);
		t_pos = t_pos -> next;
	}
	g_list_free(list);
	closedir(dp);
}

void
item_expanded(GtkTreeItem *treeitem, dir_node *node)
{
	if (g_list_length(GTK_TREE(node -> subtree) -> children) <= 0)
	{
		dirtree_expand_subtree(node -> tree, GTK_TREE(node -> subtree),
			node -> dir_name, NULL);
	}
	dirtree_item_set_pixmap(node->parent, GTK_TREE_ITEM(treeitem), TRUE);
}

void
item_collapsed(GtkTreeItem *treeitem, dir_node *node)
{
	gtk_tree_select_child(GTK_TREE(node->tree), GTK_WIDGET(treeitem));
}

void
item_selected(GtkTreeItem *treeitem, sel_node *node)
{
	strcpy(node->tree->selected_dir, node->new_dir);
	dirtree_item_set_pixmap(node->parent, GTK_TREE_ITEM(treeitem), TRUE);
}

void
item_deselected(GtkTreeItem *treeitem, sel_node *node)
{
	if (! (treeitem -> expanded))
	{
		dirtree_item_set_pixmap(node->parent, GTK_TREE_ITEM(treeitem), FALSE);
	}
}

gchar*
dirtree_get_dir(Dirtree *tree)
{
	return tree->selected_dir;
}

GtkWidget*
dirtree_new(GtkWidget *parent)
{
	char *wd;
	wd = g_malloc(256 * sizeof(char));
	getcwd((char *)wd, 256);
	return GTK_WIDGET(dirtree_new_by_dir(parent, (char *)wd));
}

GtkWidget*
dirtree_new_by_dir(GtkWidget *parent, gchar *dir)
{
	Dirtree *dt;
	
	dt = DIRTREE (gtk_type_new (dirtree_get_type ()));

	dt -> parent = parent;
	strcpy(dt -> selected_dir, dir);
	gtk_tree_set_view_mode(GTK_TREE(dt), GTK_TREE_VIEW_ITEM);
	gtk_tree_set_selection_mode(GTK_TREE(dt), GTK_SELECTION_BROWSE);
	dirtree_create_root_item(dt, dir);
	
	return GTK_WIDGET(dt);
}

void
dirtree_create_root_item(Dirtree *dt, gchar *dir)
{
  GtkWidget /**tree,*/ *rootitem, *subtree;
	GtkRequisition requisition;
	sel_node *snode;
	dir_node *node;
	
	rootitem = dirtree_item_new(dt -> parent, "/");
	gtk_tree_append(GTK_TREE(dt), rootitem);
	
	snode = g_malloc(sizeof(sel_node));
	snode -> tree = dt;
	strcpy(snode -> new_dir, "/");
	snode -> parent = dt -> parent;
	gtk_signal_connect(GTK_OBJECT(rootitem), "select",
		GTK_SIGNAL_FUNC(item_selected), snode);
	gtk_signal_connect(GTK_OBJECT(rootitem), "deselect",
		GTK_SIGNAL_FUNC(item_deselected), snode);
		
	subtree = gtk_tree_new();
	gtk_tree_item_set_subtree(GTK_TREE_ITEM(rootitem), subtree);
	gtk_widget_show(rootitem);
	
	node = g_malloc(sizeof(dir_node));
	strcpy(node -> dir_name, "/");
	node -> subtree = subtree;
	node -> tree = dt;
	node -> parent = dt -> parent;
	gtk_signal_connect(GTK_OBJECT(rootitem), "expand",
		GTK_SIGNAL_FUNC(item_expanded), node);
	gtk_signal_connect(GTK_OBJECT(rootitem), "collapse",
		GTK_SIGNAL_FUNC(item_collapsed), node);
		
	gtk_widget_size_request(GTK_WIDGET(rootitem), &requisition);
	dt -> item_height = requisition.height;
	if (dir[0] == '/') dir++;
	dirtree_expand_subtree(dt, GTK_TREE(subtree), "/", (char *)dir);
	gtk_tree_item_expand(GTK_TREE_ITEM(rootitem));
}

void
dirtree_item_set_pixmap(GtkWidget *parent, GtkTreeItem *treeitem, gint expanded)
{
	static GdkPixmap *folder_expanded, *folder_collapsed;
	static GdkBitmap *mask_e, *mask_c;
	static GtkStyle *style;
	
	GtkPixmap *hbox, *pixmap;
	
	if (folder_expanded == NULL || folder_collapsed == NULL)
	{
		style = gtk_widget_get_style(parent);
		folder_expanded = gdk_pixmap_create_from_xpm_d(
			parent->window,
			&mask_e, &style->bg[GTK_STATE_NORMAL],
			(gchar **)folder_expanded_xpm
		);
		folder_collapsed = gdk_pixmap_create_from_xpm_d(
			parent->window,
			&mask_c, &style->bg[GTK_STATE_NORMAL],
			(gchar **)folder_collapsed_xpm
		);
	}
	
	hbox = gtk_container_children(GTK_CONTAINER(treeitem))->data;
	pixmap = gtk_container_children(GTK_CONTAINER(hbox))->data;
	if (expanded)
	{
		gtk_pixmap_set(GTK_PIXMAP(pixmap), folder_expanded, mask_e);
	} else
	{
		gtk_pixmap_set(GTK_PIXMAP(pixmap), folder_collapsed, mask_c);
	}
}

GtkWidget*
dirtree_item_new(GtkWidget *parent, gchar *dir)
{
	static GdkPixmap *folder_pix;
	static GdkBitmap *mask;
	static GtkStyle *style;
	
	GtkWidget *hbox, *folder, *label, *treeitem;

	if (folder_pix == NULL)
	{
		style = gtk_widget_get_style(parent);
		folder_pix = gdk_pixmap_create_from_xpm_d(
			parent->window,
			&mask, &style->bg[GTK_STATE_NORMAL],
			(gchar **)folder_expanded_xpm
		);
	}	
	
	treeitem = gtk_tree_item_new();
	hbox = gtk_hbox_new(FALSE, 1);
	folder = gtk_pixmap_new(folder_pix, mask);
	gtk_box_pack_start(GTK_BOX(hbox), folder, FALSE, FALSE, 0);
	label = gtk_label_new(dir);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_container_add(GTK_CONTAINER(treeitem), hbox);
	
	dirtree_item_set_pixmap(parent, GTK_TREE_ITEM(treeitem), FALSE);
	
	gtk_widget_show(folder);
	gtk_widget_show(label);
	gtk_widget_show(hbox);

	return treeitem;
}

gchar*
dirtree_item_get_label(GtkTreeItem *treeitem)
{
	GtkWidget *hbox, *label;
	gchar *text;
	
	hbox = gtk_container_children(GTK_CONTAINER(treeitem))->data;
	label = gtk_container_children(GTK_CONTAINER(hbox))->next->data;
	gtk_label_get(GTK_LABEL(label), &text);
	return text;
}

/* return value: the y location of the item which has been selected. */
gint
dirtree_select_dir(GtkTree *tree, gchar *dir)
{
  GList *children;
  unsigned int len;
  char *label;
  GtkWidget *treeitem;
  gint pos;  
  children = gtk_container_children (GTK_CONTAINER (tree));
  if (dir[0] == '/')
    {
      if (strlen(dir) > 1)
	{
	  return 1 + 
	    dirtree_select_dir(
			       GTK_TREE(GTK_TREE_ITEM(
			       children->data)->subtree), dir + 1);
	} 
      else
	{
	  gtk_tree_select_child(GTK_TREE(tree), children->data);
	  return 0;
	}
    } 
  else
    {
      while (children) {
	if (GTK_IS_TREE_ITEM(children->data))
	  {
	    treeitem = children->data;
	    label = dirtree_item_get_label(GTK_TREE_ITEM(treeitem));
	    if (strcmp(label, dir) == 0)
	      {
		gtk_tree_select_child(GTK_TREE(tree), treeitem);
		return gtk_tree_child_position(GTK_TREE(tree), treeitem);
	      }
	  } 
	else
	  {
	    treeitem = GTK_TREE(children->data) -> tree_owner;
	    label = dirtree_item_get_label(GTK_TREE_ITEM(treeitem));
	    if (strncmp(label, dir, strlen(label)) == 0) {
	      pos = 1 + gtk_tree_child_position(GTK_TREE(tree), treeitem);
	      if ((len = strlen(label)) == strlen(dir))
		{
		  gtk_tree_select_child(GTK_TREE(tree), treeitem);
		  return pos;
		} 
	      else
		{
		  return pos + 
		    dirtree_select_dir(GTK_TREE(children->data), dir + len + 1);
		}
	      break;
	    }
	  }
	children = g_list_remove_link (children, children);
      }
    }
  //
return pos;
}
 
gint 
dirtree_cdup(Dirtree *dt)
{
	gint n = strlen(dt->selected_dir) - 1;
	while (n > 1 && dt->selected_dir[n] != '/') n--;
	if (n > 1) dt->selected_dir[n] = '\0';
	else dt->selected_dir[n + 1] = '\0';
	return dirtree_select_dir(GTK_TREE(dt), dt->selected_dir);
}

gint
dirtree_refresh(Dirtree *tree)
{
  GtkWidget *rootitem;
  gchar buffer[256];
  
  rootitem = GTK_WIDGET(GTK_TREE(tree)->children->data);
  gtk_widget_destroy (rootitem);
  
  GTK_TREE(tree)->selection = NULL;
  
  strcpy(buffer, tree->selected_dir);
  dirtree_create_root_item(tree, tree->selected_dir);
  strcpy(tree->selected_dir, buffer);
  return 0;

}
