/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998, 1999  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_dibaudio.c,v 1.6 1999/11/23 19:01:40 hercules Exp $";
#endif

/* Allow access to a raw mixing buffer */

#include <stdio.h>
#include <stdlib.h>
#include <windows.h>
#include <mmsystem.h>

#include "SDL_audio.h"
#include "SDL_mutex.h"
#include "SDL_timer.h"
#include "SDL_audio_c.h"
#include "SDL_dibaudio.h"


/* Audio driver functions */
static int DIB_OpenAudio(_THIS, SDL_AudioSpec *spec);
static void DIB_WaitAudio(_THIS);
static void DIB_PlayAudio(_THIS);
static Uint8 *DIB_GetAudioBuf(_THIS);
static void DIB_CloseAudio(_THIS);

/* Audio driver bootstrap functions */

static int Audio_Available(void)
{
	return(1);
}

static void Audio_DeleteDevice(SDL_AudioDevice *device)
{
	free(device->hidden);
	free(device);
}

static SDL_AudioDevice *Audio_CreateDevice(int devindex)
{
	SDL_AudioDevice *this;

	/* Initialize all variables that we clean on shutdown */
	this = (SDL_AudioDevice *)malloc(sizeof(SDL_AudioDevice));
	if ( this ) {
		memset(this, 0, (sizeof *this));
		this->hidden = (struct SDL_PrivateAudioData *)
				malloc((sizeof *this->hidden));
	}
	if ( (this == NULL) || (this->hidden == NULL) ) {
		SDL_OutOfMemory();
		if ( this ) {
			free(this);
		}
		return(0);
	}
	memset(this->hidden, 0, (sizeof *this->hidden));

	/* Set the function pointers */
	this->OpenAudio = DIB_OpenAudio;
	this->WaitAudio = DIB_WaitAudio;
	this->PlayAudio = DIB_PlayAudio;
	this->GetAudioBuf = DIB_GetAudioBuf;
	this->CloseAudio = DIB_CloseAudio;

	this->free = Audio_DeleteDevice;

	return this;
}

AudioBootStrap WAVEOUT_bootstrap = {
	"waveout", Audio_Available, Audio_CreateDevice
};

/*
 * Make sure that any variable that is accessed in this callback is named
 * SHARED_xxx, then make sure that any time we alter or examine SHARED_xxx
 * we first grab the var_lock mutex.
 */

/* The Win32 callback for filling the WAVE device */
static void CALLBACK FillSound(HWAVEOUT hwo, UINT uMsg, DWORD dwInstance,
						DWORD dwParam1, DWORD dwParam2)
{
	SDL_AudioDevice *this = (SDL_AudioDevice *)dwInstance;

	/* Only service "buffer done playing" messages */
	if ( uMsg != WOM_DONE )
		return;

	/* Increment which buffer is currently playing */
	SDL_mutexV(var_lock);
	--SHARED_buffers_queued;
#ifdef SOUND_DEBUG
	fprintf(stderr, "Finished playing buffer %d\n", SHARED_playing);
#endif
	SHARED_playing = (SHARED_playing+1)%NUM_BUFFERS;
	SDL_mutexV(var_lock);

	/* Signal the audio thread that a buffer is done playing */
	SDL_mutexV(audio_wait);
}

static void SetMMerror(char *function, MMRESULT code)
{
	int    err_offset;
	static char errbuf[MAXERRORLENGTH];

	err_offset = strlen(function)+strlen(": ");
	errbuf[err_offset] = 0;
	waveOutGetErrorText(code, &errbuf[err_offset],
						MAXERRORLENGTH-err_offset);
	sprintf(errbuf, "%s: %s", function, &errbuf[err_offset]);
	SDL_SetError("%s", errbuf);
	return;
}

void DIB_WaitAudio(_THIS)
{
        /* grab the var_lock mutex going in because we'll have it grabbed right
	   before we test the while loop */

        SDL_mutexP(var_lock);
	/* Queue up the wavebufs that have been finished */
	do {
	        SDL_mutexV(var_lock);
		/* Wait for the current buffer to finish playing */
		SDL_mutexP(audio_wait);

		/* Queue it up */
		last_queued = (last_queued+1)%NUM_BUFFERS;
		SDL_mutexP (var_lock);
#ifdef SOUND_DEBUG
		fprintf(stderr, "Queuing up buffer %d  (%d buffers queued)\n",
					last_queued, SHARED_buffers_queued);
#endif
		waveOutWrite(sound, &wavebuf[last_queued], sizeof(wavebuf[0]));
		++SHARED_buffers_queued;
	} while ( SHARED_buffers_queued < NUM_BUFFERS );
        SDL_mutexV(var_lock);
}

void DIB_PlayAudio(_THIS)
{
	return;		/* The sound plays via the waveOut API */
}

Uint8 *DIB_GetAudioBuf(_THIS)
{
        Uint8 *retval;

	SDL_mutexP (var_lock);
#ifdef SOUND_DEBUG
	fprintf(stderr, "Filling buffer %d\n",
				(SHARED_playing+NUM_BUFFERS-1)%NUM_BUFFERS);
#endif
	retval = (wavebuf[(SHARED_playing+NUM_BUFFERS-1)%NUM_BUFFERS].lpData);
	SDL_mutexV (var_lock);
	return retval;
}

void DIB_CloseAudio(_THIS)
{
	int i;

	if (!audio_wait)
		return;

	/* Wait for the audio to drain */
#ifdef SOUND_DEBUG
	SDL_mutexP (var_lock);
	fprintf(stderr, "Sound closing, %d buffers queued\n", SHARED_buffers_queued);
	SDL_mutexV (var_lock);
#endif
	SDL_mutexP (var_lock);
	while ( SHARED_buffers_queued > 0 ) {
	  	SDL_mutexV (var_lock);
		SDL_Delay(250);
		SDL_mutexP (var_lock);
	}
  	SDL_mutexV (var_lock);
	waveOutClose(sound);

	/* Clean up mixing buffers */
	for ( i=0; i<NUM_BUFFERS; ++i ) {
		if ( wavebuf[i].dwUser != 0xFFFF ) {
			waveOutUnprepareHeader(sound, &wavebuf[i],
						sizeof(wavebuf[i]));
			wavebuf[i].dwUser = 0xFFFF;
		}
	}
	/* Free raw mixing buffer */
	if ( mixbuf != NULL ) {
		free(mixbuf);
		mixbuf = NULL;
	}

	/* Release audio trigger */
	SDL_DestroyMutex(audio_wait);
	audio_wait = NULL;

	SDL_DestroyMutex(var_lock);
	var_lock = NULL;
}

int DIB_OpenAudio(_THIS, SDL_AudioSpec *spec)
{
	MMRESULT     result;
	int          i;
	WAVEFORMATEX waveformat;

	if (audio_wait || var_lock)
	  return (-1);

	/* Initialize the wavebuf structures for closing */
	SHARED_buffers_queued = 0;
	for ( i = 0; i < NUM_BUFFERS; ++i )
		wavebuf[i].dwUser = 0xFFFF;
	mixbuf = NULL;

	/* Set basic WAVE format parameters */
	memset(&waveformat, 0, sizeof(waveformat));
	waveformat.wFormatTag = WAVE_FORMAT_PCM;

	/* Determine the audio parameters from the AudioSpec */
	switch ( spec->format & 0xFF ) {
		case 8:
			/* Unsigned 8 bit audio data */
			spec->format = AUDIO_U8;
			waveformat.wBitsPerSample = 8;
			break;
		case 16:
			/* Signed 16 bit audio data */
			spec->format = AUDIO_S16;
			waveformat.wBitsPerSample = 16;
			break;
		default:
			SDL_SetError("Unsupported audio format");
			return(-1);
	}
	waveformat.nChannels = spec->channels;
	waveformat.nSamplesPerSec = spec->freq;
	waveformat.nBlockAlign =
		waveformat.nChannels * (waveformat.wBitsPerSample/8);
	waveformat.nAvgBytesPerSec = 
		waveformat.nSamplesPerSec * waveformat.nBlockAlign;

	/* Check the buffer size -- minimum of 1/4 second (word aligned) */
	if ( spec->samples < (spec->freq/4) )
		spec->samples = ((spec->freq/4)+3)&~3;

	/* Update the fragment size as size in bytes */
	SDL_CalculateAudioSpec(spec);

	/* Open the audio device */
	result = waveOutOpen(&sound, WAVE_MAPPER, &waveformat,
			(DWORD)FillSound, (DWORD)this, CALLBACK_FUNCTION);
	if ( result != MMSYSERR_NOERROR ) {
		SetMMerror("waveOutOpen()", result);
		return(-1);
	}

#ifdef SOUND_DEBUG
	/* Check the sound device we retrieved */
	{
		WAVEOUTCAPS caps;

		result = waveOutGetDevCaps((UINT)sound, &caps, sizeof(caps));
		if ( result != MMSYSERR_NOERROR ) {
			SetMMerror("waveOutGetDevCaps()", result);
			return(-1);
		}
		printf("Audio device: %s\n", caps.szPname);
	}
#endif

	/* Create the audio trigger event */
	audio_wait = SDL_CreateMutex();

	/* Create the mutex to prevent race conditions when accessing
	   shared variables */
	var_lock   = SDL_CreateMutex();

	if ( audio_wait == NULL || var_lock == NULL ) {
		return(-1);
	}
	SDL_mutexP(audio_wait);

	/* Create the sound buffers */
	mixbuf = (Uint8 *)malloc(NUM_BUFFERS*spec->size);
	if ( mixbuf == NULL ) {
		SDL_SetError("Out of memory");
		return(-1);
	}
	for ( i = 0; i < NUM_BUFFERS; ++i ) {
		memset(&wavebuf[i], 0, sizeof(wavebuf[i]));
		wavebuf[i].lpData = &mixbuf[i*spec->size];
		wavebuf[i].dwBufferLength = spec->size;
		wavebuf[i].dwFlags = 0;
		result = waveOutPrepareHeader(sound, &wavebuf[i],
							sizeof(wavebuf[i]));
		if ( result != MMSYSERR_NOERROR ) {
			SetMMerror("waveOutPrepareHeader()", result);
			return(-1);
		}
	}
	memset(mixbuf, spec->silence, NUM_BUFFERS*spec->size);

	/* Start playing the audio buffers -- seed with all of them */
	SDL_mutexP (var_lock);
	SHARED_playing = 0;
	for ( i=0; i<NUM_BUFFERS; ++i ) {
		result = waveOutWrite(sound, &wavebuf[i], sizeof(wavebuf[0]));
		if ( result != MMSYSERR_NOERROR ) {
			SetMMerror("waveOutWrite()", result);
			return(-1);
		}
		++SHARED_buffers_queued;
	}
	SDL_mutexV (var_lock);
	last_queued = i-1;
	return(0);
}
