/* $Id: palemu.c,v 1.12 1999/01/29 22:29:24 marcus Exp $
******************************************************************************

   Display-palemu: palette emulation on true-color modes

   Copyright (C) 1998 Andrew Apted    [andrew@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <ggi/internal/ggi-dl.h>

#include <ggi/display/palemu.h>


/**************************************************
 ***
 ***  Internal functions 
 ***
 **************************************************/


static void blitter_1(PalemuHook *ph, void *dest, void *src, int w)
{
	uint8 *s = (uint8 *) src;
	uint8 *d = (uint8 *) dest;

	for (; w > 0; w--) {
		*d++ = ph->lookup[*s++];
	}
}

static void blitter_2(PalemuHook *ph, void *dest, void *src, int w)
{
	uint8  *s = (uint8  *) src;
	uint16 *d = (uint16 *) dest;

	for (; w > 0; w--) {
		*d++ = ph->lookup[*s++];
	}
}

static void blitter_3(PalemuHook *ph, void *dest, void *src, int w)
{
	uint8 *s = (uint8 *) src;
	uint8 *d = (uint8 *) dest;

	for (; w > 0; w--) {
		ggi_pixel pix = ph->lookup[*s++];

		*d++ = pix; pix >>= 8;
		*d++ = pix; pix >>= 8;
		*d++ = pix;
	}
}

static void blitter_4(PalemuHook *ph, void *dest, void *src, int w)
{
	uint8  *s = (uint8  *) src;
	uint32 *d = (uint32 *) dest;

	for (; w > 0; w--) {
		*d++ = ph->lookup[*s++];
	}
}


/**************************************************
 ***
 ***  Exported functions 
 ***
 **************************************************/


static uint8 src_buf[8192];
static uint8 dest_buf[8192];


/* !!! flesh out all four possibilities: (a) direct access to source
 * (Y/N), and (b) direct access to destination (Y/N).
 */
		
int _ggi_palemu_Transfer(ggi_visual *vis, int x, int y, int w, int h)
{
	PalemuHook *ph = PALEMU_PRIV(vis);

	int old_r_frame = vis->r_frame_num;

	ph->mem_opdraw->setreadframe(vis, vis->d_frame_num);

	
	/* do transfer */

	for (; h > 0; h--, y++) {

		ggiGetHLine(vis, x, y, w, src_buf);
			
		(* ph->do_blit)(ph, dest_buf, src_buf, w);

		ggiPutHLine(ph->parent, x, y, w, dest_buf);
	}

	ph->mem_opdraw->setreadframe(vis, old_r_frame);

	return 0;
}

int _ggi_palemu_Flush(ggi_visual *vis)
{
	PalemuHook *ph = PALEMU_PRIV(vis);

	int sx = ph->dirty_tl.x; int sy = ph->dirty_tl.y;
	int ex = ph->dirty_br.x; int ey = ph->dirty_br.y;


	/* clear the `dirty region' */

	ph->dirty_tl.x = LIBGGI_VIRTX(vis);
	ph->dirty_tl.y = LIBGGI_VIRTY(vis);
	ph->dirty_br.x = 0;
	ph->dirty_br.y = 0;


	/* When write_frame != display_frame, then there is no need to
	 * update the parent since the affected area(s) are not visible.
	 */
	 
	if ((vis->w_frame_num == vis->d_frame_num) && 
	    (sx < ex) && (sy < ey)) {

		return _ggi_palemu_Transfer(vis, sx, sy, ex-sx, ey-sy);
	}

	return 0;
}

int _ggi_palemu_Open(ggi_visual *vis)
{
	PalemuHook *ph = PALEMU_PRIV(vis);


	GGIDPRINT("display-palemu: Open %dx%d#%dx%d\n", LIBGGI_X(vis), 
		LIBGGI_Y(vis), LIBGGI_VIRTX(vis), LIBGGI_VIRTY(vis));


	/* set the parent mode */
	
	if (ggiSetMode(ph->parent, &ph->mode) < 0) {

		GGIDPRINT("display-palemu: Couldn't set parent mode.\n");
		return -1;
	}

	GGIDPRINT("display-palemu: parent is %d/%d\n",
		GT_DEPTH(ph->mode.graphtype), 
		GT_SIZE(ph->mode.graphtype));
	

	/* setup tables and choose blitter function */

	switch ((GT_SIZE(ph->mode.graphtype) + 7) / 8) {

	case 1: ph->do_blit = &blitter_1;
		break;
		
	case 2: ph->do_blit = &blitter_2;
		break;
		
	case 3: ph->do_blit = &blitter_3;
		break;
		
	case 4: ph->do_blit = &blitter_4;
		break;
		
	default:
		GGIDPRINT("Unsupported pixel size '%d'.\n",
			GT_SIZE(ph->mode.graphtype));
		return -1;
	}


	ph->palette = _ggi_malloc(256 * sizeof(ggi_color));
	ph->lookup  = _ggi_malloc(256 * sizeof(ggi_pixel));

	ph->red_gamma = ph->green_gamma = ph->blue_gamma = 1.0;

	
	/* clear the 'dirty region' */

	ph->dirty_tl.x = LIBGGI_VIRTX(vis);
	ph->dirty_tl.y = LIBGGI_VIRTY(vis);
	ph->dirty_br.x = 0;
	ph->dirty_br.y = 0;

	return 0;
}

int _ggi_palemu_Close(ggi_visual *vis)
{
	PalemuHook *ph = PALEMU_PRIV(vis);

	if (ph->palette != NULL) {
		free(ph->palette);
	}
	if (ph->lookup != NULL) {
		free(ph->lookup);
	}

	return 0;
}
