/* $Id: visual.c,v 1.24 1999/04/01 19:09:04 marcus Exp $
******************************************************************************

   Display-memory: mode management

   Copyright (C) 1995 Andreas Beck      [becka@ggi-project.org]
   Copyright (C) 1997 Jason McMullan    [jmcc@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdlib.h>
#include <string.h>

#include <ggi/display/memory.h>

static gg_option memory_options[1] =
{
	{ "input", "" }
};

ggi_event_mask GII_memory_poll(gii_input_t inp, void *arg)
{
	struct memhooks *hook=inp->priv;
	ggi_event ev;
	int rc=0;
	
	while(hook->inputoffset!=hook->inputbuffer->writeoffset)
	{
		if (hook->inputbuffer->buffer[hook->inputoffset++]!=MEMINPMAGIC)
		{
			GGIDPRINT_MISC("OUT OF SYNC in meminput !\n");
			hook->inputoffset=0;	/* Try to resync */
			return 0;
		}
		memcpy(&ev, &(hook->inputbuffer->buffer[hook->inputoffset]),
		       hook->inputbuffer->buffer[hook->inputoffset]);
		_giiEvQueueAdd(inp, &ev);
		hook->inputoffset += ev.any.size;
		rc |= 1<<ev.any.type;
		if (hook->inputoffset >= INPBUFSIZE-sizeof(ggi_event)-sizeof(hook->inputbuffer->writeoffset)-10) {
			hook->inputoffset=0;
		}
	}
	return rc;
}

int GII_memory_send(gii_input_t inp, ggi_event *event)
{
	struct memhooks *hook=inp->priv;
	int size;
	
	hook->inputbuffer->buffer[hook->inputbuffer->writeoffset++]=MEMINPMAGIC;
	memcpy(&(hook->inputbuffer->buffer[hook->inputbuffer->writeoffset]),
		event,size=event->any.size);
	hook->inputbuffer->writeoffset+=size;
	if (hook->inputbuffer->writeoffset>=INPBUFSIZE-sizeof(ggi_event)-sizeof(hook->inputbuffer->writeoffset)-10)
		hook->inputbuffer->writeoffset=0;
	hook->inputbuffer->buffer[hook->inputbuffer->writeoffset]=MEMINPMAGIC-1;	/* "break"-symbol */

	return 0;
}


int GGIdlinit(ggi_visual *vis,const char *args,void *argptr)
{
	LIBGGI_FD(vis)=-1;

	LIBGGI_GC(vis) = _ggi_malloc(sizeof(ggi_gc));

	GGIDPRINT("display-memory coming up.\n");
	
	/* Allocate descriptor for screen memory */
	LIBGGI_PRIVATE(vis)=malloc(sizeof(struct memhooks));
	MEMLIB_PRIV(vis)->memtype=MT_MALLOC;	/* Default to mallocing. */
	MEMLIB_PRIV(vis)->inputbuffer=NULL;	/* Default to no input */
	MEMLIB_PRIV(vis)->inputoffset=0;	/* Setup offset. */

	if (args) {
		args = ggParseOptions((char *) args, memory_options, 1);
		if (args == NULL) {
			fprintf(stderr, "display-memory: error in "
				"arguments.\n");
		}
	}

	if (args && *args)	/* We have parameters. Analyze them. */
	{
		GGIDPRINT("display-memory has args.\n");
#ifdef HAVE_SHM
		if (strncmp(args,"shmid:",6)==0)
		{
			sscanf(args+6,"%i",&(MEMLIB_PRIV(vis)->shmid));
			GGIDPRINT("display-memory has shmid-arg:%d.\n",
				MEMLIB_PRIV(vis)->shmid);
			MEMLIB_PRIV(vis)->memptr=shmat(MEMLIB_PRIV(vis)->shmid,NULL,0);
			GGIDPRINT("display-memory: shmat at %p.\n",
				MEMLIB_PRIV(vis)->memptr);
			if (MEMLIB_PRIV(vis)->memptr!=(void *)-1) 
			{
				MEMLIB_PRIV(vis)->memtype=MT_SHMID;
				if (memory_options[0].result[0])
				{
					MEMLIB_PRIV(vis)->inputbuffer=MEMLIB_PRIV(vis)->memptr;
					MEMLIB_PRIV(vis)->memptr=(char *)MEMLIB_PRIV(vis)->memptr+INPBUFSIZE;
					GGIDPRINT("display-memory: moved mem to %p for input-buffer.\n",
						MEMLIB_PRIV(vis)->memptr);
				}
			}
		}
		else if (strncmp(args,"keyfile:",8)==0)
		{
			int size;
			char id;
			char filename[1024];

			sscanf(args+8,"%d:%c:%s",&size,&id,filename);
			GGIDPRINT("display-memory has keyfile-arg:%d:%c:%s.\n",
				size,id,filename);

			MEMLIB_PRIV(vis)->shmid=shmget(ftok(filename,id), size, IPC_CREAT|0666);
			GGIDPRINT("display-memory has shmid:%d.\n",
				MEMLIB_PRIV(vis)->shmid);

			MEMLIB_PRIV(vis)->memptr=shmat(MEMLIB_PRIV(vis)->shmid,NULL,0);
			GGIDPRINT("display-memory: shmat at %p.\n",
				MEMLIB_PRIV(vis)->memptr);
			if (MEMLIB_PRIV(vis)->memptr!=(void *)-1) 
			{
				MEMLIB_PRIV(vis)->memtype=MT_SHMID;
				if (memory_options[0].result[0])
				{
					MEMLIB_PRIV(vis)->inputbuffer=MEMLIB_PRIV(vis)->memptr;
					MEMLIB_PRIV(vis)->memptr=(char *)MEMLIB_PRIV(vis)->memptr+INPBUFSIZE;
					GGIDPRINT("display-memory: moved mem to %p for input-buffer.\n",
						MEMLIB_PRIV(vis)->memptr);
				}
			}
		}
#endif
		else if (strncmp(args,"pointer",7)==0)
		{
			MEMLIB_PRIV(vis)->memptr = argptr;
			if (MEMLIB_PRIV(vis)->memptr)
				MEMLIB_PRIV(vis)->memtype=MT_EXTERN;
		}
	}

	vis->opdisplay->getmode=GGI_memory_getmode;
	vis->opdisplay->setmode=GGI_memory_setmode;
	vis->opdisplay->getapi=GGI_memory_getapi;
	vis->opdisplay->checkmode=GGI_memory_checkmode;
	vis->opdisplay->setflags=GGI_memory_setflags;

	if (MEMLIB_PRIV(vis)->inputbuffer)
	{
		gii_input *inp;

		MEMLIB_PRIV(vis)->inputbuffer->visx=
		MEMLIB_PRIV(vis)->inputbuffer->visy=
		MEMLIB_PRIV(vis)->inputbuffer->virtx=
		MEMLIB_PRIV(vis)->inputbuffer->virty=
		MEMLIB_PRIV(vis)->inputbuffer->frames=
		MEMLIB_PRIV(vis)->inputbuffer->visframe=0;

		GGIDPRINT_MISC("Adding gii to shmem-memtarget\n");

		/* First allocate a new gii_input descriptor. */

		if (NULL==(inp=_giiInputAlloc()))
		{
			GGIDPRINT_MISC("giiInputAlloc failure.\n");
			goto out;
		}
		GGIDPRINT_MISC("gii inp=%p\n",inp);

		/* Now fill in the blanks. */

		inp->priv=MEMLIB_PRIV(vis);	/* We need that in poll() */
		MEMLIB_PRIV(vis)->inputbuffer->writeoffset=0;	/* Not too good, but ... */
		inp->targetcan= emAll;
		inp->GIIseteventmask(inp,inp->targetcan);
		inp->maxfd=0;	/* This is polled. */
		inp->flags|=GII_FLAGS_HASPOLLED;

		inp->GIIeventpoll=GII_memory_poll;
		inp->GIIsendevent=GII_memory_send;

		/* Now join the new event source in. */
		vis->input=giiJoinInputs(vis->input,inp);
		out:
		{};
	}
	
	return GGI_DL_OPDISPLAY;
}

int GGIdlcleanup(ggi_visual *vis)
{
	_GGI_memory_resetmode(vis);

	switch(MEMLIB_PRIV(vis)->memtype) {
	case MT_MALLOC:
	case MT_EXTERN:	/* Nothing to be done. */
	  	break;
#ifdef HAVE_SHM		
	case MT_SHMKEYFILE: /* FIXME ? Should we RMID the area ? */
	case MT_SHMID: 
	  	shmdt(MEMLIB_PRIV(vis)->memptr);
	  	break;
#endif	
	default:
		break;
	}

	free(LIBGGI_PRIVATE(vis));
	free(LIBGGI_GC(vis));

	return 0;
}

#include <ggi/internal/ggidlinit.h>
