/*
 * a PhotoCD viewer, for svgalib and X11
 *
 *   (c) 1996 Gerd Knorr <kraxel@goldbach.in-berlin.de>
 *
 */

#define _BSD_SOURCE
#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#else
extern int      optind;
extern char    *optarg;

#endif

#include <vga.h>
#include <vgagl.h>
#include "svga.h"

#include "pcd.h"
#include "dither.h"

#define TRUE  1
#define FALSE 0

/* ---------------------------------------------------------------------- */

/* variables for read_image */
unsigned long  *lut_red = NULL, *lut_green = NULL, *lut_blue = NULL;
int             type, dither = FALSE, res = 3, gray = FALSE;

/* file list */
char          **files;
int             fileanz, filenr;

/* svga font stuff */
int             font_height;
unsigned char  *font;
char           *fontfile = "/usr/lib/kbd/consolefonts/lat1-16.psf";

/* ---------------------------------------------------------------------- */

void
usage(char *name)
{
    char           *h;

    h = strrchr(name, '/');
    fprintf(stderr,
	    "This program shows PhotoCD images using svgalib for\n"
	    "displaying the images.\n"
	    "  usage: %s [ options ] file ...\n"
	    "    -h      print this text\n"
	    "    -m n    use graphics mode nr n\n"
	 "    -c n    Use a graphics mode with n colors. Allowed here:\n"
	    "            gray, 256, 32k, 64k and 16m\n"
	    "    -l      list available graphics modes\n"
	    "    -f file read font from file for printing text\n"
	    "            default: %s\n"
	    "    -q      be quiet: don't print anything at all\n"
	    "    -t n    timeout: load next image after n sec without any keypress\n"
	    "    -r n    select resolution [1..5]\n"
	    "\n"
	    "Large images can be scrolled using the cursor keys. Use ESC or 'q' to quit.\n"
	    "Space and PgDn show the next, PgUp shows the previous image. Jumping to a\n"
	    "image works with <number>g\n"
	    "\n"
	    "This program tries to find the best graphics mode for displaying the image,\n"
	    "depending on its size. All graphic modes are likely to be switched on.\n"
	    "SO BE SURE YOUR MONITOR CAN DISPLAY ALL MODES YOU HAVE IN YOUR SVGALIB CONFIG\n"
	    "FILE, YOUR MONITOR MAY GET DAMAGED ELSE.\n"
	    "\n"
	    ,h ? h + 1 : name, fontfile);
}

/* ---------------------------------------------------------------------- */

void
read_image(char *filename, char *image,
	   int res, int *width, int *height)
{
    int             left, top;
    struct PCD_IMAGE img;

    left = top = *width = *height = 0;
    if (0 != pcd_open(&img, filename) ||
	-1 == pcd_select(&img, res, 0, (type == PCD_TYPE_GRAY), 0,
			 pcd_get_rot(&img, 0),
			 &left, &top, width, height) ||
	-1 == pcd_decode(&img)) {
	fprintf(stderr, "libpcd: %s\n", pcd_errmsg);
	exit(1);
    }
    if (dither) {
	unsigned char  *rgb_line = malloc((*width) * 3);
	int             y;

	for (y = 0; y < *height; y++) {
	    if (-1 == pcd_get_image_line(&img, y, rgb_line, type, 0)) {
		fprintf(stderr, "libpcd: %s\n", pcd_errmsg);
		exit(1);
	    }
	    dither_line(rgb_line, image + y * (*width), y, *width);
	}
	free(rgb_line);
    } else {
	if (type == PCD_TYPE_LUT_SHORT ||
	    type == PCD_TYPE_LUT_LONG)
	    pcd_set_lookup(&img, lut_red, lut_green, lut_blue);
	if (-1 == pcd_get_image(&img, image, type, 0)) {
	    fprintf(stderr, "libpcd: %s\n", pcd_errmsg);
	    exit(1);
	}
    }
    pcd_close(&img);
}

/* ---------------------------------------------------------------------- */

void
svga_readfont()
{
    FILE           *fp = fopen(fontfile, "r");

    if (NULL == fp)
	return;
    if (fgetc(fp) != 0x36 ||
	fgetc(fp) != 0x04 ||
	fgetc(fp) != 0x00)
	return;

    font_height = fgetc(fp);
    font = malloc(font_height * 256);
    fread(font, 256, font_height, fp);
    fclose(fp);
}

/* ---------------------------------------------------------------------- */

int
main(int argc, char *argv[])
{
    int             mode = 0, colors = 0, timeout = 0, verbose = 1;
    int             width, height;
    unsigned char  *image;
    vga_modeinfo   *vgamode;
    int             c, rc;
    int             need_read;
    char            message[256], *basename;

    for (;;) {
	c = getopt(argc, argv, "hlqm:c:r:t:f:");
	if (c == -1)
	    break;
	switch (c) {
	case 'h':
	    usage(argv[0]);
	    exit(1);
	case 'q':
	    verbose = 0;
	    break;
	case 'm':
	    mode = atoi(optarg);
	    break;
	case 'c':
	    switch (atoi(optarg)) {
	    case 256:
		colors = 256;
		break;
	    case 32:
	    case 32 * 1024:
		colors = 32 * 1024;
		break;
	    case 64:
	    case 64 * 1024:
		colors = 64 * 1024;
		break;
	    case 16:
	    case 16 * 1024 * 1024:
		colors = 16 * 1024 * 1024;
		break;
	    default:
		if (strcasecmp(optarg, "gray") == 0 ||
		    strcasecmp(optarg, "grey") == 0) {
		    colors = 256;
		    gray = TRUE;
		} else {
		    colors = 0;
		}
	    }
	    if (colors == 0)
		fprintf(stderr, "I don't understand `-c %s', ignoring.\n",
			optarg);
	    break;
	case 'r':
	    res = atoi(optarg);
	    break;
	case 'f':
	    fontfile = optarg;
	    break;
	case 't':
	    timeout = atoi(optarg);
	    break;
	case 'l':
	    vga_init();
	    svga_list_modes();
	    exit(0);
	default:
	    exit(1);
	}
    }

    if (optind == argc) {
	usage(argv[0]);
	exit(1);
    }
    files = argv + optind;
    fileanz = argc - optind;
    filenr = 0;
    need_read = 1;

    vga_init();
    vgamode = svga_guess_mode(PCD_WIDTH(res, 0), PCD_HEIGHT(res, 0), colors, mode);
    if (NULL == vgamode) {
	if (mode)
	    fprintf(stderr, "Can't use mode %i, sorry.\n", mode);
	else
	    fprintf(stderr, "I found no graphics mode.\n");
	exit(1);
    }
    if (verbose) {
	/* set default in case readfont failes */
	font = gl_font8x8;
	font_height = 8;
	svga_readfont();
	gl_setwritemode(WRITEMODE_OVERWRITE | FONT_COMPRESSED);
	gl_setfont(8, font_height, font);
	gl_setfontcolors(0, 0xffffffff);
    }
    switch (vgamode->bytesperpixel) {
    case 1:
	if (gray) {
	    type = PCD_TYPE_GRAY;
	    svga_gray_palette();
	} else {
	    type = PCD_TYPE_RGB;
	    svga_dither_palette(8, 8, 4);
	    dither = TRUE;
	    init_dither(8, 8, 4, 2);
	}
	break;
    case 2:
	type = PCD_TYPE_LUT_SHORT;
	if (vgamode->colors == 32 * 1024) {
	    lut_red = LUT_15_red;
	    lut_green = LUT_15_green;
	    lut_blue = LUT_15_blue;
	} else {
	    lut_red = LUT_16_red;
	    lut_green = LUT_16_green;
	    lut_blue = LUT_16_blue;
	}
	break;
    case 3:
	type = (vgamode->flags&RGB_MISORDERED) ? PCD_TYPE_RGB : PCD_TYPE_BGR;
	break;
    case 4:
	type = PCD_TYPE_LUT_LONG;
	if (vgamode->flags&RGB_MISORDERED) {
		lut_red   = LUT_24_blue;
		lut_green = LUT_24_green;
		lut_blue  = LUT_24_red;
	} else {
		lut_red   = LUT_24_red;
		lut_green = LUT_24_green;
		lut_blue  = LUT_24_blue;
	}
	break;
    default:
	fprintf(stderr, "Oops: %i bytes/pixel ???\n",
		vgamode->bytesperpixel);
	exit(1);
    }

    if (verbose) {
	sprintf(message, "video mode:   %ix%i, %i colors",
		vgamode->width, vgamode->height, vgamode->colors);
	gl_write(0, 0, message);
	gl_write(0, 2 * font_height, "ESC,Q:        quit");
	gl_write(0, 3 * font_height, "PgDn,space:   next image");
	gl_write(0, 4 * font_height, "PgUp:         previous image");
	gl_write(0, 5 * font_height, "<number>G:    jump to image <number>");
	gl_write(0, 6 * font_height, "cursor keys:  scroll large images");
    }
    image = malloc(PCD_WIDTH(res, 0) * PCD_HEIGHT(res, 0) *
		   vgamode->bytesperpixel);

    /* svga main loop */
    for (;;) {
	if (need_read) {
	    need_read = 0;
	    if (verbose) {
		basename = strrchr(files[filenr], '/');
		basename = basename ? basename + 1 : files[filenr];
		sprintf(message, "loading %s... ", basename);
		gl_write(0, vgamode->height - font_height, message);
	    }
	    read_image(files[filenr], image, res, &width, &height);
	    if (width < vgamode->width || height < vgamode->height)
		svga_cls(vgamode);
	}
	fcntl(0,F_SETFL,0);
	switch (rc = svga_show(vgamode, image, width, height, timeout)) {
	case KEY_ESC:
	case KEY_Q:
	case KEY_EOF:
	    vga_setmode(TEXT);
	    exit(0);
	    break;
	case KEY_SPACE:
	    if (filenr == fileanz - 1) {
		vga_setmode(TEXT);
		exit(0);
	    }
	    /* fall throuth: space exits after the last image, PgDn not */
	case KEY_PGDN:
	    if (filenr < fileanz - 1)
		filenr++, need_read = 1;
	    break;
	case KEY_PGUP:
	    if (filenr > 0)
		filenr--, need_read = 1;
	    break;
	case KEY_TIMEOUT:
	    need_read = 1;
	    filenr++;
	    if (filenr == fileanz)
		filenr = 0;
	    break;
	default:
	    if (rc > 0 && rc <= fileanz)
		filenr = rc - 1, need_read = 1;
	    break;
	}
    }
}
