/*
 * win.c: Windowing/graphics functions for XTux.
 * Copyright 1999 David Lawrence (philaw@camtech.net.au)
 * Last modified July 26.
 */

#include <X11/keysym.h>
#include <sys/types.h>
#include <sys/utsname.h>
#include <dirent.h>
#include "header.h"
#include "entity.h"
#include "main.h"
#include "tile.h"
#include "timing.h"
#include "win.h"
#include "image.h"
#include "sound.h"

extern int map_dirty, status_dirty, trigger; /* External flags */
extern entity *root, *player;

/* Used for almost ever X call */
int screen, depth;
Display *display;
Window win, text_win;

/* Colormap related stuff */
Colormap cmap;
ColorTable idx[256];

Pixmap buffer, status_buf, map_buf, tmp_buf;
XFontStruct  *med_font_18, *bold_font_14, *bold_font_24;
GC gc, masked_gc, solid_gc, offset_gc, text_gc, text_win_gc;
int text_win_up = 0;

/* Initialises all of the graphics related stuff, creates windows,
   loads images, fonts, colormap, icon. Called at program initialisation. */
void create_window(void)
{

  const char *program_name = XTUXVERSION;
  int screen, depth;
  Pixmap icon_pixmap, icon_mask, splash_screen;
  XSizeHints size_hints;
  XWMHints wm_hints;
  XClassHint class_hints;

  if( (display = XOpenDisplay(NULL)) == NULL) {
    fprintf(stderr, "Error: Can't open display.\n");
    exit(1);
  }

  screen = DefaultScreen(display);
  depth = DefaultDepth(display, screen);
  cmap = DefaultColormap(display, screen);

#if VERBOSE
  printf("Creating Window..\n");
#endif  

  win = XCreateSimpleWindow(display, RootWindow(display, screen), 0, 0,
			    SPLASH_W, SPLASH_H, 0, BlackPixel(display, screen),
			    BlackPixel(display,screen));

  text_win = XCreateSimpleWindow(display, win,(WIN_W-TWINW)/ 2,(WIN_H-TWINH)/2,
				 TWINW, TWINH, 0, WhitePixel(display, screen),
				 WhitePixel(display, screen));

  XSelectInput(display, win, KeyPressMask | KeyReleaseMask | ExposureMask);

  setup_gcs();
  create_private_colormap();

  /* Create buffers */
  buffer = XCreatePixmap(display, win, VIEW_W, VIEW_H, depth);
  map_buf = XCreatePixmap(display, win, VIEW_W+TILE_W, VIEW_H+TILE_H, depth);
  status_buf = XCreatePixmap(display, win, WIN_W, STATUS_HEIGHT, depth);

  /* Clear buffer & status_buf */
  clear_area(buffer,0,0, VIEW_W, VIEW_H, WhitePixel(display, screen));
  clear_area(status_buf,0,0, WIN_W, STATUS_HEIGHT,BlackPixel(display, screen));

#if VERBOSE
  printf("Loading Fonts..\n");
#endif
  load_fonts();
  icon_pixmap = load_image("images/icon.xpm", &icon_mask);
  splash_screen = load_image("images/tuxbat.xpm", NULL);

  class_hints.res_name = NULL;
  class_hints.res_class = NULL;

  size_hints.flags = PSize | PMinSize | PMaxSize;
  size_hints.min_width = SPLASH_W;
  size_hints.max_width = SPLASH_W;
  size_hints.min_height = SPLASH_H;
  size_hints.max_height = SPLASH_H;

  wm_hints.flags = IconPixmapHint | IconMaskHint;
  wm_hints.icon_pixmap = icon_pixmap;
  wm_hints.icon_mask = icon_mask;

#if VERBOSE
  printf("Setting WM properties...\n");
#endif
  XmbSetWMProperties(display, win, "Splash Screen", "Splash Screen", 0, 0,
		     &size_hints, &wm_hints, &class_hints);  

#if VERBOSE
  printf("Drawing splash-screen\n");
#endif
  blit(splash_screen, buffer, 0, 0, SPLASH_IMG_W, SPLASH_IMG_H);

  XMapRaised(display, win);
  XClearWindow(display, win);
  XFlush(display);
  update_window();

#if VERBOSE
  printf("Loading Images..\n");
#endif
  if( !load_images() )
    die("Error loading images!\n", 1);

  size_hints.min_width = WIN_W;
  size_hints.max_width = WIN_W;
  size_hints.min_height = WIN_H;
  size_hints.max_height = WIN_H;

#if VERBOSE
  printf("Setting WM properties(again)...\n");
#endif  

  XmbSetWMProperties(display, win, program_name, program_name, 0, 0,
		     &size_hints, &wm_hints, &class_hints);

  XResizeWindow(display, win, WIN_W, WIN_H);
  /* We want white text in the game, it's currently black */
  XSetForeground(display, text_gc, WhitePixel(display, screen));

  /* For text win */
  tmp_buf = XCreatePixmap(display, text_win, TWINW, TWINH, depth);
  XSetForeground(display, text_win_gc, WhitePixel(display, screen));
  XFillRectangle(display, tmp_buf, text_win_gc, 0, 0, TWINW, TWINH);

  XMapRaised(display, win);
  XClearWindow(display, win);

  /* Free unneeded pixmaps */
  XFreePixmap(display, splash_screen);

} /* Create_window */


/* Create and initilise the graphics contexts used in the game */
void setup_gcs(void)
{

  gc = XCreateGC(display, win, 0, NULL);
  XSetFillStyle(display, gc, FillTiled);

  masked_gc = XCreateGC(display, win, 0, NULL);
  XSetFillStyle(display, masked_gc, FillTiled);

  solid_gc = XCreateGC(display, win, 0, NULL);
  XSetFillStyle(display, solid_gc, FillSolid);

  text_gc = XCreateGC(display, win, 0, NULL);
  XSetForeground(display, text_gc, BlackPixel(display, screen));

  offset_gc = XCreateGC(display, win, 0, NULL);
  XSetFillStyle(display, offset_gc, FillTiled);

  text_win_gc = XCreateGC(display, text_win, 0, NULL);

}

void load_fonts(void) {

  if( !(med_font_18 = (XLoadQueryFont(display, MED_FONT_18)) )) {
    fprintf(stderr, "Error loading font:%s\n", MED_FONT_18);
    die("Font error!\n", 1);
  }

  if( !(bold_font_14 = (XLoadQueryFont(display, BOLD_FONT_14)) )) {
    fprintf(stderr, "Error loading font:%s\n", BOLD_FONT_14);
    die("Font error!\n", 1);
  }

  if( !(bold_font_24 = (XLoadQueryFont(display, BOLD_FONT_24)) )) {
    fprintf(stderr, "Error loading font:%s\n", BOLD_FONT_24);
    die("Font error!\n", 1);
  }

}

/* Assign RGB value (r,g,b) to colormap index i. Code for this function and
 create_private_colormap comes from the examples from Paul Coates Linux/X11
 game writes page */
void alloc_color(int i, int r, int g, int b)
{

  XColor col;
  
  col.red = r; col.green = g; col.blue = b;
  col.flags = DoRed | DoGreen | DoBlue;
  if (XAllocColor(display, cmap, &col))
    idx[i].pixel = col.pixel;
  else
    if (cmap == DefaultColormap(display, screen)) {
      cmap = XCopyColormapAndFree(display, cmap);
      XSetWindowColormap(display, win, cmap);
      col.red =	r; col.green = g; col.blue = b;
      col.flags = DoRed | DoGreen | DoBlue;
      if (XAllocColor(display, cmap, &col))
	idx[i].pixel = col.pixel;
    }
  
}

/* Creates a private colormap of 256 colors, which is used on the game window.
   >8 bit displays will use this colormap and the default, while 8 bit displays
   will use this colormap for the entire screen when the window has the color-
   map installed */
void create_private_colormap(void)
{

  int i, j, k;
  int count = 0;
  
  for (i=0; i<256; i++)
    idx[i].pixel = 0;
  
  alloc_color(count++, 0, 0, 0);
  alloc_color(count++, 255, 255, 255);
  alloc_color(count++, 0, 0, 0);
  
  for (i=0; i<5; i++)
    for (j=0; j<5; j++)
      for (k=0; k<5; k++) {
	/* Red */
	idx[count].r = (1<<16) - ( i*(1<<14) );
	if(idx[count].r < 0)
	  idx[count].r = 0;
	/* Green */
	idx[count].g = (1<<16) - ( j*(1<<14) );
	if(idx[count].g < 0)
	  idx[count].g = 0;
	/* Blue */
	idx[count].b = (1<<16) - ( k*(1<<14) );
	if(idx[count].b < 0)
	  idx[count].b = 0;
	alloc_color(count++, idx[count].r, idx[count].g, idx[count].b);
     }
  
  for (i=0; i<4; i++)
    for (j=0; j<4; j++)
      for (k=0; k<4; k++) {
	/* Red */
	idx[count].r = 60415 - ( (i*1<<14) );
	if(idx[count].r < 0)
	  idx[count].r = 0;
	/* Green */
	idx[count].g = 60415 - ( (j*1<<14) );
	if(idx[count].g < 0)
	  idx[count].g = 0;
	/* Blue */
	idx[count].b = 60415 - ( (k*1<<14) );
	if(idx[count].b < 0)
	  idx[count].b = 0;
	alloc_color(count++, idx[count].r, idx[count].g, idx[count].b);

	/* Red */
	idx[count].r = 55295 - ( (i*1<<14) );
	if(idx[count].r < 0)
	  idx[count].r = 0;
	/* Green */
	idx[count].g = 55295 - ( (j*1<<14) );
	if(idx[count].g < 0)
	  idx[count].g = 0;
	/* Blue */
	idx[count].b = 55295 - ( (k*1<<14) );
	if(idx[count].b < 0)
	  idx[count].b = 0;
	alloc_color(count++, idx[count].r, idx[count].g, idx[count].b);
      }
}


/* Displays things while splash screen is up */
void splash_screen_print(char *msg)
{

  int title_width = XTextWidth( med_font_18, msg, strlen(msg));

  /* Remove other traces of text */
  clear_area(buffer, 0, SPLASH_IMG_H, SPLASH_W, SPLASH_H - SPLASH_IMG_H,
	     WhitePixel(display, screen));

  print(msg, buffer, (SPLASH_W-title_width)/2, SPLASH_IMG_H,
	med_font_18);

  update_window();

}


/***************************************************************************
 * Draws a menu screen, where the user can select what character and map they
 * want to play on. At the moment, user-input is handled by this functions very
 * own input handler, though this will change in future.
 * Player drops back down to the menu when they die or quit from a map.
 ***************************************************************************/
int menu(char *map_name, int *type)
{

  /* Constant strings */
  char *title = XTUXVERSION;
  char *inst = "< Up/Down selects map, Left/Right selects character >";
  char *current_map = "Current map: ";
  char *ego_stroke0 = "Code/Design: David Lawrence";
  char *ego_stroke1 = "Art/Design: James Andrews";

  XEvent event;
  KeySym key;
  DIR *dp;
  struct dirent *current;
  int chosen = 0, found_map = 0;
  int i, txt_w0, txt_w1;
  int title_width, title_height, map_name_width, inst_width, cur_map_width;
  /* Fake entity is used to draw the characters on the screen */
  entity fake;
#define NUM_CHARS 3
  const int character[NUM_CHARS] = { TUX,
				     GOWN,
				     BSD };

  clear_area(buffer,0,0, VIEW_W, VIEW_H, BlackPixel(display, screen));
  clear_area(status_buf,0,0, WIN_W, STATUS_HEIGHT,BlackPixel(display, screen));

  blit( buffer, win, 0, 0, VIEW_W, VIEW_H);
  blit( status_buf, win, 0, VIEW_H, VIEW_W, STATUS_HEIGHT);
  status_dirty = 1;

  title_width = XTextWidth( bold_font_24, title, strlen(title));
  title_height = bold_font_24->ascent + bold_font_24->descent;
  inst_width = XTextWidth( bold_font_14, inst, strlen(inst));
  cur_map_width = XTextWidth( bold_font_14, current_map, strlen(current_map));
  txt_w0 = XTextWidth( bold_font_14, ego_stroke0, strlen(ego_stroke0));
  txt_w1 = XTextWidth( bold_font_14, ego_stroke1, strlen(ego_stroke1));

  fake.mode = ALIVE;
  fake.dir = 4; /* Facing downwards */
  fake.frame = 0;
  fake.img_no = 0;

  /* Draw initial characters */
  for( i=TUX ; i<NUM_CHARS ; i++ ) {
    fake.type = character[i];
    draw_entity( &fake, buffer, (2*i+1)*(VIEW_W/(2*NUM_CHARS + 1)), VIEW_H/2);
  }

  if( (dp = opendir("maps")) == NULL) {
    fprintf(stderr, "can't open %s\n", "maps");
    die("Forced Exit!\n", 1);
  }

  i = TUX;
  while( !chosen ) {
    while( !found_map ) {

      if( (current = readdir(dp)) == NULL ) {
	rewinddir(dp);
	continue;
      }
      
      /* We're only interested in .map files */
      if( strcmp(current->d_name+(strlen(current->d_name) - 4), ".map") )
	continue;
      
#if VERBOSE
      printf("Current selection is %s\n", current->d_name);
#endif      

      clear_area(buffer, 0, 4*title_height, VIEW_W, title_height,
		 BlackPixel(display, screen));
      
      /* Print the name of the map on the screen */
      map_name_width = XTextWidth( bold_font_14, current->d_name,
				   strlen(current->d_name));
      print(current_map, buffer, VIEW_W/2 - cur_map_width,
	    4*title_height, bold_font_14);
      print(current->d_name, buffer, VIEW_W/2,
	    4*title_height, bold_font_14);
      found_map = 1;
    }
    
    /* Print the name of the map on the screen */
    print(title, buffer, (VIEW_W-title_width)/2, title_height, bold_font_24);
    /* Instructions */
    print(inst, buffer, (VIEW_W-inst_width)/2, VIEW_H - 4*title_height,
	  bold_font_14);
    /* Credits */
    print(ego_stroke0, buffer, (VIEW_W-txt_w0)/2, VIEW_H - 2*title_height,
	  bold_font_14);
    print(ego_stroke1, buffer, (VIEW_W-txt_w1)/2, VIEW_H - title_height,
	  bold_font_14);

    XFlush(display);
    while( XPending(display) ) {
      XNextEvent(display, &event);
      if( event.type == KeyPress ) {
	key = XLookupKeysym(&event.xkey, 0);
	switch( key ) {
	  /* Easter eggs, you can play as a bunny or a GNU */
	case XK_G:
	case XK_g:
	  *type = GNU;
	  strcpy( map_name, current->d_name );
	  return 1;
	case XK_B:
	case XK_b:
	  *type = BUNNY;
	  strcpy( map_name, current->d_name );
	  return 1;
	case XK_Escape:
	case XK_Q:
	case XK_q:
	  die("Quitting\n", 0);
	  break;
	case XK_Return:
	case XK_space:
	  chosen = 1;
#if USESOUND
	  play_sound(S_AMMO_PICKUP);
#endif
	  break;
	  /* Up & Down controls map */
	case XK_Up:
	case XK_KP_Up:
	case XK_k:
	case XK_Down:
	case XK_KP_Down:
	case XK_j:
	  found_map = 0; /* Search for next map */
	  break;
	  /* Left & Right switches players */
	case XK_Left:
	case XK_KP_Left:
	case XK_h:
	  if( --i < 0 )
	    i = NUM_CHARS - 1;
	  fake.dir = 4;
	  fake.frame = 0;
	  break;
	case XK_Right:
	case XK_KP_Right:
	case XK_l:
	  if( ++i >= NUM_CHARS )
	    i = 0;
	  fake.dir = 4;
	  fake.frame = 0;
	  break;
	}
      }
    }

    clear_area(buffer, (2*i+1)*(VIEW_W/(2*NUM_CHARS + 1)), VIEW_H/2, TILE_W,
	       TILE_H, BlackPixel(display, screen));

    fake.type = character[i];
    draw_entity( &fake, buffer, (2*i+1)*(VIEW_W/(2*NUM_CHARS + 1)), VIEW_H/2);

    /* Just an option we were toying with. It looks better when they just spin
       on the spot IMHO */
#if CHARS_RUN_IN_MENU
    delay( DEFAULT_FRAME_LEN * 2 );

    if( ++fake.frame > 2 )
      fake.frame = 0;
    if( !fake.frame )
#else
      delay( DEFAULT_FRAME_LEN );
#endif
    if( ++fake.dir > 7 )
      fake.dir = 0;
    
    update_window();

  } /* quit=1, user has chosen map & character */
  
#if VERBOSE
  printf("Final choice = %s\n", current->d_name); 
#endif

  /* Set what we needed and return */
  strcpy( map_name, current->d_name );
  *type = fake.type;
  closedir(dp);

  return 1;

} /* Select map */


/* Draws those annoying pop-up windows when you run over a message. */
void draw_text_window(char *mesg)
{

  const char *pressenter = "<Enter>";
  int font_offset;
  int font_width;
  int quit = 0;
  int st, line, end; /* start and end of mesg */
  XEvent event;
 
  font_offset = med_font_18->ascent + med_font_18->descent;
  font_width = XTextWidth( med_font_18, pressenter, strlen(pressenter));

  /* Make a papery background */
  XSetFillStyle(display, text_win_gc, FillTiled);
  XSetTile(display, text_win_gc, misc_image(PAPER_TEXTURE) );
  XFillRectangle(display, tmp_buf, text_win_gc, 0, 0, TWINW, TWINH);

  XSetFillStyle(display, text_win_gc, FillSolid);
  XSetForeground(display, text_win_gc, BlackPixel(display, screen));
  XSetFont(display, text_win_gc, med_font_18->fid);

  /* Deal with newlines. X probably does this for me, but I don't know how! */
  st = line = end = 0;

  do {

    line++;

    while( *(mesg+end) != '\n' && *(mesg+end) != '/' && *(mesg+end) != '\0')
      end++;

    XDrawString(display, tmp_buf, text_win_gc, 0, font_offset*line,
		(mesg+st), end-st);
    st = end++;

  } while( *(mesg+st++) ); /* Don't wanna draw \n next line. */
  
  /* Draw "enter" at bottom of window */
  XDrawString(display, tmp_buf, text_win_gc, (TWINW-font_width)/2,
	      TWINH-font_offset/2, pressenter, strlen(pressenter));

#if USESOUND
      play_sound(S_MSG_UP);
#endif
#if VERBOSE
  printf("Mapping text window\n");
#endif
  XMapRaised(display, text_win);

  /* Wait for enter key to be pressed */
  do {

    text_win_up = 1;
    update_window();
    delay( DEFAULT_FRAME_LEN );

    XFlush(display);
    if( XPending(display) ) {
      XNextEvent(display, &event);
      if( event.type == KeyPress )
	if( XLookupKeysym(&event.xkey, 0) == XK_Return )
	  quit = 1;
    }

  } while( !quit );

  /* Reset players movement speed */
  trigger = 0;
  player->x_v = 0;
  player->y_v = 0;

  text_win_up = 0;
  XUnmapWindow(display, text_win);
  /* map_dirty = 1; */

} 


/* Draws the map onto buffer, it only calls draw_map to update map_buf when
   it needs to, which tends to be around 20% of the time. Draw_map is pretty
   darn processor intensive (uses lots of small blits) */
void update_screen(struct map_t *map, int x_off, int y_off)
{

  static int old_x_off, old_y_off;

  if( !map_dirty )
    if( old_x_off/TILE_W != x_off/TILE_W || old_y_off/TILE_H != y_off/TILE_H )
      map_dirty = 1;

  if( map_dirty ) {
    old_x_off = x_off;
    old_y_off = y_off;
    draw_map( map, x_off, y_off );
    map_dirty = 0;
  }

  offset_blit(map_buf, buffer, 0, 0, VIEW_W , VIEW_H,
	      x_off? -x_off%TILE_W  : 0, y_off? -y_off%TILE_H : 0);

}


/* Draw all entities that are visible to the player (ie inside the view window)
   onto the buffer */
void draw_entities(int x_off, int y_off)
{

  int x,y;
  entity *ent;

  ent = player;

  do {

    /* Only draw entities on screen */
    if( ent->x+TILE_W  >= x_off && ent->x < x_off + VIEW_W )
      if( ent->y+TILE_H >= y_off && ent->y < y_off + VIEW_H ) {
	/* Translate from global posn to screen position */
	x = ent->x - x_off;
	y = ent->y - y_off;
	
	draw_entity(ent, buffer, x, y);

      }

    ent = ent->next;
    
  } while( ent != NULL );

}

/* Updates the actual window on the desktop. Usually t wll just copy the buffer
   straight onto the window, but if the status bar is obscured, or the text
   window is ment to be viewed, it will draw those too */
void update_window(void)
{

#define MAGIC -22 /* I don't know where the heck this comes from!!! */
		  
  blit(buffer, win, 0, 0, VIEW_W, VIEW_H);

  if( status_dirty ) {
    offset_blit(status_buf, win, 0, VIEW_H, WIN_W, STATUS_HEIGHT , 0, MAGIC);
    status_dirty = 0;
  }

  if( text_win_up ) {
     XSetFillStyle(display, text_win_gc, FillTiled);
     XSetTile(display, text_win_gc, tmp_buf);
     XFillRectangle(display, text_win, text_win_gc, 0, 0, TWINW, TWINH);
  }

}

/* Draws a text string. Bloody X draws your string so that the bottom left is
   at (x,y). This draws it from the top right. */
void print(char *msg, Drawable dest, int x, int y, XFontStruct *font)
{

  int font_offset = font->ascent + font->descent;
  y += font_offset;

  XSetFont(display, text_gc, font->fid);
  XDrawString(display, dest, text_gc, x, y, msg, strlen(msg));
  
}


/* Clears an area of a DRAWABLE. Not to be confused with XClearArea, which
   clears a window */
void clear_area(Drawable drwbl, int x, int y, int width, int height, int color)
{

  XSetForeground(display, solid_gc, color);
  XFillRectangle(display, drwbl, solid_gc, x, y, width, height);
  
}

/* Copies drawable src onto dest at position x,y with length width, height */
void blit(Drawable src, Drawable dest, int x, int y, int width, int height)
{

  XGCValues xgcv;

  xgcv.ts_x_origin = x;
  xgcv.ts_y_origin = y;
  XChangeGC(display, gc, GCTileStipXOrigin | GCTileStipYOrigin, &xgcv);
  XSetTile(display, gc, src);
  XFillRectangle(display, dest, gc, x, y, width, height);

}

/* Like blit, but drawn with transparency ON. This is slower than straight
   copying, so we don't do this by default */
void trans_blit(Drawable src, Drawable dest, int x, int y, int width,
		int height, Drawable clip_mask)
{

  XGCValues xgcv;

  xgcv.ts_x_origin = x;
  xgcv.ts_y_origin = y;
  xgcv.clip_x_origin = x;
  xgcv.clip_y_origin = y;
  XChangeGC(display, masked_gc, GCClipXOrigin | GCClipYOrigin |
	    GCTileStipXOrigin | GCTileStipYOrigin, &xgcv);

  XSetClipMask(display, masked_gc, clip_mask);

  XSetTile(display, masked_gc, src);
  XFillRectangle(display, dest, masked_gc, x, y, width, height);

}


/* Draws src onto dest but offset by x_o, y_o */
void offset_blit(Drawable src, Drawable dest, int x, int y,
		 int width, int height,  int x_o, int y_o)
{
 
  XGCValues xgcv;

  xgcv.ts_x_origin = x_o;
  xgcv.ts_y_origin = y_o;

  XChangeGC(display, offset_gc, GCTileStipXOrigin | GCTileStipYOrigin,
	    &xgcv);

  XSetTile(display, offset_gc, src);
  XFillRectangle(display, dest, offset_gc, x, y, width, height);

}


/* Draws the status bar at the bottom of the screen */
void draw_status_bar(struct map_t *map, int weap_type)
{

  char tmp_str[MAP_STR_LEN];
  int font_width;
  Pixmap weap_img;

  clear_area(status_buf,0,0, WIN_W, STATUS_HEIGHT,BlackPixel(display, screen));

  /* Caffeine image */
  blit(misc_image(CAFFEINE_BAR), status_buf, 0, 0, BAR_WIDTH, BAR_HEIGHT);

  switch( weap_type ) {
  case WEAP_CD:
    weap_img = misc_image(CD_PIC);
    break;
  case WEAP_FIREBALL:
    weap_img = misc_image(FIRE_PIC);
    break;
  case WEAP_NUKE:
    weap_img = misc_image(RADIOACTIVE_PIC);
    break;
  default:
    fprintf(stderr, "draw_status_bar: No image for type %d!!!\n", weap_type);
    die("Exiting\n", 1);
  }

  offset_blit( weap_img, status_buf, WEAP_IMG_X, 0, 48, 48, 4, 0);

  sprintf(tmp_str, "Map: %s", map->name); 
  font_width = XTextWidth( STATUS_FONT, tmp_str, strlen(map->name) + 5);
  print(tmp_str, status_buf, VIEW_W - font_width, BAR_Y, med_font_18);

}


/* Draw's energy bar at the bottom of the screen */
void update_caff_bar(int health)
{

  /* Clear old energy bar. */
  clear_area(status_buf, BAR_X+BAR_XOFF, BAR_Y+BAR_YOFF,
	     CAFF_BAR_WIDTH, CAFF_BAR_HEIGHT, BlackPixel(display, screen));
  
  /* Draw new one */
  clear_area(status_buf, BAR_X+BAR_XOFF, BAR_Y+BAR_YOFF,
	     health, CAFF_BAR_HEIGHT, BAR_COLOR);

}

/* Flashes screen black if c == 0, otherwise flash white */
void flash_color(int c)
{

  clear_area(buffer, 0, 0, VIEW_W, VIEW_H, c? WhitePixel(display, screen) :
	     BlackPixel(display, screen));

}


void draw_map(struct map_t *map, int x_off, int y_off)
{
  
  int x, y, x_inc, y_inc;

  /* Don't buffer outside of the map. */
  if( x_off + VIEW_W >= map->width * TILE_W )
    x_inc = 0;
  else x_inc = 1;

  if( y_off + VIEW_H >= map->height * TILE_H )
    y_inc = 0;
  else y_inc = 1;

  /* Draw the map. */
  for( y=y_off/TILE_H ; y<y_off/TILE_H+VIEW_H/TILE_H + y_inc ; y++ ) {
    for( x=x_off/TILE_W ; x < x_off/TILE_W + VIEW_W/TILE_W + x_inc ; x++) {

      /* Draw base layer. */
      if( *(map->base + y*map->width + x) == T_BLANK )
	clear_area( map_buf, (x-x_off/TILE_W)*TILE_W, (y-y_off/TILE_H)*TILE_H,
		    TILE_W, TILE_H, BlackPixel(display, screen) );
      else if( *(map->base + y*map->width + x) == T_WHITE )
	clear_area( map_buf, (x-x_off/TILE_W)*TILE_W, (y-y_off/TILE_H)*TILE_H,
		    TILE_W, TILE_H, WhitePixel(display, screen) );
      else
	blit( tile_pixmap(map, BASE, 0,x, y), map_buf,(x-x_off/TILE_W)*TILE_W,
	      (y-y_off/TILE_H)*TILE_H, TILE_W, TILE_H);

      /* Draw Object layer */
      /* Only draw if not O_NULL */
      if( *(map->object + y*map->width + x) ) {
	trans_blit( tile_pixmap( map, OBJECT, 0, x, y), map_buf,
		    (x-x_off/TILE_W)*TILE_W, (y-y_off/TILE_H)*TILE_H,
		    TILE_W, TILE_H, tile_pixmap(map,OBJECT,1,x,y));
      }

    }
  }

}


void die(char *str, int exit_status)
{

  struct utsname sname;

  remove_all_entities();
  free(player);
  free(root);
  fflush(NULL);

  /* If an error, print out system info */
  if( exit_status ) {
    uname(&sname);
    fprintf(stderr,
	    "An error occured, please report the following to\n"
	    "philaw@camtech.net.au (David Lawrence)\n"
	    "----------------------\n"
	    "       OS: %9s\n"
	    "  Release: %9s\n"
	    "  Architecture: %s\n"
	    "  Display depth: %d\n"
	    "  Error: %s\n"
	    "  Exit status: %d\n"
	    "----------------------\n\n",
	    sname.sysname, sname.release, sname.machine,
	    DefaultDepth(display, screen), str, exit_status); 

  } else
    fprintf(stderr, "%s", str); /* Print error message */

  XCloseDisplay(display);
  exit(exit_status);

}
