/* quick command to do a sanity check on a QUETZAL file
 *
 * 28/04/97
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#define GOT_HEADER	0x01
#define GOT_MEMORY	0x02
#define GOT_STACKS	0x04
#define GOT_CMEM	0x10
#define GOT_UMEM	0x20
#define GOT_ALL		0x07

#ifndef EXIT_SUCCESS
#define EXIT_SUCCESS 0
#define EXIT_FAILURE 1
#endif

unsigned char read_byte (FILE *fp)
    {
    int c;
    if ((c = getc (fp)) == EOF)
        {
        printf ("*** Premature EOF.\n");
        exit (EXIT_FAILURE);
	}
    return (unsigned char) c;
    }

int main (int argc, char **argv)
    {
    FILE *fp = stdin;
    int i;
    unsigned char status = 0;
    char id[5];
    unsigned short errors = 0;
    unsigned long filelen, cklen;

    if (argc == 2 && strcmp (argv[1],"-"))
        {
        if ((fp = fopen (argv[1],"rb")) == NULL)
            {
            perror(argv[1]);
            exit(EXIT_FAILURE);
            }
        }
    else if (isatty(0))
        {
        fprintf (stderr,"usage: %s [file]\n\n",argv[0]);
        fprintf (stderr,"Checks a QUETZAL/IFZS file for conformance to the standard.\n");
        fprintf (stderr,"This does not do in-depth checking, but makes sure all chunk lengths are\n");
        fprintf (stderr,"correct, and ensures all necessary chunks appear and in the correct order.\n");
        exit (EXIT_FAILURE);
	}

    /* print details of FORM header */
    if (getc (fp) != (int)'F' || getc (fp) != (int)'O' ||
        getc (fp) != (int)'R' || getc (fp) != (int)'M')
        {
        printf ("*** No FORM header: not an IFZS file.\n");
        exit (EXIT_FAILURE);
        }
    for (filelen=0, i=0; i<4; ++i)
        filelen = (filelen << 8) | read_byte(fp);
    if (getc (fp) != (int)'I' || getc (fp) != (int)'F' ||
        getc (fp) != (int)'Z' || getc (fp) != (int)'S')
        {
        printf ("*** No IFZS type: not an IFZS file.\n");
        exit (EXIT_FAILURE);
        }
    fprintf (stdout,"FORM %lu IFZS\n",filelen);
    filelen -= 4;
    if (filelen & 1)
        {
        printf ("*** FORM length is even in all legal files\n");
        ++errors;
	}

    /* loop while there's still something left in the IFZS chunk */
    while (filelen > 0)
        {
        /* check there's enough of the IFZS chunk left */
        if (filelen < 8)
            {
            printf ("*** IFZS chunk too short to contain another chunk (%lu bytes left)\n",filelen);
            exit (EXIT_FAILURE);
	    }

        /* read this chunk's details */
        for (i=0; i<4; ++i)
            id[i] = read_byte (fp);
        for (i=0; i<4; ++i)
            if (id[i] < 0x20 || id[i] > 0x7E)
                {
                printf ("*** Illegal chunk ID: 0x%02X%02X%02X%02X\n",
                    (unsigned short)id[0], (unsigned short)id[1],
                    (unsigned short)id[2], (unsigned short)id[3]);
                exit (EXIT_FAILURE);
                }
        for (cklen=0, i=0; i<4; ++i)
            cklen = (cklen<<8) | read_byte(fp);
        id[4] = (unsigned char)0;
        printf ("  %s %6lu",id,cklen);

        /* if it's a known chunk id, print more information */
        if (!strncmp (id,"IFhd",4))
            {
            printf (" (QUETZAL header)\n");
            if (status & GOT_HEADER)
                {
                printf ("*** Only one IFhd chunk is allowed.\n");
                ++errors;
                }
            if (status & (GOT_STACKS | GOT_MEMORY))
                {
                printf ("*** warning: IFhd must come before CMem, UMem, or Stks.");
                ++errors;
                }
            status |= GOT_HEADER;
            }
        else if (!strncmp(id,"CMem",4))
            {
            printf (" (compressed memory)\n");
            if (status & GOT_CMEM)
                {
                printf ("*** warning: only one CMem chunk is allowed.\n");
                ++errors;
                }
            status |= GOT_MEMORY | GOT_CMEM;
            }
        else if (!strncmp (id,"UMem",4))
            {
            printf (" (uncompressed memory)\n");
            if (status & GOT_UMEM)
                {
                printf ("*** warning: only one UMem chunk is allowed.\n");
                ++errors;
                }
            status |= GOT_MEMORY | GOT_UMEM;
            }
        else if (!strncmp (id,"Stks",4))
            {
            printf (" (stacks)\n");
            if (status & GOT_STACKS)
                {
                printf ("*** warning: only one Stks chunk is allowed.\n");
                ++errors;
                }
            status |= GOT_STACKS;
            }
        else if (!strncmp(id,"IntD",4))
            printf (" (interpreter-dependent)\n");
        else if (!strncmp(id,"ANNO",4))
            printf (" (annotation)\n");
        else if (!strncmp(id,"AUTH",4))
            printf (" (author)\n");
        else if (!strncmp(id,"NAME",4))
            printf (" (name of content)\n");
        else if (!strncmp(id,"(c) ",4))
            printf (" (copyright on content)\n");
        else if (!strncmp(id,"    ",4))
            printf (" (filler)\n");
        else
            {
            printf ("\n*** Unknown chunk type found.\n");
            ++errors;
            }
        filelen -= 8;

        if (cklen > filelen)
            {
            printf ("*** Chunk extends past end of IFZS chunk (%lu left in IFZS)\n",filelen);
            exit (EXIT_FAILURE);
	    }

        cklen = (cklen + 1) & ~1;
        filelen -= cklen;
        for (; cklen > 0; --cklen)
            (void) read_byte (fp);	/* skip chunk contents */
        }

    if (getc(fp) != EOF)
        {
        for (filelen=1; getc (fp) != EOF; ++filelen);
        printf ("*** Spurious data (%lu bytes) past specified chunk length\n",filelen);
        ++errors;
	}

    if (argc == 2 && strcmp (argv[1],"-"))
        fclose (fp);

    i = EXIT_SUCCESS;

    if ((status & GOT_ALL) != GOT_ALL)
        {
        printf ("\n*** Missing chunks:");
        if (!(status & 0x01)) printf (" IFhd");
        if (!(status & 0x02)) printf (" Stks");
        if (!(status & 0x04)) printf (" CMem/UMem");
        printf ("\n");
        i = EXIT_FAILURE;
        }

    if (errors>0)
        {
        printf ("\n*** %u error%s.\n",errors,(errors>1)?"s":"");
        i = EXIT_FAILURE;
	}

    exit (i);
    }
