/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 by the Free Software Foundation.               *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */




#include <assert.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>

#define BUILDING_GNUGO_ENGINE  /* bodge to access private fns and variables */
#include "liberty.h"

#include "main.h"
#include "sgf.h"
#include "ttsgf.h"
#include "interface.h"


/*PUBLIC_VARIABLE  SGFNodeP sgf_head;*/

static FILE *sgfout = NULL;



/* A move has been made : write out the potential moves and this move */

void 
sgf_move_made(int i, int j, int who, int value)
{
  int m,n;
  
  int done_label=0;
  if (!sgfout)
    return;

  for (m=0; m < board_size; ++m)
  {
    for (n=0; n<board_size; ++n)
    {
      if (potential_moves[m][n] > 0)
      {
	if (!done_label)
	{
	  fprintf(sgfout, "\nLB");
	  done_label=1;
	}
	fprintf(sgfout, "[%c%c:%d]", 'a'+n, 'a'+m, potential_moves[m][n]);
      }
    }
  }

  if (value) fprintf(sgfout, "\nC[Value of move: %d]", value);

  /* if it is a pass move */
  if ((i==j) && (i==board_size))
  {
    if (board_size>19)
      fprintf(sgfout, "\n;%c[]\n", who==WHITE ? 'W' : 'B');
    else
      fprintf(sgfout, "\n;%c[tt]\n", who==WHITE ? 'W' : 'B');
  }
  else
    fprintf(sgfout, "\n;%c[%c%c]\n", who==WHITE ? 'W' : 'B', 'a'+j, 'a'+i);

  fflush(sgfout);  /* in case cgoban terminates us without notice */
}  

/* mark dead and critical dragons in sgf file */

void 
sgf_dragon_status(int i, int j, int status)
{
  if (sgfout)
  {
    switch(status)
    {
      case DEAD:
	fprintf(sgfout, "LB[%c%c:X]\n", 'a'+j, 'a'+i);
	break;
      case CRITICAL:
	fprintf(sgfout, "LB[%c%c:!]\n", 'a'+j, 'a'+i);
	break;
    }
  }
}


/* write line to sgf file */

int 
sgf_write_line(const char * line, ...)
{
  va_list ap;
  if (!sgfout) return 0;

  va_start(ap, line);
  vfprintf(sgfout, line, ap);
  va_end(ap);

  return sgf_flush_file();
}

/* write header information to sgf file */

int 
sgf_write_game_info(int bsize, int handicap, float komi, int seed, const char *gametype)
{
   if (!sgfout) return 0;
   fprintf(sgfout, "(;GM[1]FF[4]");
   fprintf(sgfout, "RU[%s]", "Japanese");
   fprintf(sgfout, "SZ[%d]", bsize);
   fprintf(sgfout, "\n");

   fprintf(sgfout, "PW[GNU Go]PB[GNU Go %s %s]", VERSION, gametype);
   fprintf(sgfout, "HA[%d]", handicap);
   fprintf(sgfout, "KM[%.1f]", komi);
   fprintf(sgfout, "GN[GNU Go %s %s ", VERSION, gametype);
   fprintf(sgfout, "Random Seed %d", seed);
   fprintf(sgfout, "] ");
   fprintf(sgfout, "\n");
   return sgf_flush_file();
}

/* open sgf file for output */

int 
sgf_open_file(const char *sgf_filename)
{
   if (strcmp(sgf_filename, "-")==0) 
   {
     sgfout = stdout;
   } else {
     sgfout = fopen(sgf_filename, "w");
   }
   if (sgfout) signal(SIGTERM, sigterm_handler);
   if (!sgfout) return 0;
   else return 1;
}

/* flush buffered output to sgf file */

int 
sgf_flush_file()
{
   if (!sgfout) return 0;
   fflush(sgfout);
   return 1;
}

/* close sgf file for output */

int 
sgf_close_file()
{
   if (!sgfout) return 0;
   fprintf(sgfout, ")\n");
   fclose(sgfout);
   sgfout=NULL;
   return 1;
}

/* begin_sgfdump begins outputting all moves considered by
 * trymove and trysafe to an sgf file.
 */

void 
begin_sgfdump(const char *filename)
{
  sgf_open_file(filename);

  fprintf(sgfout,"(;SZ[%d];", get_boardsize() );
  sgf_printboard(-1);
  fprintf(sgfout, "\n(;");
  sgf_dump=1;
}

/* begin_sgfdump begins outputting all moves considered by
 * trymove and trysafe to an sgf file.
 */

void 
end_sgfdump()
{
  while (stackp>0)
    popgo();
  fprintf(sgfout,")\n)\n");
  fclose(sgfout);
  count_variations=0;
  sgf_dump=0;
  sgfout=NULL;
}

/* sgf_decidestring writes the variations considered in attacking
 * or defending the string at (m,n) to an sgf file.
 */

void
sgf_decidestring(int m, int n, const char *sgf_output)
{
  int i, j;

  if (sgf_output)
    begin_sgfdump(sgf_output);
  count_variations=1;

  if (attack(m, n, &i, &j)) {
    TRACE("%m can be attacked at %m (%d variations)\n", 
	  m, n, i, j, count_variations);
    count_variations=1;
    if (find_defense(m, n, &i, &j))
      TRACE("%m can be defended at %m (%d variations)\n", 
	    m, n, i, j, count_variations);
    else
      TRACE("%m cannot be defended (%d variations)\n", 
	    m, n, count_variations);
  }
  else 
    TRACE("%m cannot be attacked (%d variations)\n", 
	  m, n, count_variations);

  if (sgf_output)
    end_sgfdump();
}

void
sgf_write_comment(const char *comment)
{
   sgfAddComment(NULL,comment);
   if (!sgfout) return ;
   fprintf(sgfout,"C[%s]",comment);
}

/*adding a stone to the outputfile*/
void
sgf_set_stone(int i, int j, int who)
{
  if (sgfout)
    fprintf(sgfout, "A%c[%c%c]", who==WHITE ? 'W' : 'B', 'a'+j, 'a'+i);
}


/*
 * sgf_printboard writes the current board position to the output file
 */
void sgf_printboard(int next) 
{
  int i,j;
  int start = 0;
  if (!sgfout) return;

  for (i=0; i<board_size; i++) 
  {
    for (j=0; j< board_size; j++) 
    {
      if (p[i][j] == WHITE) 
      {
	if (!start) 
        {
	  fprintf(sgfout, "AW");
	  start = 1;
	}
	fprintf(sgfout, "[%c%c]", j+'a', i+'a');
      }
    }
  }
  fprintf(sgfout, "\n");
  start = 0;
  for (i=0; i<board_size; i++) 
  {
    for (j=0; j< board_size; j++) 
    {
      if (p[i][j] == BLACK) 
      {
	if (!start) 
        {
	  fprintf(sgfout, "AB");
	  start = 1;
	}
	fprintf(sgfout, "[%c%c]", j+'a', i+'a');
      }
    }
  }
  fprintf(sgfout, "\n");
  if (next != WHITE && next != BLACK) return;
  if (next == WHITE) 
    fprintf(sgfout, "PL[W]\n"); 
  else if (next == BLACK)
    fprintf(sgfout, "PL[B]\n"); 
  start = 0;
  for (i=0; i<board_size; i++) 
  {
    for (j=0; j< board_size; j++) 
    {
      if (p[i][j] == EMPTY && !legal(i,j,next)) 
      {
	if (!start) 
        {
	  fprintf(sgfout, "IL");
	  start = 1;
	}
	fprintf(sgfout, "[%c%c]", j+'a', i+'a');
      }
    }
  }
  fprintf(sgfout, "\n");
}

/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */
