#include "STDTypes.hh"
#include <limits.h>
#include "SavantlineType.hh"
#include "AccessVariable.hh"
#include <math.h>
#include <stdlib.h>

const UniversalInteger UNIVERSAL_TRUE(1);
const UniversalInteger UNIVERSAL_FALSE(0);

UniversalInteger
UniversalInteger::typeCast(VHDLData& toCast){
  switch(toCast.get_universal_kind()){
  case UNIVERSAL_REAL:
    return UniversalInteger((int)(rint((((UniversalReal &)toCast).val))));
    break;
  case UNIVERSAL_LONG_LONG_INTEGER:
    return UniversalInteger((int)(rint((((UniversalLongLongInteger &)toCast).val))));
    break;
  case UNIVERSAL_INTEGER:
    return UniversalInteger((int)((UniversalInteger &)toCast).val);
    break;
  default:
    cerr << toCast.get_universal_kind() 
	 << " cannot be type casted into an UniversalInteger ! " << endl;
    return UniversalInteger(0);
  }
}

UniversalReal
UniversalReal::typeCast(VHDLData& toCast){
  switch(toCast.get_universal_kind()){
  case UNIVERSAL_INTEGER:
    return UniversalReal((float)(((UniversalInteger &)toCast).val));
    break;
  case UNIVERSAL_LONG_LONG_INTEGER:
    return UniversalReal((float)(((UniversalLongLongInteger &)toCast).val));
    break;
  case UNIVERSAL_REAL:
    return UniversalReal((float)((UniversalReal &)toCast).val);
    break;
  default:
    cerr << toCast.get_universal_kind() 
	 << " cannot be type casted into an UniversalReal ! " << endl;
    return UniversalReal(0);
  }
}

UniversalLongLongInteger
UniversalLongLongInteger::typeCast(VHDLData& toCast){
  switch(toCast.get_universal_kind()){
  case UNIVERSAL_INTEGER:
    return UniversalLongLongInteger((longlongint)
				    (((UniversalInteger &)toCast).val));
    break;
  case UNIVERSAL_REAL:
    return UniversalLongLongInteger((longlongint)
				    (rint(((UniversalReal &)toCast).val)));
    break;
  case UNIVERSAL_LONG_LONG_INTEGER:
    return UniversalLongLongInteger(((UniversalLongLongInteger &)toCast).val);
    break;
  default:
    cerr << toCast.get_universal_kind() 
	 << " cannot be type casted into an UniversalLongLongInteger ! " << endl;
    return UniversalLongLongInteger(0);
  }
}

bool
savantAnd(const UniversalBoolean& lhs, const UniversalBoolean& rhs) {
  return INT_TO_BOOL(lhs.val && rhs.val);
}

bool
savantOr(const UniversalBoolean& lhs, const UniversalBoolean& rhs) {
  return INT_TO_BOOL(lhs.val || rhs.val);
}

bool
savantNand(const UniversalBoolean& lhs, const UniversalBoolean& rhs) {
  return INT_TO_BOOL(!(lhs.val && rhs.val));
}

bool
savantNor(const UniversalBoolean& lhs, const UniversalBoolean& rhs) {
  return INT_TO_BOOL(!(lhs.val ||rhs.val));
}


bool
savantXor(const UniversalBoolean& lhs, const UniversalBoolean& rhs) {
  return INT_TO_BOOL(lhs.val != rhs.val);
}

bool
savantXnor(const UniversalBoolean& lhs, const UniversalBoolean& rhs) {
  return INT_TO_BOOL(lhs.val == rhs.val);
}

bool
savantNot(const UniversalBoolean& lhs) {
  return INT_TO_BOOL(!lhs.val);
}

bool
savantEqual(const UniversalBoolean& lhs, const UniversalBoolean& rhs) {
  return INT_TO_BOOL(lhs.val == rhs.val);
}

bool
savantNotEqual(const UniversalBoolean& lhs, const UniversalBoolean& rhs) {
  return INT_TO_BOOL(lhs.val != rhs.val);
}

bool
savantLessThan(const UniversalBoolean& lhs, const UniversalBoolean& rhs) {
  return INT_TO_BOOL(lhs.val < rhs.val);
}

bool
savantLessThanOrEqual(const UniversalBoolean& lhs, const UniversalBoolean& rhs) {
  return INT_TO_BOOL(lhs.val <= rhs.val);
}

bool
savantGreaterThan(const UniversalBoolean& lhs, const UniversalBoolean& rhs) {
  return INT_TO_BOOL(lhs.val > rhs.val);
}

bool
savantGreaterThanOrEqual(const UniversalBoolean& lhs, const UniversalBoolean& rhs) {
  return INT_TO_BOOL(lhs.val >= rhs.val);
}





bool
savantEqual(const UniversalCharacter& lhs, const UniversalCharacter& rhs) {
  return INT_TO_BOOL(lhs.val == rhs.val);
}

bool
savantNotEqual(const UniversalCharacter& lhs, const UniversalCharacter& rhs) {
  return INT_TO_BOOL(lhs.val != rhs.val);
}

bool
savantLessThan(const UniversalCharacter& lhs, const UniversalCharacter& rhs) {
  return INT_TO_BOOL(lhs.val < rhs.val);
}

bool
savantLessThanOrEqual(const UniversalCharacter& lhs, const UniversalCharacter& rhs) {
  return INT_TO_BOOL(lhs.val <= rhs.val);
}

bool
savantGreaterThan(const UniversalCharacter& lhs, const UniversalCharacter& rhs) {
  return INT_TO_BOOL(lhs.val > rhs.val);
}

bool
savantGreaterThanOrEqual(const UniversalCharacter& lhs, const UniversalCharacter& rhs) {
  return INT_TO_BOOL(lhs.val >= rhs.val);
}




bool
savantEqual(const _savant_severity_level& lhs, const _savant_severity_level& rhs) {
  return INT_TO_BOOL(lhs.val == rhs.val);
}

bool
savantNotEqual(const _savant_severity_level& lhs, const _savant_severity_level& rhs) {
  return INT_TO_BOOL(lhs.val != rhs.val);
}

bool
savantLessThan(const _savant_severity_level& lhs, const _savant_severity_level& rhs) {
  return INT_TO_BOOL(lhs.val < rhs.val);
}

bool
savantLessThanOrEqual(const _savant_severity_level& lhs, const _savant_severity_level& rhs) {
  return INT_TO_BOOL(lhs.val <= rhs.val);
}

bool
savantGreaterThan(const _savant_severity_level& lhs, const _savant_severity_level& rhs) {
  return INT_TO_BOOL(lhs.val > rhs.val);
}

bool
savantGreaterThanOrEqual(const _savant_severity_level& lhs, const _savant_severity_level& rhs) {
  return INT_TO_BOOL(lhs.val >= rhs.val);
}


// Reducing the range by one so that we are able to detect when we
// overflow.
const UniversalInteger UniversalInteger::MAX(INT_MAX);
const UniversalInteger UniversalInteger::MIN(INT_MIN);

UniversalInteger::UniversalInteger(const UniversalReal& r) {
  val = int(r.val);
  overflow_flag = false;
}

UniversalInteger::UniversalInteger(const UniversalLongLongInteger& lli) {
  val = int(lli.val);
  overflow_flag = false;
}

VHDLData&
UniversalInteger::operator=(const VHDLData& d) {
  //val = (((UniversalInteger *) ((VHDLData *) &d))->val);
  val = ((UniversalInteger &) d).val;
  return ((VHDLData &) (*this));
}

UniversalInteger&
UniversalInteger::operator=(const UniversalInteger& i) {
  //val = (((UniversalInteger *) ((VHDLData *&) i))->val);
  val = i.val;
  return *this;
}

bool
UniversalInteger::operator!=(const VHDLData &d) const {
  UniversalInteger i = (UniversalInteger &) d;
  int x = (((UniversalInteger *) ((VHDLData *) &i))->val);
  
  return INT_TO_BOOL(val != x);
}

UniversalBoolean::UniversalBoolean(const VHDLData& v) {
  this->operator=(v);
}

UniversalCharacter::UniversalCharacter(const VHDLData& v) {
  this->operator=(v);
}

UniversalInteger::UniversalInteger(const VHDLData& v) {
  this->operator=(v);
}

UniversalLongLongInteger::UniversalLongLongInteger(const VHDLData& v) {
  this->operator=(v);
}

UniversalReal::UniversalReal(const VHDLData& v) {
  this->operator=(v);
}

bool
savantEqual(const UniversalInteger& lhs, const UniversalInteger& rhs) {
  return INT_TO_BOOL(lhs.val == rhs.val);
}

bool
savantNotEqual(const UniversalInteger& lhs, const UniversalInteger& rhs) {
  return INT_TO_BOOL(lhs.val != rhs.val);
}

bool
savantLessThan(const UniversalInteger& lhs, const UniversalInteger& rhs) {
  return INT_TO_BOOL(lhs.val < rhs.val);
}

bool
savantLessThanOrEqual(const UniversalInteger& lhs, const UniversalInteger& rhs) {
  return INT_TO_BOOL(lhs.val <= rhs.val);
}

bool
savantGreaterThan(const UniversalInteger& lhs, const UniversalInteger& rhs) {
  return INT_TO_BOOL(lhs.val > rhs.val);
}

bool
savantGreaterThanOrEqual(const UniversalInteger& lhs, const UniversalInteger& rhs) {
  return INT_TO_BOOL(lhs.val >= rhs.val);
}

UniversalInteger
savantUnaryPlus(const UniversalInteger& lhs) {
  return UniversalInteger(lhs.val);
}

UniversalInteger
savantUnaryMinus(const UniversalInteger& lhs) {
  return UniversalInteger(-lhs.val);
}


UniversalInteger
savantAbs(const UniversalInteger& lhs ) {
  return UniversalInteger( abs(lhs.val));
}

UniversalInteger
savantPlus(const UniversalInteger& lhs, const UniversalInteger& rhs) {
  return UniversalInteger(lhs.val + rhs.val);
}

UniversalInteger
savantMinus(const UniversalInteger& lhs, const UniversalInteger& rhs) {
  return UniversalInteger(lhs.val - rhs.val);
}

UniversalInteger
savantMultiply(const UniversalInteger& lhs, const UniversalInteger& rhs) {
  return UniversalInteger(lhs.val * rhs.val);
}
UniversalInteger
savantDivide(const UniversalInteger& lhs, const UniversalInteger& rhs) {
  return UniversalInteger(lhs.val / rhs.val);
}
// VHDL LRM Section 7.2.6: 
// The result of the modulus operation is such that (A mod B) has the sign
// of B and an absolute value less than the absolute vaue of B; in
// addition, for some integer value N, this result must satisfy the
// relation:
//     A = B*N + (A mod B)
UniversalInteger
savantMod(const UniversalInteger& lhs, const UniversalInteger& rhs) {
  int lval = lhs.val;
  int rval = rhs.val;
  return UniversalInteger((lval*rval < 0) ? (rval + lval%rval) : (lval%rval));
}
// VHDL LRM Section 7.2.6: 
// Integer division and remainder are defined by the following relation:
//     A = (A/B)*B + (A rem B)
// where (A rem B) has the sign of A and an absolute value less than the
// absolute value of B.
UniversalInteger
savantRem(const UniversalInteger& lhs, const UniversalInteger& rhs) {
  return UniversalInteger(lhs.val % rhs.val);
}

UniversalInteger
savantPow(const UniversalInteger& lhs, const UniversalInteger& rhs) {
  return UniversalInteger( (int)pow((double)lhs.val, (double)rhs.val));
}

const UniversalLongLongInteger UniversalLongLongInteger::MAX((LONG) LLONG_MAX);
const UniversalLongLongInteger UniversalLongLongInteger::MIN((LONG) LLONG_MIN);

UniversalLongLongInteger::UniversalLongLongInteger(const UniversalReal& r) {
  val = (long long int)r.val;
}

bool
savantEqual(const UniversalLongLongInteger& lhs, const UniversalLongLongInteger& rhs) {
  return INT_TO_BOOL(lhs.val == rhs.val);
}

bool
savantNotEqual(const UniversalLongLongInteger& lhs, const UniversalLongLongInteger& rhs) {
  return INT_TO_BOOL(lhs.val != rhs.val);
}

bool
savantLessThan(const UniversalLongLongInteger& lhs, const UniversalLongLongInteger& rhs) {
  return INT_TO_BOOL(lhs.val < rhs.val);
}

bool
savantLessThanOrEqual(const UniversalLongLongInteger& lhs, const UniversalLongLongInteger& rhs) {
  return INT_TO_BOOL(lhs.val <= rhs.val);
}

bool
savantGreaterThan(const UniversalLongLongInteger& lhs, const UniversalLongLongInteger& rhs) {
  return INT_TO_BOOL(lhs.val > rhs.val);
}

bool
savantGreaterThanOrEqual(const UniversalLongLongInteger& lhs, const UniversalLongLongInteger& rhs) {
  return INT_TO_BOOL(lhs.val >= rhs.val);
}

UniversalLongLongInteger
savantUnaryPlus(const UniversalLongLongInteger& lhs) {
  return UniversalLongLongInteger(lhs.val);
}

UniversalLongLongInteger
savantUnaryMinus(const UniversalLongLongInteger& lhs) {
  return UniversalLongLongInteger(-lhs.val);
}


UniversalLongLongInteger
savantAbs(const UniversalLongLongInteger& lhs ) {
  if(lhs.val < 0) {
    return UniversalLongLongInteger(-1*lhs.val);
  }
  else {
    return UniversalLongLongInteger(lhs.val);
  }
  //I don't know why the following operation returns a wrong value
  //So commenting the following.
  //return UniversalLongLongInteger( llabs(lhs.val));
}

UniversalLongLongInteger
savantPlus(const UniversalLongLongInteger& lhs, const UniversalLongLongInteger& rhs) {
  return UniversalLongLongInteger(lhs.val + rhs.val);
}

UniversalLongLongInteger
savantMinus(const UniversalLongLongInteger& lhs, const UniversalLongLongInteger& rhs) {
  return UniversalLongLongInteger(lhs.val - rhs.val);
}

UniversalLongLongInteger
savantMultiply(const UniversalLongLongInteger& lhs, const UniversalLongLongInteger& rhs) {
  return UniversalLongLongInteger(lhs.val * rhs.val);
}
UniversalLongLongInteger
savantMultiply(const UniversalLongLongInteger& lhs, const UniversalReal& rhs) {
  return UniversalLongLongInteger((LONG) (lhs.val * rhs.val));
}
UniversalLongLongInteger
savantMultiply(const UniversalLongLongInteger& lhs, const UniversalInteger& rhs) {
  return UniversalLongLongInteger((LONG) (lhs.val * rhs.val));
}

UniversalLongLongInteger
savantDivide(const UniversalLongLongInteger& lhs, const UniversalLongLongInteger& rhs) {
  return UniversalLongLongInteger(lhs.val / rhs.val);
}

UniversalLongLongInteger
savantDivide(const UniversalLongLongInteger& lhs, const UniversalInteger& rhs) {
  return UniversalLongLongInteger(lhs.val / rhs.val);
}

UniversalLongLongInteger
savantDivide(const UniversalLongLongInteger& lhs, const UniversalReal& rhs) {
  return UniversalLongLongInteger(lhs.val / rhs.val);
}

UniversalLongLongInteger
savantMod(const UniversalLongLongInteger& lhs, const UniversalLongLongInteger& rhs) {
  return UniversalLongLongInteger(lhs.val / rhs.val);
}
UniversalLongLongInteger
savantRem(const UniversalLongLongInteger& lhs, const UniversalLongLongInteger& rhs) {
  return UniversalLongLongInteger(lhs.val % rhs.val);
}

UniversalLongLongInteger
savantPow(const UniversalLongLongInteger& lhs, const UniversalLongLongInteger& rhs) {
  return UniversalLongLongInteger( (int)pow((double)lhs.val, (double)rhs.val));
}

// Here, (DBL_MAX > 1.0E38) has to hold, Required by LRM.
const UniversalReal UniversalReal::MAX(1.0E38);
const UniversalReal UniversalReal::MIN(-1.0E38);

bool
savantEqual(const UniversalReal& lhs, const UniversalReal& rhs) {
  return INT_TO_BOOL(lhs.val == rhs.val);
}

bool
savantNotEqual(const UniversalReal& lhs, const UniversalReal& rhs) {
  return INT_TO_BOOL(lhs.val != rhs.val);
}

bool
savantLessThan(const UniversalReal& lhs, const UniversalReal& rhs) {
  return INT_TO_BOOL(lhs.val < rhs.val);
}

bool
savantLessThanOrEqual(const UniversalReal& lhs, const UniversalReal& rhs) {
  return INT_TO_BOOL(lhs.val <= rhs.val);
}

bool
savantGreaterThan(const UniversalReal& lhs, const UniversalReal& rhs) {
  return INT_TO_BOOL(lhs.val > rhs.val);
}

bool
savantGreaterThanOrEqual(const UniversalReal& lhs, const UniversalReal& rhs) {
  return INT_TO_BOOL(lhs.val >= rhs.val);
}

UniversalReal
savantUnaryPlus(const UniversalReal& lhs) {
  return UniversalReal(lhs.val);
}

UniversalReal
savantUnaryMinus(const UniversalReal& lhs) {
  return UniversalReal(-lhs.val);
}


UniversalReal
savantAbs(const UniversalReal& lhs ) {
  return UniversalReal( fabs(lhs.val));
}

UniversalReal
savantPlus(const UniversalReal& lhs, const UniversalReal& rhs) {
  return UniversalReal(lhs.val + rhs.val);
}

UniversalReal
savantMinus(const UniversalReal& lhs, const UniversalReal& rhs) {
  return UniversalReal(lhs.val - rhs.val);
}

UniversalReal
savantMultiply(const UniversalReal& lhs, const UniversalReal& rhs) {
  return UniversalReal(lhs.val * rhs.val);
}
UniversalReal
savantDivide(const UniversalReal& lhs, const UniversalReal& rhs) {
  return UniversalReal(lhs.val / rhs.val);
}

UniversalReal
savantPow(const UniversalReal& lhs, const UniversalInteger& rhs) {
  return UniversalReal( pow(lhs.val, rhs.val));
}



bool
savantEqual(const _savant_file_open_kind& lhs, const _savant_file_open_kind& rhs) {
  return INT_TO_BOOL(lhs.val == rhs.val);
}

bool
savantNotEqual(const _savant_file_open_kind& lhs, const _savant_file_open_kind& rhs) {
  return INT_TO_BOOL(lhs.val != rhs.val);
}

bool
savantLessThan(const _savant_file_open_kind& lhs, const _savant_file_open_kind& rhs) {
  return INT_TO_BOOL(lhs.val < rhs.val);
}

bool
savantLessThanOrEqual(const _savant_file_open_kind& lhs, const _savant_file_open_kind& rhs) {
  return INT_TO_BOOL(lhs.val <= rhs.val);
}

bool
savantGreaterThan(const _savant_file_open_kind& lhs, const _savant_file_open_kind& rhs) {
  return INT_TO_BOOL(lhs.val > rhs.val);
}

bool
savantGreaterThanOrEqual(const _savant_file_open_kind& lhs, const _savant_file_open_kind& rhs) {
  return INT_TO_BOOL(lhs.val >= rhs.val);
}


bool
savantEqual(const _savant_file_open_status& lhs, const _savant_file_open_status& rhs) {
  return INT_TO_BOOL(lhs.val == rhs.val);
}

bool
savantNotEqual(const _savant_file_open_status& lhs, const _savant_file_open_status& rhs) {
  return INT_TO_BOOL(lhs.val != rhs.val);
}

bool
savantLessThan(const _savant_file_open_status& lhs, const _savant_file_open_status& rhs) {
  return INT_TO_BOOL(lhs.val < rhs.val);
}

bool
savantLessThanOrEqual(const _savant_file_open_status& lhs, const _savant_file_open_status& rhs) {
  return INT_TO_BOOL(lhs.val <= rhs.val);
}

bool
savantGreaterThan(const _savant_file_open_status& lhs, const _savant_file_open_status& rhs) {
  return INT_TO_BOOL(lhs.val > rhs.val);
}

bool
savantGreaterThanOrEqual(const _savant_file_open_status& lhs, const _savant_file_open_status& rhs) {
  return INT_TO_BOOL(lhs.val >= rhs.val);
}

// The following functions are necessary since bit is an enum type and all
// enums are implemented as integers within the kernel.
//The following functons are only for bit types

const UniversalInteger
savantAnd(const UniversalInteger& lhs, const UniversalInteger& rhs) {
  return(lhs.val & rhs.val);
}

const UniversalInteger
savantOr(const UniversalInteger& lhs, const UniversalInteger& rhs) {
  return(lhs.val | rhs.val);
}

const UniversalInteger
savantNand(const UniversalInteger& lhs, const UniversalInteger& rhs) {
  return(savantMinus(UniversalInteger(1),UniversalInteger(lhs.val & rhs.val)));
}

const UniversalInteger
savantNor(const UniversalInteger& lhs, const UniversalInteger& rhs) {
  return(savantMinus(UniversalInteger(1),UniversalInteger(lhs.val |rhs.val)));
}

const UniversalInteger
savantXor(const UniversalInteger& lhs, const UniversalInteger& rhs) {
  return(lhs.val != rhs.val);
}

const UniversalInteger
savantXnor(const UniversalInteger& lhs, const UniversalInteger& rhs) {
  return(lhs.val == rhs.val);
}

const UniversalInteger
savantNot(const UniversalInteger& lhs) {
  if(lhs.val == 0) {
    return UniversalInteger(1);
  } else {
    return UniversalInteger(0);
  }
}


int
UniversalBoolean::savantwrite(AccessVariable <char*> &line) const {
  ostrstream tempStr;
  char *temp_char_str;
  
  if (line.val != NULL) {
    tempStr << line.val << (int)val << FILE_DELIMITER << '\0' << ends;
  }
  else {
    tempStr << (int)val << FILE_DELIMITER << '\0' << ends;
  }
  
  if (line.val != NULL) {
    delete line.val;
  }

  temp_char_str = tempStr.str();

  line.val = new char[strlen(temp_char_str)];
  strcpy(line.val, temp_char_str);
  
  delete [] temp_char_str;
  return NORMAL_RETURN;
}

int
UniversalBoolean::savantwrite(SavantlineType &line) const {
  ostrstream tempStr;
  char * temp_char_str;

  if (line.val != NULL) {
    tempStr << line << (int)this->val << FILE_DELIMITER << '\0' << ends;
  }
  else {
    tempStr << (int)this->val << FILE_DELIMITER << '\0' << ends;
  }
  
  if (line.val != NULL) {
    delete line.val;
  }
  
  temp_char_str = tempStr.str();
  line.val = new SavantstringType(ObjectBase::VARIABLE, temp_char_str);

  delete [] temp_char_str;
  return NORMAL_RETURN;
}

int
UniversalBoolean::savantread(AccessVariable <char*> &) {
  return NORMAL_RETURN;
}

int
UniversalBoolean::savantread(SavantlineType &line) {

  char *workString = line.val->getString();
  char *input      = new char[strlen(workString) + 1];
  strstream input_stream;
  
  extractToDelimiter(workString, FILE_DELIMITER, input);
  
  int tempInt;

  input_stream << input;
  input_stream >> tempInt;
  delete [] input_stream.str();
  //sscanf(input, "%d", &tempInt);

  if (tempInt == 1) {
    this->val = true;
  }
  else {
    this->val = false;
  }
  
  delete line.val;

  if (strlen(workString) > 0) {
    line.val = new SavantstringType(ObjectBase::VARIABLE, workString);
  }
  else {
    line.val = NULL;
  }

  delete workString;
  delete input;
  
  return NORMAL_RETURN;
}

int
UniversalCharacter::savantwrite(AccessVariable <char*> &line) const {
  ostrstream tempStr;
  char *temp_char_str;
  
  if (line.val != NULL) {
    tempStr << line.val << *this << FILE_DELIMITER << '\0' << ends;
  }
  else {
    tempStr << *this << FILE_DELIMITER << '\0' << ends;
  }
  
  if (line.val != NULL) {
    delete line.val;
  }

  temp_char_str = tempStr.str();
  line.val = new char[strlen(temp_char_str)];
  strcpy(line.val, temp_char_str);
  
  delete [] temp_char_str;
  return NORMAL_RETURN;
}

int
UniversalCharacter::savantwrite(SavantlineType &line) const {
  ostrstream tempStr;
  char *temp_char_str;

  if (line.val != NULL) {
    tempStr << line << *this << FILE_DELIMITER << '\0' << ends;
  }
  else {
    tempStr << *this << FILE_DELIMITER << '\0' << ends;
  }
  
  if (line.val != NULL) {
    delete line.val;
  }
  
  temp_char_str = tempStr.str();
  line.val = new SavantstringType(ObjectBase::VARIABLE, temp_char_str);

  delete [] temp_char_str;
  return NORMAL_RETURN;
}

int
UniversalCharacter::savantread(AccessVariable <char*> &) {
  return NORMAL_RETURN;
}

int
UniversalCharacter::savantread(SavantlineType &line) {
  char *workString = line.val->getString();
  char *input      = new char[strlen(workString) + 1];
  strstream input_stream;
  
  extractToDelimiter(workString, FILE_DELIMITER, input);

  input_stream << input;
  input_stream >> this->val;
  delete [] input_stream.str();
  //sscanf(input, "%c", &(this->val));

  delete line.val;

  if (strlen(workString) > 0) {
    line.val = new SavantstringType(ObjectBase::VARIABLE, workString);
  }
  else {
    line.val = NULL;
  }

  delete workString;
  delete input;
    
  return NORMAL_RETURN;
}

int
UniversalInteger::savantwrite(AccessVariable <char*> &line) const {
  ostrstream tempStr;
  char *temp_char_str;
  
  if (line.val != NULL) {
    tempStr << line.val << *this << FILE_DELIMITER << '\0' << ends;
  }
  else {
    tempStr << *this << FILE_DELIMITER << '\0' << ends;
  }
  
  if (line.val != NULL) {
    delete line.val;
  }

  temp_char_str = tempStr.str();
  line.val = new char[strlen(temp_char_str)];
  strcpy(line.val, temp_char_str);
  
  delete [] temp_char_str;
  return NORMAL_RETURN;
}

int
UniversalInteger::savantwrite(SavantlineType &line) const {
  ostrstream tempStr;
  char *temp_char_str;

  if (line.val != NULL) {
    tempStr << line << *this << FILE_DELIMITER << '\0' << ends;
  }
  else {
    tempStr << *this << FILE_DELIMITER << '\0' << ends;
  }
  
  if (line.val != NULL) {
    delete line.val;
  }
  
  temp_char_str = tempStr.str();
  line.val = new SavantstringType(ObjectBase::VARIABLE, temp_char_str);

  delete [] temp_char_str;
  return NORMAL_RETURN;
}

int
UniversalInteger::savantread(AccessVariable <char*> &) {
  return NORMAL_RETURN;
}

int
UniversalInteger::savantread(SavantlineType &line) {
  char *workString = line.val->getString();
  char *input      = new char[strlen(workString) + 1];
  strstream input_stream;
  
  extractToDelimiter(workString, FILE_DELIMITER, input);

  input_stream << input;
  input_stream >> this->val;
  delete [] input_stream.str();
  //  sscanf(input, "%d", &(this->val));

  delete line.val;

  if (strlen(workString) > 0) {
    line.val = new SavantstringType(ObjectBase::VARIABLE, workString);
  }
  else {
    line.val = NULL;
  }

  delete [] workString;
  delete [] input;
  
  return NORMAL_RETURN;
}

int
UniversalLongLongInteger::savantwrite(AccessVariable <char*> &line) const {
  ostrstream tempStr;
  char *temp_char_str;
  
  if (line.val != NULL) {
    tempStr << line.val << *this << FILE_DELIMITER << '\0' << ends;
  }
  else {
    tempStr << *this << '\0' << ends;
  }
  
  if (line.val != NULL) {
    delete line.val;
  }

  temp_char_str = tempStr.str();
  line.val = new char[strlen(temp_char_str)];
  strcpy(line.val, temp_char_str);
  
  delete [] temp_char_str;
  return NORMAL_RETURN;
}

int
UniversalLongLongInteger::savantwrite(SavantlineType &line) const {
  ostrstream tempStr;
  char *temp_char_str;

  if (line.val != NULL) {
    tempStr << line << *this << FILE_DELIMITER << '\0' << ends;
  }
  else {
    tempStr << *this << FILE_DELIMITER << '\0' << ends;
  }
      
  if (line.val != NULL) {
    delete line.val;
  }
  
  temp_char_str = tempStr.str();
  line.val = new SavantstringType(ObjectBase::VARIABLE, temp_char_str);

  delete [] temp_char_str;
  return NORMAL_RETURN;
}

int
UniversalLongLongInteger::savantread(AccessVariable <char*> &) {
  return NORMAL_RETURN;
}

int
UniversalLongLongInteger::savantread(SavantlineType &line) {
  char *workString = line.val->getString();
  char *input      = new char[strlen(workString) + 1];
  strstream input_stream;
  
  extractToDelimiter(workString, FILE_DELIMITER, input);

  input_stream << input ;
  input_stream >> this->val;
  delete [] input_stream.str();
  //sscanf(input, "%lld", &(this->val));

  delete line.val;

  if (strlen(workString) > 0) {
    line.val = new SavantstringType(ObjectBase::VARIABLE, workString);
  }
  else {
    line.val = NULL;
  }

  delete [] workString;
  delete [] input;
    
  return NORMAL_RETURN;
}

int
UniversalReal::savantwrite(AccessVariable <char*> &line) const {
  ostrstream tempStr;
  char *temp_char_str;
  
  if (line.val != NULL) {
    tempStr << line.val << FILE_DELIMITER << *this << '\0' << ends;
  }
  else {
    tempStr << *this << '\0' << ends;
  }
  
  if (line.val != NULL) {
    delete line.val;
  }

  temp_char_str = tempStr.str();
  line.val = new char[strlen(temp_char_str)];
  strcpy(line.val, temp_char_str);
  
  delete [] temp_char_str;
  return NORMAL_RETURN;
}

int
UniversalReal::savantwrite(SavantlineType &line) const {
  ostrstream tempStr;
  char *temp_char_str;
  
  tempStr << line << *this << FILE_DELIMITER << '\0' << ends;

  if (line.val != NULL) {
    delete line.val;
  }
  
  temp_char_str = tempStr.str();
  line.val = new SavantstringType(ObjectBase::VARIABLE, temp_char_str);

  delete [] temp_char_str;
  return NORMAL_RETURN;
}

int
UniversalReal::savantread(AccessVariable <char*> &) {
  return NORMAL_RETURN;
}

int
UniversalReal::savantread(SavantlineType &line) {
  char *workString = line.val->getString();
  char *input      = new char[strlen(workString) + 1];
  strstream input_stream;
  
  extractToDelimiter(workString, FILE_DELIMITER, input);

  input_stream << input;
  input_stream >> this->val;
  delete [] input_stream.str();
  //sscanf(input, "%le", &(this->val));

  delete line.val;

  if (strlen(workString) > 0) {
    line.val = new SavantstringType(ObjectBase::VARIABLE, workString);
  }
  else {
    line.val = NULL;
  }

  delete [] workString;
  delete [] input;
    
  return NORMAL_RETURN;
}

void extractToDelimiter(char *source, int delimiter, char *destination)
{
  int  counter = 0, destinationPos = 0;
  char found = 0;
  char inQuotes = 0;
  
  while ((!found) && (*(source + counter) != '\0')) {
    if (inQuotes) {
      // We are currently searching withing a string. Wait for another quote
      if (*(source + counter) == '\\') {
	// Ignore this and use the next character as the actual one
	counter++;
	*(destination + destinationPos++) = *(source + counter);
      }
      else {
	if (*(source + counter) == '\"') {
	  inQuotes = 0;
	}
      }
    }
    else {
      if (*(source + counter) == (char) delimiter) {
	found = 1;
      }
      else {
	if (*(source + counter) == '\"') {
	  // Okay. Opening of a quote here
	  inQuotes = 1;
	}
	else {
	  *(destination + destinationPos++) = *(source + counter);
	}
      }
    }
    counter++;
  }

  // Now we should have got something. Finish the destination string
  // and delete the stuff from the source.

  strcpy(source, (source + counter));
  *(destination + destinationPos) = '\0';
}
