//---------------------------------------------------------------------------
// Copyright (c) 1995-1999 Ohio Board of Regents and the University of
// Cincinnati.  All Rights Reserved.

// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.


//
// $Id: MarkedQueue.cc,v 1.24 1999/10/16 17:33:20 dmadhava Exp $
//
//---------------------------------------------------------------------------
#include "MarkedQueue.hh"
#include <new.h>

extern "C" {
  int memcmp(const void *, const void *, size_t);
}

void MarkedQueue::insert(SigEvent* newevent) {
  MarkedEvent *newMarkedEvent = (MarkedEvent *) new char[sizeof(MarkedEvent)];
  new(newMarkedEvent) MarkedEvent(newevent);
  SortedList<MarkedEvent>::insert(newMarkedEvent);
  mark(newMarkedEvent);
}


void MarkedQueue::mark(MarkedEvent* newevent) {
  MarkedEvent *prevptr = NULL;
  MarkedEvent *ptr = NULL;
  const VTime lvt = newevent->event->sendTime;

#ifdef VHDLDBG
  cout << "--------------------------------------------------" << endl;

  cout << "Inserting: " << newevent << " " << *newevent->event << endl;
#endif

  // At this point all transactions in the list are unmarked, since they
  // are unmarked at "new", and all the old transactions are unmarked at
  // the end of this function (by the last call to this function).

  setCurrentToInsert();

  // perform 1st bit of marking on all signals, either INERTIAL or TRANSPORT
  for (ptr = seek(1, CURRENT); ptr != NULL; ptr = seek(1, CURRENT)) {
    // find messages sent in past and received in future of newevent
    // this is marking step 1.1
    // step 1.2 was the previous inserting of newevent into the markedQ
    if ((ptr->event->sigId == newevent->event->sigId) &&
	(ptr->event->recvTime >= newevent->event->recvTime)) {
      // Container<MarkedEvent> *savedPos = currentPos;
      twHandle->sendNegEvent((BasicEvent*)(ptr->event));
    }
  }

  // Check if the signal assignment has inertial delay.  Signals with
  // inertial delay have reject time > ZERO.
  if (newevent->event->rejTime == ZERO) {
#ifdef VHDLDBG
    cout << "Marked Queue is: " << endl;
    print(cout);
    cout << "--------------------------------------------------" << endl;
#endif
    return;
  }
  // only perform the rest of this on INERTIAL delays

  // Marking step 2.1
  newevent->mark();

  for (ptr = seek(0, END);
       ((ptr != NULL) &&  (ptr->event->recvTime.time > lvt.time));
       ptr = seek(-1, CURRENT)) {
    if ((ptr->event->sigId == newevent->event->sigId) &&
	(ptr->event->recvTime.time < 
	 (newevent->event->recvTime.time - newevent->event->rejTime.time))) {
      ptr->mark();
    }
  }    
    
  prevptr = ptr = seek(0, END);
  while((prevptr != NULL) &&
	(prevptr->event->sigId != newevent->event->sigId)) {
    prevptr = seek(-1, CURRENT);
  }
  for (ptr = seek(-1, END);
       ((ptr != NULL) && (ptr->event->recvTime.time > lvt.time)); 
       ptr = seek(-1, CURRENT)) {
    if (ptr->event->sigId == newevent->event->sigId &&
	!ptr->isMarked() && prevptr->isMarked() &&
	0 == memcmp((ptr->event + 1), (prevptr->event + 1),
		    prevptr->event->size - sizeof(SigEvent))) {
      ptr->mark();
    }
    if(ptr->event->sigId == newevent->event->sigId) {
      prevptr = ptr;
    }
  }    

  // locate the current driving event for this signal--needed for marking,
  // step 2.4.  It will be the latest event with a recvtime <= now, which
  // is the sendtime of the current message we're marking off of.
  for (ptr = seek(0, END); (ptr != NULL); ptr = seek(-1, CURRENT)) {
    if ((ptr->event->sigId == newevent->event->sigId) &&
	(ptr->event->recvTime.time <= lvt.time)) {
      // We have located the transaction that determines the current value
      // of the driver.  Mark this: Rule 2.4
      
      ptr->mark();
      break;
    }
  }
  
  // Cancel the transactions that are not marked.
  for (ptr = seek(0, END); (ptr != NULL); ptr = seek(-1, CURRENT)) {
    if ((ptr->event->sigId == newevent->event->sigId) &&
	(ptr->event->recvTime.time > lvt.time) && 
	(ptr->isMarked() == false)) {
      twHandle->sendNegEvent((BasicEvent*)(ptr->event));
    } else {
      // Unmark all the marked transactions, in preparation for the next
      // call to mark.
      ptr->marked = false;
    }
  }
  
#ifdef VHDLDBG
  cout << "Marked Queue is: " << endl;
  print(cout);
  cout << "--------------------------------------------------" << endl;
#endif
}


SigEvent* 
MarkedQueue::find(VTime findTime, findMode_t mode  ) {
  static SigEvent sigEventkey;
  static MarkedEvent key(&sigEventkey);
  MarkedEvent* findEvent;

  key.setsigevent(&sigEventkey);
  key.event->recvTime = findTime;
  findEvent = SortedList<MarkedEvent>::find(&key, mode);

  // since SigEvent *event in MarkedEvent is always allocated using
  // new char[..]. it was being deleted as a delete [] char. but in this case
  // the sigEventkey is a static instance and not a physically newed one
  // so setting the SigEvent *event pointer in MarkedEvent to NULL , so
  // that it is not being deleted there in MarkedEvent when the destructor
  // for static MarkedEvent key is being called.
  key.setsigevent((SigEvent *)NULL);
  return(findEvent->event);
}


void
MarkedQueue::gcollect(VTime gtime) {
  Container<MarkedEvent> *saveCurrent;
  MarkedEvent *delptr;

  saveCurrent = getCurrent();
  delptr = seek(0, START);
  
  while (delptr != NULL)  {
    if (delptr->event->sendTime <= gtime) {
      delptr = removeCurrent(); // this advances currentPos and deletes container
      ((MarkedEvent *) delptr)->~MarkedEvent();
      delete [] (char*)delptr; // delete actual object
      delptr = get();  // get new element at currentPos
    }
    else {
      delptr = seek(1, CURRENT);
    }
  }
  
  setCurrent(saveCurrent);
}


void
MarkedQueue::rollbackTo(VTime& rollbackTime) {
  MarkedEvent *delptr;


  delptr = seek(0, START);
  while (delptr != NULL) {
    if (delptr->event->sendTime > rollbackTime) {
#ifdef VHDLDBG
      cout << "markedQ::rollbackTo() -- Event deleted " << delptr << endl;
#endif
      delptr = removeCurrent(); // advances currentPos and deletes container
      ((MarkedEvent *)delptr)->~MarkedEvent();
      delete [] (char*)delptr;
    }
    else {
#ifdef VHDLDBG
      cout << "markedQ::rollbackTo -- Event not deleting " << delptr << endl;
#endif
      delptr = seek(1, CURRENT);
    }
    delptr = get();  // get new element at currentPos
  }
}
