(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Modified from m3totex by Michel Dagenais 15 June 1995                     *)

(* The m3tosgml program optionally receives an input Modula-3 source code
   file name, and an output file name as command line argument. The default
   is to filter from standard input to standard output. The input file is
   formatted according to the rules below and produces SGML markup tags
   accordingly. Several document types may be supported although only
   HTML is currently implemented.

   All lines up to and including the first blank line
   are skipped.  (This makes it easy to omit the copyright notice
   from the typeset version, for example.) The first blank line
   must not be within a comment.

   The portions of the file that are not in comments will be typeset
   verbatim, in typewriter font, obeying line breaks and indentation.
   Any characters that have special significance to HTML will be quoted.

   The portions of the file that are in comments will be treated
   differently depending on the position of the comment.  There are 
   three cases:

   Comments that do not begin in the first column are typeset in a
   fixed-width font obeying line breaks and indentation, just like 
   program text. In the following example, OUT will be processed
   like the rest of the line:

|    PROCEDURE P(VAR (*OUT*) x: INTEGER);

   Comments that begin in the first column and are not preceded
   by a blank line are indented and typeset in slanted roman text,
   except that double-quoted expressions are typeset in slanted
   typewriter font.  The purpose of this is to allow a brief comment
   after a procedure declaration, indicating what the procedure does.
   For example, here is the specification of "Path.LineTo":

|    PROCEDURE LineTo(path: T; READONLY p: Point.T);
|    (* Extend "path" with a linear segment that starts
|       at its current point and ends at "p". *)

   It is recommended that these slanted comments be kept brief;
   additional details can be provided in longer type 3 comments. 

   Comments that begin in the first column and are preceeded by a
   blank line are typeset in a roman font and justified.
   Items enclosed in double-quotes are set in unslanted 
   typewriter font; for example, program identifiers 
   should be double-quoted this way.  Line breaks and
   indentation have no effect in these comments.

   Sometimes you want a portion of a comment to be treated verbatim, like
   program text, for example, when including an example program or a
   table.  Therefore, any line that starts with "| " (vertical bar 
   followed by space) is treated like program text, typesetting it 
   verbatim in typewriter font.

   Comments of type 2 and 3 may contain HTML tags, for example 
   to annotate section headings or emphasize text.
   
   Lines in type-three comments that begin with "| " are treated just
   like program text with one exception: you can slip in words in roman
   font by enclosing them in back-quotes.
       
|    |   ELSIF s `was generated by` LineTo THEN

*)

MODULE Main;

IMPORT Stdio, Rd, Wr, Params, Text, Thread, OSError, Lex, 
    TextWr, FileRd, FileWr;

<*FATAL Thread.Alerted*>

TYPE
  T = OBJECT
      rd: Rd.T;
      wr: Wr.T;
      oldwr: Wr.T := NIL;
      target: TargetDtd := HTML;
      nameFound: BOOLEAN := FALSE;
    END;

  TargetDtd = RECORD
      startFile, betweenTitle, endTitle, endFile, startProgramSection,
      endProgramSection, startCode, endCode, 
      startComment1, startComment2, endComment2, 
      startDisplay, endDisplay,
      paragraph, startQuote, endQuote, startBQuote, endBQuote, startFileHeader,
      endFileHeader: TEXT;
    END;

CONST
  AlphaNumerics = SET OF CHAR{'A'..'Z', 'a'..'z'};

(* The target Dtd is configurable. A table associates each different
   context with a string containing one or more tags. The possible values
   for the attribute CLASS include skip, skiptag, startProg, display, quote,
   and backquote. They may be used to recover the original program text
   from the SGML file.  *)

  HTML = TargetDtd{
      "<!DOCTYPE HTML><HTML CLASS=\"m3tosgml.skiptag\"><HEAD CLASS=\"m3tosgml.skip\"><TITLE>",
      "</TITLE></HEAD><BODY CLASS=\"m3tosgml.skiptag\"><H1 CLASS=\"m3tosgml.skip\">",
      "</H1>",
      "</BODY></HTML>",
      "<DIV CLASS=\"m3tosgml.startProg\">",
      "</DIV>",
      "<PRE CLASS=\"m3tosgml.skiptag\"><TT CLASS=\"m3tosgml.skiptag\">",
      "</TT></PRE>",
      "<P CLASS=\"m3tosgml.skiptag\">",
      "<DIV CLASS=\"m3tosgml.comment2\"><P CLASS=\"m3tosgml.skiptag\">",
      "</P></DIV>",
      "<PRE CLASS=\"m3tosgml.display\"><TT CLASS=\"m3tosgml.skiptag\">",
      "</TT></PRE><P CLASS=\"m3tosgml.skiptag\">",
      "<P CLASS=\"m3tosgml.skiptag\">",
      "<TT CLASS=\"m3tosgml.quote\">",
      "</TT>",
      "<I CLASS=\"m3tosgml.bquote\">",
      "</I>",
      "<!--",
      "-->"};

(* Read the file header followed by alternating program and comment sections *)

PROCEDURE Trans(t: T) RAISES {Rd.Failure, Wr.Failure} =
  VAR
    type: [2..3];
  BEGIN
    TRY
      (* Dont really output anything until the interface/module name has
         been found. *)

      t.oldwr := t.wr;
      t.wr := TextWr.New();

      (* skip the file head *)
      Wr.PutText(t.wr,t.target.startFileHeader);
      AdvanceToBlankLine(t);
      Wr.PutText(t.wr,t.target.endFileHeader);

      (* alternate between program segments and comments *)
      LOOP 
        type := Prog(t); 
        Comment(type,t);
      END;
    EXCEPT
      Rd.EndOfFile =>
    END;
    Wr.PutText(t.wr,t.target.endFile);
  END Trans;

(* Put the first lines up to the first blank line in SGML comments. *)

PROCEDURE AdvanceToBlankLine(t: T) RAISES {Rd.Failure, Wr.Failure} =
  VAR blank: BOOLEAN; c: CHAR;
  BEGIN
    TRY
      blank := TRUE;
      LOOP
        c := Rd.GetChar(t.rd);
        IF c = '\n' THEN
          IF blank THEN EXIT 
          ELSE blank := TRUE;
          END;
        ELSIF c > ' ' THEN 
          blank := FALSE; 
        END;
        Wr.PutChar(t.wr,c);
      END;
      Rd.UnGetChar(t.rd);
    EXCEPT
      Rd.EndOfFile => (*skip*)
    END
  END AdvanceToBlankLine;

(* Put the code within a pre formatted HTML section. Issue a <TT>
   command to mark the begin/end of comments section. Issue a <PRE> only
   when a non empty program section is found; empty <PRE> sections are
   considered bad manner in HTML, probably more so than empty <TT>. *)

PROCEDURE Prog(t: T): [2..3] RAISES {Rd.EndOfFile, Rd.Failure, Wr.Failure} =

  (* Special processing at the beginning of non empty code sections. *)

  PROCEDURE StartProg(c: CHAR) RAISES {Rd.Failure, Wr.Failure} =
    BEGIN
      (* We do have a non empty code section *)

      progStarted := TRUE;
      Wr.PutText(t.wr, t.target.startCode);
      Wr.PutChar(t.wr,c);

      (* Process the interface/module statement when encountered.
         Until it is found, all output is retained to an in-memory text writer.
         Once the interface/module name is found, the file TITLE may be output
         followed by the accumulated content of the text writer. *)

      IF NOT t.nameFound THEN
        VAR
          t1 := Lex.Scan(t.rd,AlphaNumerics);
          t2 := Lex.Scan(t.rd,Lex.Blanks);
          t3 := Lex.Scan(t.rd,AlphaNumerics);
          interface := FALSE;
          title: TEXT;
        BEGIN
          (* The first character has been read already. Anyway it must be
             MODULE, INTERFACE or GENERIC *)

          IF Text.Equal(t1,"ENERIC") THEN
            Wr.PutText(t.wr,t1);
            Wr.PutText(t.wr,t2);
            t1 := t3;
            t2 := Lex.Scan(t.rd,Lex.Blanks);
            t3 := Lex.Scan(t.rd,AlphaNumerics);
            IF Text.Equal(t1,"INTERFACE") THEN interface := TRUE; END;
          ELSIF Text.Equal(t1,"NTERFACE") THEN 
            interface := TRUE; 
          END;

          Wr.PutText(t.wr,t1);
          Wr.PutText(t.wr,t2);
          Wr.PutText(t.wr,t3);
  
          (* Compute the TITLE *)
          IF interface THEN title := "The " & t3 & " interface";
          ELSE title := "The " & t3 & " module";
          END;

          (* Output the TITLE, which is repeated in TITLE and H1 for HTML *)
          Wr.PutText(t.oldwr,t.target.startFile);
          Wr.PutText(t.oldwr,title);
          Wr.PutText(t.oldwr,t.target.betweenTitle);
          Wr.PutText(t.oldwr,title);
          Wr.PutText(t.oldwr,t.target.endTitle);

          (* Output all the accumulated content of the text writer. *)
          Wr.PutText(t.oldwr,TextWr.ToText(t.wr));

          (* Stop diverting the output to a text writer. *)
          t.wr := t.oldwr;
          t.oldwr := NIL;
          t.nameFound := TRUE;
        END;
      END;
    END StartProg;

  VAR 
    c: CHAR;
    previousChar := ' ';
    inQuote := FALSE;
    vspace := 2; 
    hspace := 0; 
    startOfLine := TRUE;
    progStarted := FALSE;
  BEGIN
    Wr.PutText(t.wr, t.target.startProgramSection);
    TRY
      LOOP
        c := Rd.GetChar(t.rd);

        CASE c OF
        (* remember the number of consecutive white lines *)
        | '\n' => 
            INC(vspace); 
            hspace := 0;
            startOfLine := TRUE;
            Wr.PutChar(t.wr,c);

        (* compute the position of the first non blank character *)
        | ' '  => 
            INC(hspace);
            Wr.PutChar(t.wr,c);

        (* do not consider an escaped quote as an end of quoted text *)
        | '\\' =>
            IF NOT progStarted THEN StartProg(c);ELSE Wr.PutChar(t.wr, c); END;
            vspace := 0;
            startOfLine := FALSE;
            (* the backslash itself was escaped and should not be treated
               as an escape character *)
            IF previousChar # '\\' THEN previousChar := '\\';
            ELSE previousChar := ' ';
            END;

        (* characters within quotes are not examined for end of comment *)
        | '"' =>
            IF NOT progStarted THEN StartProg(c);ELSE Wr.PutChar(t.wr, c); END;
            vspace := 0;
            startOfLine := FALSE;
            IF previousChar # '\\' THEN
              inQuote := NOT inQuote;
            END;

        (* This character must be escaped in HTML *)
        | '<' =>
            IF NOT progStarted THEN StartProg('&'); Wr.PutText(t.wr,"lt;");
            ELSE Wr.PutText(t.wr,"&lt;");
            END;
            vspace := 0;
            startOfLine := FALSE;

        (* This character must be escaped in HTML *)
        | '&' =>
            IF NOT progStarted THEN 
              StartProg('&'); Wr.PutText(t.wr,"amp;");
            ELSE 
              Wr.PutText(t.wr,"&amp;");
            END;
            vspace := 0;
            startOfLine := FALSE;

        (* It may be a comment end *)
        | '(' =>
            WITH d = Rd.GetChar(t.rd) DO
              (* A type 2 or type 3 comment is starting since no blank lines
                 separate it from the last line of code. *)

              IF d = '*' AND startOfLine AND hspace = 0 AND (NOT inQuote) THEN
                EXIT;
              ELSE
                Rd.UnGetChar(t.rd);
                IF NOT progStarted THEN StartProg(c); 
                ELSE Wr.PutChar(t.wr, c);
                END;
                vspace := 0;
                startOfLine := FALSE;
              END;
            END;
        ELSE
          IF NOT progStarted THEN StartProg(c);ELSE Wr.PutChar(t.wr, c); END;
          vspace := 0;
          startOfLine := FALSE;
        END;
        (* when c is a backslash, previousChar has already been computed *)
        IF c # '\\' THEN previousChar := c; END;
      END;
    FINALLY
      IF progStarted THEN Wr.PutText(t.wr, t.target.endCode); END;
      Wr.PutText(t.wr, t.target.endProgramSection);
    END;

    (* The code segment ends. If there is no blank line, it is a type 2
       comment *)
    IF vspace < 2 THEN RETURN 2;
    ELSE RETURN 3;
    END;
  END Prog;

(* Process the different comment types. *)

PROCEDURE Comment(type: [2..3]; t: T) RAISES {Rd.EndOfFile, 
    Wr.Failure, Rd.Failure} =

  (* Wait until it is determined if the line is a display starting with "|".
     Then output the few blank characters read up to there. *)

  PROCEDURE PrintLineStart() RAISES {Wr.Failure} =
    BEGIN
      Wr.PutChar(t.wr,'\n');
      FOR i := 0 TO column - 1 DO Wr.PutChar(t.wr,' '); END;
      startOfLine := FALSE;
    END PrintLineStart;

  VAR 
    c: CHAR; 
    startOfLine := FALSE; 
    inDisplay := FALSE;
    inTag := FALSE;
    dblquoteparity := 0; 
    backquoteparity := 0;
    column := 0;
  BEGIN
    (* For type 2 comments, an indented block (NOTE) is used *)
    IF type = 2 THEN Wr.PutText(t.wr, t.target.startComment2); 
    ELSE Wr.PutText(t.wr,t.target.startComment1);
    END;

    LOOP
      c := Rd.GetChar(t.rd);

      (* end of a section of lines starting with "|" and treated as verbatim *)

      IF inDisplay AND startOfLine AND (c # '|') AND (c # ' ') THEN
        inDisplay := FALSE;
        Wr.PutText(t.wr, t.target.endDisplay);
      END;

      CASE c OF
      | '\n' =>
          IF startOfLine THEN
            PrintLineStart();
            IF NOT inTag THEN Wr.PutText(t.wr, t.target.paragraph); END;
          END;
          startOfLine := TRUE;
          column := 0;
      | ' ' =>
          IF startOfLine THEN INC(column);
          ELSE Wr.PutChar(t.wr,c);
          END;
      | '&' =>
          IF startOfLine THEN PrintLineStart(); END;
          IF NOT inDisplay THEN 
            Wr.PutChar(t.wr,c);
          (* We are in a display, this character must be escaped in HTML *)
          ELSE Wr.PutText(t.wr,"&amp;");
          END;
      | '<' =>
          IF startOfLine THEN PrintLineStart(); END;
          (* This is the beginning of an HTML tag *)
          IF NOT inDisplay THEN 
            Wr.PutChar(t.wr,c);
            inTag := TRUE;
          (* We are in a display, this character must be escaped in HTML *)
          ELSE Wr.PutText(t.wr,"&lt;");
          END;
      | '>' =>
          IF startOfLine THEN PrintLineStart(); END;
          (* This is the end of a tag *)
          IF inTag THEN 
            inTag := FALSE;
          END;
          Wr.PutChar(t.wr,c);
      | '\"' =>
          IF startOfLine THEN PrintLineStart(); END;
          (* quotes are used as a short hand for requesting a TT font.
             This is not true however within a tag or a display *)
          IF (NOT inDisplay) AND (NOT inTag) THEN
            IF dblquoteparity = 0 THEN
              Wr.PutText(t.wr, t.target.startQuote);
            ELSE
              Wr.PutText(t.wr, t.target.endQuote);
            END;
            dblquoteparity := (dblquoteparity + 1) MOD 2;
          ELSE
            Wr.PutChar(t.wr,c);
          END;
      | '`' =>
          IF startOfLine THEN PrintLineStart(); END;
          (* back quotes are used within display to get italics *)
          IF inDisplay THEN
            IF backquoteparity = 0 THEN
              Wr.PutText(t.wr, t.target.startBQuote);
            ELSE
              Wr.PutText(t.wr, t.target.endBQuote);
            END;
            backquoteparity := (backquoteparity + 1) MOD 2;
          ELSE
            Wr.PutChar(t.wr,c);
          END;
      | '|' =>
          (* | in the first column is used for pre formatted sections *)
          IF startOfLine THEN 
            PrintLineStart();
            IF NOT inDisplay THEN
              Wr.PutText(t.wr, t.target.startDisplay);
              inDisplay := TRUE;
            END;
          ELSE 
            Wr.PutText(t.wr, "|");
          END;
      | '*' =>
          IF startOfLine THEN PrintLineStart(); END;
          IF NOT inDisplay THEN
            WITH d = Rd.GetChar(t.rd) DO
              (* The comment ends *)
              IF d = ')' THEN
                IF inDisplay THEN Wr.PutText(t.wr, t.target.endDisplay); END;
                IF type = 2 THEN Wr.PutText(t.wr, t.target.endComment2); END;
                Wr.PutText(t.wr, t.target.paragraph);
                RETURN;
              ELSE
                Rd.UnGetChar(t.rd);
                Wr.PutChar(t.wr, c);
              END;
            END;
          ELSE
            Wr.PutText(t.wr,"*");
          END;
      ELSE
        IF startOfLine THEN PrintLineStart(); END;
        Wr.PutChar(t.wr, c);
      END;
    END;
  END Comment;

EXCEPTION UsageError(TEXT);

(* Arguments not starting with a - are treated as file names. *)

VAR
  currentParam := 1;

PROCEDURE ParseOptions(): BOOLEAN RAISES {UsageError} = 
  VAR
    arg: TEXT;
    file := 0;
  BEGIN
    in := Stdio.stdin;
    out := Stdio.stdout;

    IF currentParam >= Params.Count THEN RETURN FALSE; END;

    WHILE (currentParam < Params.Count AND file < 2) DO
      arg := Params.Get (currentParam); 
      INC (currentParam);
      IF Text.FindChar(arg,'-') # 0 THEN
        TRY
          IF file = 0 THEN in := FileRd.Open(arg);
          ELSIF file = 1 THEN out := FileWr.Open(arg);
          ELSE RAISE UsageError("");
          END;
        EXCEPT
        | OSError.E =>
            RAISE UsageError("Unable to open file " & arg);
        END;
        INC(file);
      ELSIF Text.Equal (arg, "-html") THEN
      ELSE RAISE UsageError("Unrecognized option " & arg);
      END;
    END;
    RETURN TRUE;
  END ParseOptions;

VAR
  in: Rd.T;
  out: Wr.T;
  firstPass := TRUE;

<*FATAL Wr.Failure*>

BEGIN
  TRY
    WHILE ParseOptions() OR firstPass DO
      firstPass := FALSE;
      Trans(NEW(T, rd := in, wr := out));
      Rd.Close(in);
      Wr.Close(out);
    END;
  EXCEPT
  | Rd.Failure => Wr.PutText(Stdio.stderr, "? can't read file\n");
  | UsageError(t) =>
      Wr.PutText(Stdio.stderr,
          t & "\n? usage: m3tosgml [-html] [infile] [outfile]\n");
  | Wr.Failure =>
      Wr.PutText(Stdio.stderr, "? can't write file\n");
  END;
END Main.


