/*
 * SXPDB.C - command processors for PDBLib extensions in SX
 *
 * Source Version: 3.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "sx.h"

#define LITERAL   1
#define NOPADDING 2

g_file
 *SX_file_list = NULL;

PFInt
 SX_file_type_hook = NULL;

HASHTAB
 *PDB_Chrt;

static FILE
 *text_fp_f0 = NULL;

static int
 SX_string_mode = LITERAL;

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_TARGET - set the data standard and alignment for the next file to be
 *            - created. If an argument is 0 or missing, the corresponding
 *            - value is not changed.
 */

static void _SX_target(data, align)
   int data, align;
   {int data_max, align_max;

    data_max  = 0;
    align_max = 0;

    if (data == 0)
       {if (REQ_STANDARD != NULL)
           while (REQ_STANDARD != PD_std_standards[data++]);}
    else
       {while (PD_std_standards[data_max++]);
        if ((data < 1) || (data >= data_max))
           SS_error("UNKNOWN DATA STANDARD - _SX_TARGET",
                    SS_mk_integer((BIGINT)data));};
        
    if (align == 0)
       {if (REQ_ALIGNMENT != NULL)
           while (REQ_ALIGNMENT != PD_std_alignments[align++]);}
    else
       {while (PD_std_alignments[align_max++]);
       if ((align < 1) || (align >= align_max))
          SS_error("UNKNOWN DATA ALIGNMENT - _SX_TARGET",
                   SS_mk_integer((BIGINT)align));};

    if (data != 0)
       REQ_STANDARD  = PD_std_standards[data - 1];
    if (align != 0)
       REQ_ALIGNMENT = PD_std_alignments[align - 1];

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_TARGET - set the data standard and alignment for subsequently created
 *           - files. If an argument is 0 or missing, the corresponding
 *           - value is not changed. The data standard and alignment are
 *           - returned as a pair. If the data standard or alignment was
 *           - never set, 0 is returned.
 */

static object *SX_target(arg)
   object *arg;
   {int data = 0;
    int align = 0;

    SS_args(arg,
            SC_INTEGER_I, &data,
            SC_INTEGER_I, &align,
            0);

    _SX_target(data, align);

    return(SS_mk_cons(SS_mk_integer((BIGINT)data), SS_mk_integer((BIGINT)align)));}

/*--------------------------------------------------------------------------*/

/*                          RAW BINARY FILE I/O                             */

/*--------------------------------------------------------------------------*/

/* SX_OPEN_RAW_FILE - open a file as a binary input port */

static object *SX_open_raw_file(argl)
   object *argl;
   {int data, align;
    char *name, *mode, *type;
    PDBfile *file;
    FILE *fp;
    SC_THREAD_ID(_t_index);

#ifdef HAVE_THREADS
    if (_PD_nthreads <= 0)
       PD_init_threads(1, NULL);
#endif

    data  = 0;
    align = 0;
    type  = NULL;
    SS_args(argl,
            SC_STRING_I, &name,
            SC_STRING_I, &mode,
            SC_STRING_I, &type,
            SC_INTEGER_I, &data,
            SC_INTEGER_I, &align,
            0);

    _SX_target(data, align);

    file = NULL;
    fp   = NULL;
    switch (setjmp(_PD_OPEN_ERR(_t_index)))
       {case ABORT :
	     io_close(fp);
	     return(SS_f);
        default :
	     memset(PD_ERR(_t_index), 0, MAXLINE);
	     break;};

    if (*mode == 'w')
       fp = io_open(name, BINARY_MODE_WPLUS);
    else
       fp = io_open(name, BINARY_MODE_RPLUS);

    if (fp == NULL)
       {*mode = 'r';
        fp = io_open(name, BINARY_MODE_R);
	if (fp == NULL)
	   SS_error("CAN'T OPEN FILE READ-ONLY - SX_OPEN_RAW_FILE", argl);};

    file = _PD_mk_pdb(name);
    if (file == NULL)
       SS_error("CAN'T ALLOCATE PDBFILE - SX_OPEN_RAW_FILE", argl);

/* initialize the PDB file */
    file->std            = _PD_copy_standard(file->host_std);
    file->align          = _PD_copy_alignment(file->host_align);
    file->chrtaddr       = 0;
    file->system_version = PDB_SYSTEM_VERSION;
    file->date           = SC_date();
    file->stream         = fp;

    if (*mode == 'a')
       file->mode = PD_APPEND;
    else
       file->mode = PD_OPEN;

#ifdef HAVE_THREADS
    _PD_smp_init(file);
#endif

    if (REQ_STANDARD != NULL)
       {if (!_PD_compare_std(REQ_STANDARD, file->std,
                             REQ_ALIGNMENT, file->align))
           {_PD_rl_standard(file->std);
	    file->std   = _PD_copy_standard(REQ_STANDARD);
	    _PD_rl_alignment(file->align);
            file->align = _PD_copy_alignment(REQ_ALIGNMENT);};

        REQ_STANDARD  = NULL;
        REQ_ALIGNMENT = NULL;};

    _PD_init_chrt(file);

    return(SX_mk_gfile(_SX_mk_file(name, SX_PDBFILE_S, type, file)));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SEEK_RAW_FILE - seek in a file */

static object *SX_seek_raw_file(argl)
   object *argl;
   {off_t addr;
    int whence, ret;
    PDBfile *file;
    FILE *fp;
    g_file *po;

    addr   = 0;
    whence = 0;
    SS_args(argl,
            G_FILE, &po,
            SC_OFF_T_I, &addr,
            SC_INTEGER_I, &whence,
            0);

    if (po == NULL)
       SS_error("BAD FILE POINTER - SX_SEEK_RAW_FILE", argl);
       
    file = FILE_FILE(PDBfile, po);
    PD_GET_PTR_STREAM(file, fp);

    switch (whence)
       {default :
        case -1 :
             ret = io_seek(fp, addr, SEEK_SET);
             break;
        case -2 :
             ret = io_seek(fp, addr, SEEK_CUR);
             break;
        case -3 :
             ret = io_seek(fp, addr, SEEK_END);
             break;};

    if (ret != 0)
       SS_error("CAN'T SEEK IN FILE - SX_SEEK_RAW_FILE", argl);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_CLOSE_RAW_FILE - close a file as a binary input port */

static object *SX_close_raw_file(argl)
   object *argl;
   {PDBfile *file;
    g_file *po;
    FILE *fp;

    SS_args(argl,
            G_FILE, &po,
            0);

    if (po == NULL)
       SS_error("BAD FILE POINTER - SX_CLOSE_RAW_FILE", argl);
       
    file = FILE_FILE(PDBfile, po);
    PD_GET_PTR_STREAM(file, fp);

    if (io_close(fp) != 0)
       SS_error("CAN'T CLOSE FILE - SX_CLOSE_RAW_FILE", argl);

/* free the space */
    _PD_rl_pdb(file);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_RD_RAW - read from a raw binary file
 *           - (read-binary <file> <address> <nitems>
 *           -              <file-type> [<mem-type>])
 */

static object *SX_rd_raw(argl)
   object *argl;
   {PDBfile *file;
    FILE *fp;
    syment *ep;
    g_file *po;
    byte *vr;
    long nitems, ni;
    off_t addr;
    int ret;
    char *intype, *outtype;
    SC_THREAD_ID(_t_index);

    if (_SS_length(argl) > 4)
       SS_args(argl,
               G_FILE, &po,
               SC_OFF_T_I, &addr,
               SC_LONG_I, &nitems,
               SC_STRING_I, &intype,
               SC_STRING_I, &outtype,
               0);
    else
       {SS_args(argl,
                G_FILE, &po,
                SC_OFF_T_I, &addr,
                SC_LONG_I, &nitems,
                SC_STRING_I, &intype,
                0);

        outtype = intype;};

    if (po == NULL)
       SS_error("BAD FILE POINTER - SX_RD_RAW", argl);
       
    file = FILE_FILE(PDBfile, po);
    PD_GET_PTR_STREAM(file, fp);

/* NOTE: this changes the file position on the CRAYS
 *       which is very bad for SEEK_CUR
    if (io_flush(fp))
       SS_error("FFLUSH FAILED BEFORE READ - SX_RD_RAW", argl);
*/

    switch (addr)
       {case -1 :
             ret = io_seek(fp, (off_t)0, SEEK_SET);
             break;
        case -2 :
             ret = FALSE;
             break;
        case -3 :
             ret = io_seek(fp, (off_t)0, SEEK_END);
             break;
        default :
             ret = io_seek(fp, addr, SEEK_SET);
             break;};

    addr = io_tell(fp);
    ep   = _PD_mk_syment(intype, nitems, addr, NULL, NULL);

    if (ret)
       SS_error("FSEEK FAILED TO FIND REQUESTED ADDRESS - SX_RD_RAW",
                argl);

/* pad char arrays with 2 null characters - for printing and SC_firsttok */
    if (strcmp(outtype, "char") == 0)
       vr = _PD_alloc_entry(file, outtype, nitems + 2L);
    else
       vr = _PD_alloc_entry(file, outtype, nitems);

    if (vr == NULL)
       SS_error("CAN'T ALLOCATE MEMORY - SX_RD_RAW", SS_null);

    switch (setjmp(_PD_READ_ERR(_t_index)))
       {case ABORT    : return(SS_f);
        default       : memset(PD_err, 0, MAXLINE);
                        break;};

    ni = _PD_rd_syment(file, ep, outtype, vr);
/*    ni = _PD_rd_tree(file, vr, nitems, intype, outtype); */

    return(_SX_mk_gpdbdata("data", vr, ep, file));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_WR_RAW - write to a raw binary file
 *           - (write-binary <file> <data> <address> <nitems>
 *           -               <file-type> [<mem-type>])
 */

static object *SX_wr_raw(argl)
   object *argl;
   {PDBfile *file;
    object *obj;
    FILE *fp;
    g_file *po;
    byte *vr;
    int ret;
    long nitems;
    off_t addr;
    char *intype, *outtype;
    SC_THREAD_ID(_t_index);

    if (_SS_length(argl) > 5)
       SS_args(argl,
               G_FILE, &po,
               SS_OBJECT_I, &obj,
               SC_OFF_T_I, &addr,
               SC_LONG_I, &nitems,
               SC_STRING_I, &intype,
               SC_STRING_I, &outtype,
               0);
    else
       {SS_args(argl,
                G_FILE, &po,
                SS_OBJECT_I, &obj,
                SC_OFF_T_I, &addr,
                SC_LONG_I, &nitems,
                SC_STRING_I, &intype,
                0);

        outtype = intype;};

    if (po == NULL)
       SS_error("BAD FILE POINTER - SX_WR_RAW", argl);
       
    if (SX_PDBDATAP(obj))
       vr = (byte *) PDBDATA_DATA(obj);
    else
       SS_error("CAN'T HANDLE THIS DATA - SX_WR_RAW", argl);

    file = FILE_FILE(PDBfile, po);
    PD_GET_PTR_STREAM(file, fp);

    if (io_flush(fp))
       SS_error("FFLUSH FAILED BEFORE READ - SX_WR_RAW", argl);

    switch (addr)
       {case -1 :
             ret = io_seek(fp, (off_t)0, SEEK_SET);
             break;
        case -2 :
             ret = FALSE;
             break;
        case -3 :
             ret = io_seek(fp, (off_t)0, SEEK_END);
             break;
        default :
             ret = io_seek(fp, addr, SEEK_SET);
             break;};

    if (ret)
       SS_error("FSEEK FAILED TO FIND REQUESTED ADDRESS - SX_WR_RAW",
                argl);

    switch (setjmp(_PD_WRITE_ERR(_t_index)))
       {case ABORT    : return(SS_f);
        default       : memset(PD_err, 0, MAXLINE);
                        break;};

    _PD_wr_syment(file, vr, nitems, intype, outtype);

    return(SS_t);}

/*--------------------------------------------------------------------------*/

/*                              PDB FILE I/O                                */

/*--------------------------------------------------------------------------*/

/* _SX_SPEC_INSTANCE - convert the data specification in the arg list to
 *                   - an instance in memory and return it
 *                   -
 *                   - Syntax:
 *                   -
 *                   -    ((type <type-name> [<dim1> ... <dimn>])
 *                   -     <item1> ... <itemn>)
 *                   -
 *                   - each pointered type has the data in a list
 *                   -
 *                   - Examples:
 *                   -
 *                   - (make-defstr* nil "bird"
 *                   -               (def-member integer i)
 *                   -               (def-member integer *i1)
 *                   -               (def-member char c 10) 
 *                   -               (def-member char *a)
 *                   -               (def-member char **s)
 *                   -               (def-member float f))
 *                   -
 *                   - ((type "bird" 3)
 *                   -  '(10 ((1 2))
 *                   -    "test" ("doggie") (("big" "ugly")) 20.0)
 *                   -  '(20 ((3 4 5 6))
 *                   -    "baz"  ("kitty")  (("nice" "soft" "warm")) 22.0)
 *                   -  '(30 (7)
 *                   -    "foo"  ("birdy")  (("small")) 24.0)))
 */

static syment *_SX_spec_instance(file, argl)
   PDBfile *file;
   object *argl;
   {object *data;
    SC_address val;
    long number;
    char *type, *label;
    dimdes *dims;
    syment *ep;

    ep = _PD_mk_syment(NULL, 0L, 0L, NULL, NULL);
    if (ep != NULL)
       {val.memaddr = NULL;

	data = SS_car(argl);
	argl = SS_cdr(argl);

	if (!SS_consp(data))
	   SS_error("SHOULD BE LIST - _SX_SPEC_INSTANCE", data);

	label = SC_strsavef(SS_get_string(SS_car(data)),
			    "char*:_SX_SPEC_INSTANCE:label");
	data  = SS_cdr(data);
	if (strcmp(label, "type") == 0)
	   {type        = SC_strsavef(SS_get_string(SS_car(data)),
				      "char*:_SX_SPEC_INSTANCE:type");
	    dims        = _SX_make_dims_dimdes(file, SS_cdr(data));
	    number      = _PD_comp_num(dims);
	    val.memaddr = _PD_alloc_entry(file, type, number);
	    if (val.memaddr == NULL)
	       SS_error("CAN'T ALLOCATE TYPE - _SX_SPEC_INSTANCE", data);

	    _SX_rd_tree_list(argl, file, val.memaddr, number, type, dims);

	    PD_entry_type(ep)       = SC_strsavef(type,
						  "char*:_SX_SPEC_INSTANCE:type");
            PD_entry_dimensions(ep) = dims;
            PD_entry_number(ep)     = number;
            PD_entry_address(ep)    = val.diskaddr;

	    SC_mark(val.memaddr, 1);};};

    return(ep);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_PDBFILEP - return TRUE iff an object is a G_FILE of external type PDB */

int SX_pdbfilep(obj)
   object *obj;
   {if (SX_FILEP(obj))
       {if (strcmp(FILE_EXT_TYPE(obj), SX_PDBFILE_S) == 0)
           return(TRUE);};

    return(FALSE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_IPDBFILEP - return TRUE iff an object is a G_FILE of internal type PDB*/

int SX_ipdbfilep(obj)
   object *obj;
   {if (SX_FILEP(obj))
       {if (strcmp(FILE_TYPE(obj), SX_PDBFILE_S) == 0)
           return(TRUE);};

    return(FALSE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_CREATE_PDBFILE - create up the named PDBFile */

object *SX_create_pdbfile(arg)
   object *arg;
   {return(SX_open_file(arg, (PFPByte) PD_open,
                        SX_PDBFILE_S, SX_PDBFILE_S, "w", SX_pdbfilep));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_OPEN_PDBFILE - open up the named PDBFile */

object *SX_open_pdbfile(argl)
   object *argl;
   {object *obj;
    char mode[3], *md;

    strcpy(mode, "r");
    if (SS_consp(argl))
       {obj  = SS_car(argl);
        argl = SS_cdr(argl);
        if (SS_consp(argl))
           {md      = SS_get_string(SS_car(argl));
            mode[0] = *md;
            mode[1] = '\0';};}
    else
       obj = argl;

    return(SX_open_file(obj, (PFPByte) PD_open, SX_PDBFILE_S, SX_PDBFILE_S,
                        mode, SX_pdbfilep));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_OPEN_FILE - open up the named G_FILE and
 *              - cons its object to SX_file_list.
 */

object *SX_open_file(arg, fun, type, ext_type, mode, type_hook)
   object *arg;
   PFPByte fun;
   char *type, *ext_type, *mode;
   PFInt type_hook;
   {char *name;
    g_file *po;

    name = NULL;
    SS_args(arg,
            SC_STRING_I, &name,
            0);

    if (name == NULL)
       SS_error("BAD FILE NAME - SX_OPEN_FILE", arg);

/* search for an existing file by this name */
    for (po = SX_file_list; po != NULL; po = po->next)
        {if (strcmp(name, po->name) == 0)
             return(po->file_object);};

/* add to file_list */
    SX_file_list = _SX_mk_open_file(fun, name, type, ext_type,
                                    mode, type_hook);

    return(SX_file_list->file_object);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_CLOSE_PDBFILE - close a PDBFile and
 *                  - remove the object from SX_file_list
 */

object *SX_close_pdbfile(arg)
   object *arg;
   {return(SX_close_file(arg, (PFInt) PD_close));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_FLUSH_PDBFILE - flush a PDBFile */

object *SX_flush_pdbfile(arg)
   object *arg;
   {g_file *po;
    PDBfile *file;

    po = NULL;
    SS_args(arg,
            G_FILE, &po,
            0);

    if (po != NULL)
       {file = FILE_FILE(PDBfile, po);
	if (PD_flush(file))
	   return(SS_t);};

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DEF_COMMON_TYPES - define common internal SX types to the file */

static object *SX_def_common_types(arg)
   object *arg;
   {g_file *po;
    PDBfile *file;

    po = NULL;
    SS_args(arg,
            G_FILE, &po,
            0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    _SX_init_hash_objects(file);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_ENTRY_NUMBER - return the number of items for the named variable */

static object *SX_entry_number(argl)
   object *argl;
   {char *name;
    g_file *po;
    PDBfile *file;
    object *obj;
    syment *ep;

    po   = NULL;
    name = NULL;
    SS_args(argl,
            G_FILE, &po,
            SC_STRING_I, &name,
            0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    if (name == NULL)
       return(obj = SS_null);

    else
       {ep = _PD_effective_ep(file, name, TRUE, NULL);
        if (ep == NULL)
           return(obj = SS_null);

        return(obj = SS_mk_integer((BIGINT)PD_entry_number(ep)));};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_RESET_PTRS - call PD_reset_ptr_list */

object *SX_reset_ptrs(argl)
   object *argl;
   {PDBfile *file;
    g_file *po;

    po = NULL;
    SS_args(argl,
            G_FILE, &po,
            0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    PD_reset_ptr_list(file);

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_FAMILY_FILE - family the file */

object *SX_family_file(argl)
   object *argl;
   {int ifl;
    PDBfile *file;
    g_file *po;
    object *fl;

    po = NULL;
    fl = SS_t;
    SS_args(argl,
            G_FILE, &po,
            SS_OBJECT_I, &fl,
            0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    ifl  = SS_true(fl);
    file = PD_family(file, ifl);

    return(SX_mk_gfile(_SX_mk_file(file->name,
				   SX_PDBFILE_S, SX_PDBFILE_S,
				   file)));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_FILE_INFO - return the named info about the file
 *              - available info is:
 *              -    name (char *)               - file name
 *              -    mode (int)                  - file mode
 *              -    default-offset (int)        - index default offset
 *              -    conversions (int)           - whether conversions will be done
 *              -    flushed (int)               - whether file has been flushed
 *              -    virtual-internal (int)      - whether file is "internal"
 *              -    system-version (int)        - PDB system version
 *              -    major-order (int)           - major order for indexes
 *              -    header-address (off_t)      - address of header info
 *              -    chart-address (off_t)       - address of structure chart
 *              -    symbol-table-address (off_t)- address of symbol table
 *              -    date (char *)               - file creation date
 *              -    maximum_size (off_t)        - threshold for new file
 *              -    previous_file (char *)      - previous file in family
 */

static object *SX_file_info(argl)
   object *argl;
   {char *name;
    g_file *po;
    PDBfile *file;
    object *obj;

    po   = NULL;
    name = NULL;
    SS_args(argl,
            G_FILE, &po,
            SC_STRING_I, &name,
            0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

/* put some help here - return a list of available info or something */
    if (name == NULL)
       obj = SS_null;

    else if (strcmp(name, "position") == 0)
       {FILE *fp;

        PD_GET_PTR_STREAM(file, fp);
        if (fp != NULL)
           obj = SS_mk_integer((BIGINT) io_tell(fp));
        else
           obj = SS_mk_integer((BIGINT) -1);}

    else if (strcmp(name, "name") == 0)
       obj = SS_mk_string(file->name);

    else if (strcmp(name, "mode") == 0)
       obj = SS_mk_integer((BIGINT) file->mode);    

    else if (strcmp(name, "default-offset") == 0)
       obj = SS_mk_integer((BIGINT) file->default_offset);    

    else if (strcmp(name, "flushed") == 0)
       obj = SS_mk_integer((BIGINT) file->flushed);    

    else if (strcmp(name, "virtual-internal") == 0)
       obj = SS_mk_integer((BIGINT) file->virtual_internal);    

    else if (strcmp(name, "system-version") == 0)
       obj = SS_mk_integer((BIGINT) file->system_version);    

    else if (strcmp(name, "major-order") == 0)
       obj = SS_mk_integer((BIGINT) file->major_order);    

    else if (strcmp(name, "maximum-size") == 0)
       obj = SS_mk_integer((BIGINT) file->maximum_size);    

    else if (strcmp(name, "header-address") == 0)
       obj = SS_mk_integer((BIGINT) file->headaddr);

    else if (strcmp(name, "chart-address") == 0)
       obj = SS_mk_integer((BIGINT) file->chrtaddr);    

    else if (strcmp(name, "symbol-table-address") == 0)
       obj = SS_mk_integer((BIGINT) file->symtaddr);    

    else if (strcmp(name, "date") == 0)
       obj = SS_mk_string(file->date);    

    else if (strcmp(name, "previous-file") == 0)
       obj = SS_mk_string(file->previous_file);    

    else if (strcmp(name, "track-pointers") == 0)
       obj = SS_mk_integer((BIGINT) file->track_pointers);    

    else
       obj = SS_null;

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DEFAULT_OFFSET - set the default offset for the given file */

static object *SX_default_offset(arg)
   object *arg;
   {g_file *po;
    PDBfile *file;
    int offset;
    int nargs;

    po = NULL;
    nargs = SS_args(arg,
                    G_FILE, &po,
                    SC_INTEGER_I, &offset,
                    0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    if (nargs >= 2)
       file->default_offset = offset;

    return(SS_mk_integer((BIGINT)file->default_offset));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MAJOR_ORDER - set the file to row or column major order */

static object *SX_major_order(arg)
   object *arg;
   {g_file *po;
    PDBfile *file;
    char *order;

    po = NULL;
    order = NULL;
    SS_args(arg,
            G_FILE, &po,
            SC_STRING_I, &order,
            0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    if (order != NULL)
       {if ((order[0] == 'r') || (order[0] == 'R'))
           file->major_order = ROW_MAJOR_ORDER;
        else
           file->major_order = COLUMN_MAJOR_ORDER;}

    return(SS_mk_integer((BIGINT)file->major_order));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_FILE_MODE - set the file mode */

static object *SX_file_mode(arg)
   object *arg;
   {g_file *po;
    PDBfile *file;
    char *mode;

    po = NULL;
    mode = NULL;
    SS_args(arg,
            G_FILE, &po,
            SC_STRING_I, &mode,
            0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    if (mode != NULL)
       {if ((mode[0] == 'r') || (mode[0] == 'R'))
           file->mode = PD_READ;
        else if ((mode[0] == 'w') || (mode[0] == 'W'))
           file->mode = PD_WRITE;
        else
           file->mode = PD_APPEND;}

    return(SS_mk_integer((BIGINT)file->mode));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_PDBFILE_TO_LIST - return a PDBfile as a list
 *                     - (symtab chart headaddr symtaddr chrtaddr)
 *
 *  Do not garbage collect the hashtables because they are
 *  still needed by the pdbfile.
 */

object *_SX_pdbfile_to_list(file)
   PDBfile *file;
   {object *obj, *obj1;

    obj = SS_null;
    obj = SS_mk_cons(SS_mk_integer((BIGINT)file->chrtaddr), obj);
    obj = SS_mk_cons(SS_mk_integer((BIGINT)file->symtaddr), obj);
    obj = SS_mk_cons(SS_mk_integer((BIGINT)file->headaddr), obj);

    obj1 = SS_mk_hash_table(file->chart);
    SS_UNCOLLECT(obj1);
    obj = SS_mk_cons(obj1, obj);

    obj1 = SS_mk_hash_table(file->host_chart);
    SS_UNCOLLECT(obj1);
    obj = SS_mk_cons(obj1, obj);

    obj1 = SS_mk_hash_table(file->symtab);
    SS_UNCOLLECT(obj1);
    obj = SS_mk_cons(obj1, obj);

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_SYMENT_TO_LIST - returns the syment as a list of objects
 *                    - (type diskaddr dimensions)
 */

object *_SX_syment_to_list(ep)
   syment *ep;
/*
   {object *obj;

    obj = SS_make_list(SC_STRING_I, PD_entry_type(ep),
                       SC_OFF_T_I, &PD_entry_address(ep),
                       0);

    return(SS_mk_cons(obj, _SX_make_dims_obj(PD_entry_dimensions(ep))));}
*/
   {object *obj;

    obj = _SX_make_dims_obj(PD_entry_dimensions(ep));
    obj = SS_mk_cons(SS_mk_integer((BIGINT)PD_entry_address(ep)), obj);
    obj = SS_mk_cons(SS_mk_string(PD_entry_type(ep)), obj);

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_DEFSTR_TO_LIST - returns the defstr as a list of objects
 *                    - (name (members))
 *                    - members (type name dimensions)
 */

object *_SX_defstr_to_list(dp)
   defstr *dp;
   {object *obj, *obj1;
    memdes *lst;

    obj = SS_null;
    obj = SS_mk_cons(SS_mk_string(dp->type), obj);

/* convert to dimension from a dimdes to a cons list */
    for (lst = dp->members; lst != NULL; lst = lst->next)
        {obj1 = SS_make_list(SC_STRING_I, lst->type,
                             SC_STRING_I, lst->name,
                             0);

         obj1 = SS_mk_cons(obj1,
                           _SX_make_dims_obj(lst->dimensions));

         obj = SS_mk_cons(obj1, obj);};

    return(SS_reverse(obj));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_MEMDES_TO_LIST - returns the memdes as a list of objects
 *                    - member := (type name dimensions)
 */

static object *_SX_memdes_to_list(mp)
   memdes *mp;
   {object *obj;

    obj = SS_make_list(SC_STRING_I, mp->type,
                       SC_STRING_I, mp->name,
                       SC_INTEGER_I, &mp->number,
                       SS_OBJECT_I, _SX_make_dims_obj(mp->dimensions),
                       0);

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_PDBDATA_TO_LIST - convert a pdbdata type to a list.
 *                     - (name (type dimension) data)
 */

object *_SX_pdbdata_to_list(name, vr, ep, file)
   char *name;
   byte *vr;
   syment *ep;
   PDBfile *file;
   {object *obj, *obj1;

    obj1 = _SX_make_dims_obj(PD_entry_dimensions(ep));
    obj1 = SS_mk_cons(SS_mk_string(PD_entry_type(ep)), obj1);
    obj1 = SS_mk_cons(SS_mk_string("type"), obj1);

    obj = _SX_make_list_syment(file, vr, PD_entry_number(ep), PD_entry_type(ep));
    obj = SS_mk_cons(obj1, obj);
    obj = SS_mk_cons(SS_mk_string(name), obj);

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_PDB_TO_LIST - convert a pdb type to a list
 *                -   syment to (type dimensions number diskaddr)
 *                -   defstr to (type size members)
 *                -   pdbdata to (file name defstr data)
 *                -   pdbfile to (symtab chart headaddr symtaddr chrtaddr)
 */

object *SX_pdb_to_list(arg)
   object *arg;
   {object *obj;
    g_pdbdata *pp;

    switch (SS_OBJECT_TYPE(arg))
       {default        :
        case G_FILE    : if (SX_ipdbfilep(arg))
                            obj = _SX_pdbfile_to_list(FILE_STREAM(PDBfile, arg));
                         else
                            obj = _SX_pdbfile_to_list(SX_vif);
                         break;
        case G_DEFSTR  : obj = _SX_defstr_to_list(SS_GET(defstr, arg));
                         break;
        case G_SYMENT  : obj = _SX_syment_to_list(SS_GET(syment, arg));
                         break;
        case G_MEMDES  : obj = _SX_memdes_to_list(SS_GET(memdes, arg));
                         break;
        case G_PDBDATA : pp = SS_GET(g_pdbdata, arg);
                         obj = _SX_pdbdata_to_list(pp->name, pp->data,
                                                   pp->ep, pp->file);
                         break;};

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_PARSE_TYPE - return a list of items describing a C/PDB type variable
 *               - definition
 */

static object *SX_parse_type(argl)
   object *argl;
   {char *def;

    def = NULL;
    SS_args(argl,
            SC_STRING_I, &def,
            0);

    if (def != NULL)
       {memdes *desc;
        object *ret;

        desc = _PD_mk_descriptor(def, 0);
        ret  = _SX_memdes_to_list(desc);
        _PD_rl_descriptor(desc);

        return(ret);}

    else
       return(SS_null);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_CLOSE_FILE - close a G_FILE and
 *               - remove the object from SX_file_list
 */

object *SX_close_file(arg, fun)
   object *arg;
   PFInt fun;
   {g_file *po, *prev;

    po = NULL;
    SS_args(arg,
            G_FILE, &po,
            0);

    if (po == NULL)
       return(SS_f);

/* remove from list */
    if (po == SX_file_list)
       SX_file_list = SX_file_list->next;

/* find the link preceding this one */
    else
       {for (prev = SX_file_list; prev != NULL; prev = prev->next)
            if (prev->next == po)
               break;

        if (prev != NULL)
           prev->next = po->next;};

    _SX_rel_open_file(fun, po);

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_LIST_FILE - return a list of open files */

object *SX_list_file()
   {g_file *po;
    object *obj;

    obj = SS_null;
    for (po = SX_file_list; po != NULL; po = po->next)
        obj = SS_mk_cons(po->file_object, obj);

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_PDBFP - Scheme level version of SX_pdbfilep function */

object *SX_pdbfp(obj)
   object *obj;
   {return(SX_pdbfilep(obj) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DEFP - function version of SX_DEFSTRP macro */

object *SX_defp(obj)
   object *obj;
   {return(SX_DEFSTRP(obj) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_PDBDATAP - function version of SX_PDBDATAP macro */

object *SX_pdbdatap(obj)
   object *obj;
   {return(SX_PDBDATAP(obj) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SYMP - function version of SX_SYMENTP macro */

object *SX_symp(obj)
   object *obj;
   {return(SX_SYMENTP(obj) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_LIST_SYMTAB - return a list of the symbol table entries */

object *SX_list_symtab(argl)
   object *argl;
   {object *args, *obj;
    char *patt;
    g_file *po;
    PDBfile *file;

    po   = NULL;
    patt = NULL;
    SS_args(argl,
            G_FILE, &po,
            SC_STRING_I, &patt,
            0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    if (patt == NULL)
       args = SS_mk_cons(SS_mk_hash_table(file->symtab),
                         SS_null);
    else
       args = SS_mk_cons(SS_mk_hash_table(file->symtab),
                         SS_mk_cons(SS_mk_string(patt),
                                    SS_null));

    obj = SS_hash_dump(args);
    
    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_LIST_VARIABLES - return a list of the variables in a file directory */

object *SX_list_variables(argl)
   object *argl;
   {object *obj;
    char *patt;
    char *type;
    char **names;
    int i, num;
    g_file *po;
    PDBfile *file;

    po   = NULL;
    patt = NULL;
    type = NULL;
    SS_args(argl,
            G_FILE, &po,
            SC_STRING_I, &patt,
            SC_STRING_I, &type,
            0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    names = PD_ls(file, patt, type, &num);

    obj = SS_null;
    for (i = num - 1; i >= 0; i--)
        {SS_Assign(obj, SS_mk_cons(SS_mk_string(names[i]), obj));};

    SFREE(names);

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_CHANGE_DIRECTORY - change the current working directory in a file */

object *SX_change_directory(argl)
   object *argl;
   {char *dirname;
    g_file *po;
    PDBfile *file;

    po      = NULL;
    dirname = NULL;
    SS_args(argl,
            G_FILE, &po,
            SC_STRING_I, &dirname,
            0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    if (dirname == NULL)
       dirname = "/";

    if (!PD_cd(file, dirname))
       SS_error("BAD DIRECTORY NAME - SX_CHANGE_DIRECTORY", argl);

    return(SS_mk_string(dirname));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MAKE_DIRECTORY - make a directory in a file */

object *SX_make_directory(argl)
   object *argl;
   {char *dirname;
    g_file *po;
    object *errf;
    PDBfile *file;

    po      = NULL;
    dirname = NULL;
    errf    = SS_t;
    SS_args(argl,
            G_FILE, &po,
            SC_STRING_I, &dirname,
            SS_OBJECT_I, &errf,
            0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    if (!PD_mkdir(file, dirname))
       {PD_err[0] = '\0';
	if (SS_true(errf))
           SS_error("UNABLE TO CREATE DIRECTORY - SX_MAKE_DIRECTORY",
		    argl);};

    return(SS_mk_string(dirname));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_CURRENT_DIRECTORY - return the current working directory in a file */

object *SX_current_directory(arg)
   object *arg;
   {g_file *po;
    PDBfile *file;

    po      = NULL;
    SS_args(arg,
            G_FILE, &po,
            0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    return(SS_mk_string(PD_pwd(file)));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_CREATE_LINK - create a link to a variable in a file */

object *SX_create_link(argl)
   object *argl;
   {char *oldname;
    char *newname;
    g_file *po;
    PDBfile *file;

    po      = NULL;
    oldname = NULL;
    newname = NULL;
    SS_args(argl,
            G_FILE, &po,
            SC_STRING_I, &oldname,
            SC_STRING_I, &newname,
            0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    if (!PD_ln(file, oldname, newname))
       SS_error("UNABLE TO CREATE LINK - SX_CREATE_LINK", argl);

    return(SS_mk_string(newname));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_LIST_DEFSTRS - return a list of the derived data types in a file */

object *SX_list_defstrs(argl)
   object *argl;
   {object *obj, *sort;
    g_file *po;
    PDBfile *strm;

    po   = NULL;
    sort = SS_t;
    SS_args(argl,
            G_FILE, &po,
	    SS_OBJECT_I, &sort,
            0);

    if ((po == NULL) || (po == SX_gvif))
       strm = SX_vif;
    else
       strm = FILE_FILE(PDBfile, po);

    obj = SS_make_list(SS_OBJECT_I, SS_mk_hash_table(strm->chart),
                       SS_OBJECT_I, SS_null,
                       SS_OBJECT_I, sort,
                       0);

    obj = SS_hash_dump(obj);

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DEF_PRIM - Scheme version of PD_DEFIX, PD_DEFLOAT, PD_DEFNCV
 *             -
 *             - (make-prim file name "ncv" bytespitem align)
 *             - (make-prim file name "fix" bytespitem align flg)
 *             - (make-prim file name "fp"  bytespitem align
 *             -            order-list expb mantb sbs sbe sbm hmb bias)
 */
 
static object *SX_def_prim(argl)
   object *argl;
   {int n;
    char *name, *type;
    long bytespitem;
    int align;
    PDBfile *file;
    defstr *dp;
    g_file *po;

    po         = NULL;
    name       = NULL;
    type       = NULL;
    bytespitem = 0L;
    align      = 0;

    n = SS_args(argl,
                G_FILE, &po,
                SC_STRING_I, &name,
                SC_STRING_I, &type,
                SC_LONG_I, &bytespitem,
                SC_INTEGER_I, &align,
                0);

    if ((po == NULL) || (po == SX_gvif))
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    if (strcmp(type, "ncv") == 0)
       dp = PD_defncv(file, name, bytespitem, align);

    else if (strcmp(type, "fix") == 0)
       {int flg;
        object *rest;

        rest = _SS_lst_tail(argl, n);
        SS_args(rest,
                SC_INTEGER_I, &flg,
                0);

        dp = PD_defix(file, name, bytespitem, align, flg);}

    else if (strcmp(type, "fp") == 0)
       {int *ordr;
        long i, no, expb, mantb, sbs, sbe, sbm, hmb, bias;
        object *rest, *ord;

        rest = _SS_lst_tail(argl, n);
        SS_args(rest,
                SS_OBJECT_I, &ord,
                SC_LONG_I, &expb,
                SC_LONG_I, &mantb,
                SC_LONG_I, &sbs,
                SC_LONG_I, &sbe,
                SC_LONG_I, &sbm,
                SC_LONG_I, &hmb,
                SC_LONG_I, &bias,
                0);

        if (SS_nullobjp(ord))
           SS_error("BAD BYTE ORDERING - SX_DEF_PRIM", ord);

        no = _SS_length(ord);
        if (no != bytespitem)
           SS_error("INCONSISTENT SIZE - SX_DEF_PRIM", ord);

        ordr = FMAKE_N(int, no, "SX_DEF_PRIM:ordr");
        for (i = 0L; i < no; i++, ord = SS_cdr(ord));
            ordr[i] = SS_INTEGER_VALUE(SS_car(ord));

        dp = PD_defloat(file, name, bytespitem, align, ordr,
                        expb, mantb, sbs, sbe, sbm, hmb, bias);};

    return(_SX_mk_gdefstr(dp));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_READ_DEFSTR - read a defstr from a file
 *                - Usage: (read-defstr file name)
 */
 
static object *SX_read_defstr(argl)
   object *argl;
   {char *name;
    g_file *po;
    PDBfile *file;
    defstr *dp;

    po   = NULL;
    name = NULL;
    SS_args(argl,
            G_FILE, &po,
            SC_STRING_I, &name,
            0);

    if ((po == NULL) || (po == SX_gvif))
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    dp   = PD_inquire_type(file, name);
    if (dp == NULL)
       SS_error("BAD TYPE - SX_READ_DEFSTR", argl);

    return(_SX_mk_gdefstr(dp));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_WRITE_DEFSTR - write a defstr to a file
 *                 - Usage: (write-defstr file defstr)
 */
 
static object *SX_write_defstr(argl)
   object *argl;
   {g_file *po;
    PDBfile *file;
    memdes *desc;
    defstr *dp;

    po = NULL;
    dp = NULL;
    SS_args(argl,
            G_FILE, &po,
            G_DEFSTR, &dp,
            0);

    if ((po == NULL) || (po == SX_gvif))
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    desc = PD_copy_members(dp->members);
    if (desc == NULL)
       {if (PD_inquire_table_type(file->chart, dp->type) == NULL)
           _PD_d_install(dp->type, dp, file->chart);
        if (PD_inquire_table_type(file->host_chart, dp->type) == NULL)
           _PD_d_install(dp->type, dp, file->host_chart);}
    else
       _PD_defstr_inst(dp->type, desc, -1, NULL, NULL,
		       file->chart, file->host_chart,
		       file->align, file->host_align,
		       TRUE);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MAKE_DEFSTR - Create a new derived type
 *                - Usage: (make-defstr file name [member(s)]+)
 *                -
 *                - member = (type name dimension(s))
 */
 
static object *SX_make_defstr(argl)
   object *argl;
   {char *name, *mname, *type, *memtemp;
    char member[80];
    memdes *desc, *lst, *prev;
    dimdes *dims, *dim0;
    object *member_obj, *dim_obj;
    g_file *po;
    PDBfile *file;
    defstr *dp;

    po   = NULL;
    name = NULL;
    SS_args(argl,
            G_FILE, &po,
            SC_STRING_I, &name,
            0);

    if ((po == NULL) || (po == SX_gvif))
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    argl = SS_cddr(argl);
    lst  = NULL;
    for (; !SS_nullobjp(argl); argl = SS_cdr(argl))
        {member_obj = SS_car(argl);
	 if (!SS_consp(member_obj))
	    SS_error("MEMBER MUST BE A LIST - SX_MAKE_DEFSTR",
		     member_obj);

         mname = NULL;
         type  = NULL;
         SS_args(member_obj,
                 SC_STRING_I, &type,
                 SC_STRING_I, &mname,
                 0);

	 sprintf(member, "%s %s", type, mname);
	 dim_obj = SS_cddr(member_obj);
	 if (!SS_nullobjp(dim_obj))
	    {dims = _SX_make_dims_dimdes(file, dim_obj);
	     memtemp = member + strlen(member);
	     *memtemp++ = '(';
	     for (; dims != NULL; dims = dim0)
	         {if (dims->index_min == file->default_offset)
		     {sprintf(memtemp, "%ld", dims->number);
		      memtemp += strlen(memtemp);}
		  else
		     {sprintf(memtemp, "%ld:%ld",
			      dims->index_min, dims->index_max);
		      memtemp += strlen(memtemp);};

		  dim0 = dims->next;
		  if (dim0 != NULL)
		     *memtemp++ = ',';
		  SFREE(dims);};
	     *memtemp++ = ')';
	     *memtemp = '\000';};
 
	 desc = _PD_mk_descriptor(member, file->default_offset);

	 if (lst == NULL)
	    lst = desc;
	 else
	    prev->next = desc;
	 prev = desc;};

    dp = _PD_defstr_inst(name, lst, -1, NULL, NULL,
			 file->chart, file->host_chart,
			 file->align, file->host_align,
			 FALSE);

    return(_SX_mk_gdefstr(dp));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MAKE_TYPEDEF - Scheme version of PD_TYPEDEF
 *                 -
 *                 - (make-typedef file old new)
 */
 
static object *SX_make_typedef(argl)
   object *argl;
   {char *ntype, *otype;
    PDBfile *file;
    g_file *po;

    po    = NULL;
    ntype = NULL;
    otype = NULL;
    SS_args(argl,
            G_FILE, &po,
            SC_STRING_I, &otype,
            SC_STRING_I, &ntype,
            0);

    if ((po == NULL) || (po == SX_gvif))
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    return(_SX_mk_gdefstr(PD_typedef(file, otype, ntype)));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MAKE_CAST - Scheme version of PD_CAST
 *              -
 *              - (make-cast file type member controller)
 */
 
static object *SX_make_cast(argl)
   object *argl;
   {char *type, *memb, *contr;
    PDBfile *file;
    g_file *po;

    po   = NULL;
    type  = NULL;
    memb  = NULL;
    contr = NULL;
    SS_args(argl,
            G_FILE, &po,
            SC_STRING_I, &type,
            SC_STRING_I, &memb,
            SC_STRING_I, &contr,
            0);

    if ((po == NULL) || (po == SX_gvif))
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    return(PD_cast(file, type, memb, contr) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_FILE_VARP - return #t iff the given variable name/path refers or 
 *              - could possibly refer to a variable in the given file
 */

object *SX_file_varp(argl)
   object *argl;
   {int flag, ret;
    char *name;
    object *fobj;
    PDBfile *file;
    g_file *po;

    po   = NULL;
    name = NULL;
    fobj = SS_f;
    SS_args(argl,
            G_FILE, &po,
            SC_STRING_I, &name,
            SS_OBJECT_I, &fobj,
            0);

    if ((po == NULL) || (po == SX_gvif))
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

/* check for the flag */
    flag = (fobj != SS_f);

    ret = _SX_file_varp(file, name, flag);
    SFREE(name);

    return(ret ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_FILE_VARP - return TRUE iff the named variable is in the file */

int _SX_file_varp(file, name, flag)
   PDBfile *file;
   char *name;
   int flag;
   {char *token, *s;
    syment *ep;

    if (flag)
       {ep = _PD_effective_ep(file, name, TRUE, NULL);
        if (ep != NULL)
           _PD_rl_syment_d(ep);}

    else
       {token = SC_firsttok(name, ".([ ");
	s = SC_strstr(token, "->"); 
	if (s != NULL)
	   s[0] = '\0';
        ep = PD_inquire_entry(file, token, TRUE, NULL);};

    return((ep == NULL) ? FALSE : TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_RD_SYMENT - read a syment from the given file
 *              -
 *              - (read-syment <file> <name>)
 *              -
 *              - This is a low level function not intended for the
 *              - general user. A syment is also created and written
 *              - when a variable is written using write-pdbdata.
 *              -
 *              - if the file is not specified, the internal virtual
 *              - file is used
 *              - the syment is looked up
 */

static object *SX_rd_syment(argl)
   object *argl;
   {char *name, *s;
    g_file *po;
    PDBfile *file;
    syment *ep;
    dimdes *dp;
    object *err;

    po   = NULL;
    name = NULL;
    SS_args(argl,
            G_FILE, &po,
            SC_STRING_I, &name,
            SS_OBJECT_I, &err,
            0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    s = name;
    name = _PD_expand_hyper_name(file, s);
    SFREE(s);
    if (name == NULL)
       return(SS_null);

    ep = _PD_effective_ep(file, name, TRUE, NULL);
    if (ep == NULL)
       {SFREE(name);
	if (SS_true(err))
	   return(SS_null);
	else
	   SS_error(PD_err, SS_cadr(argl));}

    else if (name[strlen(name) - 1] == ']')
       {dp = PD_entry_dimensions(ep);
        PD_entry_dimensions(ep) = _PD_hyper_dims(file, name, dp);

/* GOTCHA: this set of dimensions may still be valid for the original ep
 *         they weren't copied - its certainly TRUE in cases with VIF's
	_PD_rl_dimensions(dp);
 */
        };

/* GOTCHA: the syment EP will never be GC'd as the coding stands
 *         the error case above does not count!
 */

    return(_SX_mk_gsyment(ep));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_MAKE_BLOCKS - make a symblock array from the given disk address
 *                 - information
 */

static symblock *_SX_make_blocks(alst, numb)
   object *alst;
   long numb;
   {symblock *sp;
    int i, n;
    off_t addr, ni, tot;

    if (SS_consp(alst))
       {n   = _SS_length(alst) >> 1;
        sp  = FMAKE_N(symblock, n, "_SX_MAKE_BLOCKS:sp");
        tot = 0L;
        for (i = 0; i < n; i++)
            {SS_args(alst,
                     SC_OFF_T_I, &addr,
                     SC_LONG_I, &ni,
                     0);
             sp[i].diskaddr = addr;
             sp[i].number   = ni;

             tot += ni;

             alst = SS_cddr(alst);};

        if (tot != numb)
           SS_error("INCONSISTENT DISCONTIGUOUS ENTRY - _SX_MAKE_BLOCKS",
                    alst);}

    else
       {sp = FMAKE(symblock, "_SX_MAKE_BLOCKS:sp");

        sp->number = numb;
        SS_args(alst,
                SC_OFF_T_I, &sp->diskaddr,
                0);};

    return(sp);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_WR_SYMENT - install a syment
 *              -
 *              - (write-syment <file> <name> <type> <diskaddr> [<dimensions>])
 *              -
 *              - This is a low level function not intended for the
 *              - general user. A syment is also created and written
 *              - when a variable is written using write-pdbdata.
 *              -
 *              - If the file is not specified, the internal virtual
 *              - file is used. A syment is created.
 *              - The number of items is computed from the dimension.
 */

static object *SX_wr_syment(argl)
   object *argl;
   {char *name, *type;
    char s[MAXLINE];
    long numb;
    g_file *po;
    PDBfile *file;
    syment *ep;
    dimdes *dims;
    object *alst;
    symblock *sp;

    po   = NULL;
    name = NULL;
    type = NULL;
    alst = SS_null;
    SS_args(argl,
            G_FILE, &po,
            SC_STRING_I, &name,
            SC_STRING_I, &type,
            SS_OBJECT_I, &alst,
            0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

/* get optional dimensions */
    argl = SS_cddr(SS_cddr(argl));

    dims = _SX_make_dims_dimdes(file, argl);
    numb = _PD_comp_num(dims);
    sp   = _SX_make_blocks(alst, numb);
    ep   = _PD_mk_syment(type, numb, 0L, NULL, dims);
    strcpy(s, _PD_fixname(file, name));

    _PD_e_install(s, ep, file->symtab, TRUE);

    SFREE(PD_entry_blocks(ep));
    PD_entry_blocks(ep) = sp;
    SC_mark(sp, 1);

    return(_SX_mk_gsyment(ep));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_PSEUDO_SUITABLE - is the specified entry suitable for writing to
 *                     - a "psuedo" file (i.e. a buffer) or only with
 *                     - the more restricted VIF
 *                     - return TRUE iff suitable for "pseudo" file
 */

static int _SX_pseudo_suitable(file, et)
   PDBfile *file;
   syment *et;
   {int ok;
    char *type;
    defstr *dp;
    memdes *dm;

    type = PD_entry_type(et);

    ok = !_PD_indirection(type);
    dp = _PD_lookup_type(type, file->host_chart);
    for (dm = dp->members; dm != NULL && ok; dm = dm->next)
        ok &= !_PD_indirection(dm->type);

    return(ok);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_WRITE_FILEDATA - write a pdbdata object from a PDB file obtained by
 *                  - the function passed in.
 *                  -
 *                  - Syntax:
 *                  -    (write-filedata file name (type dimension) data)
 *                  - or
 *                  -    (write-filedata file name pdbdata)
 *                  -
 *                  - This routine will write a pdbdata object
 *                  - If file is missing, the internal virtual file will
 *                  - be used.
 */

static object *SX_write_filedata(argl, fun)
   object *argl;
   PFPObject fun;
   {int wrfl, new;
    long number;
    char *type, *ntype, *vrin, *t;
    char fullpath[MAXLINE], bf[MAXLINE];
    PDBfile *file, *fp;
    syment *ep, *et, *eo;
    dimdes *dims;
    defstr *dp, *ndp;
    SC_address addr, b;
    object *name_obj, *syment_obj;
    
    if (!SS_consp(argl))
       SS_error("BAD ARGUMENT LIST - SX_WRITE_FILEDATA", argl);

    argl = (*fun)(argl, &file);

/* get the name of the variable */
    name_obj = SS_car(argl);
    strcpy(fullpath, _PD_fixname(file, SS_get_string(name_obj)));
    argl     = SS_cdr(argl);

/* if no other arguments its an error */
    if (SS_nullobjp(argl))
       SS_error("INSUFFICIENT ARGUMENTS - SX_WRITE_FILEDATA", argl);

/* if the next item is a pdbdata, use its info */
    syment_obj = SS_car(argl);
    if (SX_PDBDATAP(syment_obj))
       {SS_MARK(syment_obj);
        ep           = PDBDATA_EP(syment_obj);
        addr.memaddr = (char *) PDBDATA_DATA(syment_obj);
        type         = SC_strsavef(PD_entry_type(ep),
                         "char*:SX_WRITE_FILEDATA:type");
        number       = PD_entry_number(ep);
        dims         = PD_copy_dims(PD_entry_dimensions(ep));
        if (file->virtual_internal == TRUE)
           {vrin         = addr.memaddr;
            addr.memaddr = _PD_alloc_entry(file, type, number);
            _SX_copy_tree(file, vrin, addr.memaddr, number, type);
            ep = _PD_mk_syment(type, number, addr.diskaddr, NULL, dims);
            SC_mark(addr.memaddr, 1);
	    _PD_e_install(fullpath, ep, file->symtab, TRUE);}
        else
           {dp    = PD_inquire_host_type(file, type);
	    ndp   = _PD_container_type(file, dp);
	    ntype = (ndp == NULL) ? type : ndp->type;
	    ep    = _PD_write(file, fullpath, ntype, ntype,
			      addr.memaddr, dims, FALSE, &new);
            if (ep == NULL)
	       {SFREE(type);
		SFREE(dims);
		SS_error(PD_err, name_obj);};
	    if (new)
	       ep = PD_copy_syment(ep);};}
        
/* otherwise the next thing should be a cons */
    else
       {ep = _SX_spec_instance(file, argl);

	addr.diskaddr = PD_entry_address(ep);
        if (addr.memaddr == NULL)
           SS_error("UNKNOWN DATA - SX_WRITE_FILEDATA", argl);

/* "write" to correct file */
	wrfl = TRUE;
	fp   = file;
        if (file->virtual_internal == TRUE)
	   {et = _PD_effective_ep(file, fullpath, FALSE, NULL);
	    if (et == NULL)
	       {_PD_e_install(fullpath, ep, file->symtab, TRUE);
		wrfl = FALSE;}

	    else if (!_SX_pseudo_suitable(file, et))
	       {_PD_rl_syment(et);
		wrfl = FALSE;}

	    else
	       {_PD_rl_syment(et);

		strcpy(bf, fullpath);
		t  = SC_firsttok(bf, "[]() ");
		eo = PD_inquire_entry(file, t, TRUE, NULL);

		b.diskaddr = PD_entry_address(eo);

		fp = PN_open(file, b.memaddr);

                et = _PD_mk_syment(PD_entry_type(eo),
				   PD_entry_number(eo),
				   b.mdiskaddr,
				   NULL,
				   PD_entry_dimensions(eo));

		_PD_e_install(t, et, fp->symtab, TRUE);};};

	if (wrfl)
           {if (fp->mode == PD_OPEN)
	       SS_error("FILE OPENED READ-ONLY - SX_WRITE_FILEDATA", SS_null);
            ep = _PD_write(fp, fullpath, PD_entry_type(ep), PD_entry_type(ep),
                           addr.memaddr, PD_entry_dimensions(ep), FALSE, &new);
            if (ep == NULL)
	       SS_error(PD_err, name_obj);
	    if (new)
	       ep = PD_copy_syment(ep);};

	if (fp != file)
	   PN_close(fp);};

    return(_SX_mk_gpdbdata(fullpath, addr.memaddr, ep, file));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_WRITE_PDBDATA - write a pdbdata object
 *                  -
 *                  - Syntax:
 *                  -    (write-pdbdata file name (type dimension) data)
 *                  - or
 *                  -    (write-pdbdata file name pdbdata)
 *                  -
 *                  - This routine will write a pdbdata object
 *                  - If file is missing, the internal virtual file will
 *                  - be used.
 */

static object *SX_write_pdbdata(argl)
   object *argl;
   {return(SX_write_filedata(argl, SX_get_pdbfile));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_READ_FILEDATA - read a pdbdata object from a PDB file obtained by
 *                  - the function passed in
 *                 - Syntax: (read-filedata file name)
 *                 -
 *                 - This routine will make a pdbdata object
 *                 - If file is nil, the internal virtual file will be used.
 */

static object *SX_read_filedata(argl, fun)
   object *argl;
   PFPObject fun;
   {char *name, *s, *type;
    int n, err;
    long number;
    object *name_obj;
    PDBfile *file;
    syment *ep, *tep;
    SC_address addr;
    dimdes *dims;
    SC_THREAD_ID(_t_index);

    if (!SS_consp(argl))
       SS_error("BAD ARGUMENT LIST - SX_READ_FILEDATA", argl);

    argl = (*fun)(argl, &file);

/* get the name of the variable */
    name_obj = SS_car(argl);
    name = SC_strsavef(_PD_fixname(file, SS_get_string(name_obj)),
		       "char*:SX_READ_FILEDATA:name");
    s = name;
    name = _PD_expand_hyper_name(file, s);
    SFREE(s);
    if (name == NULL)
       SS_error("BAD SUBSCRIPT EXPRESSION - SX_READ_FILEDATA", name_obj);

    ep = _PD_effective_ep(file, name, FALSE, NULL);
    if (ep == NULL)
       {SFREE(name);
	SS_error(PD_err, name_obj);};
/*
    if (ep == NULL)
       {PRINT(stdout, "%s\n", PD_err);
	return(SS_null);};
*/
    type   = PD_entry_type(ep);
    number = PD_entry_number(ep);

    if (file == SX_vif)

/* GOTCHA: this protects against problems with pointers into the middle
 *         of SCORE allocated arrays - which are fierce
 *         at the cost of being unable to edit memory with PDBView
 */
       {SC_address ad;

	if (!_PD_indirection(type))
           {addr.memaddr = _PD_alloc_entry(file, type, number);
	    ad.diskaddr = PD_entry_address(ep);
	    memcpy(addr.memaddr,
		   ad.memaddr,
		   SC_arrlen(addr.memaddr));}
        else
	   addr.diskaddr = PD_entry_address(ep);

/* GOTCHA: until PDB is made to do the reference counting in detail
 *         the best we can do is to assume that things with NO
 *         reference counts are incorrect and have at least 1 reference
 */
        if (SC_ref_count(addr.memaddr) < 1)
	   SC_mark(addr.memaddr, 1);}

    else
       {switch (setjmp(_PD_READ_ERR(_t_index)))
           {case ABORT    :
                 SS_error("_PD_HYPER_READ FAILED - SX_READ_FILEDATA",
                          name_obj);
            default       :
                 memset(PD_err, 0, MAXLINE);
                 break;};

        addr.memaddr = _PD_alloc_entry(file, type, number);

        err = _PD_hyper_read(file, name, type, ep, addr.memaddr);
        if (!err)
           SS_error("PROBLEMS READING DATA - SX_READ_FILEDATA", name_obj);};

    if (name[strlen(name) - 1] == ']')
       {type = _PD_hyper_type(name, type);
	dims = _PD_hyper_dims(file, name, PD_entry_dimensions(ep));
	tep  = _PD_mk_syment(type, number, addr.diskaddr, NULL, dims);
	_PD_rl_syment_d(ep);}
    else
       tep = ep;

    if ((strcmp(type, SC_CHAR_S) == 0) &&
	(SX_string_mode != LITERAL))
       {s = (char *) addr.memaddr;
	for (n = number - 1; n >= 0; n--)
	    {if (isprint(s[n]) && !isspace(s[n]))
	        {s[n+1] = '\0';
		 break;};};
	if (n < 0)
	   s[0] = '\0';};

    return(_SX_mk_gpdbdata(name, addr.memaddr, tep, file));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_READ_PDBDATA - read a pdbdata object
 *                 -
 *                 - Syntax: (read-pdbdata file name)
 *                 -
 *                 - This routine will make a pdbdata object
 *                 - If file is nil, the internal virtual file will be used.
 */

static object *SX_read_pdbdata(argl)
   object *argl;
   {return(SX_read_filedata(argl, SX_get_pdbfile));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_WRT_ULTRA_CURVE - write an ULTRA curve specified by some numeric
 *                    - arrays
 */

object *SX_wrt_ultra_curve(argl)
   object *argl;
   {int npts;
    PDBfile *file;
    g_file *po;
    C_array *xarr, *yarr;
    REAL *xval, *yval;
    char *labl;
    static int ic = 0;

    po   = NULL;
    file = NULL;
    labl = NULL;
    npts = 0;
    xarr = NULL;
    yarr = NULL;
    SS_args(argl,
            G_FILE, &po,
            G_NUM_ARRAY, &xarr,
            G_NUM_ARRAY, &yarr,
            SC_STRING_I, &labl,
            SC_INTEGER_I, &npts,
            0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    xval = (REAL *) xarr->data;
    yval = (REAL *) yarr->data;
    PD_wrt_pdb_curve(file, labl, npts, xval, yval, ic++);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_GET_PDBFILE - examine the argument list and return either the
 *                - PDBfile which is first on the list or 
 *                - the default internal file if there is no file on the
 *                - argument list
 */

object *SX_get_pdbfile(argl, pfile)
   object *argl;
   PDBfile **pfile;
   {PDBfile *file;
    g_file *gf;
    object *file_obj;

    file_obj = SS_car(argl);
    if (SX_FILEP(file_obj))
       {gf = SS_GET(g_file, file_obj);
        if (gf->type_hook == NULL)
           file = FILE_FILE(PDBfile, gf);

        else if ((*gf->type_hook)(file_obj))
           file = FILE_FILE(PDBfile, gf);}

    else
       file = SX_vif;

    argl   = SS_cdr(argl);
    *pfile = file;

    return(argl);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_GET_FILE - examine the argument list and return either the
 *             - g_file which is first on the list or 
 *             - the default internal file if there is no file on the
 *             - argument list
 */

object *SX_get_file(argl, pfile)
   object *argl;
   g_file **pfile;
   {object *file_obj;
    g_file *file;

    file_obj = SS_car(argl);
    if (SX_FILEP(file_obj))
       {file   = SS_GET(g_file, file_obj);
        argl   = SS_cdr(argl);
        *pfile = file;}

    else if (SS_nullobjp(file_obj))
       {argl   = SS_cdr(argl);
        *pfile = SX_gvif;};
       

    return(argl);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SIZEOF - return the size in bytes of the given object */

static object *SX_sizeof(argl)
   object *argl;
   {g_file *po;
    PDBfile *file;
    char *type;
    long bytespitem;
    int flg;

    po   = NULL;
    file = NULL;
    type = NULL;
    flg  = FALSE;
    SS_args(argl,
            SC_STRING_I, &type,
            G_FILE, &po,
            SC_INTEGER_I, &flg,
            0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    if (flg)
       bytespitem = _PD_lookup_size(type, file->host_chart);
    else
       bytespitem = _PD_lookup_size(type, file->chart);

    return(SS_mk_integer((BIGINT)bytespitem));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_OPEN_TEXT_FILE - open a text file */

static object *SX_open_text_file(arg)
    object *arg;
    {char *name, *mode, *type;

     name = NULL;
     mode = NULL;
     type = "text";
     SS_args(arg,
	     SC_STRING_I, &name,
	     SC_STRING_I, &mode,
	     0);

     if (name == NULL)
        SS_error("BAD FILE NAME - SX_OPEN_TEXTFILE", arg);

     text_fp_f0 = fopen(name, mode);
     if (text_fp_f0 != NULL)
        return(SS_t);

     PRINT(stdout, "Couldn't open %s %s %s\n", name, mode, type);
     return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_CLOSE_TEXT_FILE - close an opened text file */

static object *SX_close_text_file(arg)
    object *arg;
    {if (text_fp_f0 != NULL)
        {fclose(text_fp_f0);
	 text_fp_f0 = NULL;
	 return(SS_t);};

     PRINT(stdout, "No opened file\n");
     return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_WRITE_PDB - display/write content of a pdb object */

static object *_SX_write_pdb(f0, argl)
   FILE *f0;
   object *argl;
   {g_pdbdata *pp;
    object *obj;

    obj  = SS_car(argl);
    argl = SS_cdr(argl);

    switch (SS_OBJECT_TYPE(obj))
       {default        :
	case G_FILE    :
             if (SX_ipdbfilep(obj))
                PD_write_extras(f0, FILE_STREAM(PDBfile, obj));
	     else
	        PD_write_extras(f0, SX_vif);
             break;
        case G_DEFSTR  :
             PD_write_defstr(f0, SS_GET(defstr, obj));
             break;
        case G_SYMENT  :
             PD_write_syment(f0, SS_GET(syment, obj));
             break;
        case G_PDBDATA :
             pp = SS_GET(g_pdbdata, obj);
             if (SS_consp(argl))
                {SX_GET_INTEGER_FROM_LIST(PD_tolerance, argl,
                                          "BAD PRECISION - SX_SHOW_PDB");};
             PD_write_entry(f0,
                            pp->file, pp->name,
                            pp->data, pp->ep);
             break;};

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_WR_TO_TEXTFILE - write content of a pdb object */

object *SX_wr_to_textfile(argl)
   object *argl;
   {object *obj;

    PC_io_connect(PC_LOCAL);   /* state = PC_REMOTE before this */
    obj = _SX_write_pdb(text_fp_f0, argl);

    PRINT(text_fp_f0, "\n");   /* to add newline for next data */

    PC_io_connect(PC_REMOTE);  /* back to the original state */

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SHOW_PDB - display content of a pdb object */

object *SX_show_pdb(argl)
   object *argl;
   {return(_SX_write_pdb(stdout, argl));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_TO_PDBDATA - return a pdbdata object from the arguments */

static object *SX_to_pdbdata(argl)
   object *argl;
   {SC_address val;
    syment *ep;

    ep = _SX_spec_instance(SX_vif, argl);

    val.diskaddr = PD_entry_address(ep);

    return(_SX_mk_gpdbdata("none", val.memaddr, ep, NULL));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DESC_VARIABLE - print out the syment in a object in human
 *                  - readable form.
 */

object *SX_desc_variable(obj)
   object *obj;
   {if (!SX_PDBDATAP(obj))
       return(SS_null);

    PD_write_syment(stdout, PDBDATA_EP(obj));

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MAKE_DIMS_DIMDES - take a list of integers and make a dimension
 *                     - descriptor
 *                     - (type dimdes)
 */

dimdes *_SX_make_dims_dimdes(file, argl)
   PDBfile *file;
   object *argl;
   {long mini, leng;
    object *dim_obj;
    dimdes *dims, *next, *prev;

    dims = NULL;
    for ( ; !SS_nullobjp(argl); argl = SS_cdr(argl))
        {dim_obj = SS_car(argl);
         if (SS_integerp(dim_obj))
            {mini = (long) file->default_offset;
             leng = SS_INTEGER_VALUE(dim_obj);}
         else if (SS_floatp(dim_obj))
            {mini = (long) file->default_offset;
             leng = SS_FLOAT_VALUE(dim_obj);}
         else if (SS_consp(dim_obj))
            {mini = SS_INTEGER_VALUE(SS_car(dim_obj));
             leng = SS_INTEGER_VALUE(SS_cdr(dim_obj)) - mini + 1;}
         else
            SS_error("DIMENSIONS MUST BE INTEGERS - _SX_MAKE_DIMS_DIMDES",
                     dim_obj);

         next = _PD_mk_dimensions(mini, leng);
         if (dims == NULL)
            dims = next;
         else
            {prev->next = next;
	     SC_mark(next, 1);};

         prev = next;};

    return(dims);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_MAKE_DIMS_OBJ - take an object and make a list of integers */

object *_SX_make_dims_obj(dims)
   dimdes *dims;
   {object *obj;

/* if not dimensions, return nothing */
    if (dims == NULL)
       return(SS_null);

    for (obj = SS_null; dims != NULL; dims = dims->next)
        obj = SS_mk_cons(SS_mk_cons(SS_mk_integer((BIGINT) dims->index_min), 
                                    SS_mk_integer((BIGINT) dims->index_max)),
                         obj);

    return(SS_reverse(obj));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_PDBDATA_TO_HASH - convert a pdbdata to a hash table.
 *                    - (pdbdata->hash data)
 */

object *SX_pdbdata_to_hash(arg)
   object *arg;
   {if (!SX_PDBDATAP(arg))
       return(SS_null);

    return(SS_mk_hash_table(PDBDATA_DATA(arg)));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_HASH_TO_PDBDATA - convert a hash table to a pdbdata object. 
 *                    - (hash->pdbdata hash-table file name)
 */

object *SX_hash_to_pdbdata(argl)
   object *argl;
   {HASHTAB *tab;
    char *name;
    g_file *po;
    PDBfile *file;

    tab  = NULL;
    file = NULL;
    name = NULL;
    SS_args(argl,
            G_FILE, &po,
            HASH_TABLE, &tab,
            SC_STRING_I, &name,
            0);

    if (po == NULL)
       SS_error("BAD FILE - SX_HASH_TO_PDBDATA", argl);

    file = FILE_FILE(PDBfile, po);

    if (tab == NULL)
       SS_error("BAD HASH TABLE - SX_HASH_TO_PDBDATA", argl);

    if (name == NULL)
       SS_error("BAD HASH TABLE NAME - SX_HASH_TO_PDBDATA", argl);

    return(SX_pdbdata_handler(file, name, "HASHTAB", tab, FALSE));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_PDBDATA_HANDLER - given a list with a PDB file object and a name,
 *                    - a type, and a pointer to some data
 *                    - encapsulate the pointer in a PDBDATA object and
 *                    - return a pointer to it
 */

object *SX_pdbdata_handler(file, name, type, vr, flag)
   PDBfile *file;
   char *name, *type;
   byte *vr;
   int flag;
   {syment *ep;
    SC_address data;
    object *ret;
    char fullpath[MAXLINE];
    int new;

    data.memaddr = (char *) vr;
    strcpy(fullpath, _PD_fixname(file, name));

/* if the next object is a pdbfile, use it, otherwise, use default file */
    if ((file == NULL) || (file->virtual_internal == TRUE))

/* if its an indirection, chances are high that vr is on the stack as something
 * like &f where f is a local variable
 * this had better be guarded against
 */
       {if (_PD_indirection(type))
           {char **p;
            p  = FMAKE(char *, "SX_PDBDATA_HANDLER:p");
            *p = DEREF(vr);
            data.memaddr = (char *) p;
	    SC_mark(*p, 1);
            SC_mark(p, 1);};

        ep = _PD_mk_syment(type, 1L, data.diskaddr, NULL, NULL);

        _PD_e_install(fullpath, ep, file->symtab, TRUE);}

    else
       {ep = _PD_write(file, fullpath, type, type,
		       data.memaddr, NULL, FALSE, &new);
        if (ep == NULL)
           SS_error(PD_err, SS_null);
	if (new)
	   ep  = PD_copy_syment(ep);

        if (flag)
           PD_reset_ptr_list(file);};
    
    ret = _SX_mk_gpdbdata(fullpath, data.memaddr, ep, file);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SET_SWITCH - set a switch */

object *SX_set_switch(argl)
   object *argl;
   {int indx, val;

    indx = -1;
    val  = -1;
    SS_args(argl,
            SC_INTEGER_I, &indx,
            SC_INTEGER_I, &val,
            0);

    if (indx < 0)
       SS_error("BAD INDEX - SX_SET_SWITCH", argl);

    PD_print_controls[indx] = val;

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SET_BUFFER_SIZE - set the buffer_size */

object *SX_set_buffer_size(argl)
   object *argl;
   {int v;

    v = -1;
    SS_args(argl,
            SC_INTEGER_I, &v,
            0);

    PD_set_buffer_size(v);

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_GET_BUFFER_SIZE - get the buffer_size */

object *SX_get_buffer_size()
   {BIGINT v;

    v = PD_get_buffer_size(v);

    return(SS_mk_integer(v));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_GET_ERROR - get the PD_err */

object *SX_get_error()
   {char *v;

    v = PD_get_error();

    return(SS_mk_string(v));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SET_MAX_SIZE - set the file->maximum_size */

object *SX_set_max_size(argl)
   object *argl;
   {int v;
    g_file *po;
    PDBfile *file;

    po = NULL;
    v  = -1;
    SS_args(argl,
            G_FILE, &po,
            SC_INTEGER_I, &v,
            0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    PD_set_max_file_size(file, v);

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SET_TRACK_POINTERS - set the file->track_pointers */

object *SX_set_track_pointers(argl)
   object *argl;
   {int v;
    g_file *po;
    PDBfile *file;

    po = NULL;
    v  = -1;
    SS_args(argl,
            G_FILE, &po,
            SC_INTEGER_I, &v,
            0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    PD_set_track_pointers(file, v);

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SET_FORMAT - set a format */

object *SX_set_format(argl)
   object *argl;
   {char *field, *format;

    field  = NULL;
    format = NULL;
    SS_args(argl,
            SC_STRING_I, &field,
            SC_STRING_I, &format,
            0);

    if (strcmp(field, "integer") == 0)
       {if (PD_user_formats1[0] != NULL)
           SFREE(PD_user_formats1[0]);
        PD_user_formats1[0] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");
        if (PD_user_formats2[0] != NULL)
           SFREE(PD_user_formats2[0]);
        PD_user_formats2[0] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");}

    else if (strcmp(field, "integer1") == 0)
       {if (PD_user_formats1[0] != NULL)
           SFREE(PD_user_formats1[0]);
        PD_user_formats1[0] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");}

    else if (strcmp(field, "integer2") == 0)
       {if (PD_user_formats2[0] != NULL)
           SFREE(PD_user_formats2[0]);
        PD_user_formats2[0] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");}

    else if (strcmp(field, "long") == 0)
       {if (PD_user_formats1[1] != NULL)
           SFREE(PD_user_formats1[1]);
        PD_user_formats1[1] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");
        if (PD_user_formats2[1] != NULL)
           SFREE(PD_user_formats2[1]);
        PD_user_formats2[1] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");}

    else if (strcmp(field, "long1") == 0)
       {if (PD_user_formats1[1] != NULL)
           SFREE(PD_user_formats1[1]);
        PD_user_formats1[1] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");}

    else if (strcmp(field, "long2") == 0)
       {if (PD_user_formats2[1] != NULL)
           SFREE(PD_user_formats2[1]);
        PD_user_formats2[1] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");}

    else if (strcmp(field, "float") == 0)
       {if (PD_user_formats1[2] != NULL)
           SFREE(PD_user_formats1[2]);
        PD_user_formats1[2] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");
        if (PD_user_formats2[2] != NULL)
           SFREE(PD_user_formats2[2]);
        PD_user_formats2[2] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");}

    else if (strcmp(field, "float1") == 0)
       {if (PD_user_formats1[2] != NULL)
           SFREE(PD_user_formats1[2]);
        PD_user_formats1[2] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");}

    else if (strcmp(field, "float2") == 0)
       {if (PD_user_formats2[2] != NULL)
           SFREE(PD_user_formats2[2]);
        PD_user_formats2[2] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");}

    else if (strcmp(field, "double") == 0)
       {if (PD_user_formats1[3] != NULL)
           SFREE(PD_user_formats1[3]);
        PD_user_formats1[3] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");
        if (PD_user_formats2[3] != NULL)
           SFREE(PD_user_formats2[3]);
        PD_user_formats2[3] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");}

    else if (strcmp(field, "double1") == 0)
       {if (PD_user_formats1[3] != NULL)
           SFREE(PD_user_formats1[3]);
        PD_user_formats1[3] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");}

    else if (strcmp(field, "double2") == 0)
       {if (PD_user_formats2[3] != NULL)
           SFREE(PD_user_formats2[3]);
        PD_user_formats2[3] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");}

    else if (strcmp(field, "short") == 0)
       {if (PD_user_formats1[4] != NULL)
           SFREE(PD_user_formats1[4]);
        PD_user_formats1[4] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");
        if (PD_user_formats2[4] != NULL)
           SFREE(PD_user_formats2[4]);
        PD_user_formats2[4] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");}

    else if (strcmp(field, "short1") == 0)
       {if (PD_user_formats1[4] != NULL)
           SFREE(PD_user_formats1[4]);
        PD_user_formats1[4] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");}

    else if (strcmp(field, "short2") == 0)
       {if (PD_user_formats2[4] != NULL)
           SFREE(PD_user_formats2[4]);
        PD_user_formats2[4] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");}

    else if (strcmp(field, "char") == 0)
       {if (PD_user_formats1[5] != NULL)
           SFREE(PD_user_formats1[5]);
        PD_user_formats1[5] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");
        if (PD_user_formats2[5] != NULL)
           SFREE(PD_user_formats2[5]);
        PD_user_formats2[5] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");}

    else if (strcmp(field, "char1") == 0)
       {if (PD_user_formats1[5] != NULL)
           SFREE(PD_user_formats1[5]);
        PD_user_formats1[5] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");}

    else if (strcmp(field, "char2") == 0)
       {if (PD_user_formats2[5] != NULL)
           SFREE(PD_user_formats2[5]);
        PD_user_formats2[5] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");}

    else if (strcmp(field, "user_int") == 0)
       {if (PD_user_formats1[7] != NULL)
           SFREE(PD_user_formats1[7]);
        PD_user_formats1[7] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");
        if (PD_user_formats2[7] != NULL)
           SFREE(PD_user_formats2[7]);
        PD_user_formats2[7] = SC_strsavef(format,
                              "char*:SX_SET_FORMAT:format");}

    else if (strcmp(field, "default") == 0)
       {if (PD_user_formats1[0] != NULL)
           SFREE(PD_user_formats1[0]);
        if (PD_user_formats1[1] != NULL)
           SFREE(PD_user_formats1[1]);
        if (PD_user_formats1[2] != NULL)
           SFREE(PD_user_formats1[2]);
        if (PD_user_formats1[3] != NULL)
           SFREE(PD_user_formats1[3]);
        if (PD_user_formats1[4] != NULL)
           SFREE(PD_user_formats1[4]);
        if (PD_user_formats1[5] != NULL)
           SFREE(PD_user_formats1[5]);

        if (PD_user_formats2[0] != NULL)
           SFREE(PD_user_formats2[0]);
        if (PD_user_formats2[1] != NULL)
           SFREE(PD_user_formats2[1]);
        if (PD_user_formats2[2] != NULL)
           SFREE(PD_user_formats2[2]);
        if (PD_user_formats2[3] != NULL)
           SFREE(PD_user_formats2[3]);
        if (PD_user_formats2[4] != NULL)
           SFREE(PD_user_formats2[4]);
        if (PD_user_formats2[5] != NULL)
           SFREE(PD_user_formats2[5]);}

    else
       SS_error("UNKNOWN TYPE - SX_SET_FORMAT", argl);

    SFREE(field);
    SFREE(format);

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_PDB_TYPE - make a type cons for make-pdbdata */

object *SX_pdb_type(argl)
   object *argl;
   {return(SS_mk_cons(SS_mk_string("type"), argl));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_UNP_BITSTRM - read a variable as a bitstream and unpack it into
 *                - the specified type in a numeric array
 *                -
 *                - Form: (pd-unpack-bitstream <file> <name> <type> <nbits>
 *                -                            <padsz> <fpp>)
 *                - 
 *                -  <type>  = data type of destination
 *                -  <nbits> = # bits per field
 *                -  <padsz> = # pad bits
 *                -  <fpp>   = # fields per pad
 *                -            <pad> <field> <field> <field> ... <pad> ...
 */

static object *SX_unp_bitstrm(argl)
   object *argl;
   {C_array *arr;
    g_file *po;
    char *name, *type, *data;
    PDBfile *file;
    int nbits, padsz, fpp;
    long anumb, offs, nitems;

    po    = NULL;
    name  = NULL;
    type  = NULL;
    nbits = 8;
    padsz = 0;
    fpp   = 1;
    offs  = 0L;
    SS_args(argl,
            G_FILE, &po,
            SC_STRING_I, &name,
            SC_STRING_I, &type,
            SC_LONG_I, &nitems,
            SC_INTEGER_I, &nbits,
            SC_INTEGER_I, &padsz,
            SC_INTEGER_I, &fpp,
            SC_LONG_I, &offs,
            0);

    if ((po == NULL) || (po == SX_gvif))
       file = SX_vif;

    else if (strcmp(po->type, SX_PDBFILE_S) == 0)
       file = FILE_FILE(PDBfile, po);

    else
       SS_error("BAD FILE - SX_UNP_BITSTRM", argl);

    arr = NULL;
    if (!_PD_rd_bits(file, name, type, nitems, FALSE, nbits, padsz, fpp, offs,
		     &anumb, &data))
       SS_error("_PD_RD_BITS FAILED - SX_UNP_BITSTRM", argl);

    arr = FMAKE(C_array, "SX_UNP_BITSTRM:arr");
    arr->type   = SC_strsavef(type, "char*:SX_UNP_BITSTRM:type");
    arr->length = anumb;
    arr->data   = data;

    return(SX_mk_C_array(arr));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_CONTAINER - function to hang on SC_contain_hook */

void SX_container(dtype, stype)
   char *dtype, *stype;
   {defstr *sp, *dp;
    PDBfile *file;
    g_file *po;

    dtype[0] = '\0';

    po = NULL;
    SS_args(SS_lk_var_val(SX_curfile, SS_Env),
	    G_FILE, &po,
	    0);

    if (po == NULL)
       file = SX_vif;
    else
       file = FILE_FILE(PDBfile, po);

    sp = PD_inquire_type(file, stype);
    if (sp != NULL)
       {dp = _PD_container_type(file, sp);
	if (dp != NULL)
	   strcpy(dtype, dp->type);
	else
	   strcpy(dtype, stype);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_CONVERT - function to hang on SC_convert_hook */

int SX_convert(dtype, pd, stype, s, n, flag)
   char *dtype;
   byte **pd;
   char *stype;
   byte *s;
   int n, flag;
   {int conv, ret, nb, nbi, fls, fld;
    long ino, outo;
    long *lt;
    char *d;
    double *dt;
    defstr *sp, *dp;
    HASHTAB *fc, *hc;
    data_standard *fs, *hs;
    PDBfile *file;
    g_file *po;
    SC_THREAD_ID(_t_index);

    ret = FALSE;

    if (strcmp(stype, dtype) == 0)
       {nbi  = SIZEOF(stype);
	conv = FALSE;}

    else
       {po = NULL;
	SS_args(SS_lk_var_val(SX_curfile, SS_Env),
		G_FILE, &po,
		0);

	if (po == NULL)
	   file = SX_vif;
	else
	   file = FILE_FILE(PDBfile, po);

	sp = PD_inquire_type(file, stype);
	if (sp == NULL)
	   return(ret);

	dp = _PD_container_type(file, sp);
	if (dp == NULL)
	   dp = PD_inquire_type(file, dtype);

	nbi  = dp->size;
	conv = TRUE;

	fls = (sp->format != NULL);
	fld = (dp->format != NULL);

/* if the source is floating point but not the destination */
	if (fls && !fld)
	   {lt = NULL;
	    SC_convert(SC_LONG_S, (byte **) &lt, stype, s, n, flag);

	    s     = (byte *) lt;
	    sp    = PD_inquire_type(file, SC_LONG_S);
	    stype = sp->type;

	    flag = TRUE;
	    conv = strcmp(dtype, SC_LONG_S);}

/* if the source is integral but not the destination */
	else if (fld && !fls)
	   {dt = NULL;
	    SC_convert(SC_DOUBLE_S, (byte **) &dt, stype, s, n, flag);

	    s     = (byte *) dt;
	    sp    = PD_inquire_type(file, SC_DOUBLE_S);
	    stype = sp->type;

	    flag = TRUE;
	    conv = strcmp(dtype, SC_DOUBLE_S);};};

/* the number of bytes we are dealing with */
    nb = n*nbi;

/* get the destination data setup */
    d = (char *) *pd;
    if (d == NULL)
       {d   = FMAKE_N(char, nb, "SX_CONVERT-B:d");
	*pd = (byte *) d;};

/* decide what to do with the data now that the types
 * have been sorted out
 */
    if (conv)

/* GOTCHA: PDB considers a char a NO_CONV type so we have to 
 *         treat character destinations specially for now
 *         we should probably just add a DEFIX type such as int1
 */
       {if (strcmp(dtype, SC_CHAR_S) == 0)
	   SC_convert(SC_CHAR_S, pd, stype, s, n, flag);

	else
	   {switch (setjmp(_PD_TRACE_ERR(_t_index)))
	       {case ABORT :
		     return(ret);
		case ERR_FREE :
		     return(ret);
		default :
		     memset(PD_err, 0, MAXLINE);
		     break;};

/* GOTCHA: this was this way for a long time and although it seems
 * pretty obviously wrong it worked for whatever cases went
 * through here
	    hc = file->chart;
	    fc = file->host_chart;
 */
	    fc = file->chart;
	    hc = file->host_chart;

	    fs = file->std;
	    hs = file->host_std;

	    ino = outo = 0L;

	    PD_convert(&d, (char **) &s,
		       stype, dtype, n, fs, hs, hs,
		       &ino, &outo,
		       fc, hc, 0, PD_TRACE);

	    ret = TRUE;};}

    else
       {memcpy(d, s, nb);

        ret = TRUE;};

    if (flag && (ret == 1))
       SFREE(s);

    return(ret);}

/*--------------------------------------------------------------------------*/

/*                         PDB FUNCTION INSTALLATION                        */

/*--------------------------------------------------------------------------*/
 
/* SX_INSTALL_PDB_FUNCS - install the PDB extensions to Scheme */
 
void SX_install_pdb_funcs()
   {

    SS_install("buffer-size",
               "Get the default buffer size for I/O files",
               SS_zargs,
               SX_get_buffer_size, SS_PR_PROC);

    SS_install("change-directory",
               "Change current working directory in a file",
               SS_nargs,
               SX_change_directory, SS_PR_PROC);

    SS_install("close-pdbfile",
               "Close a PDB file",
               SS_sargs,
               SX_close_pdbfile, SS_PR_PROC);

    SS_install("close-raw-binary-file",
               "Close a raw binary file",
               SS_nargs,
               SX_close_raw_file, SS_PR_PROC);

    SS_install("create-link",
               "Create a link to a variable in a file",
               SS_nargs,
               SX_create_link, SS_PR_PROC);

    SS_install("create-pdbfile",
               "Create a PDB file, return PDBfile",
               SS_sargs,
               SX_create_pdbfile, SS_PR_PROC);

    SS_install("current-directory",
               "Return the current working directory in a file",
               SS_sargs,
               SX_current_directory, SS_PR_PROC);

    SS_install("def-common-types",
               "Define some common internal SX data types to the given file",
               SS_sargs,
               SX_def_common_types, SS_PR_PROC);

    SS_install("def-member",
               "Create a member list",
               SS_nargs, 
               SS_list, SS_UR_MACRO);

    SS_install("def-primitive",
               "Define a new primitive type which is either: ncv, fix, fp",
               SS_nargs,
               SX_def_prim, SS_PR_PROC);

    SS_install("default-offset",
               "Set the default offset for the given file",
               SS_nargs,
               SX_default_offset, SS_PR_PROC);

    SS_install("defstr?",
               "Returns #t if the object is a defstr, and #f otherwise",
               SS_sargs, 
               SX_defp, SS_PR_PROC);

    SS_install("desc-variable",
               "Describe a pdb varible",
               SS_sargs, 
               SX_desc_variable, SS_PR_PROC);

    SS_install("diff-variable",
               "Compare an entry in two files\nFORM: (diff-variable <file1> <file2> <name>)",
               SS_nargs, 
               SX_diff_var, SS_PR_PROC);

    SS_install("display-differences",
               "Display information returned by diff-variable",
               SS_nargs, 
               SX_display_diff, SS_PR_PROC);

    SS_install("display-menu",
               "Display a menu of mappings, images, and curves",
               SS_nargs, 
               SX_menu, SS_PR_PROC);

    SS_install("pd-error-message",
               "Get the last PDB error message",
               SS_zargs,
               SX_get_error, SS_PR_PROC);

    SS_install("pd-entry-number",
               "Return the number of items for the named variabled",
               SS_nargs, 
               SX_entry_number, SS_PR_PROC);

    SS_install("family-file",
               "Check the file size and if necessary open the next family member",
               SS_nargs,
               SX_family_file, SS_PR_PROC);

    SS_install("file-info",
               "Return the named information about the given file",
               SS_nargs, 
               SX_file_info, SS_PR_PROC);

    SS_install("file-mode",
               "Set the mode of the given file",
               SS_nargs, 
               SX_file_mode, SS_PR_PROC);

    SS_install("file-variable?",
               "Return #t iff the named object is a variable in the given file",
               SS_nargs,
               SX_file_varp, SS_PR_PROC);

    SS_install("find-types",
               "Return a list of the derived data types in a variable",
               SS_sargs,
               SX_find_types, SS_PR_PROC);

    SS_install("flush-pdbfile",
               "Flush a PDB file",
               SS_sargs,
               SX_flush_pdbfile, SS_PR_PROC);

    SS_install("hash->pdbdata",
               "Convert a hash table to a pdbdata object",
               SS_nargs, 
               SX_hash_to_pdbdata, SS_PR_PROC);

    SS_install("indirection",
               "Create a type list",
               SS_nargs, 
               SS_list, SS_PR_PROC);

    SS_install("list-defstrs",
               "Return a list of the data types in a file",
               SS_nargs,
               SX_list_defstrs, SS_PR_PROC);

    SS_install("list-file",
               "Return a list of open pdbfiles",
               SS_zargs,
               SX_list_file, SS_PR_PROC);

    SS_install("list-symtab",
               "Return a list of the symbol table entries",
               SS_nargs,
               SX_list_symtab, SS_PR_PROC);

    SS_install("list-variables",
               "Return a list of the variables in a file directory",
               SS_nargs,
               SX_list_variables, SS_PR_PROC);

    SS_install("make-cast",
               "Specify that the real type of a member comes from another member",
               SS_nargs,
               SX_make_cast, SS_UR_MACRO);

    SS_install("make-cast*",
               "Specify that the real type of a member comes from another member",
               SS_nargs,
               SX_make_cast, SS_PR_PROC);

    SS_install("read-defstr",
               "Read a DEFSTR from the given file (macro version)",
               SS_nargs,
               SX_read_defstr, SS_UR_MACRO);

    SS_install("write-defstr",
               "Write a DEFSTR to the given file (macro version)",
               SS_nargs,
               SX_write_defstr, SS_UR_MACRO);

    SS_install("make-defstr",
               "Create a DEFSTR from the list (macro version)",
               SS_nargs,
               SX_make_defstr, SS_UR_MACRO);

    SS_install("read-defstr*",
               "Read a DEFSTR from the given file (procedure version)",
               SS_nargs,
               SX_read_defstr, SS_PR_PROC);

    SS_install("write-defstr*",
               "Write a DEFSTR to the given file (procedure version)",
               SS_nargs,
               SX_write_defstr, SS_PR_PROC);

    SS_install("make-defstr*",
               "Create a DEFSTR from the list (procedure version)",
               SS_nargs,
               SX_make_defstr, SS_PR_PROC);

    SS_install("make-directory",
               "Create a directory in a file",
               SS_nargs,
               SX_make_directory, SS_PR_PROC);

    SS_install("make-typedef",
               "Create a type alias for the given type (macro version)",
               SS_nargs,
               SX_make_typedef, SS_UR_MACRO);

    SS_install("make-typedef*",
               "Create a type alias for the given type (procedure version)",
               SS_nargs,
               SX_make_typedef, SS_PR_PROC);

    SS_install("major-order",
               "Set the given file to row or column major order",
               SS_nargs,
               SX_major_order, SS_PR_PROC);

    SS_install("open-pdbfile",
               "Open a PDB file, return PDBfile",
               SS_nargs,
               SX_open_pdbfile, SS_PR_PROC);

    SS_install("open-raw-binary-file",
               "Open a binary file as an input port",
               SS_nargs,
               SX_open_raw_file, SS_PR_PROC);

    SS_install("parse-declaration",
               "Parse a C style definition/declaration and return a list of elements",
               SS_nargs, 
               SX_parse_type, SS_PR_PROC);

    SS_install("pd-unpack-bitstream",
               "Read a variable from a file as a bit stream and unpack it",
               SS_nargs, 
               SX_unp_bitstrm, SS_PR_PROC);

    SS_install("pdb->list",
               "Convert some pdb type to a list",
               SS_sargs, 
               SX_pdb_to_list, SS_PR_PROC);

    SS_install("pdb-read-numeric-data",
               "Read a numeric array from a PDB file",
               SS_nargs,
               SX_read_numeric_data, SS_PR_PROC);

    SS_install("pdbdata",
               "Convert arguments to a pdb variable",
               SS_nargs, 
               SX_to_pdbdata, SS_PR_PROC);

    SS_install("pdbdata?",
               "Returns #t if the object is a pdbdata, and #f otherwise",
               SS_sargs, 
               SX_pdbdatap, SS_PR_PROC);

    SS_install("pdbdata->hash",
               "Convert a pdbdata object to a hash object",
               SS_sargs, 
               SX_pdbdata_to_hash, SS_PR_PROC);

    SS_install("pdbfile?",
               "Returns #t if the object is a pdb file, and #f otherwise",
               SS_sargs, 
               SX_pdbfp, SS_PR_PROC);

    SS_install("read-binary",
               "Read binary data from a raw binary file",
               SS_nargs,
               SX_rd_raw, SS_PR_PROC);

    SS_install("read-pdbdata",
               "Read PDB data from a file and encapsulate is as a PDBDATA object",
               SS_nargs,
               SX_read_pdbdata, SS_PR_PROC);

    SS_install("read-syment",
               "Get a syment from the specified file",
               SS_nargs,
               SX_rd_syment, SS_PR_PROC);

    SS_install("reset-pointer-list!",
               "Reset the pointer lists for the given file",
               SS_nargs,
               SX_reset_ptrs, SS_PR_PROC);

    SS_install("seek-file",
               "Seek in a binary file",
               SS_nargs,
               SX_seek_raw_file, SS_PR_PROC);

    SS_install("set-format",
               "Set an edit format",
               SS_nargs, 
               SX_set_format, SS_PR_PROC);

    SS_install("set-switch",
               "Set a code switch",
               SS_nargs, 
               SX_set_switch, SS_PR_PROC);

    SS_install("set-buffer-size!",
               "Set the default file buffer size",
               SS_nargs, 
               SX_set_buffer_size, SS_PR_PROC);

    SS_install("set-maximum-file-size!",
               "Set the maximum file size for a PDB file",
               SS_nargs, 
               SX_set_max_size, SS_PR_PROC);

    SS_install("set-track-pointers!",
               "Set the track_pointers flag for a PDB file",
               SS_nargs, 
               SX_set_track_pointers, SS_PR_PROC);

    SS_install("sizeof",
               "Return the size in bytes of the specified type\nUsage: (sizeof <type> [<file> [1]])",
               SS_nargs, 
               SX_sizeof, SS_PR_PROC);

    SS_install("show-pdb",
               "Display the contents of a PDB variable",
               SS_nargs, 
               SX_show_pdb, SS_PR_PROC);

    SS_install("open-text-file",
               "Open a text file",
               SS_nargs,
               SX_open_text_file, SS_PR_PROC);

    SS_install("close-text-file",
               "Close a text file",
               SS_nargs,
               SX_close_text_file, SS_PR_PROC);

    SS_install("wr-to-textfile",
               "Write the contents of a PDB variable to a text file",
               SS_nargs,
               SX_wr_to_textfile, SS_PR_PROC);

    SS_install("syment?",
               "Returns #t if the object is a syment, and #f otherwise",
               SS_sargs, 
               SX_symp, SS_PR_PROC);

    SS_install("target",
               "Set the data standard and alignment for the next file opened",
               SS_nargs,
               SX_target, SS_PR_PROC);

    SS_install("type",
               "Create a type list for make-pdbdata",
               SS_nargs, 
               SX_pdb_type, SS_PR_PROC);

    SS_install("write-binary",
               "Write binary data to a raw binary file",
               SS_nargs,
               SX_wr_raw, SS_PR_PROC);

    SS_install("write-pdbdata",
               "Write PDB data to a file and encapsulate is as a PDBDATA object",
               SS_nargs,
               SX_write_pdbdata, SS_PR_PROC);

    SS_install("write-syment",
               "Install a syment in the specified file",
               SS_nargs,
               SX_wr_syment, SS_PR_PROC);

    SS_install("write-ultra-curve",
               "Create an ULTRA curve",
               SS_nargs,
               SX_wrt_ultra_curve, SS_PR_PROC);


    SS_install_cf("string-mode",
                  "Variable: Mode for string reads - literal or nopadding",
                  SS_acc_int,
                  &SX_string_mode);

    SS_install_cf("comparison-precision",
		  "Variable: Comparison precision for floats",
		  SS_acc_int,
		  &PD_tolerance);

    SS_install_cf("display-individual-differences",
		  "Variable: Difference display mode flag",
		  SS_acc_int,
                  &SX_disp_individ_diff);
/*
    SS_install_cv("comparison-precision", &PD_tolerance, SC_INTEGER_I);

    SS_install_cv("display-individual-differences",
                  &SX_disp_individ_diff, SC_INTEGER_I);
*/
    SX_install_pdb_attr_funcs();

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
