;
; SXDM.SCM - define routines to compare PM_mappings in PDB files
;          - and display mappings whose difference measure exceeds
;          - the specified tolerance (default 1.0e-8)
;
; Source Version: 2.0
; Software Release #92-0043
;
; #include <pact-copyright.h>
; 

(load "pdbview.scm")

(define HUGE 1.0e100)
(define SMALL 1.0e-100)
(define TOLERANCE 1.0e-8)
(define PRECISION 1.0e15)

(define displays-off #t)
(define dev-a  (pg-make-device "unix:0.0" "COLOR" "New Result"))
(define dev-b  (pg-make-device "unix:0.0" "COLOR" "Standard Result"))
(define dev-c  (pg-make-device "unix:0.0" "COLOR" "Difference"))
;(define dev-d  (pg-make-device "unix:0.0" "COLOR" "Integral of Difference"))

(define contour    11)
(define image      12)
(define wire-frame 13)
(define shaded     14)
(define vector     15)

(define theta 45.0)
(define phi -45.0)

; --------------------------------------------------------------------------
; --------------------------------------------------------------------------

; DIFF-MEASURE - something like the chi squared between two mappings

(define (diff-measure a b)
   (let* ((tc (norm a))
	  (td (norm b))
	  (diff (norm (- a b)))
	  (den (+ tc td))

; diff variables
	  (domain-volume (pm-set-volume (pm-mapping-domain diff)))
	  (diff-range-volume (pm-set-volume (pm-mapping-range diff)))

; den variables
	  (ave-range-volume (pm-set-volume (pm-mapping-range den)))

	  meas integ)

     (cond ((eqv? diff-range-volume 0.0)
	    (list 0.0 nil nil))

	   ((or (> (- diff-range-volume ave-range-volume) PRECISION)
		(> diff-range-volume HUGE)
		(> ave-range-volume HUGE))
	    (list PRECISION nil nil))

	   (#t (set! meas (/ diff (pm-shift-range den SMALL)))
	       (set! integ (integrate meas))
	       (list (/ (pm-set-volume (pm-mapping-range integ))
			domain-volume)
		     meas integ)))))

; --------------------------------------------------------------------------
; --------------------------------------------------------------------------

; DIFF-MAPPINGS - the read-eval-print loop for the validation process
;               -
;               - the special commands are:
;               -
;               - NEXT/NO - these curves don't compare and go on to the next pair
;               - OK/YES  - these curves do compare and go on to the next pair
;

(define (diff-mappings file1 file2)
  (let* ((correct 0)
	 (differ nil))

    (define (get-response n a dc frst)
        (if frst
	    (begin
	        (printf nil
			"\n[ Map# | Frac Diff | Label ]->  y-accept, n-reject, (command)\n")
		(set! frst #f)))
        (printf nil  "\n[%4d | %10.3e | %s ]-> "
		n dc a)
	(let (form fl)
	     (set! form (read))
	     (set! fl (substring (sprintf "%s" form) 0 1))
	     (cond ((eqv? fl "n")
		    (set! differ (cons n differ)))
		   ((eqv? fl "y")
		    (set! correct (+ correct 1)))
		   (#t
		    (interactive 1)
		    (eval form)
		    (interactive 0)
		    (set! frst (get-response n a dc frst)))))
	frst)

    (define (diff-mappings-aux n file1 file2 frst)
        (let ((a (pdbdata->pm-mapping file1 n))
	      (b (pdbdata->pm-mapping file2 n))
	      c d dc form)
	     (if (and (not (null? a)) (not (null? b))
		      (not (pm-grotrian-mapping? a))
		      (not (pm-grotrian-mapping? b)))
		 (begin
		    (set! form (diff-measure a b))
		    (set! dc (car form))
		    (set! c  (cadr form))
		    (set! d  (caddr form))
		    (if (< dc TOLERANCE)
			(begin
			  (set! correct (+ correct 1))
			  (diff-mappings-aux (+ n 1) file1 file2 frst))
			(begin
			  (if displays-off
			      (open-devices))
			  (show-mappings a b c d)
			  (set! frst (get-response n (pm-mapping-name a) dc frst))
			  (diff-mappings-aux (+ n 1)
					     file1 file2 frst)))))))

    (diff-mappings-aux 1 file1 file2 #t)
    (if (not displays-off)
	(close-devices))
    (newline)
    (list correct differ)))

; --------------------------------------------------------------------------
; --------------------------------------------------------------------------

; OPEN-DEVICES - open four devices to show the mapping from the
;              - new and old files, the difference, and the integral
;              - of the difference

(define-macro (open-devices)
    (nxm 2 2 "New Result" "Std Result" "Diff" "Int")
    (set! displays-off #f))

; --------------------------------------------------------------------------
; --------------------------------------------------------------------------

; CLOSE-DEVICES - close the four devices

(define-macro (close-devices)
   (clw)
   (set! displays-off #t))

; --------------------------------------------------------------------------
; --------------------------------------------------------------------------

; SHOW-MAPPINGS - display the four mappings

(define (show-mappings a b c d)
   (let* ((dev (window-device current-window)))
        (cv "New Result")
	(dl 1)
	(if (not (null? a))
	    (begin (display-mapping* a)
		   (viewport-update)))
	(cv "Std Result")
	(dl 1)
	(if (not (null? b))
	    (begin (display-mapping* b)
		   (viewport-update)))
	(cv "Diff")
	(dl 1)
	(if (not (null? c))
	    (begin (display-mapping* c)
		   (viewport-update)))
	(cv "Int")
	(dl 1)
	(if (not (null? d))
	    (begin (display-mapping* d)
		   (viewport-update)))))

; --------------------------------------------------------------------------
; --------------------------------------------------------------------------

; COMPARE - read in two files and display each pair of matching curves
;          - as prompted
;          - when all have been checked kill all of them and erase the
;          - screen
;          - there is some development to be done here

(define-macro (compare new old . rest)
  (if (and (ultra-file? new) (ultra-file? old))
      (system (sprintf "pdbdiff -u %s %s" new old))
      (if (ascii-file? new)
	  (printf nil "File %s is ascii\n\n" new)
	  (if (ascii-file? old)
	      (printf nil "File %s is ascii\n\n" old)
	      (let* (n-total n-correct n-differ differing result
		     (tolerance (if rest
				    (list-ref rest 0)
				    TOLERANCE))
		     (file1 (open-pdbfile new))
		     (file2 (open-pdbfile old))
		     (snew (if (symbol? new)
			       (symbol->string new)
			       new))
		     (file (string-append snew ".chk"))
		     (log (open-output-file file)))

		(set! TOLERANCE tolerance)

		(define-global current-file file1)

		(define (report-fail file differing n-differ n-total new old)
		  (printf file
			  "%d mapping(s) out of %d differ between %s and %s:\n"
			  n-differ n-total new old)
		  (for-each '(lambda (n) (printf file " %d" n))
			    differing)
		  (printf file "\n\n"))

		(define (report-success file new old tol)
		  (printf file "All mappings in %s and %s agree to within one part in %s\n\n"
			  new old tol))

		(printf nil "\nComparing %s and %s\n" new old)
		(set! result (diff-mappings file1 file2))
		(set! n-correct (car result))
		(set! differing (reverse (cadr result)))
		(set! n-differ (if (pair? differing) (length differing) 0))
		(set! n-total (+ n-correct n-differ))
		(if (pair? differing)

; some differ
		    (begin
		      (report-fail log differing n-differ n-total new old)
		      (report-fail nil differing n-differ n-total new old))

; all match
		    (begin
		      (report-success log new old TOLERANCE)
		      (report-success nil new old TOLERANCE)))

		(close-output-file log)
		(= n-differ 0))))))

; --------------------------------------------------------------------------
; --------------------------------------------------------------------------

