/*
 * PCEXEC.C - Portable Process Control system exec utility
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "ppc.h"

static PROCESS
 *pp = NULL;

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* CHILD_HAS_TXT - handle text from the child process */

static void child_has_txt(fd)
   int fd;
   {char s[MAXLINE];

    while (PC_gets(s, MAXLINE, pp) != NULL)
       PRINT(stdout, "%s", s);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TTY_HAS_TXT - handle text from the tty */

static void tty_has_txt(fd)
   int fd;
   {char s[MAXLINE];

    if (fgets(s, LRG_TXT_BUFFER, stdin) != NULL)
       PC_printf(pp, "%s", s);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PROCESS_END - return TRUE if the process has ended
 *             - do the PC_close now since the TTY may be in
 *             - RAW mode and we want the output to
 *             - look nice (PTY's do this)
 */

static int process_end(name, pr, quiet)
   char *name;
   int *pr, quiet;
   {int status, reason;

    if (PC_status(pp) != RUNNING)
       {status = pp->status;
        reason = pp->reason;

        *pr = reason;

/* get anything remaining from the child */
        child_has_txt(pp->in);

        PC_close(pp);

        if (!quiet)
           PRINT(stdout, "\nProcess %s terminated (%d %d)\n",
	         name, status, reason);

        return(TRUE);}

    else
       return(FALSE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* POLL_MODE - poll the process and the tty */

static int poll_mode(name, quiet)
   char *name;
   int quiet;
   {char s[LRG_TXT_BUFFER], *t;
    int ret, err;

    ret = -1;
    while (TRUE)
       {while (PC_gets(s, LRG_TXT_BUFFER, pp) != NULL)
           PRINT(stdout, "%s", s);

        if (process_end(name, &ret, quiet))
           break;

        err = PC_unblock_file(stdin);
        PC_err[0] = '\0';

        t = fgets(s, LRG_TXT_BUFFER, stdin);

        err = PC_block_file(stdin);
        PC_err[0] = '\0';

        if (t != NULL)
           PC_printf(pp, "%s", s);

        if (PC_err[0] != '\0')

/* close now since the TTY may be in RAW mode and we want the output to
 * look nice (PTY's do this)
 */
           {PC_close(pp);
	    if (!quiet)
               PRINT(stdout, "%s\n\n", PC_err);
            break;};};

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* INTERRUPT_MODE - handle the child and tty I/O via interrupts */

static int interrupt_mode(name, quiet)
   char *name;
   int quiet;
   {int pi, ret;

    PC_io_interrupts_on = TRUE;

    pi  = PC_io_callback_file(stdin, tty_has_txt);
    pi &= PC_io_callback_fd(pp->in, child_has_txt);

    PC_io_interrupts_on = pi;

    if (pi)
       PC_catch_io_interrupts(PC_io_interrupts_on);

    ret = -1;
    while (TRUE)
       {if (process_end(name, &ret, quiet))
	   break;

	PC_poll_descriptors(-1);};

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* ERROR_HANDLER - trap various signals and restore the terminal */

static void error_handler(sig)
   int sig;
   {int err;

    if (PC_original_terminal_state != NULL)
       {PC_set_term_state(PC_original_terminal_state, -1);
        SFREE(PC_original_terminal_state);};

    err = PC_block_file(stdin);

    if (sig == SIGALRM)
       {PRINT(stdout, "PCEXEC timed out\n");
	exit(123);}
    else
       {PRINT(stdout, "PCEXEC exiting with signal %d\n", sig);
	exit(1);};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* MAIN - test PPC */

int main(argc, argv, envp)
   int argc;
   char **argv, **envp;
   {int i, interrupts, ret, quiet, access_file, log, parallel, to;
    char mode[5];

/* process the command line arguments */
    quiet       = FALSE;
    interrupts  = TRUE;
    parallel    = FALSE;
    access_file = FALSE;
    log         = FALSE;
    to          = 1000000;
    strcpy(mode, "w");
    if (argc < 2)
       {printf("\nUsage: pcexec [-pst] [-c n] [-i] <prog> [<arg1> ...]\n");
        printf("       pcexec -f <host>\n\n");
        printf("       pcexec -r [<arg1> ...]\n\n");
        printf("   The first form runs <prog> as a child process.\n");
        printf("   The second form acts as a remote file access server.\n");
        printf("   The third form acts as a parallel communications server.\n");
        printf("   The forms for <prog> are:\n");
        printf("       <name>              - run <name> on local <host>\n");
        printf("       <host>:<name>       - run <name> on remote <host>\n");
        printf("       <CPU>@<name>        - run <name> on processor <CPU>\n");
        printf("       <host>:<CPU>@<name> - run <name> on processor <CPU>\n");
        printf("   The last two are not yet completed.\n");
        printf("\n");
        printf("   The full syntax for <host> is:\n");
        printf("       <hostname>                         or\n");
        printf("       <hostname>,<username>              or\n");
        printf("       <hostname>,<username>,<passwd>\n");
        printf("   Note: whitespace is NOT allowed\n");
        printf("   Options:\n");
        printf("      c - timeout after n seconds\n");
        printf("      i - poll explicitly instead of using system call\n");
        printf("      l - when acting as a file server, log transactions to\n");
        printf("          PC_fs.log in your home directory\n");
        printf("\n");
        printf("      p - use pipes for communications\n");
        printf("      q - print only messages from the child\n");
        printf("      s - use sockets for communications\n");
        printf("      t - use pseudo terminals for communications\n");
        printf("   These only apply to processes on the same CPU.\n");
        printf("\n");

        return(1);};

    for (i = 1; argv[i] != NULL; i++)
        {if (argv[i][0] == '-')
            {switch (argv[i][1])
                {case 'c' : to = SC_stoi(argv[++i]);
                            break;
                 case 'f' : access_file = TRUE;
                            break;
                 case 'i' : interrupts = FALSE;
                            break;
                 case 'l' : log = TRUE;
                            break;
                 case 'r' : parallel = TRUE;
		            i++;
		            break;
                 case 'p' : strcpy(mode, "wp");
                            break;
                 case 'q' : quiet = TRUE;
                            break;
                 case 's' : strcpy(mode, "ws");
                            break;
                 case 't' : strcpy(mode, "wt");
                            break;};}
         else
            break;};

/* trap the following signals to restore the terminal state */

#ifdef UNIX
    SIGNAL(SIGSEGV, error_handler);
    SIGNAL(SIGABRT, error_handler);
    SIGNAL(SIGTERM, error_handler);
    SIGNAL(SIGQUIT, error_handler);
    SIGNAL(SIGILL, error_handler);
    SIGNAL(SIGINT, error_handler);
#endif

    if (access_file)
       ret = PC_file_access(log);

    else if (parallel)
       ret = PC_process_access(argv, "rsb+");

    else
       {if ((strcmp(argv[i], "ftp") == 0) ||
	    (strcmp(argv[i], "telnet") == 0))
	   {if (strcmp(mode, "wt") != 0)
	       {if (!quiet)
		   PRINT(stdout,
			 "\nWarning: run with -t flag\n\n");};};

/* print this before we potentially go into RAW mode (PTY's do this) */
	if (!quiet)
	   PRINT(stdout, "\nRunning process: %s\n\n", argv[i]);

/* set the alarm */
        PC_alarm(to, error_handler);

	pp = PC_open(argv+i, envp, mode);
	if (pp == NULL)
	   {if (!quiet)
	       {PRINT(stdout, "%s\n", PC_err);
		PRINT(stdout, "\nFailed to open: %s\n\n", argv[i]);};
	    error_handler(0);};

/* reset the alarm */
        PC_alarm(0, NULL);

	SC_setbuf(stdout, NULL);

/* set the alarm */
        PC_alarm(to, error_handler);

	if (interrupts)
	   ret = interrupt_mode(argv[i], quiet);

	else
	   ret = poll_mode(argv[i], quiet);

/* reset the alarm */
        PC_alarm(0, NULL);

	if (!quiet)
	   PRINT(stdout, "Process test %s ended\n\n", argv[i]);};

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
