/*
 * GSSCAT.C - scatter plot rendering routines for PGS
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pgs.h"

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_SETUP_POINTS - limit the range and domain then allocate
 *                  - and return spaces to hold the correct number
 *                  - of points
 */

static void _PG_setup_points(pnp, px, py, pz, rl, dl, rx, ry, rf)
   int *pnp;
   REAL **px, **py, **pz;
   REAL *rl, *dl, *rx, *ry, *rf;
   {int i, n, np;
    REAL xmn, xmx, ymn, ymx, fmn, fmx;
    REAL xc, yc, fc;
    REAL *x, *y, *z;

    np = *pnp;
    
    if (dl == NULL)
       {xmn = -HUGE;
	xmx = HUGE;
	ymn = -HUGE;
	ymx = HUGE;}
    else
       {xmn = dl[0];
	xmx = dl[1];
	ymn = dl[2];
	ymx = dl[3];};

    if (rl == NULL)
       {fmn = -HUGE;
	fmx = HUGE;}
    else
       {fmn = rl[0];
	fmx = rl[1];};

    x = FMAKE_N(REAL, np, "_PG_SETUP_POINTS:x");
    y = FMAKE_N(REAL, np, "_PG_SETUP_POINTS:y");
    z = FMAKE_N(REAL, np, "_PG_SETUP_POINTS:z");

    n = 0L;
    for (i = 0L; i < np; i++)
        {xc = rx[i];
	 yc = ry[i];
	 fc = rf[i];

	 if ((xmn <= xc) && (xc <= xmx) &&
	     (ymn <= yc) && (yc <= ymx) &&
	     (fmn <= fc) && (fc <= fmx))
	    {x[n] = xc;
	     y[n] = yc;
	     z[n] = fc;
	     n++;};};

    REMAKE_N(x, REAL, n);
    REMAKE_N(y, REAL, n);
    REMAKE_N(z, REAL, n);

    *pnp = n;
    *px  = x;
    *py  = y;
    *pz  = z;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DRAW_SCATTER - draw a scatter plot given
 *                  - a scalar array and view angle
 */

static void _PG_draw_scatter(dev, nd, f, rextr, dextr, x, y, nn,
			     theta, phi, chi)
   PG_device *dev;
   int nd;
   REAL **f, *rextr, *dextr, *x, *y;
   int nn;
   double theta, phi, chi;
   {int i, j, k, ig, npp;
    int color;
    REAL xc, yc, fc, z1, z2, t;
    REAL p1[9], p2[9], p3[9], lextr[2];
    REAL *tx, *ty, *tz;
    REAL *rx, *ry, *rz, *ppx, *ppy;
    PG_palette *pl;

    pl = dev->current_palette;

    nn = max(nn, 12);

/* allocate space for the range/domain limited points */
    _PG_setup_points(&nn, &tx, &ty, &tz, rextr, dextr, x, y, f[0]);

/* allocate space for the rotated points */
    rx = FMAKE_N(REAL, nn, "_PG_DRAW_SCATTER:rx");
    ry = FMAKE_N(REAL, nn, "_PG_DRAW_SCATTER:ry");
    rz = FMAKE_N(REAL, nn, "_PG_DRAW_SCATTER:rz");

/* rotate X into rX */
    PG_rotate_3d_WC(tx, ty, tz,
		    dextr[0], dextr[1], dextr[2], dextr[3],
		    rextr[0], rextr[1],
                    theta, phi, chi, rx, ry, rz, nn, TRUE);

/* determine the WC including the axis limits */
    PG_axis_3d_limits(dev, tx, ty, tz, nn, theta, phi, chi,
		      TRUE, FALSE, TRUE,
		      dextr[0], dextr[1], dextr[2], dextr[3],
		      rextr[0], rextr[1],
		      p1, p2, p3);

    PM_convex_hull(p1, p2, 8, &ppx, &ppy, &npp);

    PG_set_clipping(dev, FALSE);

/* sort f, rx, and ry into ascending rz order */
    for (ig = nn >> 1; ig > 0; ig >>= 1)
        for (i = ig; i < nn; i++)
            for (j = i-ig; j >= 0; j -= ig)
	        {k = j + ig;
		 z1 = rz[j];
		 z2 = rz[k];
		 if (z1 <= z2)
                    break;

		 rz[j] = z2;
		 rz[k] = z1;

		 t     = rx[j];
		 rx[j] = rx[k];
		 rx[k] = t;

		 t     = ry[j];
		 ry[j] = ry[k];
		 ry[k] = t;};

    lextr[0] = rz[0];
    lextr[1] = rz[nn-1];

/* draw a dot at each point */
    for (i = 0; i < nn; i++)
        {xc = rx[i];
	 yc = ry[i];
	 fc = rz[i];

	 color = PG_select_color(dev, 1, &fc, lextr);
         
	 PG_set_color_line(dev, color, FALSE);

	 PG_draw_markers(dev, 1, &xc, &yc, 1);};

    PG_set_palette(dev, "standard");
    PG_set_line_color(dev, dev->WHITE);

    PG_draw_polyline(dev, ppx, ppy, npp, FALSE);

    dev->current_palette = pl;

/* release the temporaries */
    SFREE(tx);
    SFREE(ty);
    SFREE(tz);

    SFREE(rx);
    SFREE(ry);
    SFREE(rz);

    SFREE(ppx);
    SFREE(ppy);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SCATTER_HAND - draw a scatter plot */

static void PG_scatter_hand(dev, g)
   PG_device *dev;
   PG_graph *g;
   {int i, n_nodes, npts, nd, same;
    int centering, rexfl;
    char bf[MAXLINE], *mtype, *s, *pn;
    REAL theta, phi, chi, dl[4];
    REAL *rx, *ry, *pm;
    REAL **r, *dextr, *rextr, **afd;
    byte **afs;
    PM_mapping *h;
    PM_set *domain, *range;
    pcons *alst, *info;

    pn   = NULL;
    pm   = NULL;
    info = (pcons *) g->info;
    if ((info != NULL) && (strcmp(g->info_type, SC_PCONS_P_S) == 0))
       {REAL *pt, *pp, *px;

	pt = NULL;
	pp = NULL;
	px = NULL;
        SC_assoc_info(info,
                      "THETA", &pt,
                      "PHI", &pp,
                      "CHI", &px,
		      "PALETTE", &pn,
		      "MARKER-SCALE", &pm,
                      NULL);

	theta = (pt == NULL) ? 0.0 : *pt;
	phi   = (pp == NULL) ? 0.0 : *pp;
	chi   = (px == NULL) ? 0.0 : *px;}

    else
       {theta = 0.0;
	phi   = 0.0;
	chi   = 0.0;};

    if (pn != NULL)
       PG_set_palette(dev, pn);

    if (pm != NULL)
       PG_set_marker_scale(dev, *pm);

    h      = g->f;
    domain = h->domain;

    strcpy(bf, domain->element_type);
    mtype   = SC_strtok(bf, " *", s);
    npts    = domain->n_elements;
    r       = (REAL **) domain->elements;
    n_nodes = npts;

    rx = PM_array_real(domain->element_type, r[0], npts, NULL);
    ry = PM_array_real(domain->element_type, r[1], npts, NULL);

/* setup the domain limits */
    if (dev->autodomain)
       {info = (pcons *) g->info;
	SC_assoc_info(info,
		      "LIMITS", &dextr,
		      NULL);
	if (dextr == NULL)
	   dextr = PM_array_real(domain->element_type, domain->extrema, 4, NULL);}
    else
       {PG_get_viewport_WC(dev, dl, dl+1, dl+2, dl+3);
	dextr = dl;};

    range = h->range;
    strcpy(bf, range->element_type);
    mtype = SC_strtok(bf, " *", s);
    same  = (strcmp(mtype, SC_REAL_S) == 0);
    npts  = range->n_elements;
    nd    = range->dimension_elem;

    afd   = FMAKE_N(REAL *, nd, "PG_SCATTER_HAND:afd");
    afs   = (byte **) range->elements;

/* find the range limits if any */
    rextr = PM_get_limits(range);

/* setup the range limits */
    rexfl = (rextr == NULL);
    if (same)
       {if (rexfl)
           rextr = (REAL *) range->extrema;
        for (i = 0; i < nd; i++)
            afd[i] = (REAL *) afs[i];}
    else
       {for (i = 0; i < nd; i++)
            CONVERT(SC_REAL_S, &afd[i], mtype, afs[i], npts, FALSE);

        if (rexfl)
           {rextr = NULL;
	    CONVERT(SC_REAL_S, &rextr, mtype, range->extrema,
		    2*nd, FALSE);};};

    PG_register_range_extrema(dev, nd, rextr);

/* find the additional mapping information */
    centering = N_CENT;
    alst = PM_mapping_info(h,
			   "CENTERING", &centering,
			   NULL);

    if (centering == N_CENT)
       _PG_draw_scatter(dev, nd, afd, rextr, dextr,
			rx, ry, n_nodes,
			theta, phi, chi);

    SFREE(rx);
    SFREE(ry);

    if (!same)
       {for (i = 0; i < nd; i++)
            SFREE(afd[i]);

        if (rexfl)
           SFREE(rextr);};

    SFREE(afd);
    SFREE(dextr);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SETUP_PICTURE_SCATTER - setup a window for a scatter rendering
 *                          - NOTE: no drawing of any kind is to be done here
 */

PG_picture_desc *PG_setup_picture_scatter(dev, data, save, clear)
   PG_device *dev;
   PG_graph *data;
   int save, clear;
   {int rendering, nde, nre, change;
    REAL xend;
    REAL *dpex, *ddex, *pdx, *rpex, *rdex, *prx;
    PG_picture_desc *pd;
    PG_par_rend_info *pri;
    PG_device *dd;
    pcons *alst;

    change = !dev->supress_setup;

    pd = PG_get_rendering_properties(dev, data);

    pd->legend_fl = FALSE;

    rendering = pd->rendering;
    alst      = pd->alist;
    pri       = dev->pri;
    if (pri != NULL)
       {dd = pri->dd;
	if (dd != NULL)
	   {dd->pri->alist  = alst;
	    dd->pri->render = rendering;};};

/* setup the viewport */
    if (change)
       {xend = 0.725 - _PG_palette_width;
	PG_set_viewport(dev, 0.175, xend, 0.175, 0.825);};

    if (change)

/* find the extrema for this frame */
       {PG_find_extrema(data, 0.0, &dpex, &rpex, &nde, &ddex, &nre, &rdex);

/* setup the domain limits */
	pdx = ((dev->autodomain == TRUE) || (dpex == NULL)) ? ddex : dpex;
	PG_set_window(dev, pdx[0], pdx[1], pdx[2], pdx[3]);

/* setup the range limits */
	prx = ((dev->autorange == TRUE) || (rpex == NULL)) ? rdex : rpex;
	PG_register_range_extrema(dev, nre, prx);

	SFREE(ddex);
	SFREE(rdex);

/* GOTCHA: this isn't the only time we want 3d axes */
	if ((pd->theta != HUGE) || (nde == 3))
	   pd->ax_type = CARTESIAN_3D;

/* set scatter plot attribute values */
	PG_set_clipping(dev, FALSE);};

    return(pd);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_SCATTER_PLOT - main scatter plot control routine */

#ifdef PCC

void PG_scatter_plot(dev, data, va_alist)
   PG_device *dev;
   PG_graph *data;
   va_dcl

#endif

#ifdef ANSI

void PG_scatter_plot(PG_device *dev, PG_graph *data, ...)

#endif

   {PG_graph *g;
    PG_picture_desc *pd;

    data->rendering = PLOT_SCATTER;

    pd = PG_setup_picture(dev, data, TRUE, TRUE, TRUE);

    switch (PG_hl_clear_mode)
       {case CLEAR_SCREEN :
             PG_clear_window(dev);
             break;
        case CLEAR_VIEWPORT :
             PG_clear_viewport(dev);
             break;
        case CLEAR_FRAME :
             PG_clear_frame(dev);
             break;};

/* plot all of the current functions */
    for (g = data; g != NULL; g = g->next)
        PG_scatter_hand(dev, g);

    PG_finish_picture(dev, data, pd);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_SCATTER - draw a scatter plot given a known mesh type, a
 *                 - suitable scanning function, and a scalar array
 *                 - rotate to view angle, plot points
 */

void PG_draw_scatter(dev, a1, a2, nn, ne, x,
		     theta, phi, chi, width, color, style,
		     type, name, alist)
   PG_device *dev;
   REAL *a1, *a2;
   int nn, ne;
   REAL **x;
   double theta, phi, chi, width;
   int color, style, type;
   char *name;
   pcons *alist;
   {int diml[1];
    PM_set *domain, *range;
    PG_graph *g;
    pcons *alst;

/* build the domain set */
    diml[0] = nn;
    domain = PM_make_set_alt("{D}", SC_REAL_S, FALSE,
			     1, diml, ne, (byte **) x);

/* build the range set */
    if (a2 == NULL)
       range = PM_make_set("{a}", SC_REAL_S, FALSE, 1, nn, 1, a1);
    else
       range = PM_make_set("{u,v}", SC_REAL_S, FALSE, 1, nn, 2, a1, a2);

/* add the attributes */
    alst = SC_copy_alist(alist);
    SC_ADD_VALUE_ALIST(alst, REAL, SC_REAL_P_S, "THETA", theta);
    SC_ADD_VALUE_ALIST(alst, REAL, SC_REAL_P_S, "PHI", phi);
    SC_ADD_VALUE_ALIST(alst, REAL, SC_REAL_P_S, "CHI", chi);
    
    SC_ADD_VALUE_ALIST(alst, int, SC_INTEGER_P_S, "LINE-COLOR", color);
    SC_ADD_VALUE_ALIST(alst, int, SC_INTEGER_P_S, "LINE-STYLE", style);
    SC_ADD_VALUE_ALIST(alst, REAL, SC_REAL_P_S, "LINE-WIDTH", width);

    g = PG_make_graph_from_sets(name, domain, range, N_CENT,
				SC_PCONS_P_S, alst, 'A', NULL);

    g->rendering = PLOT_SCATTER;

    PG_scatter_plot(dev, g);

    PG_rl_graph(g, FALSE, FALSE);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
