/*
 * GSDVB.C - discrete value boundary rendering routines for PGS
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pgs.h"

typedef struct s_PG_dvb_data PG_dvb_data;

struct s_PG_dvb_data
   {PG_device *dev;
    int nd;
    int *a;
    int *aext;
    REAL *x;
    REAL *y;
    byte *cnnct;
    pcons *alist;};

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SETUP_PICTURE_DV_BND - setup a window for a dvb rendering
 *                         - NOTE: no drawing of any kind is to be done here
 */

PG_picture_desc *PG_setup_picture_dv_bnd(dev, data, save, clear)
   PG_device *dev;
   PG_graph *data;
   int save, clear;
   {int nde, nre, change;
    REAL nvxmn, nvxmx, nvymn, nvymx;
    REAL *dpex, *ddex, *pdx, *rpex, *rdex, *prx, *vwprt;
    PG_picture_desc *pd;
    PG_par_rend_info *pri;
    PG_device *dd;
    pcons *alst;

    change = !dev->supress_setup;

    pd = PG_get_rendering_properties(dev, data);

    pd->legend_fl  = FALSE;
    pd->palette_fl = FALSE;

    alst = pd->alist;
    pri  = dev->pri;
    if (pri != NULL)
       {dd = pri->dd;
	if (dd != NULL)
	   {dd->pri->alist  = alst;
	    dd->pri->render = PLOT_DV_BND;};};

/* setup the viewport */
    vwprt = pd->viewport;
    if (vwprt != NULL)
       {nvxmn = vwprt[0];
        nvxmx = vwprt[1];
        nvymn = vwprt[2];
        nvymx = vwprt[3];}
    else
       {nvxmn = 0.175;
	nvxmx = 0.825;
	nvymn = 0.175;
	nvymx = 0.825;};

    if (change)
       {PG_set_viewport(dev, nvxmn, nvxmx, nvymn, nvymx);

/* find the extrema for this frame */
	PG_find_extrema(data, 0.0, &dpex, &rpex, &nde, &ddex, &nre, &rdex);

/* setup the domain limits */
	pdx = ((dev->autodomain == TRUE) || (dpex == NULL)) ? ddex : dpex;
	PG_set_window(dev, pdx[0], pdx[1], pdx[2], pdx[3]);

/* setup the range limits */
	prx = ((dev->autorange == TRUE) || (rpex == NULL)) ? rdex : rpex;
	PG_register_range_extrema(dev, nre, prx);

	SFREE(ddex);
	SFREE(rdex);};

    return(pd);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DVB_HAND - setup and dispatch to the correct DVB renderer */

static void PG_dvb_hand(dev, g, fnc_zc, fnc_nc)
   PG_device *dev;
   PG_graph *g;
   PFByte fnc_zc, fnc_nc;
   {int npts, centering, clip, prec, same;
    int *afd, *aext;
    char bf[MAXLINE], *mtype, *s;
    REAL charspace, chupx, chupy;
    REAL chpthx, chpthy;
    REAL *rx, *ry, *afs, **r;
    REAL dextr[4], *fextr, *rextr, *aextr;
    byte *cnnct;
    PM_mapping *h;
    PM_set *domain, *range;
    PG_dev_attributes *attr;
    pcons *alst;

    fextr = dev->range_extrema;

    for (h = g->f; h != NULL; h = h->next)
        {domain = h->domain;
	 npts   = domain->n_elements;
	 r      = (REAL **) domain->elements;

	 rx = PM_array_real(domain->element_type, r[0], npts, NULL);
	 ry = PM_array_real(domain->element_type, r[1], npts, NULL);

	 PM_array_real(domain->element_type, domain->extrema, 4, dextr);

	 range = h->range;
	 strcpy(bf, range->element_type);
	 mtype = SC_strtok(bf, " *", s);
	 npts  = range->n_elements;

	 same = (strcmp(mtype, SC_INTEGER_S) == 0);
	 if (same)
	    afd = *((int **) range->elements);
	 else
	    {afs = *((REAL **) range->elements);
	     afd = NULL;
	     CONVERT(SC_INTEGER_S, &afd, mtype, afs, npts, FALSE);};

/* find the range limits if any */
	 rextr = PM_get_limits(range);
	 aextr = ((rextr != NULL) && !dev->autorange) ? rextr : fextr;
	 aext  = NULL;
	 CONVERT(SC_INTEGER_S, &aext, mtype, aextr, 2, FALSE);

/* find the additional mapping information */
	 centering = N_CENT;
	 alst = PM_mapping_info(h,
				"CENTERING", &centering,
				NULL);

/* save user's values for various attributes */
	 attr = PG_get_attributes(dev);

/* set attribute values */
	 clip      = TRUE;
	 prec      = TEXT_CHARACTER_PRECISION;
	 charspace = 0.0;
	 chupx     = 0.0;
	 chupy     = 1.0;
	 chpthx    = 1.0;
	 chpthy    = 0.0;
	 PG_set_clipping(dev, clip);
	 PG_set_char_precision(dev, prec);
	 PG_set_char_path(dev, chpthx, chpthy);
	 PG_set_char_up(dev, chupx, chupy);
	 PG_set_char_space(dev, charspace);
  
	 cnnct = PM_connectivity(h);

	 switch (centering)
	    {case Z_CENT :
	          (*fnc_zc)(dev, 1, afd, rx, ry, npts, aext, cnnct, alst);
		  break;

	     case N_CENT :
                  (*fnc_nc)(dev, 1, afd, rx, ry, npts, aext, cnnct, alst);
		  break;

             case F_CENT :
             case U_CENT :
             default     :
                  break;};

	 PG_draw_domain_boundary(dev, h);

	 PG_update_vs(dev);

	 PG_set_clipping(dev, FALSE);

/* reset user's values for various attributes */
	 PG_set_attributes(dev, attr);
	 SFREE(attr);

	 SFREE(rx);
	 SFREE(ry);
	 if (!same)
	    SFREE(afd);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DVB_CHUNK_ZC_LR - do a chunk of the DVB rendering work */

static void *_PG_dvb_chunk_zc_lr(arg)
   void *arg;
   {int in, iz, k, l, nn, nmap, nz;
    int kz, lz, kzm, lzm, dk;
    int lmn, lmx, kmax, lmax, eflag;
    int a0c, a1c, a2c, a3c, a4c;
    int *a, *maxes, *aext, *pc;
    int *a1, *a2, *a3, *a4;
    REAL xmn, xmx, ymn, ymx;
    REAL x1c, x2c, x3c, x4c;
    REAL y1c, y2c, y3c, y4c;
    REAL *x1, *x2, *x3, *x4;
    REAL *y1, *y2, *y3, *y4;
    REAL *x, *y;
    char *emap;
    byte *cnnct, *rv;
    pcons *alist;
    PG_device *dev;
    PG_dvb_data *par;

    par = (PG_dvb_data *) arg;

    dev   = par->dev;
    a     = par->a;
    x     = par->x;
    y     = par->y;
    aext  = par->aext;
    cnnct = par->cnnct;
    alist = par->alist;

    maxes = (int *) cnnct;
    kmax  = maxes[0];
    lmax  = maxes[1];
    nn    = kmax*lmax;
    nmap  = (kmax - 1) * (lmax - 1);
    nz    = SC_arrlen(a)/sizeof(int);

    LR_MAPPING_INFO(alist, nmap);

    SC_assoc_info(alist,
		  "CENTERING", &pc,
		  NULL);

    PM_CHECK_EMAP(alist, nmap, eflag, emap);

    PM_LOGICAL_ZONE(x, x1, x2, x3, x4, kmax);
    PM_LOGICAL_ZONE(y, y1, y2, y3, y4, kmax);

    PG_get_window(dev, &xmn, &xmx, &ymn, &ymx);

    kz  = kmax - 1;
    lz  = lmax - 1;
    if (*pc == N_CENT)
       {kzm = kz - 1;
	lzm = lz - 1;
        dk  = kmax;}
    else
       {kzm = kz - 1;
	lzm = lz - 1;
        dk  = kz;};

/* template:
 *                 .    .     .    .
 *                         2
 *                 .    .     .    .
 *                    3    0    1
 *                 .    .     .    .
 *                         4
 *                 .    .     .    .
 */
    a1 = a + 1;
    a2 = a + kz;
    a3 = a - 1;
    a4 = a - kz;

    SC_chunk_split(&lmn, &lmx, &rv);

    for (l = lmn; l < lmx; l++)
        for (k = 0; k < kz; k++)
            {in = l*kmax + k;
	     iz = l*dk + k;
	     if (emap[iz] == 0)
	        continue;

	     a0c = a[iz];
	     a1c = a1[iz];
	     a2c = a2[iz];
	     a3c = a3[iz];
	     a4c = a4[iz];

	     x1c = x1[in];
	     x2c = x2[in];
	     x3c = x3[in];
	     x4c = x4[in];

	     y1c = y1[in];
	     y2c = y2[in];
	     y3c = y3[in];
	     y4c = y4[in];

	     if ((a0c != a1c) && (k != kzm))
	        PG_draw_line(dev, x1c, y1c, x2c, y2c);

	     if ((a0c != a2c) && (l != lzm))
	        PG_draw_line(dev, x2c, y2c, x3c, y3c);

	     if ((a0c != a3c) && (k != 0))
	        PG_draw_line(dev, x3c, y3c, x4c, y4c);

	     if ((a0c != a4c) && (l != 0))
	        PG_draw_line(dev, x4c, y4c, x1c, y1c);};

    if (eflag)
       SFREE(emap);

    return(rv);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DV_BND_ZC_LR - do DVB rendering assuming a zone centered
 *                 - Logical-Rectangular mesh
 *                 - the mesh topology is always ignored and is there
 *                 - to be call compatible with the AC version
 */

void PG_dv_bnd_zc_lr(dev, nd, a, x, y, npts, aext, cnnct, alist)
   PG_device *dev;
   int nd;
   int *a;
   REAL *x, *y;
   int npts, *aext;
   byte *cnnct;
   pcons *alist;
   {int kz, lz;
    int *maxes, kmax, lmax;
    PG_dvb_data par;

    maxes = (int *) cnnct;
    kmax  = maxes[0];
    lmax  = maxes[1];

    kz = kmax - 1;
    lz = lmax - 1;

    par.dev   = dev;
    par.nd    = nd;
    par.a     = a;
    par.x     = x;
    par.y     = y;
    par.aext  = aext;
    par.cnnct = cnnct;
    par.alist = alist;

    SC_chunk_loop(_PG_dvb_chunk_zc_lr, 0, lz, TRUE, &par);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DV_BND_NC_LR - draw DVB segments for the variable A on the
 *                 - Logical-Rectangular mesh specified by coordinates
 *                 - X and Y which is KMAX by LMAX
 */

static void PG_dv_bnd_nc_lr(dev, nd, a, x, y, npts, aext, cnnct, alist)
   PG_device *dev;
   int nd;
   int *a;
   REAL *x, *y;
   int npts, *aext;
   byte *cnnct;
   pcons *alist;
   {int *ia;
    REAL *ra, *ap;

    ra = NULL;
    CONVERT(SC_REAL_S, &ra, SC_INTEGER_S, a, npts, FALSE);

    ap = PM_node_zone_lr_2d(ra, cnnct, alist);

    ia = NULL;
    CONVERT(SC_INTEGER_S, &ia, SC_REAL_S, ap, npts, FALSE);

    PG_dv_bnd_zc_lr(dev, nd, ia, x, y, npts, aext, cnnct, alist);

    SFREE(ap);
    SFREE(ra);
    SFREE(ia);

    return;}
  
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DV_BND_ZC_AC - do DVB rendering assuming a zone centered
 *                 - Arbitrarily-Connected mesh of with mesh topology MT
 */

void PG_dv_bnd_zc_ac(dev, nd, a, x, y, npts, aext, cnnct, alist)
   PG_device *dev;
   int nd;
   int *a;
   REAL *x, *y;
   int npts, *aext;
   byte *cnnct;
   pcons *alist;
   {int is, os, iz, oz, is1, is2, in1, in2;
    int ia, ip, jp, jn1, jn2, a1, a2, nb;
    int nz, nzp, nsp, npt, npto;
    int *nc, *np, *nd1, *nd2, *p1, *p2;
    long **cells, *zones, *sides;
    REAL xmn, xmx, ymn, ymx;
    PM_mesh_topology *mt;

    mt = (PM_mesh_topology *) cnnct;

    cells = mt->boundaries;
    zones = cells[2];
    sides = cells[1];

    nc = mt->n_cells;
    nz = nc[2];

    np  = mt->n_bound_params;
    nzp = np[2];
    nsp = np[1];

    PG_get_window(dev, &xmn, &xmx, &ymn, &ymx);

/* can compute this by looping over the sides */
    npt = 4*nz;
    nd1 = FMAKE_N(int, npt, "PG_DV_BND_ZC_AC:nd1");
    nd2 = FMAKE_N(int, npt, "PG_DV_BND_ZC_AC:nd2");

    a1 = aext[0];
    a2 = aext[1];

    npt  = 0;
    npto = 0;
    for (ia = a1; ia <= a2; ia++)
        {p1 = nd1 + npt;
	 p2 = nd2 + npt;

/* add all the zone boundaries for this IN */
	 for (iz = 0; iz < nz; iz++)
	     {if (a[iz] != ia)
		 continue;

	      oz  = iz*nzp;
	      is1 = zones[oz];
	      is2 = zones[oz+1];

	      for (is = is1; is <= is2; is++)
		  {os    = is*nsp;
		   *p1++ = sides[os];
		   *p2++ = sides[os+1];
		   npt++;};};

/* remove all duplicates */
	 for (ip = 0; ip < npt; ip++)
	     {in1 = nd1[ip];
	      in2 = nd2[ip];

	      nb  = ip + 1;
	      nb  = max(nb, npto);
	      for (jp = nb; jp < npt; jp++)
		  {jn1 = nd1[jp];
		   jn2 = nd2[jp];
		   if (((in1 == jn1) && (in2 == jn2)) ||
		       ((in1 == jn2) && (in2 == jn1)))
		      {npt--;
		       for ( ; jp < npt; jp++)
			   {nd1[jp] = nd1[jp+1];
			    nd2[jp] = nd2[jp+1];};};};};

	 npto = npt;};

/* plot what's left */
    for (ip = 0; ip < npt; ip++)
        {in1 = nd1[ip];
	 in2 = nd2[ip];

	 PG_draw_line(dev, x[in1], y[in1], x[in2], y[in2]);};

    SFREE(nd1);
    SFREE(nd2);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DV_BND_NC_AC - draw DVB segments for the variable A on the
 *                 - Arbitrarily-Connected mesh specified by coordinates
 *                 - X and Y with topology MT
 */

static void PG_dv_bnd_nc_ac(dev, nd, a, x, y, npts, aext, cnnct, alist)
   PG_device *dev;
   int nd;
   int *a;
   REAL *x, *y;
   int npts, *aext;
   byte *cnnct;
   pcons *alist;
   {int *ia;
    REAL *ra, *ap;

    ra = NULL;
    CONVERT(SC_REAL_S, &ra, SC_INTEGER_S, a, npts, FALSE);

    ap = PM_node_zone_ac_2d(ra, cnnct, alist);

    ia = NULL;
    CONVERT(SC_INTEGER_S, &ia, SC_REAL_S, ap, npts, FALSE);

    PG_dv_bnd_zc_ac(dev, nd, ia, x, y, npts, aext, cnnct, alist);

    SFREE(ap);
    SFREE(ia);
    SFREE(ra);

    return;}
  
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_DVB_CORE - DVB plot skeleton routine */

static void _PG_dvb_core(dev, data, fnc_zc, fnc_nc)
   PG_device *dev;
   PG_graph *data;
   PFByte fnc_zc, fnc_nc;
   {PG_graph *g;
    PG_picture_desc *pd;

    data->rendering = PLOT_DV_BND;

    pd = PG_setup_picture(dev, data, TRUE, TRUE, TRUE);

/* plot all of the current functions */
    for (g = data; g != NULL; g = g->next)
        PG_dvb_hand(dev, g, fnc_zc, fnc_nc);

    PG_finish_picture(dev, data, pd);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_DV_BND_PLOT - main DVB plot control routine */

#ifdef PCC

void PG_dv_bnd_plot(dev, data, va_alist)
   PG_device *dev;
   PG_graph *data;
   va_dcl

#endif

#ifdef ANSI

void PG_dv_bnd_plot(PG_device *dev, PG_graph *data, ...)

#endif

   {

    if (strcmp(data->f->category, PM_LR_S) == 0)
       _PG_dvb_core(dev, data, PG_dv_bnd_zc_lr, PG_dv_bnd_nc_lr);

    else if (strcmp(data->f->category, PM_AC_S) == 0)
       _PG_dvb_core(dev, data, PG_dv_bnd_zc_ac, PG_dv_bnd_nc_ac);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
