/*
 * GSDLIN.C - line drawing routines for PGS
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pgs.h"

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_DRAW_POLYLINE - draw a polyline in WC */
 
void PG_draw_polyline(dev, x, y, n, clip)
   PG_device *dev;
   REAL *x, *y;
   int n, clip;
   {PG_curve *crv;
    double xo, yo;

    PtoS(dev, 0, 0, xo, yo);
    StoW(dev, xo, yo);
    
    crv = PG_make_curve(dev, WORLDC, FALSE, n, xo, yo, x, y);

    PG_draw_curve(dev, crv, clip);

    PG_release_curve(crv);

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_MULTIPLE_LINE - Draws multiple line segments.  Two lines are
 *                       - specified and the number of line segments to
 *                       - be drawn between them.  This routine is useful
 *                       - in drawing grid lines, cross hatching, and
 *                       - shading.  It is not limited to drawing parallel
 *                       - lines; the two delimiting lines may be at any
 *                       - orientation to each other.
 *                       - nlines - specifies the number of lines to
 *                                  draw between the two given
 *                                  lines.  If n=0, only the two
 *                                  given lines will be drawn
 *                       - (x1, y1), (x2, y2) - defines the two end points
 *                                  for the first line
 *                       - (x3, y3), (x4, y4) - defines the two end points
 *                                  for the second line
 */

void PG_draw_multiple_line(dev, nlines, x1, y1, x2, y2,
                           x3, y3, x4, y4, dx)
   PG_device *dev;
   int nlines;
   double x1, y1, x2, y2, x3, y3, x4, y4;
   REAL *dx;
   {REAL position, dx13, dy13, dx24, dy24, inv_nlines;
    REAL *x, *y, *px, *py;
    int i, n;

    if (nlines <= 0)
       nlines = 1;

    if (nlines > 1)
       inv_nlines = 1.0/((REAL) (nlines - 1));
    else
       inv_nlines = 1.0;

    dx13 = x3 - x1;
    dy13 = y3 - y1;
    dx24 = x4 - x2;
    dy24 = y4 - y2;

    n = nlines << 1;
    x = FMAKE_N(REAL, n, "PG_DRAW_MULTIPLE_LINE:x");
    y = FMAKE_N(REAL, n, "PG_DRAW_MULTIPLE_LINE:y");

/* plot lines in between, if any */
    px = x;
    py = y;
    if (dx == NULL)
       {for (i = 0; i < nlines; ++i)
            {position = ((REAL) i)*inv_nlines;
             *px++ = x1 + position*dx13;
             *py++ = y1 + position*dy13;
             *px++ = x2 + position*dx24;
             *py++ = y2 + position*dy24;};}
    else
       {for (i = 0; i < nlines; ++i)
            {position = ABS(dx[i]);
             *px++ = x1 + position*dx13;
             *py++ = y1 + position*dy13;
             *px++ = x2 + position*dx24;
             *py++ = y2 + position*dy24;};};

    PG_draw_disjoint_polyline_2(dev, x, y, nlines, FALSE, TRUE);

    SFREE(x);
    SFREE(y);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_DISJOINT_POLYLINE_3 - draws disjoint three dimensional line
 *                             - segments specified in WC
 *                             - N is the number of segments hence half
 *                             - the number of points
 *                             - X[2*i] is one endpoint of the ith segment
 *                             - and X[2*i+1] is the other endpoint
 *                             - NORM specifies whether segments should be
 *                             - normalized to NC
 *                             - FLAG specifies wether the set is clipped
 *                             - to the viewport limits
 */

void PG_draw_disjoint_polyline_3(dev, x, y, z, theta, phi, chi,
				 n, flag, norm)
   PG_device *dev;
   REAL *x, *y, *z;
   double theta, phi, chi;
   long n;
   int flag, norm;
   {int i, m;
    REAL *rx, *ry, *rz;
    REAL xmn, xmx, ymn, ymx, zmn, zmx;
    REAL xv, yv, zv;

    PG_make_device_current(dev);

    m = n << 1;

    rx = FMAKE_N(REAL, m, "PG_DRAW_DISJOINT_POLYLINE_3:rx");
    ry = FMAKE_N(REAL, m, "PG_DRAW_DISJOINT_POLYLINE_3:ry");
    rz = FMAKE_N(REAL, m, "PG_DRAW_DISJOINT_POLYLINE_3:rz");

    xmn =  HUGE_REAL;
    ymn =  HUGE_REAL;
    zmn =  HUGE_REAL;
    xmx = -HUGE_REAL;
    ymx = -HUGE_REAL;
    zmx = -HUGE_REAL;
    for (i = 0; i < m; i++)
        {xv = x[i];
         yv = y[i];
         zv = z[i];
         xmn = min(xmn, xv);
         xmx = max(xmx, xv);
         ymn = min(ymn, yv);
         ymx = max(ymx, yv);
         zmn = min(zmn, zv);
         zmx = max(zmx, zv);};

    PG_rotate_3d_WC(x, y, z, xmn, xmx, ymn, ymx, zmn, zmx,
                    theta, phi, chi, rx, ry, rz, m, norm);

    PG_draw_disjoint_polyline_2(dev, rx, ry, n, flag, TRUE);

    PG_release_current_device(dev);

    SFREE(rx);
    SFREE(ry);
    SFREE(rz);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_RAD - draw a radial line from rmin to rmax along the direction
 *             - given with respect to positive x axis
 *             - (increasing counter-clockwise)
 *             - angle units (radians or degrees) specified by unit
 *             - from central point (x, y)
 */

void PG_draw_rad(dev, rmin, rmax, a, x, y, unit)
   PG_device *dev;
   double rmin, rmax, a, x, y;
   int unit;
   {REAL xn, xx, yn, yx;
    REAL ca, sa;

    if (unit == DEGREE)
       a *= DEG_RAD;

    ca = cos(a);
    sa = sin(a);

    xn = x + rmin*ca;
    yn = y + rmin*sa;
    xx = x + rmax*ca;
    yx = y + rmax*sa;
    PG_draw_line(dev, xn, yn, xx, yx);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_ARC - draw an arc with radius, r, from angles a1 to a2
 *             - given with respect to positive x axis
 *             - (increasing counter-clockwise)
 *             - angle units (radians or degrees) specified by unit
 *             - from central point (x, y)
 */

void PG_draw_arc(dev, r, a1, a2, x, y, unit)
   PG_device *dev;
   double r, a1, a2, x, y;
   int unit;
   {REAL a, da, ca, sa;
    REAL xr[N_ANGLES], yr[N_ANGLES];
    int i;

    if (r <= 0.0)
       return;

    if (unit == DEGREE)
       {a1 *= DEG_RAD;
        a2 *= DEG_RAD;};

    da = (a2-a1)/(N_ANGLES-1);
    for (i = 0, a = a1; i < N_ANGLES; a += da, i++)
        {ca = cos(a);
         sa = sin(a);
         xr[i] = x + r*ca;
         yr[i] = y + r*sa;};

    PG_draw_polyline(dev, xr, yr, N_ANGLES, TRUE);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

