(*
 *                     The OCaml-gtk interface
 *
 * Copyright (c) 1997-99   David Monniaux, Pascal Cuoq, Sven Luther
 *
 * This file is distributed under the conditions described in
 * the file LICENSE.  
 *)

open Gtk
open Glib
open Gtk.Unsafe

class gtkobject (gtkobject : Gtk.Unsafe.gtkobject) =
object
  val gtkobject = gtkobject
  method get_gtkobject = gtkobject
  method disconnect_slot n = signal_disconnect gtkobject n
  method destroy = object_destroy gtkobject
end

class widget (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit gtkobject gtkobject
  method show = widget_show gtkobject
  method hide = widget_hide gtkobject
  method set_tip tip_text tip_private = 
    tooltips_set_tip (tooltips_new ()) gtkobject tip_text tip_private
  method realize = widget_realize gtkobject 
  method unrealize = widget_unrealize gtkobject 
  method set_usize x y = widget_set_usize gtkobject x y
  method set_sensitive sens = widget_set_sensitive gtkobject sens
  method window_of = window_of gtkobject
  method style_of = style_of gtkobject
  method set_events events = widget_set_events gtkobject events
  method unparent = widget_unparent gtkobject
  method reparent (new_parent : widget) =
    widget_reparent gtkobject (new_parent #get_gtkobject)
  method connect_expose (func : gtkobject->int->int->int->int->bool) =
    let lowlevel_handler gtkobj args = 
      ( match args with
        [Pointer ptr] ->
          let x = Gdk.Event.Extract.expose_x ptr
          and y = Gdk.Event.Extract.expose_y ptr
          and w = Gdk.Event.Extract.expose_w ptr
          and h = Gdk.Event.Extract.expose_h ptr
          in Bool (func (new gtkobject gtkobj) x y w h)
      | _ -> assert false )
    in
    signal_connect gtkobject "expose_event" lowlevel_handler
      
  method connect_configure (func : gtkobject->int->int->int->int->bool) =
    let lowlevel_handler gtkobj args =
      ( match args with
        [Pointer ptr] ->
          let x = Gdk.Event.Extract.expose_x ptr
          and y = Gdk.Event.Extract.expose_y ptr
          and w = Gdk.Event.Extract.expose_w ptr
          and h = Gdk.Event.Extract.expose_h ptr
          in Bool (func (new gtkobject gtkobj) x y w h)
      | _ -> assert false )
    in
    signal_connect gtkobject "configure_event" lowlevel_handler

  method connect_button_press (func : gtkobject->int->int->int->int->bool) =
    let lowlevel_handler gtkobj args =
      ( match args with
        [Pointer ptr] ->
          let x = Gdk.Event.Extract.button_x ptr
          and y = Gdk.Event.Extract.button_y ptr
          and button = Gdk.Event.Extract.button_button ptr
          and state = Gdk.Event.Extract.button_state ptr
            in Bool (func (new gtkobject gtkobj) x y button state)
      | _ -> assert false )
    in
    signal_connect gtkobject "button_press_event" lowlevel_handler

  method connect_motion_notify (func : gtkobject->int->int->bool->int->bool) =
    let lowlevel_handler gtkobj args =
      ( match args with
        [Pointer ptr] ->  
          let x = Gdk.Event.Extract.motion_x ptr
          and y = Gdk.Event.Extract.motion_y ptr
          and is_hint = Gdk.Event.Extract.motion_is_hint ptr
          and state = Gdk.Event.Extract.motion_state ptr
          in Bool (func (new gtkobject gtkobj) x y is_hint state)
      | _ -> assert false ) 
    in
    signal_connect gtkobject "motion_notify_event" lowlevel_handler

  method connect_key_press (func : gtkobject->int->int->bool) =
    let lowlevel_handler gtkobj args =
      ( match args with
        [Pointer ptr] ->  
          let keyval = Gdk.Event.Extract.key_keyval ptr
          and state = Gdk.Event.Extract.key_state ptr
          in Bool (func (new gtkobject gtkobj) keyval state)
      | _ -> assert false )
    in
    signal_connect gtkobject "key_press_event" lowlevel_handler

  method connect_destroy (func : gtkobject->Glib.Pointer.t->unit) =
    let lowlevel_handler gtkobj args =
      ( match args with
        [Pointer ptr] -> 
          ignore (func (new gtkobject gtkobj) ptr); Unit
      | _ -> assert false )   
    in
    signal_connect gtkobject "destroy" lowlevel_handler
      
  (* UNFINISHED *)
end

(* 5.13 Container widget *)
class container (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit widget gtkobject
  method border_width = container_border_width gtkobject
  method add (child : widget) = container_add gtkobject (child #get_gtkobject)
  method remove (child : widget) = 
    container_remove gtkobject (child #get_gtkobject)
  method block_resize = container_block_resize gtkobject
  method unblock_resize = container_unblock_resize gtkobject
  method need_resize = container_need_resize gtkobject
  (* UNFINISHED difficult to make foreach type-safe *)
  method focus = container_focus gtkobject
  (* UNFINISHED difficult to make children type-safe *)
end

(* 5.4 Bin widget *)
class bin (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit container gtkobject
end

(* 5.1 Alignment widget *)
class alignment (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit bin gtkobject
  method set = alignment_set gtkobject
end

(* 5.42 Misc widget *) (* UNFINISHED *)
class misc (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit widget gtkobject
end

(* 5.46 Pixmap widget *) (* UNFINISHED *)
class pixmap (gtkobject : Gtk.Unsafe.gtkobject) =
  object
    inherit misc gtkobject
  end
 
(* 5.2 Arrow widget *)
class arrow (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit misc gtkobject
  method set = arrow_set gtkobject
end

(* 5.23 Frame widget *) 
class frame (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit bin gtkobject
  method set_shadow_type = frame_set_shadow_type gtkobject
  method set_label = frame_set_label gtkobject
  method set_label_align = frame_set_label_align gtkobject
end

(* 5.3 Aspect frame widget *)
class aspect_frame (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit frame gtkobject
  method set = aspect_frame_set gtkobject
end

(* 5.5 Box widget *)
class box (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit container gtkobject
  method pack_start (child : widget) =
    box_pack_start gtkobject (child #get_gtkobject)
  method pack_end (child : widget) =
    box_pack_end gtkobject (child #get_gtkobject)
  method pack_start_defaults (child : widget) =
    box_pack_start_defaults gtkobject (child #get_gtkobject)
  method pack_end_defaults (child : widget) =
    box_pack_end_defaults gtkobject (child #get_gtkobject)
  method set_homogeneous = box_set_homogeneous gtkobject
  method set_spacing = box_set_spacing gtkobject
  method reorder_child (child : widget) =
    box_reorder_child gtkobject (child #get_gtkobject)
  method query_child_packing (child : widget) =
    box_query_child_packing gtkobject (child #get_gtkobject)
end

(* 5.6 Button box widget *)
class button_box (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit box gtkobject
  method set_child_size = button_box_set_child_size gtkobject
  method set_child_ipadding = button_box_set_child_ipadding gtkobject
  method set_layout = button_box_set_layout gtkobject
  method get_spacing = button_box_get_spacing gtkobject
  method get_child_size = button_box_get_child_size gtkobject
  method get_child_ipadding = button_box_get_child_ipadding gtkobject
  method get_layout = button_box_get_layout gtkobject
end

(* ### 5.7 Button widget *)
class button (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit container gtkobject
  (* signals completed *)
  method connect_pressed (func : unit -> unit) =
    signal_connect gtkobject "pressed"
      (function _ -> function _ -> func (); Unit)
  method connect_released (func : unit -> unit) =
    signal_connect gtkobject "released"
      (function _ -> function _ -> func (); Unit)
  method connect_clicked (func : unit -> unit) =
    signal_connect gtkobject "clicked"
      (function _ -> function _ -> func (); Unit)
  method connect_enter (func : unit -> unit) =
    signal_connect gtkobject "enter"
      (function _ -> function _ -> func (); Unit)
  method connect_leave (func : unit -> unit) =
    signal_connect gtkobject "leave"
      (function _ -> function _ -> func (); Unit)

  method pressed = button_pressed gtkobject
  method released = button_released gtkobject
  method clicked = button_clicked gtkobject
  method enter = button_enter gtkobject
  method leave = button_leave gtkobject
end

(* ### 5.60 Toggle button widget *) 
class toggle_button (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit button gtkobject
  method set_state flag = toggle_button_set_state gtkobject flag
  method set_mode flag = toggle_button_set_mode gtkobject flag
  method toggle_button_toggled = toggle_button_toggled gtkobject 
  method set_active flag = toggle_button_set_active gtkobject flag
end

(* ### 5.8 Check button widget *)
class check_button (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit toggle_button gtkobject
end

class item (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit bin gtkobject
end

class menu_shell (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit container gtkobject
  method append (item : menu_item) =
    menu_shell_append gtkobject (item #get_gtkobject)
  method prepend (item : menu_item) =
    menu_shell_prepend gtkobject (item #get_gtkobject)
  method insert (item : menu_item) pos =
    menu_shell_insert gtkobject (item #get_gtkobject) pos
end

and menu (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit menu_shell gtkobject
  method append (item : menu_item) =
    menu_append gtkobject (item #get_gtkobject)
  method prepend (item : menu_item) =
    menu_prepend gtkobject (item #get_gtkobject)
  method insert (item : menu_item) pos =
    menu_insert gtkobject (item #get_gtkobject) pos
  method popdown () =
    menu_popdown gtkobject
end

and menu_item (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit item gtkobject
  method set_submenu (menu : menu) =
    menu_item_set_submenu gtkobject (menu #get_gtkobject)
  method connect_activate (f : unit->unit) =
    signal_connect gtkobject "activate" 
      (function _ -> function _ -> f () ; Unit) 
  method select () = menu_item_select gtkobject
  method deselect () = menu_item_deselect gtkobject
end


(* ### 5.9 Check menu item widget *)
class check_menu_item (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit menu_item gtkobject
  method set_state = check_menu_item_set_state gtkobject
  method toggled = check_menu_item_toggled gtkobject
  method connect_toggled (func : unit -> unit) =
    signal_connect gtkobject "toggled"
      (function _ -> function _ -> func (); Unit)
end

(* ### 5.10 Compound list *)
class ['a] clist (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit container gtkobject
  method set_border = clist_set_border gtkobject
  method set_selection_mode = clist_set_selection_mode gtkobject
  method freeze = clist_freeze gtkobject
  method thaw = clist_thaw gtkobject
  method column_titles_show = clist_column_titles_show gtkobject
  method column_titles_hide = clist_column_titles_hide gtkobject
  method get_text = clist_get_text gtkobject
  method set_column_title = clist_set_column_title gtkobject
  method set_column_widget i (w : widget) = 
    clist_set_column_widget gtkobject i (w #get_gtkobject)
  method set_column_justification = clist_set_column_justification gtkobject
  method set_column_width = clist_set_column_width gtkobject
  method set_row_height = clist_set_row_height gtkobject
  method set_row_moveto = clist_moveto gtkobject
  method set_text = clist_set_text gtkobject
(* clist_set_pixmap $$$ *)
(* clist_setpixtext $$$ *)
(* clist_set_foreground $$$ *)
(* clist_set_background $$$ *)
  method set_shift = clist_set_shift gtkobject
  method append_row = clist_append gtkobject
  method prepend_row = clist_prepend gtkobject
  method insert_row = clist_insert gtkobject
  method remove_row = clist_remove gtkobject
  method set_row_data row data = clist_set_row_data gtkobject row (data : 'a)
  method get_row_data row = ((clist_get_row_data gtkobject row) : 'a)
  method select_row = clist_select_row gtkobject
  method unselect_row = clist_unselect_row gtkobject
  method connect_select_row (func : int->int->unit) = 
    (* UNFINISHED *)
    let lowlevel_handler _ args = 
      ( match args with
        [ Int row; Int column; _ ] -> func row column; Unit
      | _ -> assert false )
    in
    signal_connect gtkobject "select_row" lowlevel_handler

  method connect_unselect_row (func : int->int->unit) = 
    (* UNFINISHED *)
    let lowlevel_handler _ args = 
      ( match args with
        [ Int row; Int column; _ ] -> func row column; Unit
      | _ -> assert false )
    in
    signal_connect gtkobject "unselect_row" lowlevel_handler

  method connect_click_column (func : int->unit) =
    let lowlevel_handler _ args = 
      ( match args with
        [Int column] -> func column; Unit
      | _ -> assert false )
    in
    signal_connect gtkobject "click_column" lowlevel_handler
      
  method clear = clist_clear gtkobject
end


(* ### 5.65 Vertical box widget *)
class vbox (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit box gtkobject
end

(* ### 5.11 Color selector *)
class color_selection (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit vbox gtkobject
  method set_update_policy= color_selection_set_update_policy gtkobject
  method set_color = color_selection_set_color gtkobject
  method get_color = color_selection_get_color gtkobject
  method set_opacity = color_selection_set_opacity gtkobject
  method connect_color_changed (func : unit->unit) =
    signal_connect gtkobject "color_changed"
      (function _ -> function _ -> func (); Unit)
end

class drawing_area (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit widget gtkobject
  method size width height = drawing_area_size gtkobject width height
end

and scrolled_window (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit container gtkobject
  method scrolled_window_add_with_viewport (child : widget) =
    scrolled_window_add_with_viewport gtkobject (child #get_gtkobject)
end

and paned (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit widget gtkobject
  method add1 (child : widget) =
    paned_add1 gtkobject (child #get_gtkobject)
  method add2 (child : widget) =
    paned_add2 gtkobject (child #get_gtkobject)
  method handle_size size = paned_handle_size gtkobject size
  method gutter_size size = paned_gutter_size gtkobject size
end

and vpaned (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit paned gtkobject
end

and hpaned (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit paned gtkobject
end

and list_item (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit item gtkobject
end

and window (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit bin gtkobject
  method connect_delete_event (func : unit->bool) =
    signal_connect gtkobject "delete_event"
      (function _ -> function _ -> Bool(func ()))
  method connect_move_resize (func : (*int ref->int ref->*)int->int->bool) =
    signal_connect gtkobject "move_resize"
      (fun _ args ->
        ( match args with 
          [(*(Pointer xp); (Pointer yp) ;*)_;_; (Int w); (Int h)] ->
             (*let x=ref (Pointer.int_peek xp) and y=ref (Pointer.int_peek yp)
            in let rv=func x y w h
            in Pointer.int_poke xp !x; Pointer.int_poke yp !y;
            Bool rv*) Bool(func w h)
        | _ -> assert false ) )
  method set_title s = window_set_title gtkobject s
end

and dialog (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit window gtkobject
  method get_action_area = new box (dialog_get_action_area gtkobject)
  method get_vbox = new box (dialog_get_vbox gtkobject)
end

and hbox (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit box gtkobject
end

and radio_button (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit check_button gtkobject
end

and list_box (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit container gtkobject
  method insert_items (items : list_item list) index =
    list_insert_items gtkobject
      (GList.from_ml (List.map (fun item-> item #get_gtkobject) items))
      index
  method append_items (items : list_item list) =
    list_append_items gtkobject
      (GList.from_ml (List.map (fun item-> item #get_gtkobject) items))
end

class radio_menu_item (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit check_menu_item gtkobject
end

and menu_bar (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit menu_shell gtkobject
  method append (item : menu_item) =
    menu_bar_append gtkobject (item #get_gtkobject)
  method prepend (item : menu_item) =
    menu_bar_prepend gtkobject (item #get_gtkobject)
  method insert (item : menu_item) pos =
    menu_bar_insert gtkobject (item #get_gtkobject) pos
end

class label (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit misc gtkobject
  method set_text s  = label_set_text gtkobject s
  method get () = label_get gtkobject
end
let label_new s = new label (label_new s)


class progress_bar (gtkobject : Gtk.Unsafe.gtkobject) = 
object
  inherit widget gtkobject
  method update ratio = progress_bar_update gtkobject ratio
end

and separator (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit widget gtkobject
end

and hseparator (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit separator gtkobject
end

and vseparator (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit separator gtkobject
end

and scrollbar (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit widget gtkobject
end

and hscrollbar (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit scrollbar gtkobject
end

and vscrollbar (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit scrollbar gtkobject
end

and editable (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit widget gtkobject
  method insert_text_return_position text len pos = 
    editable_insert_text gtkobject text len pos
  method insert_text text len pos = 
    let _ = editable_insert_text gtkobject text len pos in ()
  method select_region start endd = editable_select_region gtkobject start endd
  method get_chars start endd = editable_get_chars gtkobject start endd
  method get_text = editable_get_chars gtkobject 0 (-1)
  method get_position = editable_get_position gtkobject
end

and text (gtkobject : Gtk.Unsafe.gtkobject) =
object (t)
  inherit editable gtkobject
  method insert_substring font s start length = 
    text_insert gtkobject font s start length
  method insert font s = t#insert_substring font s 0 (-1)
  method set_editable flag = text_set_editable gtkobject flag
  method forward_delete n = text_forward_delete gtkobject n
  method backward_delete n = text_backward_delete gtkobject n
  method freeze () = text_freeze gtkobject
  method thaw () = text_thaw gtkobject
  method get_length () = text_get_length gtkobject
  method get_point () = text_get_point gtkobject
  method set_point p = text_set_point gtkobject p
end

and entry (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit editable gtkobject
  method connect_activate (func : entry -> unit) =
    signal_connect gtkobject "activate"
      (function gtkobj -> function _ -> func (new entry gtkobj); Unit)
  method set_text text = entry_set_text gtkobject text 
  method get_text = entry_get_text gtkobject
  method set_position pos = entry_set_position gtkobject pos
end

and combo (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit hbox gtkobject
  method get_entry = new entry (combo_get_entry gtkobject)
  method get_button = new entry (combo_get_button gtkobject)
  method combo_disable_activate = combo_disable_activate gtkobject
end

and statusbar (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit hbox gtkobject
  method get_context_id = statusbar_get_context_id gtkobject 
  method push = statusbar_push gtkobject 
  method pop = statusbar_pop gtkobject 
  method remove_mesg = statusbar_remove gtkobject 
end

and table (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit widget gtkobject
  method attach 
      (child : widget) left_attach right_attach top_attach bottom_attach
      xoptions yoptions xpadding ypadding = 
    table_attach 
      gtkobject
      (child #get_gtkobject)
      left_attach right_attach
      top_attach bottom_attach
      xoptions yoptions xpadding ypadding
end
        
and file_selection (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit window gtkobject
  method set_filename name = file_selection_set_filename gtkobject name
  method get_filename = file_selection_get_filename gtkobject 
  method get_ok_button =
    new button (file_selection_get_ok_button gtkobject)
  method get_cancel_button =
    new button (file_selection_get_cancel_button gtkobject)
end

and notebook (gtkobject : Gtk.Unsafe.gtkobject) =
object
  inherit widget gtkobject
  method append_page (page : widget) (label : widget) = 
    notebook_append_page gtkobject (page #get_gtkobject) (label #get_gtkobject)
  method prepend_page (page : widget) (label : widget) = 
    notebook_prepend_page gtkobject (page #get_gtkobject) 
      (label #get_gtkobject)
  method insert_page (page : widget) (label : widget) pos = 
    notebook_insert_page gtkobject (page #get_gtkobject) 
      (label #get_gtkobject) pos
  method remove_page pos = 
    notebook_remove_page gtkobject pos
  method set_page pos = 
    notebook_set_page gtkobject pos
  method next_page = 
    notebook_next_page gtkobject 
  method prev_page = 
    notebook_prev_page gtkobject 
  method set_tab_pos pos = notebook_set_tab_pos gtkobject pos
  method set_show_tabs show_tabs = 
    notebook_set_show_tabs gtkobject show_tabs		
  method set_show_border show_border = 
    notebook_set_show_border gtkobject show_border		
end

(******* useful function *********)

let map_fold f org =
  let rec map_fold_rec =
    function
    	[] -> org, []
      | h :: t ->
	  let now, rest = map_fold_rec t in
            let after, point = f now h in
              after, (point :: rest)
  in function l -> snd (map_fold_rec l)


(****************************************************)
(* constructors                                     *)
(****************************************************)

let button_new () = new button (button_new ())
let button_new_with_label s = new button (button_new_with_label s)

let check_button_new_with_label s = 
  new check_button (check_button_new_with_label s)
let check_button_new () = new check_button (check_button_new ())
let check_menu_item_new_with_label s = 
  new check_menu_item (check_menu_item_new_with_label s)
let check_menu_item_new () = new check_menu_item (check_menu_item_new ())
let clist_new i = new clist (clist_new i)
let color_selection_new title = new color_selection
  (color_selection_new title)
let combo_new () = new combo (combo_new ())

let dialog_new () = new dialog (dialog_new ())    
let drawing_area_new () = new drawing_area (drawing_area_new ())

let entry_new () = new entry (entry_new ())
let entry_new_with_max_length n = 
  if n>=0 & n<=65535 
  then new entry (entry_new_with_max_length n)
  else raise (Invalid_argument "entry_new_with_max_length")

let file_selection_new title = new file_selection (file_selection_new title)
let frame_new s = new frame (frame_new s)

let hbox_new homogeneous spacing = new hbox (hbox_new homogeneous spacing)
let hbutton_box_new () = new button_box (hbutton_box_new ())
let hpaned_new () = new vpaned(hpaned_new()) 
let hseparator_new () = new hseparator (hseparator_new ())

let list_item_new_with_label s = new list_item (list_item_new_with_label s)
let list_new () = new list_box (list_new ())

let menu_bar_new () = new menu_bar (menu_bar_new ())
let menu_item_new_with_label s = new menu_item (menu_item_new_with_label s)
let menu_item_new ()=
  new menu_item (menu_item_new ())
let menu_new () = new menu (menu_new ())

let notebook_new () = new notebook (notebook_new ()) 

let pixmap_new gdkpixmap mask = new pixmap (pixmap_new gdkpixmap mask)
let progress_bar_new () = new progress_bar (progress_bar_new ())

let radio_buttons_new_with_labels = 
  map_fold (fun group label ->
    let widget = radio_button_new_with_label group label in
      (Some (radio_button_group widget)), (new radio_button widget)) None
let rec radio_buttons_set_index = function
    [] -> invalid_arg "GtkObj.radio_buttons_set_index"
  | (h : radio_button)::t -> function
        0 -> h #set_state true
      |	n -> h #set_state false;
             radio_buttons_set_index t (pred n) 
let radio_menu_items_new_with_labels = 
  map_fold 
    (fun group label ->
      let widget = radio_menu_item_new_with_label group label in
      Some (radio_menu_item_group widget), (new radio_menu_item widget) )
    None

let scrolled_window_new () = new scrolled_window (scrolled_window_new ())
let statusbar_new () = new statusbar (statusbar_new ())

let table_new nbrows nbcols homogeneous = 
  new table (table_new nbrows nbcols homogeneous)
let text_new () = new text (text_new ())  
let toggle_button_new s = new toggle_button (toggle_button_new ())
let toggle_button_new_with_label s = 
  new toggle_button (toggle_button_new_with_label s)

let vbox_new homogeneous spacing = new vbox (vbox_new homogeneous spacing) 
let vbutton_box_new () = new button_box (vbutton_box_new ())
let vpaned_new () = new vpaned(vpaned_new()) 
let vscrollbar_new vadj = new vscrollbar (vscrollbar_new vadj)
let vseparator_new () = new vseparator (vseparator_new ())

let window_new window_type = new window (window_new window_type)


let ( << ) x y = x # add (y :> widget); x





