open Gdk
open GtkObj
open GtkEasy
open GtkEasy.Layout
open GtkEasy.Menu
open Gdk.Event
open Gdk.Event.Extract

(* A shorthand for a useful function ... *)
let soi = string_of_int

(* A text widget to let us know what is happening ... *)

let text =text_new ()
let mesg s = 
(
  text#freeze ();
  text#insert_text (s^"\n") ((String.length s)+1) 0;
  text#thaw ();
)

(* Drawing area definition ... *)

(* Backing pixmap *)

let pixmap = ref None
exception No_pixmap

let get_pixmap () = match !pixmap with 
  | None -> raise No_pixmap
  | Some p -> p
let free_pixmap () = match !pixmap with
  | None -> ()
  | Some p -> Gdk.pixmap_unref p; pixmap := None; ()
let set_pixmap p = match !pixmap with
  | None -> pixmap := Some p; p
  | Some p' -> Gdk.pixmap_unref p'; pixmap := Some p; p

(* Widget DrawingArea, where to draw *)
let drawingarea = drawing_area_new ()

(* The configure event takes place at initialization and every time
   the size of the widget changes *)
let drawingarea_configure drawingarea x y width height =
(
  let da = drawingarea#get_gtkobject in
  let w = Gtk.Unsafe.window_of da
  (* the Window in which to copy the pixmap *)
  and black_gc = let st = Gtk.Unsafe.style_of da in st.Gtk.black_gc
  (* The black Graphic Context to fill the empty pixmap *)
  and allocation = Gtk.Unsafe.allocation_of da
  in let width = allocation.Gtk.width
  and height = allocation.Gtk.height
  (* We get the new size of the drawingarea widget after
     the configure event *)
  in let p = set_pixmap (Gdk.pixmap_new w width height (-1))
  (* We Create a new pixmap of the new size of the drawingarea widget *)
  in let d = Gdk.drawable_from_pixmap p
  (* We get the drawable associated with the pixmap *)
  in
  (
    Gdk.draw_rectangle d black_gc true 0 0 width height;
    (* So we can draw on it ... *)
    mesg "configure ...";
    true
    (* We don't want to emit a destroy event ... *)
  )
)

(* The expose event takes place after each configure event,
   and each time the widget becomes visible again (after having been
   hidden *)
let drawingarea_expose drawingarea x y width height = 
(
  let da = drawingarea#get_gtkobject 
  and p = get_pixmap ()
  (* We get the backing pixmap
   * Note : we should put it in a gdkRectangle ...
   *)
  in let d = Gdk.drawable_from_window (Gtk.Unsafe.window_of da)
  (* We get the drawable associated with the window associated with the
   * drawingarea widget
   *)
  and fg_gc = let st = Gtk.Unsafe.style_of da
    in st.Gtk.fg_gc.(Gtk.Unsafe.state_of da)
  (* The foreground Graphic Context to draw the backing pixmap *)
  in
  (
    Gdk.draw_pixmap d fg_gc p x y x y width height;
    (* We draw the backing pixmap on the window *)
    mesg ("expose ..."^(soi x)^","^(soi y)^":"^
      soi width)^","^(soi height));
    false;
    (* Why do we send false ? if i understood correctly,
       this will make gtk to send a destroy event, ... ? *)
  )
)

let white_color = Gdk.color_of_rgb {red=65535; green=65535; blue=65535}
let red_color = Gdk.color_of_rgb {red=65535; green=0; blue=0}
let green_color = Gdk.color_of_rgb {red=0; green=65535; blue=0}
let blue_color = Gdk.color_of_rgb {red=0; green=0; blue=65535}

let current_color = ref white_color

let select_current_color color () = current_color := color

(* We draw a small square arround the clicked point *)
let draw_point x y =
  let da = drawingarea#get_gtkobject
  in let st = Gtk.Unsafe.style_of da
  in let white = st.Gtk.white_gc
  (* The white Graphic Context ... *)
  and p = get_pixmap ()
  in let d = Gdk.drawable_from_pixmap p
  and wdw = Gdk.window_from_pixmap p
  in let cmap = Gdk.window_get_colormap wdw
  and gc = Gdk.gc_new wdw
  in let _ = Gdk.colormap_alloc_color cmap !current_color true true
  and _ = Gdk.gc_set_foreground gc !current_color 
    (* The drawable where to draw ... *)
  and rect = {x=x-5; y=y-5; width=10; height=10}
  (* The rectangle to draw, and later to refresh *)
  in
    (
    Gdk.draw_rectangle d gc true rect.x rect.y rect.width rect.height;
    (* We draw the rectangle in the backing pixmap *)
    Gtk.Unsafe.widget_draw da rect
    (* And ask the widget to refresh the drawn zone. *)
  )

(* We try to get the color of a pixel. *)
(* Note : Not much success here, perhaps we should look directly
   the backing pixmap ... *)
let point_color x y =
  (* This is not very useful, becasue what is contained in pixel is not the color
      of the pixel but some strange data .... same as pixel value in *)
  let da = drawingarea#get_gtkobject
  (*
  in let w = Gtk.Unsafe.window_of da
  *)
  in let w = Gdk.window_from_pixmap (get_pixmap ())
  in let i = Gdk.image_get w 0 0 (x+1) (y+1)
  (* We get the image associated with the window. *)
  in let pixel = Gdk.image_get_pixel i x y
  (* To get the pixel (x,y) of the screen *)
  (* Note : What is the nature of the pixel returned ?
     i was said that it is the same as the pixel value in the
     GdkColor type, but in C they don't have the same type.
     Also in our case, it reports all the same values ... *)
  in mesg ("Pixel ("^(soi x)^", "^(soi y)^") : "^(soi pixel))

(* Button_press event handler *)
let button_pressed b x y =
(
  mesg ("Bouton "^(soi b)^" pressed at coordinates ("^(soi x)^
    ","^(soi y)^") ...");
  match b with
  | 1 -> draw_point x y
  | 2 -> point_color x y
  | _ -> ()
)

(* Button pressed event, we call the button_pressed function here *)
let drawingarea_button_pressed drawingarea x y button state =
(
  button_pressed button x y;
  (* We get the button that was pressed, as well as
  the coordinates of the click *)
  true
)

(* Pixmap clear function *)
let clear_pixmap () =
  let da = drawingarea#get_gtkobject
  in let st = Gtk.Unsafe.style_of da
  in let black = st.Gtk.black_gc
  (* The black Graphic Context ... *)
  and allocation = Gtk.Unsafe.allocation_of da
  (* The allocation of the widget,
   * to get the width and height of the pixmap
   *)
  and p = get_pixmap ()
  in let d = Gdk.drawable_from_pixmap p
  (* The drawable where to draw ... *)
  and rect = {x=0; y=0; width=allocation.Gtk.width; height=allocation.Gtk.height}
  (* The rectangle to draw, and later to refresh, be carefull not to use 
   * x,y = allocation.Gtk.x,y because it will not work.allocation.x,y are
   * the position of the drawing area's pixmap in the drawing area.
   *)
  in
    (
    Gdk.draw_rectangle d black true rect.x rect.y rect.width rect.height;
    (* We draw the rectangle in the backing pixmap *)
    Gtk.Unsafe.widget_draw da rect
    (* And ask the widget to refresh the drawn zone. *)
  )

let _ = drawingarea#set_events [BUTTON_PRESS_MASK; EXPOSURE_MASK]
(* We decide to catch the BUTTON_PRESS and EXPOSURE events *)
let _ = drawingarea#connect_button_press drawingarea_button_pressed
let _ = drawingarea#connect_expose drawingarea_expose
let _ = drawingarea#connect_configure drawingarea_configure
(* And associate our previously defined handlers to our widget *)
let _ = drawingarea#size 300 200
(* We change the size of our widget *)

(* A menu *)

let mesgfunc t = function () -> mesg t
let menu_def =
[ 
  Submenu (GtkEasy.Label "Tutorial 3",
  [
    Item (GtkEasy.Label "About", mesgfunc
      "A small Example of how to use the drawingarea widget in mlgtk");
    Item (GtkEasy.Label "Close", Gtk.main_quit);
  ]);
  Submenu (GtkEasy.Label "Misc",
  [
    Item (GtkEasy.Label "Options", mesgfunc "Not yet implemented.");
    Item (GtkEasy.Label "Clear", clear_pixmap);
  ]);
  Submenu (GtkEasy.Label "Color",
  [
    Item (GtkEasy.Label "White", select_current_color white_color);
    Item (GtkEasy.Label "Red", select_current_color red_color);
    Item (GtkEasy.Label "Green", select_current_color green_color);
    Item (GtkEasy.Label "Blue", select_current_color blue_color);
  ]);
  Item (GtkEasy.Label "Help", mesgfunc "No help available yet.");
]
let menu_bar = make_menu_bar menu_def

(* Toplevel stuff ... *)

let structure =
  let me =  Widget (menu_bar:>widget),
    {expand=false; fill=false; padding=1}  
  and da =  Widget (drawingarea:>widget),
    {expand=true; fill=true; padding=1}  
  and t = Widget (text:>widget),
    {expand=false; fill=false; padding=1}
  in Box (Vert, [me; da; t])
(* We put everything in a vertical box *)

let window = make_window_from_structure structure "Toplevel Window" ;;
(* We create the toplevel window ... *)
  
let _ = text#realize

let _ = window #connect_delete_event( fun () -> Gtk.main_quit(); false )
(* The handler installed with the method [#connect_delete_event]
   is called when the user tries to close the window.  *)

let main () = window #show

let _ = main ()
