(****************************************************************************)
(*                 The Calculus of InductiveConstructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA                        ENS-CNRS                *)
(*              Rocquencourt                        Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.1                                 *)
(*                               Oct 1st 1996                               *)
(*                                                                          *)
(****************************************************************************)
(*                                haskell.ml                                *)
(****************************************************************************)

(**************************************************************************)
(*        production of Haskell syntax out of an ML environment           *)
(**************************************************************************)

open Pp
open Std
open More_util
open Names
open Vernacinterp
open Ml_import
open Mlterm
open Genpp

(**************************************************************************)
(*          translation of type expressions and inductive types           *)
(**************************************************************************)

let print_typeid = function
    TYPEname id  -> [< 'sTR(string_of_id id) >]
  | TYPEparam id -> [< 'sTR(string_of_id id) >]

(* haskell_of_type : bool -> MLtype -> std_ppcmds
 * [haskell_of_type b t] formats the type expression t.
 * b=true if we need parentheses around the result. *)

let rec haskell_of_type paren_p = function
     TYarr(t1,t2) ->
       	[< open_par paren_p ;
	   haskell_of_type true t1 ; 'sTR" -> " ;
           haskell_of_type false t2 ;
	   close_par paren_p
        >]

  | TYapp cl ->
      	let n = List.length cl in
	if n=1 then 
	  haskell_of_type false (List.hd cl)
        else
	  [< open_par paren_p ;
             haskell_of_type false (List.hd cl) ; 'sTR" " ;
	     prlist_with_sep (fun () -> [< 'sTR" " >])
      	       	       	     (fun c -> haskell_of_type true c) (List.tl cl) ;
	     close_par paren_p
	  >]

  | TYvar tid ->
      	[< print_typeid tid >]

  | TYglob id ->
      	[< 'sTR(string_of_id id) >]

(* haskell_of_one_inductive : MLind -> std_ppcmds
 * [haskell_of_one_inductive mi] formats the declaration of mutual
 * inductive mi. *)

let haskell_of_one_inductive (pl,name,cl) =
  let fmt_constructor (id,l) =
    [< 'sTR(string_of_id id) ;
       match l with
         [] -> [< >] 
       | _  -> [< 'sTR" " ;
      	       	  prlist_with_sep (fun () -> [< 'sTR" " >]) 
      	       	       	          (fun s -> haskell_of_type true s) l
	       >]
    >] in

  [< 'sTR(string_of_id name) ; 'sTR" " ;
     prlist_with_sep (fun () -> [< 'sTR" " >])
      	       	     (fun id -> [< 'sTR(string_of_id id) >]) pl ;
     if pl=[] then [< >] else [< 'sTR" " >] ;
     
     v 0 [< 'sTR"= " ;
            prlist_with_sep (fun () -> [< 'fNL ; 'sTR"| ">])
                            (fun c -> fmt_constructor c)
		            cl
         >]
  >]


let haskell_of_inductive il =
  [< 'sTR"data " ;
     prlist_with_sep 
      	  (fun () -> [< 'fNL ; 'sTR"data " >])
       	  (fun i -> haskell_of_one_inductive i)
	  il ;
     'fNL
  >]

(**************************************************************************)
(*                  production of Haskell syntax for terms                *)
(**************************************************************************)

let abst = function
    [] -> [< >]
  | l  -> [< 'sTR"\\" ;
             prlist_with_sep (fun  ()-> [< 'sTR" " >])
      	       	       	     (fun id -> [< 'sTR(string_of_id id) >]) l ;
             'sTR" -> " >]


let pr_binding = function
    [] -> [< >]
  | l  -> [< 'sTR" " ; prlist_with_sep (fun () -> [< 'sTR" " >])
      	       	       	(fun id -> [< 'sTR(string_of_id id) >]) l >]


(* pp_mlast : identifier list -> bool -> std_ppcmds list
 *	      -> MLast -> std_ppcmds
 * [pp_mlast idl b args t] formats the Ml term (t a1...am)
 * in the de Bruijn environment idl, where args=[a1...am].
 * b=true if we need parentheses around the result. *)

let rec pp_mlast idl paren_p args = 

  let apply st = match args with
     [] -> st
   | _  -> hOV 2 [< open_par paren_p ; st ; 'sPC ;
                    prlist_with_sep (fun () -> [< 'sPC >]) (fun s -> s) args ;
                    close_par paren_p
                 >] in

  function
    MLrel n ->
      	 apply [< 'sTR(string_of_id (List.nth idl (n-1))) >]

  | MLapp (h, args') ->
      	 let stl = List.map (fun t -> pp_mlast idl true [] t) args' in
         pp_mlast idl paren_p (stl@args) h

  | MLlam _ as t ->
      	 let fl,t' = collect_lambda t in
	 let st = [< abst (List.rev fl) ; pp_mlast (fl@idl) false [] t' >] in
	 if args=[] then
           [< open_par paren_p ; st ; close_par paren_p >]
         else
           apply [< 'sTR"(" ; st ; 'sTR")" >]

  | MLglob id ->
      	 apply [< 'sTR(string_of_id id) >]
	
  | MLcons (_,id,args') ->
      	 if args'=[] then
	   [< 'sTR(string_of_id id) >]
	 else
	   hOV 2 [< open_par paren_p ; 'sTR(string_of_id id) ; 'sPC ;
		    prlist_with_sep (fun () -> [< 'sPC >])
	              (fun t -> pp_mlast idl true [] t) args' ;
		    close_par paren_p
                 >]

  | MLcase (t, pv) ->
      	 apply
      	 [< if args<>[] then [< 'sTR"(" >]  else open_par paren_p ;
      	    v 0 [< 'sTR"case " ; pp_mlast idl false [] t ; 'sTR" of" ;
		   'fNL ; 'sTR"  " ;
		   pp_mlpat idl pv >] ;
	    if args<>[] then [< 'sTR")" >] else close_par paren_p >]

  | MLfix (x_0,x_1,x_2,x_3) ->
      	 pp_mlfix idl paren_p (x_0,x_1,x_2,x_3) args

  | MLexn id ->
      	 [< open_par paren_p ; 'sTR"error \"" ; print_id id ; 'sTR"\"" ;
	    close_par paren_p >]

and pp_mlfix idl paren_p (j,in_p,fid,bl) args =
  let idl' = fid@idl in
  [< open_par paren_p ; v 0 [< 'sTR"let { " ;
       prlist_with_sep
      	  (fun () -> [< 'sTR";" ; 'fNL >])
	  (fun (fi,ti) -> pp_mlfunction idl' fi ti)
	  (List.combine fid bl) ;
       'sTR" }" ; 'fNL ;
       if in_p then 
      	 hOV 2 [< 'sTR"in " ; 'sTR(string_of_id (List.nth fid j)) ;
                  if args<>[] then
                    [< 'sTR" "; prlist_with_sep (fun () -> [<'sTR" ">])
                                 (fun s -> s) args >]
                  else [< >]
      	       >]
       else 
         [< >] >] ;
     close_par paren_p >]

and pp_mlfunction idl f t =
  let bl,t' = collect_lambda t in

  [< 'sTR(string_of_id f) ; pr_binding (List.rev bl) ;
     'sTR" =" ; 'fNL ; 'sTR"  " ;
     hOV 2 (pp_mlast (bl@idl) false [] t') >]

and pp_mlpat idl pv =
  let pp_one_pat (name,ids,t) =
      let paren_p = match t with
                      MLlam _  -> true
                    | MLcase _ -> true
                    | _        -> false in

      hOV 2 [< 'sTR(string_of_id name) ;
      	       if ids=[] then
		 [< >]
      	       else 
		 [< 'sTR" " ; 
		    prlist_with_sep (fun () -> [<'sTR" ">])
      	       	      (fun id -> [< 'sTR(string_of_id id) >]) 
		      (List.rev ids) >] ;
	       'sTR" ->" ; 'sPC ; pp_mlast (ids@idl) paren_p [] t
            >]

  in [< prvect_with_sep (fun () -> [< 'fNL ; 'sTR"  " >])
                        (fun p -> pp_one_pat p)
	                pv >]

(* haskell_of_decl : MLdecl -> std_ppcmds
 * [haskell_of_decl d] formats the declaration d. *)

let haskell_of_decl = function
    DECLtype il -> haskell_of_inductive il

  | DECLabbrev (id, idl, t) ->
          [< 'sTR"type " ; 'sTR(string_of_id id) ; 'sTR" " ;
	     prlist_with_sep (fun () -> [< 'sTR" " >])
	                     (fun id -> [< 'sTR(string_of_id id) >]) idl;
      	     if idl<>[] then [< 'sTR" " >] else [< >] ;'sTR"= " ;
	     haskell_of_type false t ; 'fNL >]

  | DECLglob (id0 , MLfix(n,_,idl,fl)) ->
      let id' = List.nth idl n in
      [< prlist_with_sep (fun () -> [< 'fNL >])
	   (fun (id,f) -> pp_mlfunction (List.rev idl) id f)
	   (List.combine idl fl) ;
	 'fNL ; 
	 if id0 <> id' then
	   [< 'fNL ;
              'sTR(string_of_id id0) ; 'sTR" = " ; 'sTR(string_of_id id') ;
              'fNL >]
	 else
	   [< >]
      >]

  | DECLglob (id, t) ->
      	  [< pp_mlfunction [] id t ; 'fNL >]       

(**************************************************************************)
(*             translation of an environment into Haskell syntax.            *)
(**************************************************************************)

let haskell_of_env env =
  prlist (fun d -> [< haskell_of_decl d ; 'fNL >]) env

(* Optimisation step added to Haskell extraction. Eduardo 9/6/96 *)

module Haskell_renaming =
  struct
    let rename_type_parameter = lo_caml_name_of
    let rename_type           = up_caml_name_of
    let rename_term           = lo_caml_name_of
    let rename_global_type    = up_caml_name_of
    let rename_global_constructor = up_caml_name_of
    let rename_global_term    = lo_caml_name_of
  end

module Haskell_pp : MLPP = 
  struct
    let opt prm = Optimise.haskell_optimise
    let suffixe = ".hs"
    let cofix = true
    let pp_of_env = haskell_of_env
    module Renaming = Haskell_renaming
  end

module Haskell_pp_file = Pp_to_file(Haskell_pp)

(**************************************************************************)
(*            Write Haskell File filename [ ident1 ... identn ].          *)
(**************************************************************************)

(*** TODO: remove overwriting ***)
let _ = overwriting_vinterp_add("WRITEHASKELLFILE",
  function VARG_STRING file :: rem ->
    let prm = parse_param rem in
    (fun () -> Haskell_pp_file.write_extraction_file file prm)
  | _ -> anomaly "WRITEHASKELLFILE called with bad arguments.")

(* $Id: haskell.ml,v 1.7 1999/01/22 17:05:36 filliatr Exp $ *)
