(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                typing.ml                                 *)
(****************************************************************************)

(* A type-checker of the full system with a predicative hierarchy of types *)
(* Three flavors of Prop and Type for Christine's extraction algorithm     *)
(* Inductive types                                                         *)

open Std;;
open Names;;
open Vectops;;
open Generic;;
open Impuniv;;
open Environ;;
open Pp;;
open Term;;
open Termenv;;
open Constrtypes;;
open Reduction;;
open Himsg;;

exception Singleton;;

let empty_evd = Evd.mt_evd();;

(* This type gathers the inductive data mostly used *)

type mutind =
    {fullmind:constr;
     mind:constr;
     nparams:int;
     nconstr:int;
     globargs:constr list;
     realargs:constr list;
     arity:constr};;

(* raise Induc if not an inductive type *)
let try_mutind_of sigma ty =
  let (mind,largs) = find_mrectype sigma ty in
  let mispec = mind_specif_of_mind mind in 
  let nparams = mis_nparams mispec in
  let (params,proper_args) = chop_list nparams largs in 
  {fullmind = ty;
   mind     = mind;
   nparams  = nparams;
   nconstr  = mis_nconstr mispec;
   globargs = params;
   realargs = proper_args;
   arity    = mis_arity mispec};;

(********************************************************************)

(*                         Inductive types                          *)

(********************************************************************)

(* gives the vector of constructors and of 
   types of constructors of an inductive definition 
   correctly instanciated *)

let mis_type_mconstructs mispec =
let specif = mis_lc mispec
and ntypes = mis_ntypes mispec
and nconstr = mis_nconstr mispec in
let make_Ik k = DOPN(MutInd(mispec.sp,k),mispec.args) 
and make_Ck k = DOPN(MutConstruct((mispec.sp,mispec.tyi),k+1),mispec.args) in
  (tabulate_vect make_Ck nconstr, 
   sAPPVList specif (tabulate_list make_Ik ntypes));;

let type_mconstructs sigma mind =
  let redmind  = check_mrectype_spec sigma mind in
  let mispec = mind_specif_of_mind redmind in 
  mis_type_mconstructs mispec;;


(* gives the type of one constructor of an inductive definition *)
let mis_type_mconstruct i  mispec =
let specif = mis_lc mispec
and ntypes = mis_ntypes mispec
and nconstr = mis_nconstr mispec in
let make_Ik k = DOPN(MutInd(mispec.sp,k),mispec.args) in 
 if i > nconstr then error "Not enough constructors in the type" else 
 sAPPViList (i-1) specif (tabulate_list make_Ik ntypes);;

let type_mconstruct sigma i mind =
  let redmind = check_mrectype_spec sigma mind in
  let (sp,tyi,args) = destMutInd redmind in
  let mispec = mind_specif_of_mind redmind in 
    mis_type_mconstruct i mispec
;;

(* gives the type of one constructor of an instance of an 
  inductive definition *)
let type_inst_construct sigma i mind = 
try let (mI,largs) = find_mrectype sigma mind
  in let mispec = mind_specif_of_mind mI
  in let nparams = mis_nparams mispec
  in  let tc = mis_type_mconstruct i mispec in
      let (globargs,_) = chop_list nparams largs in
      hnf_prod_applist sigma "Typing.type_construct" tc globargs
with Induc -> error_not_inductive CCI mind;;


(* compute the type of a well-formed expression *)

let is_correct_arity_dep sigma kd = 
 let rec srec ind (pt,t) =
  match (whd_betadeltaiota sigma pt,whd_betadeltaiota sigma t) with
   DOP2(Prod,a1,DLAM(_,a2)),DOP2(Prod,a1',DLAM(_,a2')) -> conv_x sigma a1 a1' &
                                    srec (applist(lift 1 ind,[Rel 1])) (a2,a2')
 | DOP2(Prod,a1,DLAM(_,a2)),_               ->
    (match whd_betadeltaiota sigma a2 with
     DOP0(Sort s) -> conv_x sigma a1 ind & List.exists (sort_cmp CONV_X s) kd
    | _ -> false)
 | _                             -> false
 in srec;;


let is_correct_arity_nodep sigma kd = 
 let rec srec (pt,t) =
  match whd_betadeltaiota sigma pt,whd_betadeltaiota sigma t with
    DOP2(Prod,a1,DLAM(_,a2)),DOP2(Prod,a1',DLAM(_,a2'))
                                  -> conv_x sigma a1 a1' & srec (a2,a2')
  | DOP2(Prod,a1,DLAM(_,a2)),_    -> false
  | (_,DOP2(Prod,_,_))            -> false
  | (DOP0(Sort s),_)              -> List.exists (sort_cmp CONV_X s) kd
  | _                             -> false
 in srec;;

let make_arity_dep k = 
 let rec mrec c m = match whd_betadeltaiota empty_evd c with 
    DOP0(Sort _)             -> mkArrow m k
  | DOP2(Prod,b,DLAM(n,c_0)) ->
                           prod_name(n,b,mrec c_0 (applist(lift 1 m,[Rel 1])))
  | _                        -> error "make_arity_dep: not an arity"
 in mrec;;

let make_arity_nodep k = 
 let rec mrec c = match whd_betadeltaiota empty_evd c with 
    (DOP0(Sort _))           -> k
  | DOP2(Prod,b,DLAM(x,c_0)) -> DOP2(Prod,b,DLAM(x,mrec c_0))
  | _                        -> error "make_arity_nodep: not an arity"
 in mrec;;

let error_elim_expln kp ki=
  if is_info_sort kp & not (is_info_sort ki)
  then "non-informative objects may not construct informative ones."
  else match (kp,ki) with 
      (DOP0(Sort (Type _)), DOP0(Sort (Prop _))) ->
        "strong elimination on non-small inductive types leads to paradoxes."
    | _ -> "wrong arity"
;;

exception Arity of (constr*constr*string) option;;

let is_correct_arity env sigma kd kn (c,p) = 
 let rec srec ind (pt,t) =
 try (match whd_betadeltaiota sigma pt,whd_betadeltaiota sigma t with
    DOP2(Prod,a1,DLAM(_,a2)),DOP2(Prod,a1',DLAM(_,a2')) 
          -> if conv sigma a1 a1' 
             then srec (applist(lift 1 ind,[Rel 1])) (a2,a2') 
             else raise (Arity None)
  | DOP2(Prod,a1,DLAM(_,a2)),ki
          -> let k = whd_betadeltaiota sigma a2 in 
             let ksort = (match k with DOP0(Sort s) -> s 
                    | _ -> raise (Arity None)) in
             if (conv sigma a1 ind) then
               if (List.exists (sort_cmp CONV_X ksort) kd)
	       then (true,k)
               else raise (Arity (Some(k,ki,error_elim_expln k ki)))
	     else raise (Arity None)
  | k,DOP2(Prod,_,_) ->  raise (Arity None)
  | k,ki   -> let ksort = (match k with DOP0(Sort s) -> s 
                         | _ ->  raise (Arity None)) in
               if List.exists (sort_cmp CONV_X ksort) kn then false,k
               else raise (Arity (Some(k,ki,error_elim_expln k ki))))
 with Arity kinds ->
   let listarity =
     (List.map (fun s -> make_arity_dep (DOP0(Sort s)) t ind) kd)
     @(List.map (fun s -> make_arity_nodep (DOP0(Sort s)) t) kn)
   in error_elim_arity CCI env ind listarity c p pt kinds
 in srec 
;;


(******** Check if the case analysis for c of type indt = APP(mind,largs)
          is compatible with the type typP of the predicate *****)
(* find_case_dep_nparams : bool * int
   le booleen dit si l'elimination est dep (raise Induc si elim incorrecte)
   l'entier donne le nombre de parametres de mind *)

let find_case_dep_nparams env sigma (c,p) (mind,largs) typP =    
  let mispec = mind_specif_of_mind mind in 
  let nparams = mis_nparams mispec
  and kd = mis_kd mispec 
  and kn = mis_kn mispec
  and t  = mis_arity mispec in 
  let (globargs,la) = chop_list nparams largs in
  let glob_t = hnf_prod_applist sigma "find_elim_boolean" t globargs in
  let arity = applist(mind,globargs) in
  let (dep,_) = is_correct_arity env sigma kd kn (c,p) arity (typP,glob_t) in
  (dep, (nparams, globargs,la));;

let find_case_dep_mis env sigma mispec (c,p) (mind,largs) typP =
  fst (find_case_dep_nparams env sigma (c,p) (mind,largs) typP);;


(****************************************************)
(* Type of elimination                              *)
(****************************************************)

(********* Case construction ***********)

let type_one_branch_dep (sigma,nparams,globargs,p) co t = 
  let rec typrec n c =
    match whd_betadeltaiota sigma c with
       DOP2(Prod,a1,DLAM(x,a2)) -> prod_name (x,a1,typrec (n+1) a2)
     | x -> let lAV = destAppL (ensure_appl x) in
            let (_,vargs) = chop_vect (nparams+1) lAV
            in applist 
               (appvect ((lift n p),vargs),
               [applist(co,((List.map (lift n) globargs)@(rel_list 0 n)))])
  in typrec 0 (hnf_prod_applist sigma "type_case" t globargs);;

let type_one_branch_nodep (sigma,nparams,globargs,p) t = 
  let rec typrec n c =
    match whd_betadeltaiota sigma c with 
       DOP2(Prod,a1,DLAM(x,a2)) -> DOP2(Prod,a1,DLAM(x,typrec (n+1) a2))
     | x -> let lAV = destAppL(ensure_appl x) in
            let (_,vargs) = chop_vect (nparams+1) lAV in
            appvect (lift n p,vargs)
  in typrec 0 (hnf_prod_applist sigma "type_case" t globargs);;

(* type_case_branches type un <p>Case c of ... end 
   ct = type de c, si inductif il a la forme APP(mI,largs), sinon raise Induc
   pt = type de p
   type_case_branches retourne (lb, lr); lb est le vecteur des types
   attendus dans les branches du Case; lr est le type attendu du resultat
 *)

let type_case_branches env sigma ct pt p c =
  try
    let ((mI,largs) as indt) = find_mrectype sigma ct in
    let (dep,(nparams,globargs,la)) =
     find_case_dep_nparams env sigma (c,p) indt pt (* check if arity is good *)
    in
    let (lconstruct,ltypconstr) = type_mconstructs sigma mI in
    if dep then (mI,
                 map2_vect (type_one_branch_dep (sigma,nparams,globargs,p))
                   lconstruct ltypconstr,
                 beta_applist (p,(la@[c])))
      else (mI,
            Array.map (type_one_branch_nodep (sigma,nparams,globargs,p)) 
              ltypconstr,
            beta_applist (p,la))
  with Induc -> error_case_not_inductive CCI env c ct;;

let type_case env sigma ct pt p c = 
  try
    let ((mI,largs) as indt) = find_mrectype sigma ct in
    let (dep,(nparams,globargs,la)) =
     find_case_dep_nparams env sigma (c,p) indt pt (* check if arity is good *)
    in
    if dep then beta_applist (p,(la@[c]))
           else beta_applist (p,la)
  with Induc -> error_case_not_inductive CCI env c ct;;

let check_branches_message env sigma (c,ct) (explft,lft) = 
  let n = Array.length explft and expn = Array.length lft in
  let rec check_conv i = 
    if i = n then () else
      if not (conv_leq sigma lft.(i) (explft.(i)))
      then error_ill_formed_branch CCI env c i (nf_betaiota lft.(i))
	  (nf_betaiota explft.(i))
      else check_conv (i+1) 
  in
    if n<>expn then error_number_branches CCI env c ct expn 
    else check_conv 0;;


let rec sort_of_arity sigma c =
     match whd_betadeltaiota sigma c with
     DOP0(Sort( _)) as c'      -> c'
   | DOP2(Prod,_,DLAM(_,c2)) -> sort_of_arity sigma c2
   | _            -> error "sort_of_arity: Not an arity";;


(* B7| Prod intro and quantification *)

(* Cumulativity: slide universes up if product with second type imposes it *)
(* Types of types are assumed to be sorts in head normal form. *)

let sort_of_product domsort rangsort =
  match rangsort with
      Prop _ -> rangsort  (* Product rule (s,Prop,Prop) *)
    | Type u2 ->
        (match domsort with
           (* Product rule (Prop,Type_i,Type_i) *)
             Prop _ -> rangsort
           (* Product rule (Type_i,Type_i,Type_i) *) 
           | Type u1 -> Type (sup(u1,u2)))
;;

let sort_of_product1 domtyp rangsort =
  match domtyp with
      DOP0(Sort domsort) -> sort_of_product domsort rangsort
    | DOP0(Implicit) -> rangsort
    | _ -> assert false
;;

let type_of_product domtyp rangtyp =
  match rangtyp with
      DOP0(Sort s) -> DOP0 (Sort (sort_of_product1 domtyp s))
    | DOP0(Implicit) -> rangtyp
    | _ -> error "type_of_product: Not a sort"
;;

let type_of_sort = function
    DOP0(Sort(Type(u))) -> mkType(super u)
  | DOP0(Sort(Prop cts)) -> mkType prop_univ
  | DOP0(Implicit)       -> mkImplicit
  | _ -> error "type_of_type: Not a sort"
;;

(* Fix introduction *)


(* Checks the type of a (co)fixpoint.
   Fix and CoFix are typed the same way; only the guard condition differs. *)
let type_fixpoint sigma lna lar vdefj =
  let lt = Array.length vdefj in 
    if Array.length lar = lt then 
      for i = 0 to lt-1 do 
        if not (conv_leq sigma (vdefj.(i))._TYPE 
		               (lift lt (body_of_type lar.(i)))) then
          error_ill_typed_rec_body CCI i lna vdefj lar
      done
;;


(*******************************************************************)
(* Checking correctness of recursives declarations                 *)
(*******************************************************************)

exception NotFound;;

(* find an integer in a sorted list *)
let find_sorted_assoc p = 
 let rec findrec = function 
   ((a,ta)::l) -> if a < p then findrec l 
             else if a = p then ta else raise NotFound
  | _     -> raise NotFound
 in findrec ;;

let map_lift_fst_n m = List.map (function (n,t)->(n+m,t));;
let map_lift_fst = map_lift_fst_n 1;;

let rec instantiate_recarg sp lrc ra = 
  match ra with 
      Mrec(j)        -> Imbr(sp,j,lrc)
    | Imbr(sp1,k,l)  -> Imbr(sp1,k, List.map (instantiate_recarg sp lrc) l)
    | Norec          -> Norec
    | Param(k)       -> List.nth lrc k
;;

(* mind_recarg is a vector giving the list of recargs for each type in
the mutually inductive definition *)

(* propagate checking for F,incorporating recursive arguments *)
let check_term mind_recvec f = 
 let rec crec n l (lrec,c) = 
   match (lrec,strip_outer_cast c) with
       (Param(_)::lr,DOP2(Lambda,_,DLAM(_,b))) -> 
         let l' = map_lift_fst l 
         in  crec (n+1) l' (lr,b)
     | (Norec::lr,DOP2(Lambda,_,DLAM(_,b))) -> 
         let l' = map_lift_fst l 
         in  crec (n+1) l' (lr,b)
     | (Mrec(i)::lr,DOP2(Lambda,_,DLAM(_,b)))  -> 
         let l' = map_lift_fst l 
         in  crec (n+1) ((1,mind_recvec.(i))::l') (lr,b)
     | (Imbr(sp,i,lrc)::lr,DOP2(Lambda,_,DLAM(_,b))) -> 
         let l' = map_lift_fst l in
         let sprecargs = mind_recargs (mkMutInd sp i [||]) in
         let lc = (Array.map 
                       (List.map (instantiate_recarg sp lrc))
                    sprecargs.(i))
         in  crec (n+1) ((1,lc)::l') (lr,b)
     | _,f_0 -> f n l f_0
 in crec ;;

let is_inst_var sigma k c = 
  match whd_betadeltaiota_stack sigma c [] with 
      (Rel n,_) -> n=k
    | _         -> false;;

(* Check if t is a subterm of Rel n, and gives its specification, 
assuming lst already gives index of
subterms with corresponding specifications of recursive arguments *)

(* A powerful notion of subterm *)

let is_subterm_specif sigma lcx mind_recvec = 
  let rec crec n lst c = 
    match whd_betadeltaiota_stack sigma c [] with 
        ((Rel k),_)         -> find_sorted_assoc k lst
      |  (DOPN(MutCase _,_) as x,_) ->
           let ( _,_,c,br) = destCase x in
             if Array.length br = 0 
             then [||] 
             else
               let lcv = (try if is_inst_var sigma n c then lcx 
                              else (crec n lst c) 
                          with NotFound -> (Array.create (Array.length br) []))
               in if Array.length br <> Array.length lcv 
	         then assert false
                 else let stl = map2_vect 
                                  (fun lc a -> 
                                     check_term mind_recvec 
                                       crec  n lst (lc,a)) lcv br 
                 in stl.(0)


	 | (DOPN(Fix(_),la) as mc,l) ->
             let (recindxs,i,typarray,funnames,bodies) = destUntypedFix mc in
             let nbfix   = List.length funnames in
             let decrArg = recindxs.(i) in 
             let theBody = bodies.(i)   in
             let (gamma,strippedBody) = decompose_lam_n (decrArg+1) theBody in
             let absTypes = List.map snd gamma in 
             let nbOfAbst = nbfix+decrArg+1 in
             let newlst = 
               if (List.length l < (decrArg+1)) 
               then ((nbOfAbst,lcx) ::
                     (map_lift_fst_n nbOfAbst lst))
               else 
                 let theDecrArg  = List.nth l decrArg in
                 let recArgsDecrArg = 
                   try (crec n lst theDecrArg)
	           with NotFound -> Array.create 0 [] 
                 in if (Array.length recArgsDecrArg)=0
                   then ((nbOfAbst,lcx) ::
                         (map_lift_fst_n nbOfAbst lst))
                   else ((1,recArgsDecrArg)::
                         (nbOfAbst,lcx) ::
                         (map_lift_fst_n nbOfAbst lst))                     
             in  (crec (n+nbOfAbst) newlst strippedBody)

      |  (DOP2(Lambda,_,DLAM(_,b)),[]) -> 
           let lst' = map_lift_fst lst 
           in crec  (n+1) lst' b

(***** Experimental change *************************)
|  (DOP0(Meta _),_)             -> [||]
(***************************************************)
|  _                        -> raise NotFound
  in crec;;


let is_subterm sigma lcx mind_recvec n lst c = 
  try is_subterm_specif sigma lcx mind_recvec n lst c; true 
  with NotFound -> false;;



(* Checking function for terms containing existential variables.
 The function noccur_with_meta consideres the fact that
 each existential variable (as well as each isevar)
 in the term appears applied to its local context,
 which may contain the CoFix variables. These occurrences of CoFix variables
 are not considered *)

let noccur_with_meta sigma n m term = let rec occur_rec n = function
    Rel(p)        -> if n<=p & p<n+m then raise Occur
  | VAR _         -> ()
  | DOPN(AppL,cl) ->
      (match strip_outer_cast cl.(0) with
        DOP0 (Meta _) -> ()
      | _             -> Array.iter (occur_rec n) cl)
  | DOPN(Const sp, cl) when Evd.in_dom sigma sp -> ()
  | DOPN(op,cl)   -> Array.iter (occur_rec n) cl
  | DOPL(_,cl)    -> List.iter (occur_rec n) cl
  | DOP0(_)       -> ()
  | DOP1(_,c)     -> occur_rec n c
  | DOP2(_,c1,c2) -> occur_rec n c1; occur_rec n c2
  | DLAM(_,c)     -> occur_rec (n+1) c
  | DLAMV(_,v)    -> Array.iter (occur_rec (n+1)) v
  in try (occur_rec n term; true) with Occur -> false;;


(*******Experimental change ***************************************)
(* noccur_bet ==> noccur_with_meta + an entry for Meta in the case*)
(******************************************************************)

(* Auxiliary function: it checks a condition f depending on a deBrujin
 index  for a certain number of abstractions *)

let rec check_subterm_rec_meta sigma vectn k def = 
  (k < 0) or
  (let nfi = Array.length vectn in 
 (* check fi does not appear in the k+1 first abstractions, 
      gives the type of the k+1-eme abstraction  *)
   let rec check_occur n def = 
      (match strip_outer_cast def with
	   DOP2(Lambda,a,DLAM(_,b)) -> 
	     if noccur_with_meta sigma n nfi a then
               if n = k+1 then (a,b) else check_occur (n+1) b
             else error "Bad occurrence of recursive call"
         | _ -> error "Not enough abstractions in the definition") in
   let (c,d) = check_occur 1 def in 
   let (mI, largs) = 
     (try find_minductype sigma c 
      with Induc -> 
	error "Recursive definition on a non inductive type") in
   let (sp,tyi,_) = destMutInd mI in
   let mind_recvec = mind_recargs mI in 
   let lcx = mind_recvec.(tyi)  in
    
(* n   = decreasing argument in the definition;
   lst = a mapping var |-> recargs
   t   = the term to be checked
*)

   let rec check_rec_call n lst t = 
            (* n gives the index of the recursive variable *)
     (noccur_with_meta sigma (n+k+1) nfi t) or 
(* no recursive call in the term *)
      (match whd_betadeltaiota_stack sigma t [] with 
	   (Rel p,l) -> 
	     if n+k+1 <= p & p < n+k+nfi+1 (* recursive call *)
	     then let glob = nfi+n+k-p in  (* the index of the recursive call *) 
		  let np = vectn.(glob) in (* the decreasing arg of the rec call *)
		    if List.length l > np then 
		      (match chop_list np l with
			   (la,(z::lrest)) -> 
	                     if (is_subterm sigma lcx mind_recvec n lst z) 
                  	     then List.for_all (check_rec_call n lst) (la@lrest)
                  	     else error "Recursive call applied to an illegal term"
		      	 | _ -> assert false)
		    else error  "Not enough arguments for the recursive call"
	     else List.for_all (check_rec_call n lst) l        
	 | (DOPN(MutCase _,_) as mc,l) ->
             let (ci,p,c_0,lrest) = destCase mc in
             let lc = (try if is_inst_var sigma n c_0 
                           then lcx 
		           else is_subterm_specif sigma lcx mind_recvec n lst c_0
		       with NotFound -> Array.create (Array.length lrest) []) in
               (for_all2eq_vect
		  (fun c_0 a -> check_term mind_recvec (check_rec_call) n lst (c_0,a))
		  lc lrest) & (List.for_all (check_rec_call n lst) (c_0::p::l)) 

(* Enables to traverse Fixpoint definitions in a more intelligent
   way, ie, the rule :

   if - g = Fix g/1 := [y1:T1]...[yp:Tp]e &
      - f is guarded with respect to the set of pattern variables S 
        in a1 ... am        &
      - f is guarded with respect to the set of pattern variables S 
        in T1 ... Tp        &
      - ap is a sub-term of the formal argument of f &
      - f is guarded with respect to the set of pattern variables S+{yp}
        in e
   then f is guarded with respect to S in (g a1 ... am).

Eduardo 7/9/98
*)
	 | (DOPN(Fix(_),la) as mc,l) ->
             (List.for_all (check_rec_call n lst) l) &
             let (recindxs,i,typarray,funnames,bodies) = destUntypedFix mc in
             let nbfix       = List.length funnames in
             let decrArg     = recindxs.(i) 
             in if (List.length l < (decrArg+1)) 
                then (for_all_vect (check_rec_call n lst) la)
                else 
                  let theDecrArg  = List.nth l decrArg in
                  let recArgsDecrArg = 
                    try (is_subterm_specif sigma lcx mind_recvec n lst theDecrArg)
	            with NotFound -> Array.create 0 [] 
                  in if (Array.length recArgsDecrArg)=0
                    then (for_all_vect (check_rec_call n lst) la)
                    else 
                 let theBody = bodies.(i)   in
                 let (gamma,strippedBody) = decompose_lam_n (decrArg+1) theBody in
                 let absTypes = List.map snd gamma in 
                 let nbOfAbst = nbfix+decrArg+1 in
                 let newlst = ((1,recArgsDecrArg)::(map_lift_fst_n nbOfAbst lst))
                 in ((for_all_vect 
			(fun t -> check_rec_call n lst t)
			typarray) &
                      (for_all_i (fun n -> check_rec_call n lst) n absTypes) &
                      (check_rec_call (n+nbOfAbst) newlst strippedBody))


	 | (DOP2(_,a,b),l) -> (check_rec_call n lst a)
                               & (check_rec_call n lst b)
                               & (List.for_all (check_rec_call n lst) l)
	 | (DOPN(_,la),l) -> (for_all_vect (check_rec_call n lst) la)
                              & (List.for_all (check_rec_call n lst) l)
	 | (DOP0 (Meta _),l) -> true
	 | (DLAM(_,t),l)  -> (check_rec_call (n+1) (map_lift_fst lst) t)
                              & (List.for_all (check_rec_call n lst) l)
	 | (DLAMV(_,vt),l)  -> 
	     (for_all_vect (check_rec_call (n+1) (map_lift_fst lst)) vt)
             & (List.for_all (check_rec_call n lst) l)
	 | (_,l)    ->   List.for_all (check_rec_call n lst) l
      ) 

  in check_rec_call 1 [] d);;    


(* vargs is supposed to be built from A1;..Ak;[f1]..[fk][|d1;..;dk|]
and vdeft is [|t1;..;tk|] such that f1:A1,..,fk:Ak |- di:ti
nvect is [|n1;..;nk|] which gives for each recursive definition 
the inductive-decreasing index 
the function checks the convertibility of ti with Ai *)

let check_fix sigma f = 
  match f with
    | DOPN(Fix(nvect,j),vargs) -> 
 	let nbfix = let nv = Array.length vargs in 
          if nv < 2 then error "Ill-formed recursive definition" else nv-1 in
  	let varit = Array.sub vargs 0 nbfix in
  	let ldef = last_vect vargs in
  	let ln = Array.length nvect and la = Array.length varit in
	  if ln <> la then error "Ill-formed fix term"
	  else let (lna,vdefs) = decomp_DLAMV_name ln ldef in 
	       let vlna = Array.of_list lna in
	       let check_type i = 
		 (try check_subterm_rec_meta sigma nvect nvect.(i) vdefs.(i) 
		  with UserError (s,str) ->
		    error_ill_formed_rec_body str CCI lna i vdefs) in
		 for i = 0 to ln-1 do check_type i done
    | _ -> assert false
;;
(*********************************************)
(* Checking correctness of CoFix definitions *)
(*********************************************)


(* The function check_guard_rec checks the guarded condition for one 
   definition of the block.
   The argument  nbfix is the number of functions defined in the block.
   The argument  def is the definition of certain function of the block.
   The argument deftype is the type of def  *)


(* Checking function without existential variables *)

let check_guard_rec sigma nbfix def deftype = 

  let rec codomain_is_coind c  =
     match whd_betadeltaiota sigma c  with
           DOP2(Prod,a,DLAM(_,b)) ->  codomain_is_coind b 
         | b  -> try find_mcoinductype sigma b
                 with Induc -> error "The codomain is not a coinductive type"
         | _ -> error "Type of Cofix function not as expected"   in

  let (mI,_) = codomain_is_coind deftype in 
  let (sp, tyi, _) = destMutInd mI in
  let vlra = (mind_recargs mI).(tyi) in  

  let rec check_rec_call alreadygrd n vlra  t = 
  if noccur_bet n nbfix t then true (* no recursive call in the term *)
  else match whd_betadeltaiota_stack sigma t [] with 

      (Rel p,l) -> 
              if n <= p & p < n+nbfix (* recursive call *)
             then if alreadygrd
                  then if List.for_all (noccur_bet n nbfix) l 
                       then true
                       else error "Nested recursive occurrences"
                  else error "Unguarded  recursive call"
             else  error "Variable applied to a recursive call "

    | (DOPN ((MutConstruct((x,y),i)),l), args)  ->
         let lra =vlra.(i-1) in 
         let mI = DOPN(MutInd(x,y),l) in
         let _,realargs = chop_list (mind_nparams mI) args in
         let rec process_args_of_constr l lra   =
           match l with
               []    -> true 
             | t::lr ->(match lra  with 
                            [] -> anomaly "a constructor with an empty list of recargs is being applied" 
                          |  (Mrec _)::lrar -> 
                               (check_rec_call true n  vlra t) &
                               (process_args_of_constr lr lrar)
				 
                          |  (Imbr(sp,i,lrc)::lrar)  -> 
                               let sprecargs = 
                                 mind_recargs (mkMutInd sp i [||]) in
                               let lc = (Array.map 
                                           (List.map 
                                              (instantiate_recarg sp lrc))
                                           sprecargs.(i))
                               in (check_rec_call true n  lc t ) &
                                  (process_args_of_constr lr lrar)
			       
                          |  _::lrar  -> 
                               if    (noccur_bet n nbfix t) 
                               then  (process_args_of_constr lr lrar)
                               else  error "Recursive call inside a non-recursive argument of constructor"
)
         in (process_args_of_constr realargs lra)

    | (DOP2(Lambda,a,DLAM(_,b)),[]) -> 
        if (noccur_bet n nbfix a)
        then (check_rec_call alreadygrd (n+1)  vlra b)
        else error "Recursive call in the type of an abstraction"
	  
    | (DOPN(MutCase _,_) as mc,l) -> 
        let (_,p,c,vrest) = destCase mc in
          if (noccur_with_meta sigma n nbfix p) 
          then if (noccur_with_meta sigma n nbfix c) 
          then if (List.for_all (noccur_with_meta sigma n nbfix) l)
          then (for_all_vect (check_rec_call alreadygrd n vlra) vrest)
          else error "Recursive call in the argument of a function defined by cases"        
          else error "Recursive call in the argument of a case expression"
          else error "Recursive call in the type of a Case expression" 
	    

    | _    ->   error "Definition not in guarded form"

in check_rec_call false 1 vlra def;;


let check_guard_rec_meta sigma nbfix def deftype = 

  let rec codomain_is_coind c  =
     match (whd_betadeltaiota sigma (strip_outer_cast c)) with
           DOP2(Prod,a,DLAM(_,b)) ->  codomain_is_coind b 
         | b  -> try find_mcoinductype sigma b
                 with Induc -> error "The codomain is not a coinductive type"
         | _ -> error "Type of Cofix function not as expected"   in

  let (mI, _) = codomain_is_coind deftype in
  let (sp,tyi,_) = destMutInd mI in
  let lvlra = (mind_recargs mI) in
  let vlra = lvlra.(tyi) in  
    
  let rec check_rec_call alreadygrd n vlra  t = 
    if (noccur_with_meta sigma n nbfix t) 
    then true 
    else match whd_betadeltaiota_stack sigma t [] with 
     
      (DOP0 (Meta _), l) -> true

    |  (Rel p,l) -> 
              if n <= p & p < n+nbfix (* recursive call *)
             then if alreadygrd
                  then if List.for_all (noccur_with_meta sigma n nbfix) l 
                       then true
                       else error "Nested recursive occurrences"
                  else error "Unguarded  recursive call"
             else  error "Va "

    | (DOPN ((MutConstruct((x,y),i)),l), args)  ->
         let lra =vlra.(i-1) in 
         let mI = DOPN(MutInd(x,y),l) in
         let _,realargs = chop_list (mind_nparams mI) args in
         let rec process_args_of_constr l lra   =
                     match l with
                         []    -> true 
                       | t::lr ->( match lra  with 
                                    [] -> error "Anomaly: a constructor with an empty list of recargs is being applied" 
                                 |  (Mrec i)::lrar -> 
                                         let newvlra = lvlra.(i)
                                         in (check_rec_call true n  newvlra t) &
                                            (process_args_of_constr lr lrar)

                                 |  (Imbr(sp,i,lrc)::lrar)  ->
                                      let sprecargs = 
                                        mind_recargs (mkMutInd sp i [||]) in
                                      let lc = (Array.map 
                                                  (List.map 
                                                     (instantiate_recarg sp lrc))
                                                  sprecargs.(i))
                                      in (check_rec_call true n lc t) &
                                         (process_args_of_constr lr lrar)
                 
                                 |  _::lrar  -> 
                                         if (noccur_with_meta sigma n nbfix t) 
                                         then (process_args_of_constr lr lrar)
                                         else error "Recursive call inside a non-recursive argument of constructor")
         in (process_args_of_constr realargs lra)
	      

    | (DOP2(Lambda,a,DLAM(_,b)),[]) -> 
               if (noccur_with_meta sigma n nbfix a)
               then (check_rec_call alreadygrd (n+1)  vlra b)
               else error "Recursive call in the type of an abstraction"


    | (DOPN(CoFix(j),vargs),l) -> 
        if  (List.for_all (noccur_with_meta sigma n nbfix) l)
        then 
          let nbfix = let nv = Array.length vargs in 
            if nv < 2 
            then error "Ill-formed recursive definition" 
            else nv-1 in
          let varit = Array.sub vargs 0 nbfix in
          let ldef  = last_vect vargs in
          let la    = Array.length varit in
          let (lna,vdefs) = decomp_DLAMV_name la ldef in
          let vlna = Array.of_list lna 
          in if  (for_all_vect (noccur_with_meta sigma n nbfix) varit)
            then (for_all_vect (check_rec_call alreadygrd (n+1) vlra) vdefs)
              &
              (List.for_all (check_rec_call alreadygrd (n+1) vlra) l) 
            else error "Recursive call in the type of a declaration"
        else error "Unguarded recursive call"

    | (DOPN(MutCase _,_) as mc,l) -> 
        let (_,p,c,vrest) = destCase mc in
          if (noccur_with_meta sigma n nbfix p) 
          then if (noccur_with_meta sigma n nbfix c) 
          then if (List.for_all (noccur_with_meta sigma n nbfix) l)
          then (for_all_vect (check_rec_call alreadygrd n vlra) vrest)
          else error "Recursive call in the argument of a function defined by cases"        
          else error "Recursive call in the argument of a case expression"
          else error "Recursive call in the type of a Case expression" 
            
    | _    -> error "Definition not in guarded form"

in check_rec_call false 1 vlra def;;

(* The  function which checks that the whole block of definitions 
   satisfies the guarded condition *)
let check_cofix sigma f = 
  match f with
    |  DOPN(CoFix(j),vargs) -> 
	 let nbfix = let nv = Array.length vargs in 
           if nv < 2 
           then error "Ill-formed recursive definition" 
           else nv-1 in
	 let varit = Array.sub vargs 0 nbfix in
	 let ldef  = last_vect vargs in
	 let la    = Array.length varit in
	 let (lna,vdefs) = decomp_DLAMV_name la ldef in
	 let vlna = Array.of_list lna in
	 let check_type i = 
	   (try check_guard_rec_meta sigma nbfix vdefs.(i)  varit.(i)
	    with UserError (s,str) -> 
	      error_ill_formed_rec_body str CCI lna i vdefs) in
	   for i = 0 to nbfix-1 do check_type i done
    | _ -> assert false
;;

(* A function which checks that a term well typed verifies both
   syntaxic conditions *)


let control_only_guard sigma = 
 let rec control_rec = function
    Rel(p)        -> ()
  | VAR _         -> ()
  | DOP0(_)       -> ()
  | DOPN(CoFix(_),cl) as cofix  -> (check_cofix sigma cofix) ; 
                                   (Array.iter control_rec cl)
  | DOPN(Fix(_),cl) as fix  -> (check_fix sigma fix) ; 
                               (Array.iter control_rec cl)
  | DOPN(_,cl)    -> Array.iter control_rec cl
  | DOPL(_,cl)    -> List.iter control_rec cl
  | DOP1(_,c)     -> control_rec c
  | DOP2(_,c1,c2) -> control_rec c1; control_rec c2
  | DLAM(_,c)     -> control_rec c
  | DLAMV(_,v)    -> Array.iter control_rec v
 in control_rec 
;;



(* $Id: typing.ml,v 1.29 1999/10/22 12:49:21 herbelin Exp $ *)
