(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                 trad.ml                                  *)
(****************************************************************************)

open Std;;
open Pp;;
open Names;;
open Vectops;;
open Impuniv;;
open Generic;;
open Evd;;
open Term;;
open Printer;;
open Reduction;;
open Termenv;;
open Typing;;
open Himsg;;
open More_util;;
open Machops;;
open Classops;;
open List;;
open Recordops;; 
open Tradevar;;

(* Pb: Mach cannot type evar in the general case (all Const must be applied
 * to VARs). But evars may be applied to Rels or other terms! This is the
 * difference between type_of_const and type_of_const2.
 *)
(* Fonctions temporaires pour relier la forme caste de la forme jugement *)
let tjudge_of_cast_safe sigma env var =
  match under_casts (nf_ise1 sigma) var with
   DOP2 (Cast, b, t) ->
     (match whd_betadeltaiota sigma t with
	  DOP0 (Sort s) -> {body=b; typ=s}
	| _ -> anomaly "Not a type (tjudge_of_cast)")
  | c -> Mach.execute_rec_type sigma env c;;
(* FIN TMP ******)

let mt_evd = Evd.mt_evd();;

let vect_lift_type = map_i_vect (fun i t -> type_app (lift i) t) 0;;

let class_of1 sigma t = class_of sigma (nf_ise1 sigma t);;


let j_nf_ise sigma {_VAL=v;_TYPE=t;_KIND=k} =
  {_VAL=nf_ise1 sigma v;_TYPE=nf_ise1 sigma t;_KIND=k}
let jl_nf_ise sigma = List.map (j_nf_ise sigma)
let jv_nf_ise sigma = Array.map (j_nf_ise sigma)


(* This code (i.e. try_solve_pb, solve_pb, etc.) takes a unification
 * problem, and tries to solve it. If it solves it, then it removes
 * all the conversion problems, and re-runs conversion on each one, in
 * the hopes that the new solution will aid in solving them.
 *
 * The kinds of problems it knows how to solve are those in which
 * the usable arguments of an existential var are all themselves
 * universal variables.
 * The solution to this problem is to do renaming for the Var's,
 * to make them match up with the Var's which are found in the
 * hyps of the existential, to do a "pop" for each Rel which is
 * not an argument of the existential, and a subst1 for each which
 * is, again, with the corresponding variable. This is done by
 * Tradevar.evar_define
 *
 * Thus, we take the arguments of the existential which we are about
 * to assign, and zip them with the identifiers in the hypotheses.
 * Then, we process all the Var's in the arguments, and sort the
 * Rel's into ascending order.  Then, we just march up, doing
 * subst1's and pop's.
 *
 * NOTE: We can do this more efficiently for the relative arguments,
 * by building a long substituend by hand, but this is a pain in the
 * ass.
 *)

let rec evar_apprec isevars stack c =
  let (t,stack) = Reduction.apprec !isevars c stack in 
  if ise_defined isevars t
  then evar_apprec isevars stack (const_value !isevars t)
  else (t,stack)
;;

let conversion_problems = ref ([] : (conv_pb * constr * constr) list);;

let reset_problems () = conversion_problems := [];;

let add_conv_pb pb = (conversion_problems := pb::!conversion_problems);;

let get_changed_pb lsp =
  let (pbs,pbs1) = List.fold_left
      (fun (pbs,pbs1) pb ->
    	if status_changed lsp pb then (pb::pbs,pbs1)
        else (pbs,pb::pbs1))
      ([],[])
      !conversion_problems in
  conversion_problems := pbs1;
  pbs
;;


(* Precondition: one of the terms of the pb is an uninstanciated evar,
 * possibly applied to arguments.
 *)
let rec solve_pb isevars pb =
  match solve_simple_eqn (evar_conv_x isevars None CONV_X) isevars pb with
    Some lsp ->
      let pbs = get_changed_pb lsp in
      List.for_all
 	(fun (pbty,t1,t2) -> evar_conv_x isevars None pbty t1 t2)
 	pbs
  | None -> (add_conv_pb pb; true)

and evar_conv_x isevars b pbty term1 term2 =
  let term1 = whd_ise1 !isevars term1
  and term2 = whd_ise1 !isevars term2 in
   if eq_constr term1 term2 then true
   else if (not(has_undefined_isevars isevars term1)) &
            not(has_undefined_isevars isevars term2)
   then fconv pbty !isevars term1 term2
   else if ise_undefined isevars term1 or ise_undefined isevars term2
   then solve_pb isevars (pbty,term1,term2)
   else 
     let (t1,l1) = evar_apprec isevars [] term1
     and (t2,l2) = evar_apprec isevars [] term2 in
     if (head_is_embedded_exist isevars t1 & not(is_eliminator t2))
         or (head_is_embedded_exist isevars t2 & not(is_eliminator t1))
     then (add_conv_pb (pbty,applist(t1,l1),applist(t2,l2)); true)
     else evar_eqappr_x isevars b pbty (t1,l1) (t2,l2)

and evar_eqappr_x isevars b pbty appr1 appr2 =
  match (appr1,appr2) with
    ((DOPN(Const sp1,al1) as term1,l1), (DOPN(Const sp2,al2) as term2,l2)) ->
      let f1 () = 
      	(ise_undefined isevars term1 or ise_undefined isevars term2) &
	if List.length l1 > List.length l2 then 
          let (deb1,rest1) = chop_list (List.length l1-List.length l2) l1
          in solve_pb isevars(pbty,applist(term1,deb1),term2)
            & for_all2eq (evar_conv_x isevars b CONV_X) rest1 l2
	else
	  let (deb2,rest2) = chop_list (List.length l2-List.length l1) l2
          in solve_pb isevars(pbty,term1,applist(term2,deb2))
            & for_all2eq (evar_conv_x isevars b CONV_X) l1 rest2
      and f2 () =
         (sp1 = sp2)
       & (for_all2eq_vect (evar_conv_x isevars b CONV_X) al1 al2)  
       & (for_all2eq (evar_conv_x isevars b CONV_X) l1 l2)
      and f3 () =
	 b<>None
       & (try conv_record isevars b 
           (try check_conv_record appr1 appr2
           with Not_found -> check_conv_record appr2 appr1)
          with _ -> false)
      and f4 () =
        if evaluable_const !isevars term2 then
          evar_eqappr_x isevars b pbty
            appr1 (evar_apprec isevars l2 (const_value !isevars term2))
        else if evaluable_const !isevars term1 then
          evar_eqappr_x isevars b pbty
            (evar_apprec isevars l1 (const_value !isevars term1)) appr2
        else false
      in ise_try isevars [f1; f2; f3; f4]

  | ((DOPN(Const _,_) as term1,l1),(t2,l2)) ->  
      let f1 () =
       	ise_undefined isevars term1 &
       	(List.length l1 <= List.length l2) &
       	let (deb2,rest2) = chop_list (List.length l2-List.length l1) l2
        in solve_pb isevars(pbty,term1,applist(t2,deb2))
          & for_all2eq (evar_conv_x isevars b CONV_X) l1 rest2
      and f2 () =
	b<>None &
 	(try conv_record isevars b (check_conv_record appr1 appr2)
        with _ -> false)
      and f3 () = 
	evaluable_const !isevars term1 &
 	evar_eqappr_x isevars b pbty
          (evar_apprec isevars l1 (const_value !isevars term1)) appr2
      in ise_try isevars [f1; f2; f3]

  | ((t1,l1),(DOPN(Const _,_) as t2,l2))  -> 
      let f1 () =
       	ise_undefined isevars t2 &
       	(List.length l2 <= List.length l1) &
       	let (deb1,rest1) = chop_list (List.length l1-List.length l2) l1
        in solve_pb isevars(pbty,applist(t1,deb1),t2)
          & for_all2eq (evar_conv_x isevars b CONV_X) rest1 l2
      and f2 () = 
	b<>None &
 	(try (conv_record isevars b (check_conv_record appr2 appr1))
        with _ -> false)
      and f3 () =
        evaluable_const !isevars t2 &
 	evar_eqappr_x isevars b pbty
          appr1 (evar_apprec isevars l2 (const_value !isevars t2))
      in ise_try isevars [f1; f2; f3]

  | ((DOPN(Abst _,_) as term1,l1),(DOPN(Abst _,_) as term2,l2)) ->
      let f1 () =
        (term1=term2) &
 	(List.length(l1) = List.length(l2)) &
 	(for_all2 (evar_conv_x isevars b CONV_X) l1 l2)
      and f2 () =
        if (evaluable_abst term2)
	then evar_eqappr_x isevars b pbty
               appr1 (evar_apprec isevars l2 (abst_value term2))
	else evaluable_abst term1
            & evar_eqappr_x isevars b pbty
                (evar_apprec isevars l1 (abst_value term1)) appr2
      in ise_try isevars [f1; f2]

  | ((DOPN(Abst _,_) as term1,l1),_) ->  
        (evaluable_abst term1)
      & evar_eqappr_x isevars b pbty
          (evar_apprec isevars l1 (abst_value term1)) appr2

  | (_,(DOPN(Abst _,_) as term2,l2))  -> 
        (evaluable_abst term2)
      & evar_eqappr_x isevars b pbty
 	  appr1 (evar_apprec isevars l2 (abst_value term2))

  | ((Rel(n),l1),(Rel(m),l2)) ->
	 n=m
       & (List.length(l1) = List.length(l2))
       & (for_all2 (evar_conv_x isevars b CONV_X) l1 l2)
  | ((DOP2(Cast,c,_),l),_) -> evar_eqappr_x isevars b pbty (c,l) appr2
  | (_,(DOP2(Cast,c,_),l)) -> evar_eqappr_x isevars b pbty appr1 (c,l)
  | ((VAR id1,l1),(VAR id2,l2)) ->
	 (id1=id2 & (List.length l1 = List.length l2)
	    & (for_all2 (evar_conv_x isevars b CONV_X) l1 l2))
  | ((DOP0(Meta(n)),l1),(DOP0(Meta(m)),l2)) ->
	 (n=m & (List.length(l1) = List.length(l2))
	    & (for_all2 (evar_conv_x isevars b CONV_X) l1 l2))
  | ((DOP0(Sort s1),[]),(DOP0(Sort s2),[])) -> sort_cmp pbty s1 s2
  | ((DOP2(Lambda,c1,DLAM(_,c2)),[]), (DOP2(Lambda,c'1,DLAM(_,c'2)),[])) -> 
	  evar_conv_x isevars b CONV_X c1 c'1
	    & evar_conv_x isevars b CONV_X c2 c'2
  | ((DOP2(Prod,c1,DLAM(n,c2)),[]), (DOP2(Prod,c'1,DLAM(_,c'2)),[])) -> 
         evar_conv_x isevars b CONV_X c1 c'1 
         & evar_conv_x isevars
	    (option_app
	       (add_rel (n,tjudge_of_cast_safe !isevars (outSOME b) c1)) b)
	    pbty c2 c'2
  | ((DOPN(MutInd _ as o1,cl1) as ind1,l'1),
     (DOPN(MutInd _ as o2,cl2) as ind2,l'2)) ->
	  o1=o2
	& for_all2eq_vect (evar_conv_x isevars b CONV_X) cl1 cl2
        & for_all2eq (evar_conv_x isevars b CONV_X) l'1 l'2
         
  | ((DOPN(MutConstruct _ as o1,cl1) as constr1,l1),
     (DOPN(MutConstruct _ as o2,cl2) as constr2,l2)) ->
	  o1=o2
        & for_all2eq_vect (evar_conv_x isevars b CONV_X) cl1 cl2
        & for_all2eq (evar_conv_x isevars b CONV_X) l1 l2

  | ((DOPN(MutCase _,_) as constr1,l'1),
     (DOPN(MutCase _,_) as constr2,l'2)) ->
	  let (_,p1,c1,cl1) = destCase constr1 in
	  let (_,p2,c2,cl2) = destCase constr2 in
	    evar_conv_x isevars b CONV_X p1 p2
	  & evar_conv_x isevars b CONV_X c1 c2
	  & (for_all2eq_vect (evar_conv_x isevars b CONV_X) cl1 cl2)
	  & (for_all2eq (evar_conv_x isevars b CONV_X) l'1 l'2)

  | ((DOPN(Fix _ as o1,cl1),l1),(DOPN(Fix _ as o2,cl2),l2))   ->
	 o1=o2 & 
	 (for_all2eq_vect (evar_conv_x isevars b CONV_X) cl1 cl2) &
	 (for_all2eq (evar_conv_x isevars b CONV_X) l1 l2)

  | ((DOPN(CoFix(i1),cl1),l1),(DOPN(CoFix(i2),cl2),l2))   ->
	 i1=i2 & 
	 (for_all2eq_vect (evar_conv_x isevars b CONV_X) cl1 cl2) &
	 (for_all2eq (evar_conv_x isevars b CONV_X) l1 l2)

  | (DOP0(Implicit),[]),(DOP0(Implicit),[]) -> true
(* added to compare easily the specification of fixed points
 * But b (optional env) is not updated!
 *)
  | (DLAM(_,c1),[]),(DLAM(_,c2),[]) -> evar_conv_x isevars b pbty c1 c2
  | (DLAMV(_,vc1),[]),(DLAMV(_,vc2),[]) ->
	 for_all2eq_vect (evar_conv_x isevars b pbty) vc1 vc2
  | _ -> false


and conv_record isevars ((Some env) as b) (c,bs,(xs,xs1),(us,us1),(ts,ts1),t) = 
   let ks =
     List.fold_left
       (fun ks b ->
	 let k = new_isevar isevars env (substl ks b) CCI in (k::ks))
       [] bs
   in
     if (for_all2eq 
	   (fun u1 u -> evar_conv_x isevars b CONV_X u1 (substl ks u))
	   us1 us)
        &
        (for_all2eq 
	   (fun x1 x -> evar_conv_x isevars b CONV_X x1 (substl ks x))
	   xs1 xs)
        & (for_all2eq (evar_conv_x isevars b CONV_X) ts ts1)
	& (evar_conv_x isevars b CONV_X t 
	     (if ks=[] then c 
	      else  (DOPN(AppL,Array.of_list(c::(rev ks))))))
     then
 (*TR*) (if !compter then (nbstruc:=!nbstruc+1;
                          nbimplstruc:=!nbimplstruc+(length ks);true)
                     else true)
     else false


and check_conv_record (t1,l1) (t2,l2) = 
  try
    let {o_DEF=c;o_TABS=bs;o_TPARAMS=xs;o_TCOMPS=us} = 
      objdef_info (cte_of_constr t1,cte_of_constr t2) in
    let xs1,t::ts = chop_list (List.length xs) l1 in
    let us1,ts1= chop_list (List.length us) l2 in
    c,bs,(xs,xs1),(us,us1),(ts,ts1),t
  with _ -> raise Not_found  (* try ... with _ -> ... *)
;;


let the_conv_x isevars env t1 t2 =
  conv_x !isevars t1 t2 or evar_conv_x isevars (Some env) CONV_X t1 t2;;

(* In the case where conv_x_leq returns true, we may also want to call
 * evar_conv_x in order to solve some undefined evars as in the
 * unification of (K O) and (K ?i) when K := [_]True.
 * These terms are convertible whatever the value of ?i, but maybe
 * one wants to infer ?i = O, which is not so clear to me.
 *)
let the_conv_x_leq isevars env t1 t2 =
  if conv_x_leq !isevars t1 t2 then
    if (has_undefined_isevars isevars t1) or (has_undefined_isevars isevars t2)
    then 
      try let _ = evar_conv_x isevars (Some env) CONV_X_LEQ t1 t2 in true
      with UserError _ -> true
    else true
  else evar_conv_x isevars (Some env) CONV_X_LEQ t1 t2     
(*
let the_conv_x_leq isevars env t1 t2 =
  conv_x_leq !isevars t1 t2
  or evar_conv_x isevars (Some env) CONV_X_LEQ t1 t2     
;; 
*)

let evar_type_fixpoint isevars env lna lar vdefj =
  let lt = Array.length vdefj in 
    if Array.length lar = lt then 
      for i = 0 to lt-1 do 
        if not (the_conv_x_leq isevars env
		  (vdefj.(i))._TYPE (lift lt (body_of_type lar.(i)))) then
          error_ill_typed_rec_body CCI i lna (jv_nf_ise !isevars vdefj) 
	    (Array.map (type_app (nf_ise1 !isevars)) lar)
      done;;


let apply_rel_list isevars env argjl funj tycon =
let rec apply_rec acc typ = function
    [] -> (match tycon with None -> ()
            | Some typ' -> let _ = the_conv_x_leq isevars env typ typ' in ());
          {_VAL=applist(j_val_only funj,List.map j_val_only (List.rev acc));
           _TYPE= typ;
           _KIND = funj._KIND}
  | hj::restjl ->
  match hnf_constr !isevars typ with
     DOP2(Prod,c1,DLAM(_,c2)) ->
     if the_conv_x_leq isevars env hj._TYPE c1 then
         apply_rec (hj::acc) (subst1 hj._VAL c2) restjl
        else error_cant_apply "Type Error" CCI env 
	  (j_nf_ise !isevars funj) (jl_nf_ise !isevars argjl)
   | _ -> error_cant_apply "Non-functional construction" CCI env
	  (j_nf_ise !isevars funj) (jl_nf_ise !isevars argjl)
in apply_rec [] funj._TYPE argjl;;

(* Inutile ?
let cast_rel isevars env cj tj =
    if the_conv_x_leq isevars env cj._TYPE tj._VAL then
        {_VAL=j_val_only cj;
         _TYPE=tj._VAL;
         _KIND = hnf_constr !isevars tj._TYPE}
   else error_actual_type CCI env (j_nf_ise !isevars cj) (j_nf_ise !isevars tj)
;;
*)
let let_path = make_path ["Core"] (id_of_string "let") CCI;;

let wrong_number_of_cases_message isevars env (c,ct) expn = 
  let c = nf_ise1 !isevars c and ct = nf_ise1 !isevars ct in
    error_number_branches CCI env c ct expn
;;

let check_branches_message isevars env (c,ct) (explft,lft) = 
  let n = Array.length explft and expn = Array.length lft in
    if n<>expn then wrong_number_of_cases_message isevars env (c,ct) expn
    else
      let rec check_conv i = 
        if i = n then () else
          if not (the_conv_x_leq isevars env lft.(i) explft.(i)) then 
            let c = nf_ise1 !isevars c
            and ct = nf_ise1 !isevars ct 
            and lfi = nf_betaiota (nf_ise1 !isevars lft.(i)) in
            error_ill_formed_branch CCI env c i lfi (nf_betaiota explft.(i))
          else check_conv (i+1) 
      in check_conv 0;;
(*
let evar_type_case isevars env ct pt lft p c =
  let (mind,bty,rslty) = type_case_branches env !isevars ct pt p c
  in check_branches_message isevars env (c,ct) (bty,lft); (mind,rslty);;
*)


let trad_metamap = ref [];;
let trad_nocheck = ref false;;

(* exemeta_rec vtcon isevars env constr tries to solve the *)
(* existential variables in constr in environment env with the *)
(* constraint vtcon (see Tradevar). *)
let rec exemeta_rec vtcon isevars env cstr =
match cstr with

  DOP0(XTRA("ISEVAR",[])) ->
  if !compter then nbimpl:=!nbimpl+1;
  (match vtcon with
    (is_ass,(Some valc, tyc)) ->
      exemeta_rec (is_ass,(None,tyc)) isevars env valc
  | (_,(None,Some ty)) ->
      let evarty = j_val_cast (exemeta_rec def_vty_con isevars env ty) in
      let k = new_isevar isevars env evarty CCI in
      exemeta_rec vtcon isevars env k
  | (true,(None,None)) ->
      let k = new_isevar isevars env (mkCast dummy_sort dummy_sort) CCI in
      exemeta_rec vtcon isevars env k
  | (false,(None,None)) -> error "There is an unknown subterm I cannot solve")

| DOP0(Meta n) ->
    let metaty =
      try List.assoc n !trad_metamap
      with Not_found -> error "A variable remains non instanciated" in
    (match kind_of_term metaty with
      IsCast (typ,kind) -> {_VAL=cstr; _TYPE=typ; _KIND=kind}
    | _ ->
        {_VAL=cstr;
          _TYPE=metaty;
          _KIND=hnf_constr !isevars
            (exemeta_rec def_vty_con isevars (gLOB(get_globals env))
	       metaty)._TYPE})

| Rel(n) ->  relative !isevars n env 
| VAR id ->
    let {body=typ;typ=s} = snd(lookup_glob id env) in
    {_VAL=cstr; _TYPE=typ; _KIND = DOP0 (Sort s)}

| DOPN(Const sp,_) ->
    (match type_of_const2 !isevars env cstr with
      (DOP2(Cast,typ,kind)) -> {_VAL=cstr;_TYPE=typ;_KIND = kind}
    | typ -> {_VAL=cstr;_TYPE=typ;
              _KIND= whd_betadeltaiota !isevars
                 (exemeta_rec def_vty_con isevars env typ)._TYPE})

| DOPN(Abst sp,cl) ->
  if sp = let_path then
      (match Array.to_list cl with
       [m;DLAM(na,b)] ->
       let mj = exemeta_rec mt_tycon isevars env m in
	 (try 
	    let mj = inh_ass_of_j isevars env mj in
	    let mb = body_of_type mj in
	    let bj =
	     exemeta_rec mt_tycon isevars (add_rel (na,mj) env) b in
	   {_VAL = DOPN(Abst sp,[|mb;DLAM(na,bj._VAL)|]);
            _TYPE = sAPP (DLAM(na,bj._TYPE)) mb;
            _KIND = pop bj._KIND }
	 with UserError _ -> 
	   exemeta_rec vtcon isevars env (abst_value cstr)) 
      | _ -> errorlabstrm "Trad.constr_of_com" [< 'sTR"Malformed ``let''" >])
   else if evaluable_abst cstr then
     exemeta_rec vtcon isevars env (abst_value cstr)
   else error "Cannot typecheck an unevaluable abstraction"

| DOPN(MutInd _,_) ->
  let {body=typ; typ=s} = type_of_mind !isevars env cstr in
    {_VAL=cstr; _TYPE=typ; _KIND=DOP0(Sort s)}
 
| DOPN(MutConstruct _,_) ->
    let (typ,kind) = destCast (type_of_mconstr !isevars env cstr) in
    {_VAL=cstr; _TYPE=typ; _KIND=kind}
 
| DOPN(MutCase _,_) ->
  let (_  (* est-ce c'est deja type ici ? tant pis, on *)
          (* recalculera mind avec check_case_branches *)
        ,p1,c1,lf) = destCase cstr in
  let cj = exemeta_rec mt_tycon isevars env c1 in
  let pj = exemeta_rec mt_tycon isevars env p1 in

  let evalct = nf_ise1 !isevars cj._TYPE
  and evalPt = nf_ise1 !isevars pj._TYPE in
  let (mind,bty,rsty) =
    type_case_branches env !isevars evalct evalPt pj._VAL cj._VAL in
  if Array.length bty <> Array.length lf then
    wrong_number_of_cases_message isevars env (cj._VAL,evalct)
      (Array.length bty)
  else
    let lfj =
      map2_vect
        (fun tyc f -> exemeta_rec (mk_tycon tyc) isevars env f) bty lf in
    let lft = (Array.map (fun j -> j._TYPE) lfj) in
    let ci = ci_of_mind mind in
      (check_branches_message isevars env (cj._VAL,evalct) (bty,lft);
       {_VAL=mkMutCaseA ci (j_val pj) (j_val cj) (Array.map j_val lfj);
        _TYPE= rsty;
        _KIND = sort_of_arity !isevars pj._TYPE})

| DOPN(Fix(vn,i),cl) ->
  let nbfix = let nv = Array.length cl in 
           if nv < 2 then error "Ill-formed recursive definition" else nv-1 in
  let lar = Array.sub cl 0 nbfix in
  let ldef = last_vect cl in
  let (lfi,vdef) = decomp_DLAMV_name nbfix ldef in
  let larj = Array.map (exemeta_rec def_vty_con isevars env) lar in
  let lara = Array.map (assumption_of_judgement !isevars env) larj in
  let newenv =
    it_vect2 (fun env name ar -> (add_rel (name,ar) env))
      env (Array.of_list (List.rev lfi)) (vect_lift_type lara) in
  let vdefj =
    map_i_vect 
      (fun i def -> (* we lift nbfix times the type in tycon, because of
                     * the nbfix variables pushed to newenv *)
        exemeta_rec (mk_tycon (lift nbfix (larj.(i)._VAL))) isevars newenv def)
      0 vdef in
  let fix = mkFix vn i lara (List.rev lfi) (Array.map j_val_only vdefj) in
    evar_type_fixpoint isevars env lfi lara vdefj;
    check_fix !isevars fix;
    make_judge fix lara.(i)

| DOPN(CoFix(i),cl) ->
  let nbfix = let nv = Array.length cl in 
           if nv < 2 then error "Ill-formed recursive definition" else nv-1 in
  let lar = Array.sub cl 0 nbfix in
  let ldef = last_vect cl in
  let (lfi,vdef) = decomp_DLAMV_name nbfix ldef in
  let larj = Array.map (exemeta_rec def_vty_con isevars env) lar in
  let lara = Array.map (assumption_of_judgement !isevars env) larj in
  let newenv =
    it_vect2 (fun env name ar -> (add_rel (name,ar) env))
      env (Array.of_list (List.rev lfi)) (vect_lift_type lara) in
  let vdefj =
    map_i_vect 
      (fun i def -> (* we lift nbfix times the type in tycon, because of
                     * the nbfix variables pushed to newenv *)
        exemeta_rec (mk_tycon (lift nbfix (larj.(i)._VAL))) isevars newenv def)
      0 vdef in
  let cofix = mkCoFix i lara (List.rev lfi) (Array.map j_val_only vdefj) in
    evar_type_fixpoint isevars env lfi lara vdefj;
    check_cofix !isevars cofix;
    make_judge cofix lara.(i)

| DOP0(Sort(Prop(c))) -> fcn_proposition c
| DOP0(Sort(Type(u))) -> fcn_type_with_universe u
| DOPN(AppL,tl) -> 
    let j = exemeta_rec mt_tycon isevars env (hd_vect tl) in
    let j = inh_app_fun isevars env j in
    let apply_one_arg (tycon,jl) c =
      let cj = exemeta_rec (app_dom_tycon isevars tycon) isevars env c in
      let rtc = app_rng_tycon isevars cj._VAL tycon in
      (rtc,cj::jl)  in
    let jl = List.rev (snd (List.fold_left apply_one_arg
			      (mk_tycon j._TYPE,[]) (list_of_tl_vect tl))) in
    inh_apply_rel_list isevars env jl j vtcon

| DOP2(Lambda,c1,DLAM(name,c2))      ->
    let j = exemeta_rec (abs_dom_valcon isevars vtcon) isevars env c1 in
    let assum = inh_ass_of_j isevars env j in
    let var = (name,assum) in
    let j' =
      exemeta_rec (abs_rng_tycon isevars vtcon) isevars
 	(add_rel var env) c2 in 
    abs_rel !isevars name assum j'

| DOP2(Prod,c1,DLAM(name,c2))        ->
    let j = exemeta_rec def_vty_con isevars env c1 in
    let assum = inh_ass_of_j isevars env j in
    let var = (name,assum) in
    let j' = exemeta_rec def_vty_con isevars (add_rel var env) c2 in
    let j'' = inh_tosort_test isevars env j' in
    gen_rel !isevars CCI env name assum j''

| DOPN(XTRA("REC",[]),cl)  ->
  let p = cl.(0)
  and c = cl.(1)
  and lf = Array.sub cl 2 ((Array.length cl) - 2) in
  let cj = exemeta_rec mt_tycon isevars env c and
      pj = exemeta_rec mt_tycon isevars env p in
  let p = pj._VAL and
      c = cj._VAL in
  let evalct = nf_ise1 !isevars cj._TYPE
  and evalPt = nf_ise1 !isevars pj._TYPE in
  let (mind,bty,rsty) = 
    Indrec.type_rec_branches true !isevars env evalct evalPt p c 
  in if Array.length bty <> Array.length lf then 
    wrong_number_of_cases_message isevars env (c,evalct) (Array.length bty)
   else 
  let lfj =
    map2_vect (fun tyc f -> exemeta_rec (mk_tycon tyc) isevars env f) bty lf in
  let lft = (Array.map (fun j -> j._TYPE) lfj) in
  (check_branches_message isevars env (c,evalct) (bty,lft);
     let rEC = Array.append [|p; c|] (Array.map j_val lfj) in
      {_VAL=Indrec.transform_rec env !isevars rEC (evalct,evalPt);
       _TYPE = rsty;
       _KIND = sort_of_arity !isevars evalPt})

| (DOPN(XTRA("MLCASE",[isrectok]),cl) as expr) ->
  let isrec = match isrectok with CoqAst.Str(_,"REC") -> true | _ -> false in
  let c = cl.(0) and  lf = tl_vect cl in
   (try match vtcon with
     (_,(_,Some pred)) -> 
       let predj = exemeta_rec mt_tycon isevars env pred in 
       let predj = inh_tosort_test isevars env predj in
       exemeta_rec vtcon isevars env
	 (Indrec.make_case_ml isrec predj._VAL c None lf)  
   | _ -> error "notype"
   with UserError _ -> (* get type information from type of branches *)
     let cj = exemeta_rec mt_tycon isevars env c in
     let (mind,_) =
       try find_mrectype !isevars cj._TYPE
       with Induc -> error_case_not_inductive CCI env
	   (nf_ise1 !isevars cj._VAL) (nf_ise1 !isevars cj._TYPE) in
     let rec findtype i =
       if i > Array.length lf then
	 let pe = pTERMINENV (env,expr) in
	 errorlabstrm "Trad.execute"
           (hOV 3 [<'sTR "Cannot infer type of expression :";'wS 1; pe >])
       else try
	 let expti = Indrec.branch_scheme !isevars isrec i cj._TYPE in
	 let fj = exemeta_rec (mk_tycon expti) isevars env lf.(i-1) in 
	 let efjt = nf_ise1 !isevars fj._TYPE in 
	 let pred = 
	   Indrec.pred_case_ml env !isevars isrec (cj._VAL,cj._TYPE) lf
	     (i,efjt)
	 in if has_ise pred then error"isevar" else pred
       with UserError _ -> findtype (i+1) in
     let pred = findtype 1
     in exemeta_rec vtcon isevars env 
          (Indrec.make_case_ml isrec pred cj._VAL (ci_of_mind mind) lf))

| DOPN(XTRA("MULTCASE",l),cl) as macro ->
    Multcase.compile_multcase
      ((fun vtyc -> exemeta_rec vtyc isevars),
       inh_ass_of_j isevars,
       (fun () -> !isevars))
      vtcon env macro

| DOP2(Cast,c,t) ->
  let tj = exemeta_rec def_vty_con isevars env t in
  let tj = inh_tosort_test isevars env tj in
  let cj =
    exemeta_rec (mk_tycon2 vtcon (assumption_of_judgement !isevars env tj).body)
      isevars env c in
  inh_cast_rel isevars env cj tj

| _ -> error_cant_execute CCI env (nf_ise1 !isevars cstr)

and

inh_app_fun isevars env j = 
 match whd_betadeltaiota !isevars j._TYPE with
    DOP2(Prod,_,DLAM(_,_)) -> j
   | _ ->
       (try
 	let t,i1 = class_of1 !isevars j._TYPE in
      	let p = lookup_path_to_fun_from i1 in
        apply_pcoercion isevars env p j t
      with _ -> j)
(* find out which exc we must trap (e.g we don't want to catch Sys.Break!) *)

and

inh_cast_rel isevars env cj tj =
    let cj' = (try inh_conv_coerce isevars env tj._VAL cj 
              with Not_found | Failure _ -> error_actual_type CCI env 
		  (j_nf_ise !isevars cj) (j_nf_ise !isevars tj)) in
      { _VAL = mkCast cj'._VAL tj._VAL;
        _TYPE = tj._VAL;
        _KIND = whd_betadeltaiota !isevars tj._TYPE}

and

inh_tosort_test isevars env j = 
  let typ = whd_betadeltaiota !isevars j._TYPE in
  match typ with
    DOP0(Sort(_)) -> j
  | _ -> (try inh_tosort isevars env j with _ -> j) (* idem inh_app_fun *)

and

inh_tosort isevars env j =
  let t,i1 = class_of1 !isevars j._TYPE in
  let p = lookup_path_to_sort_from i1 in
  apply_pcoercion isevars env p j t 

and 

inh_ass_of_j isevars env j =
   let typ = whd_betadeltaiota !isevars j._TYPE in
     match typ with
         DOP0(Sort s) -> {body=j._VAL;typ=s}
       | _ ->
           (try
             let j1 = inh_tosort isevars env j
             in assumption_of_judgement !isevars env j1 
           with _ -> error_assumption CCI env (nf_ise1 !isevars j._VAL))
                   (* try ... with _ -> ... is BAD *)

and

inh_coerce_to1 isevars env c1 v t kopt =
 let t1,i1 = class_of1 !isevars c1 in
 let t2,i2 = class_of1 !isevars t in
 let p = lookup_path_between (i2,i1) in
 let hj = (match kopt with
     None -> {_VAL=v;
              _TYPE=t;
              _KIND=(exemeta_rec def_vty_con isevars env t)._TYPE}
    | (Some k) -> {_VAL=v;_TYPE=t;_KIND=k}) in 
 let hj' = apply_pcoercion isevars env p hj t2 in
 if the_conv_x_leq isevars env hj'._TYPE c1 then hj'
 else failwith "inh_coerce_to"

and

inh_coerce_to isevars env c1 hj =
  inh_coerce_to1 isevars env c1 hj._VAL hj._TYPE (Some hj._KIND)

and

inh_conv_coerce1 isevars env c1 v t kopt =
 if the_conv_x_leq isevars env t c1 
 then (match kopt with
	   None -> 
	     {_VAL=v;
	      _TYPE=t;
	      _KIND=(exemeta_rec def_vty_con isevars env t)._TYPE}
	 | Some k -> {_VAL=v; _TYPE=t; _KIND=k})
 else try (inh_coerce_to1 isevars env c1 v t kopt)
 with _ ->  (* try ... with _ -> ... is BAD *)

   (match (hnf_constr !isevars t,hnf_constr !isevars c1) with
      | (DOP2(Prod,t1,DLAM(_,t2)),DOP2(Prod,u1,DLAM(name,u2))) -> 
          let v' = hnf_constr !isevars v in
            if (match v' with
                    DOP2(Lambda,v1,DLAM(_,v2)) ->
                      the_conv_x isevars env v1 u1 (* leq v1 u1? *)
                  |         _                -> false)
            then 
	      let (x,v1,v2) = destLambda v' in
              let jv1 = exemeta_rec def_vty_con isevars env v1 in 
	      let assv1 = assumption_of_judgement !isevars env jv1 in
              let env1 = add_rel (x,assv1) env in
              let h2 = inh_conv_coerce1 isevars env1 u2 v2 t2 None in
              abs_rel !isevars x assv1 h2
            else 
	      let ju1 = exemeta_rec def_vty_con isevars env u1 in 
	      let assu1 = assumption_of_judgement !isevars env ju1 in
              let name = (match name with Anonymous -> Name (id_of_string "x")
                            | _ -> name) in
              let env1 = add_rel (name,assu1) env in
              let h1 = 
		inh_conv_coerce1 isevars env1 t1 (Rel 1) u1 (Some ju1._TYPE) in
              let h2 = inh_conv_coerce1 isevars env1 u2  
			 (DOPN(AppL,[|(lift 1 v);h1._VAL|])) 
                         (subst1 h1._VAL t2) None in
	      abs_rel !isevars name assu1 h2
      | _ -> failwith "inh_coerce_to")
            
and

inh_conv_coerce isevars env c1 hj =
  inh_conv_coerce1 isevars env c1 hj._VAL hj._TYPE (Some hj._KIND)

and

inh_apply_rel_list isevars env argjl funj vtcon =
  let rec apply_rec acc typ = function
    [] -> let resj =
      {_VAL=applist(j_val_only funj,List.map j_val_only (List.rev acc));
       _TYPE= typ;
       _KIND = funj._KIND} in
      (match vtcon with 
      | (_,(_,Some typ')) ->
          (try inh_conv_coerce isevars env typ' resj
          with _ -> resj) (* try ... with _ -> ... is BAD *)
      | (_,(_,None)) -> resj)

  | hj::restjl ->
      match whd_betadeltaiota !isevars typ with
          DOP2(Prod,c1,DLAM(_,c2)) ->
            let hj' =
              if !trad_nocheck then hj else 
	      (try inh_conv_coerce isevars env c1 hj 
               with (Failure _ | Not_found) ->
                 error_cant_apply "Type Error" CCI env
		   (j_nf_ise !isevars funj) (jl_nf_ise !isevars argjl)) in
            apply_rec (hj'::acc) (subst1 hj'._VAL c2) restjl
        | _ ->
            error_cant_apply "Non-functional construction" CCI env
	      (j_nf_ise !isevars funj) (jl_nf_ise !isevars argjl)
  in apply_rec [] funj._TYPE argjl

and

(* appliquer le chemin de coercions p a` hj *)

apply_pcoercion isevars env p hj typ_cl =
  if !compter then (nbpathc := !nbpathc +1;
                   nbcoer := !nbcoer + (length p));
  try fst (List.fold_left
             (fun (ja,typ_cl) i -> 
                let v,b= coe_value i in
                let jv = exemeta_rec mt_tycon isevars env v in
                let jl =(List.map
                           (exemeta_rec mt_tycon isevars env)
                           (class_args_of typ_cl))@[ja] in
                let jres = apply_rel_list isevars env jl jv None in
                  (if b then {_TYPE=jres._TYPE;_KIND=jres._KIND;_VAL=ja._VAL}
                   else jres),jres._TYPE)
             (hj,typ_cl) p)
  with _ -> failwith "apply_pcoercion"
;;


let unsafe_fmachine is_ass nocheck isevars metamap env constr = 
  trad_metamap := metamap;
  trad_nocheck := nocheck;
  reset_problems ();
  let vtcon = if is_ass then def_vty_con else mt_tycon in
  exemeta_rec vtcon isevars env constr


(* [fail_evar] says how to process unresolved evars:
 *   true -> raise an error message
 *   false -> convert them into new Metas (casted with their type)
 *)
let process_evars fail_evar sigma =
  (if fail_evar then whd_ise sigma else whd_ise1_metas sigma)
;;

let j_apply f j = 
  { _VAL=strong (under_outer_cast f) j._VAL;
    _TYPE=strong f j._TYPE;
    _KIND=strong f j._KIND};;

(* TODO: comment faire remonter l'information si le typage a resolu des
       variables du sigma original. il faudrait que la fonction de typage
       retourne aussi le nouveau sigma...
*)
let ise_resolve fail_evar sigma metamap env c =
  let isevars = ref sigma in
  let j = unsafe_fmachine false false isevars metamap env c in
  j_apply (process_evars fail_evar !isevars) j
;;

let ise_resolve_type fail_evar sigma metamap env c =
  let isevars = ref sigma in
  let j = unsafe_fmachine true false isevars metamap env c in
  let tj = inh_ass_of_j isevars env j in
  type_app (strong (process_evars fail_evar !isevars)) tj;;


let ise_resolve_nocheck sigma metamap env c =
  let isevars = ref sigma in
  let j = unsafe_fmachine false true isevars metamap env c in
  j_apply (process_evars true !isevars) j
;;


let ise_resolve1 is_ass sigma env c =
  if is_ass then body_of_type (ise_resolve_type true sigma [] env c)
  else (ise_resolve true sigma [] env c)._VAL
;;

(* Endless list of alternative ways to call Trad *)

(* With dB *)

let constr_of_com_env1 is_ass sigma hyps com = 
  let c = Astterm.raw_constr_of_com sigma hyps com in
  try ise_resolve1 is_ass sigma hyps c
  with e -> Stdpp.raise_with_loc (Ast.loc com) e
;;

let constr_of_com_env sigma hyps com =
  constr_of_com_env1 false sigma hyps com;;

let fconstr_of_com_env1 is_ass  sigma hyps com = 
  let c = Astterm.raw_fconstr_of_com sigma hyps com in
    try ise_resolve1 is_ass sigma hyps c
    with e -> Stdpp.raise_with_loc (Ast.loc com) e
;;
 
let fconstr_of_com_env sigma hyps com =
  fconstr_of_com_env1 false sigma hyps com ;;


(* Without dB *)
let type_of_com sign com =
  let env = gLOB sign in
  let c = Astterm.raw_constr_of_com mt_evd env com in
  try ise_resolve_type true mt_evd [] env c
  with e -> Stdpp.raise_with_loc (Ast.loc com) e
;;

let constr_of_com1 is_ass sigma sign com = 
  constr_of_com_env1 is_ass sigma (gLOB sign) com;;

let constr_of_com sigma sign com =
  constr_of_com1 false sigma sign com;;
let constr_of_com_sort sigma sign com =
  constr_of_com1 true sigma sign com;;

let fconstr_of_com1 is_ass sigma sign com = 
  fconstr_of_com_env1 is_ass sigma (gLOB sign) com;;

let fconstr_of_com sigma sign com  =
  fconstr_of_com1 false sigma sign com;;
let fconstr_of_com_sort sigma sign com  =
  fconstr_of_com1 true sigma sign com;;

(* Note: typ is retyped *)
let constr_of_com_casted sigma sign com typ = 
  let env = gLOB sign in
  let c = Astterm.raw_constr_of_com sigma env com in
  let cc = mkCast (nf_ise1 sigma c) (nf_ise1 sigma typ) in
    try ise_resolve1 false sigma env cc
    with e -> Stdpp.raise_with_loc (Ast.loc com) e
;;


(* Typing with Trad, and re-checking with Mach *)
let fconstruct_type sigma sign com =
  let c = constr_of_com1 true sigma sign com
  in Mach.fexecute_type sigma sign c;;

let fconstruct sigma sign com =
  let c = constr_of_com1 false sigma sign com
  in Mach.fexecute sigma sign c;;

let infconstruct_type sigma (sign,fsign) cmd =
  let c = constr_of_com1 true sigma sign cmd
  in Mach.infexecute_type sigma (sign,fsign) c;;

let infconstruct sigma (sign,fsign) cmd =
  let c = constr_of_com1 false sigma sign cmd
  in Mach.infexecute sigma (sign,fsign) c;;

(* Type-checks a term with the current universe constraints, the resulting
   constraints are dropped. *)
let univ_sp = make_path ["univ"] (id_of_string "dummy") OBJ
let fconstruct_with_univ sigma sign com =
  let c = constr_of_com sigma sign com in
  let(_,j) = with_universes (Mach.fexecute sigma sign)
               (univ_sp, Constraintab.current_constraints(), c)
  in j
;;

(* Keeping universe constraints *)
let fconstruct_type_with_univ_sp sigma sign sp c =
  with_universes
    (Mach.fexecute_type sigma sign) (sp,empty_universes,c) 
;;

let fconstruct_with_univ_sp sigma sign sp c =
  with_universes
    (Mach.fexecute sigma sign) (sp,empty_universes,c) 
;;

let infconstruct_type_with_univ_sp sigma (sign,fsign) sp c =
  with_universes
    (Mach.infexecute_type sigma (sign,fsign)) (sp,empty_universes,c) 
;;

let infconstruct_with_univ_sp sigma (sign,fsign) sp c =
  with_universes
    (Mach.infexecute sigma (sign,fsign)) (sp,empty_universes,c) 
;;

(* $Id: trad.ml,v 1.42 1999/11/07 17:57:14 barras Exp $ *)
