(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                             proof_trees.mli                              *)
(****************************************************************************)

open Initial
open Names
open Term
open Constrtypes
open Pp
open Evd

(* The module proof declares the structure of proof trees and a few
   utilities on them *)

(* We keep track of whether proofs are completed or not *)
type pf_status = COMPLETE_PROOF | INCOMPLETE_PROOF

type bindOcc = 
    Dep of identifier
  | NoDep of int
  | Com

type 'a substitution = (bindOcc * 'a)   list
;;

(* prim_rules are the primitive rules of the base theory, here CIC *)
(* Ex: {name="INTRO"; hypspects=params=terms=[]; newids=["x"]} for Intro x.
       {name="EXACT"; hypspects=params=newids=[]; terms=[t]} for Exact t.
       {name="CONVERT_HYP"; hypspects=["h"]; params=newids=[]; terms=[T]} 
          for Change T in h.  *)

type prim_rule_name = 
   INTRO | INTRO_AFTER | INTRO_REPLACING | FIX | COFIX | REFINE 
 | CONVERT_CONCL | CONVERT_HYP | THIN | MOVE of bool

type prim_rule = {name : prim_rule_name;
                  hypspecs : identifier list;
                  newids : identifier list;
                  params : CoqAst.t list;
                  terms : constr list}

(* A tactic expression (tacname,args) refers to tactic tacname (indexing into
   a table in tacmach) applied to parameters args *)

type tactic_arg = (* reflects the current version of vernac *)
    COMMAND       of CoqAst.t
  | CONSTR        of constr
  | IDENTIFIER    of identifier
  | INTEGER       of int
  | CLAUSE        of identifier list
  | BINDINGS      of CoqAst.t substitution
  | CBINDINGS     of constr   substitution 
  | QUOTED_STRING of string
  | TACEXP        of CoqAst.t
  | REDEXP        of string * CoqAst.t list
  | FIXEXP        of identifier * int * CoqAst.t
  | COFIXEXP      of identifier * CoqAst.t
  | LETPATTERNS   of (int list option * (identifier * int list) list)
  | INTROPATTERN  of intro_pattern

and intro_pattern =  IdPat   of identifier          |
                     DisjPat of intro_pattern list  |
                     ConjPat of intro_pattern list  |
                     ListPat of intro_pattern list  

and tactic_expression = string * tactic_arg list

val is_bind  : tactic_arg -> bool
val cvt_arg              : CoqAst.t -> tactic_arg
val redexp_of_ast        : 'a evar_map -> context ->
                             string * CoqAst.t list -> Reduction.red_expr
val ast_of_cvt_arg       : tactic_arg ->  CoqAst.t 


(* A local constraint is just a set of section_paths *)
type local_constraints = Spset.t

val lc_toList  :  local_constraints -> section_path list


type goal =  ctxtty evar_info

and  proof_tree =
     {status : pf_status; (* the proof is complete or incomplete *)
      goal   : goal;     (* the goal *)
      ref    : (rule * proof_tree list) option; 
                      (* NONE       if the goal has still to be proved, 
                         SOME (r,l) if the rule r was applied to the goal
                                    and gave l as subproofs to be completed *)
     subproof : proof_tree option 
               (* (SOME p) if ref=(SOME(TACTIC t,l)) ;
                p is then the proof that the goal can be proven if the goals
                in l are solved *)
     }

and rule =
    PRIM        of prim_rule
  | TACTIC      of tactic_expression
  | CONTEXT     of ctxtty
  | LOCAL_CONSTRAINTS of  local_constraints
         (* for changing the lc of current goal *)

and ctxtty = {pgm    : constr option; (* for Program tactic *)
              mimick : proof_tree option;(* for a soon to exist mimick tactic *)
              lc     : local_constraints} 

and evar_declarations = ctxtty evar_map


val empty_evd : evar_declarations

val mkGOAL :  ctxtty -> context -> constr ->  goal

val mt_ctxt    :  local_constraints -> ctxtty
val get_ctxt   :  goal  ->  ctxtty
val get_pgm    :  goal -> constr option
val set_pgm    :  constr option ->  ctxtty ->  ctxtty
val get_mimick :  goal ->  proof_tree option
val set_mimick :  proof_tree option ->  ctxtty ->  ctxtty
val get_lc     :  goal -> local_constraints


val rule_of_proof     :  proof_tree ->  rule
val ref_of_proof      :  proof_tree -> ( rule *  proof_tree list)
val children_of_proof :  proof_tree ->  proof_tree list
val goal_of_proof     :  proof_tree ->  goal
val subproof_of_proof :  proof_tree ->  proof_tree
val status_of_proof   :  proof_tree -> pf_status
val is_complete_proof :  proof_tree -> bool
val is_leaf_proof     :  proof_tree -> bool
val is_tactic_proof   :  proof_tree -> bool


(*******************************************************************)
(*            Constraints for existential variables                *)
(*******************************************************************)

(* A global constraint is a mappings of existential variables
   with some extra information for the program and mimick
   tactics.
*)

type global_constraints   = evar_declarations timestamped;;


(* A readable constraint is a global constraint plus a focus set
   of existential variables and a signature.
*)

type evar_recordty = {focus : local_constraints;
                      sign  : context;
                      decls : evar_declarations}

and readable_constraints = evar_recordty timestamped

val rc_of_gc  : global_constraints -> goal -> readable_constraints
val rc_add    : readable_constraints -> section_path*goal -> readable_constraints
val get_hyps  : readable_constraints -> context
val get_focus : readable_constraints -> local_constraints
val get_decls : readable_constraints -> evar_declarations
val get_gc    : readable_constraints -> global_constraints
val remap     : readable_constraints ->section_path* goal -> readable_constraints
val ctxt_access : readable_constraints -> section_path -> bool

(* Pretty printing functions *)

val pr_goal      : goal -> std_ppcmds
val pr_subgoals  : goal list -> std_ppcmds
val pr_subgoal   : int -> goal list -> std_ppcmds

val pr_decl      : goal -> std_ppcmds
val pr_decls     : global_constraints ->std_ppcmds
val pr_evc       : readable_constraints -> std_ppcmds

val prgl         : goal -> std_ppcmds
val pr_seq       : goal -> std_ppcmds
val pr_focus     : local_constraints -> std_ppcmds
val pr_ctxt      : ctxtty -> std_ppcmds
val pr_evars     : (section_path * goal) list ->  std_ppcmds
val pr_evars_int : int -> (section_path * goal) list ->  std_ppcmds
val pr_subgoals_existential : evar_declarations -> goal list -> std_ppcmds

(* used by Intros Until *)
val pf_lookup_name_as_renamed :
    context-> constr -> identifier -> int option
val pf_lookup_index_as_renamed : constr -> int -> int option

(* $Id: proof_trees.mli,v 1.18 1999/08/06 20:49:10 herbelin Exp $ *)
