(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                  std.ml                                  *)
(****************************************************************************)

open Pp

(* System errors *)
exception Anomaly of string * std_ppcmds  (* System errors *)
let anomaly string = raise (Anomaly(string,[< 'sTR string >]))
let anomalylabstrm string pps = raise (Anomaly(string,pps))

(* User errors *)
exception UserError of string * std_ppcmds (* User errors *)
let error string = raise (UserError(string,[< 'sTR string >]))
let errorlabstrm l pps = raise (UserError(l,pps))

let enforce s f a = try if f a then () else anomaly s with _ -> anomaly s
let enforce_error s f a = try if f a then () else error s with _ -> error s

let ref_app f x = ref (f !x)

let comp f g x = f (g x)

(* dml.ml *)

let modify_vect f v = 
 let rec modify_rec i =
    if i < Array.length v then (
      v.(i) <- f(v.(i));
      modify_rec (succ i)
    )
 in modify_rec 0
  

let explode s = 
 let rec explode_rec n =
    if n >= String.length s
    then []
    else String.make 1 (String.get s n) :: explode_rec (succ n)
 in explode_rec 0
  


let implode sl =
  let len = List.fold_left (fun a b -> a+(String.length b)) 0 sl in
  let dest = String.create len in
      (List.fold_left (fun start src ->
                let src_len = String.length src in
                    (String.blit src 0 dest start src_len;
                     start + src_len))
               0 sl;
       dest)


let explode_ascii s = 
 let rec explode_rec n =
    if n >= String.length s
    then []
    else Char.code (String.get s n) :: explode_rec (succ n)
 in explode_rec 0
  


(* glob.ml *)

let null = function [] -> true | _ -> false
let car = List.hd
let cdr = List.tl
let cadr x = List.hd (List.tl x)
let cddr x = List.tl (List.tl x)
let cdddr x = List.tl (List.tl (List.tl x))

let safe f e dflt =
  try f e
  with Failure _ -> dflt

let safe_cdr l = safe cdr l []
let safe_cddr l = safe cddr l []
let safe_cdddr l = safe cdddr l []


let except e =
 let rec except_e = function
     [] -> []
   | elem::l -> if e = elem then l else elem::except_e l
 in except_e


let except_assoc e = 
 let rec except_e = function
    [] -> []
 | (x,_ as y)::l -> if x=e then l else y::except_e l
 in except_e 

(* prel.ml *)

let try_find f = 
 let rec try_find_f = function
     [] -> failwith "try_find"
   | h::t -> try f h with Failure _ -> try_find_f t
 in try_find_f


let try_find_i f = 
 let rec try_find_f n = function
     [] -> failwith "try_find_i"
   | h::t -> try f n h with Failure _ -> try_find_f (n+1) t
 in try_find_f


let index x l = 
 let rec index_x = function
     n,y::l -> if x = y then n else index_x (succ n,l)
   | _ -> failwith "index"
 in index_x (1,l)


let for_all_i p = 
 let rec for_all_p i = function
     [] -> true | a::l -> p i a & for_all_p (i+1) l
 in for_all_p


let for_all2 f x y =  
  let rec fall = function 
      [],[] -> true
    | (a::x), (b::y) -> (f a b) & fall  (x,y)
    |  _ , _ -> failwith "for_all2"
  in fall (x,y)

let for_all2eq f l1 l2 = try for_all2 f l1 l2 with Failure _ -> false


let filter p =
  let rec filter_aux = function
      [] -> []
    | x::l -> if p x then x::filter_aux l else filter_aux l
  in filter_aux


let rec fold_right_one f l b1 b2  =
  match l with
     []    ->  b1
  |  [x]   -> (b2 x)
  | (x::l) -> (f x (fold_right_one f l b1 b2))


let rec distinct = function
    h::t -> (not (List.mem h t)) & distinct t
  | _ -> true

let chop_list n l = 
  let rec chop_aux acc = function
      (0, l2) -> (List.rev acc,l2)
    | (n, (h::t)) -> chop_aux (h::acc) (pred n, t)
    | (_, []) -> failwith "chop_list"
  in chop_aux [] (n,l)
 
let split_list n l =
  let rec split_rec  i before = function
      [] -> invalid_arg "Listmap.split_list"
    | [a] -> (before,a,[]) 
    | a::l' -> if i=n then (before, a, l') else split_rec (i+1) (before@[a]) l'
  in 
    if n <= List.length l then split_rec 1 [] l 
    else invalid_arg "Listmap.split_list"


let splitby p = 
  let rec splitby_loop x y = 
    match y with 
       	[]      -> ([],[])
      | (a::l)  -> if (p a) then (x,y) else (splitby_loop (x@[a]) l)
  in splitby_loop []


let iterate f = 
  let rec iterate_f n x =
    if n <= 0 then x else iterate_f (pred n) (f x)
  in iterate_f
 

let map_i f = 
 let rec map_i_rec i = function
     [] -> [] | x::l -> let v = f i x in v::map_i_rec (i+1) l
 in map_i_rec
 


let map2_i f i l1 l2 =  
 let rec map_i i = function
     ([], []) -> []
   | ((h1::t1), (h2::t2)) -> (f i h1 h2) :: (map_i (succ i) (t1,t2))
   | (_, _) -> invalid_arg "map2_i"
 in map_i i (l1,l2)


let app_i f = 
 let rec app_i_rec i = function
     [] -> () | x::l -> let _ = f i x in app_i_rec (i+1) l
 in app_i_rec
 


let it_list_i f = 
 let rec it_list_f i a = function
     [] -> a | b::l -> it_list_f (i+1) (f i a b) l
 in it_list_f
 

let rec map_append f = function
    [] -> []
  | h::t -> (f h)@(map_append f t)

let rec map_append2 f l1 l2 =
  match l1,l2 with
    [],[] -> []
  | h1::t1, h2::t2 -> (f h1 h2)@(map_append2 f t1 t2)
  | _,_ -> invalid_arg "map_append2 : lists must have the same length"

let rec rev_append l1 l2 =
 match l1 with
     [] -> l2
   | x::l' -> rev_append l' (x::l2)


let list_assign l n e = 
 let rec assrec stk = function
     ((h::t), 0) -> rev_append stk (e::t)
   | ((h::t), n) -> assrec (h::stk) (t, n-1)
   | ([], _) -> failwith "list_assign"
 in assrec [] (l,n)




let it_vect f a v = 
 let rec it_vect_f a n =
  if n >= Array.length v then a
   else it_vect_f (f a v.(n)) (succ n)
 in it_vect_f a 0
 


let map_vect_list f v=
  let rec mvlrec l=function
     0 -> l
   | n -> let k=pred n in mvlrec ((f v.(k))::l) k
  in mvlrec [] (Array.length v)



let rec last = function
    [] -> failwith "last"
  | x::[] -> x
  | x::l -> last l


let firstn n l =
  let rec aux acc = function
      (0, l) -> List.rev acc
    | (n, (h::t)) -> aux (h::acc) (pred n, t)
    | _ -> failwith "firstn"
  in aux [] (n,l)


let lastn n l =
  let len = List.length l in
  let rec aux m l =
    if m = n then l
    else aux (m - 1) (List.tl l)
  in
    if len < n then failwith "lastn"
    else aux len l


let rec sep_last = function
    [] -> failwith "sep_last"
  | hd::[] -> (hd,[])
  | hd::tl ->
      let (l,tl) = sep_last tl in (l,hd::tl)


let words s = 
 let rec words_rec l s w i =
    if i >= String.length s then (if w = "" then l else w::l)
    else
      match String.get s i with
        ' ' | '\n' | '\t' ->
          words_rec (if w = "" then l else w::l) s "" (i+1)
      | c   ->
          words_rec l s (w^(String.make 1 c)) (i+1)
 in List.rev(words_rec [] s "" 0)
  

let scan_string s1 s2 = 
 let rec scan_rec pos =
    if pos >= String.length s1 then -1
    else
      let c = String.get s1 pos in 
 let rec mem_rec i =
        if i >= String.length s2 then scan_rec(pos+1)
        else if c == String.get s2 i then pos
        else mem_rec(i+1)
 in mem_rec 0
      
 in scan_rec
  


let compare_strings s1 s2=
  let l1= String.length s1
  and l2= String.length s2 in
  let rec cs n=
    if n >= l1 then
      if n >= l2 then 0
      else 2
    else if n >= l2 then -2
    else if (String.get s1 n) = (String.get s2 n) then cs (succ n)
    else if (String.get s1 n) < (String.get s2 n) then -1
    else 1
  in cs 0



let interval n m = 
  let rec interval_n (l,m) =
    if n > m then l else interval_n (m::l,pred m)
  in interval_n ([],m)

let range = interval 1

(* hash.ml *)

let hash_clear v = modify_vect (fun _ -> []) v
let hash_add_assoc (key,val_0 as pair) v =
  let i = (Hashtbl.hash key) mod (Array.length v) in
  v.(i) <- pair::v.(i); ()
and hash_remove_assoc (key,val_0 as pair) v =
  let i = (Hashtbl.hash key) mod (Array.length v) in
  v.(i) <- except pair v.(i); ()
and hash_assoc key v =
  List.assoc key v.((Hashtbl.hash key) mod (Array.length v))




let repeat n action arg = 
 let rec repeat_action n =
  if n <= 0 then () else (action arg;repeat_action (pred n))
 in repeat_action n
 


let break_string sep string =
  let scan (strs,chars) char =
    if char = sep then implode chars::strs,[]
    else strs,char::chars
  in fst (List.fold_left scan ([],[]) (List.rev (sep::explode string)))


let skip_string skip s = 
 let rec skip_rec i =
    if i >= String.length s then ""
    else if List.mem (String.get s i) skip then skip_rec (i+1)
    else String.sub s i (String.length s - i)
 in skip_rec 0


let skip_space_return = skip_string [' '; '\n'; '\t']

let last_n_string n s =
 let l = String.length s in
 if l < n then failwith "last_n_string"
   else String.sub s (l-n) n

and first_n_string n s =
 if String.length s < n then failwith "first_n_string"
  else String.sub s 0 n


let fst3 (a,b,c)  = a

let map_succeed f = 
 let rec map_f =
 function [] -> []
 |  h::t -> try (let x = f h in x :: map_f t) with Failure _ -> map_f t
 in map_f 

let map_succeed2 f = 
 let rec map_f  = 
   function
       [] -> (function [] -> [] | _ -> invalid_arg "map_succed2")
     | h1::t1 ->
	 (function 
	      h2::t2 -> 
	   	begin 
		  try let x = f h1 h2 in x :: map_f t1 t2
		  with Failure _ -> map_f t1 t2
		end
	    | [] -> invalid_arg "map_succed2")
 in map_f

let map_succeed_match f = 
 let rec map_f =
 function [] -> []
 |  h::t -> try (let x = f h in x :: map_f t) with Match_failure _ -> map_f t
 in map_f 

let do_listRL f = 
 let rec dorec = function
    h::t -> (dorec t;f h)
  | [] -> ()
 in dorec 

let join_map_list f = 
 let rec joinrec = function
    [] -> []
  | h::t -> (f h)@(joinrec t)
 in joinrec
    


let push l x = (l := x :: !l)

let pop l =
    match !l with
    h::tl -> l := tl
  | [] -> invalid_arg "pop"


let top l = List.hd !l
let truncate n l =
    if List.length !l > n then
        l := fst(chop_list n !l)
    else ()



type ('a,'b) union = Inl of 'a | Inr of 'b

let outl = function
    (Inl e) -> e
  | _ -> failwith "std__outl"

let outr = function
    (Inr e) -> e
  | _ -> failwith "std__outr"

let union_exn f arg =
    try Inl(f arg)
    with reraise -> Inr reraise


let compare name eqfun prargs f1 f2 arg =
    match (union_exn f1 arg,union_exn f2 arg) with
    (Inl r1,Inl r2) ->
    if eqfun r1 r2 then r1
    else anomaly ("Versions of "^name^" returned different values")
  | (Inl _,Inr _) -> anomaly ("New version of "^name^" raised exception")
  | (Inr _,Inl _) -> anomaly ("New version of "^name^" succeeded exception")
  | (Inr reraise,Inr _) -> raise reraise



let prefix_of prefl l = 
 let rec prefrec = function
     ((h1::t1), (h2::t2)) -> (h1 = h2 & prefrec (t1,t2))
   | ([], l) -> true
   | (_, _) -> false
 in prefrec (prefl,l)
    


let share_tails l1 l2 =
  let rec shr_rev acc = function
      ((x1::l1), (x2::l2)) when x1 == x2 -> shr_rev (x1::acc) (l1,l2)
    | (l1,l2) -> (List.rev l1, List.rev l2, acc)
  in shr_rev [] (List.rev l1, List.rev l2)


let lexico ord l1 l2 = 
 let rec lexrec = function
     ([], []) -> 0
   | ([], _) -> -1
   | (_, []) -> 1
   | ((h1::t1), (h2::t2)) ->
       let hord = ord h1 h2
       in if hord = 0 then lexrec (t1,t2)
       else hord
 in lexrec (l1,l2)
    

let rev_lexico ord l1 l2 =
  lexico ord (List.rev l1) (List.rev l2)


let tabulate_list f len = 
 let rec tabrec n =
    if n = len then []
    else (f n)::(tabrec (n+1))
 in tabrec 0


(**********************************************)
(**** Functions on lists representing sets ****)

let rec uniquize = function
    [] -> []
  | (h::t) -> if List.mem h t then uniquize t else h::(uniquize t)
let make_set = uniquize

let add_set a fs = if List.mem a fs then fs else (a::fs)

let rec rmv_set a ls =
  match ls with
      (h::t) -> if h = a then t else h::(rmv_set a t)
    | _ -> failwith "listset__rmv"

let intersect l1 l2 = filter (fun x -> List.mem x l2) l1

let union l1 l2 =
  let rec union_rec = function
      [] -> l2
    | a::l -> if List.mem a l2 then union_rec l else a :: union_rec l
  in union_rec l1

let unionq l1 l2 = 
 let rec urec = function
     [] -> l2
   | a::l -> if List.memq a l2 then urec l else a::urec l
 in urec l1

let union2 l1 l2 =
  let htab = Hashtbl.create 151 in
  let rec add_elt l = function
      [] -> l
    | x::ll -> try Hashtbl.find htab x; add_elt l ll
      with Not_found -> Hashtbl.add htab x ();
        add_elt (x::l) ll
  in 
  let l' = add_elt [] l1
  in add_elt l' l2

let diff_set l1 l2 =
  if l2 = [] then l1 else filter (fun x -> not (List.mem x l2)) l1
let subtract = diff_set

let subtractq l1 l2 = filter (fun x -> not (List.memq x l2)) l1

let symdiff l1 l2 = diff_set (union l1 l2) (intersect l1 l2)

let subset l1 l2 =
  let t2 = Hashtbl.create 151 in
    List.iter (fun x-> Hashtbl.add t2 x ()) l2;
    let rec look = function
	[] -> true
      | x::ll -> try Hashtbl.find t2 x; look ll
        with Not_found -> false
    in look l1

let same_members s1 s2 = subset s1 s2 & subset s2 s1

let choose p fs =
  try List.hd (filter p fs)
  with Failure "hd" -> raise Not_found

module Listset =
  struct
    type 'a t    = 'a list
    let subset   = subset
    let add      = add_set
    let rmv      = rmv_set
    let map      = List.map
    let mem      = List.mem
    let uniquize = uniquize
    let subset   = subset
    let equal    = same_members
    let elements l = l
    let for_all  = List.for_all
  end

(* $Id: std.ml,v 1.21 1999/06/29 07:47:20 loiseleu Exp $ *)
