(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                  mhs.ml                                  *)
(****************************************************************************)

open Std;;

(* Mutable Hash-Summaries *)

type ('a,'b) t = {tab : ('a,'b list) Mhm.t ;
                  mutable stk : ('a * 'b) list;
                  ord : ('b -> 'b -> int) option};;

type ('a,'b) frozen_t = ('a,'b list) Mhm.frozen_t;;

let create (n,ord) =
    {tab = Mhm.create n;
     stk = [];
     ord = ord};;

let empty s =
    (if s.stk <> [] then
         anomaly "cannot be emptying a summary which contains locally pushed values");
    Mhm.empty s.tab
;;

let map s k = Mhm.map s.tab k;;

let rec insert ordopt v l =
    match ordopt with
    None -> v::l
  | Some ord -> (
 let rec insrec = function
                 [] -> [v]
               | h::tl -> if ord v h <= 0 then v::h::tl
                          else h::(insrec tl)
 in insrec l
                 )
;;

let add s (k,v) =
    (if s.stk <> [] then
         anomaly "cannot be adding to a summary which contains locally pushed values");
    try let oval = Mhm.map s.tab k
        in Mhm.rmv s.tab k;
           Mhm.add s.tab (k,insert s.ord v oval)
    with Not_found -> Mhm.add s.tab (k,[v])
;;

let rmv s (k,v) =
    anomaly "mutable hash-summaries do not allow removal"
;;

let app f s = Mhm.app f s.tab;;

let in_dom s k = Mhm.in_dom s.tab k;;

let freeze s =
    (if s.stk <> [] then
         anomaly "cannot be emptying a summary which contains locally pushed values");
    Mhm.freeze s.tab;;

let unfreeze fs s =
    (if s.stk <> [] then
         anomaly "cannot be emptying a summary which contains locally pushed values");
    Mhm.unfreeze fs s.tab;;

let push s (k,v) =
    s.stk <- (k,v)::s.stk;
    try let oval = Mhm.map s.tab k
        in Mhm.rmv s.tab k;
           Mhm.add s.tab (k,insert s.ord v oval)
    with Not_found -> Mhm.add s.tab (k,[v])
;;

let pop s kopt =
let k = (match kopt with None -> fst(List.hd s.stk) | Some k -> k)
in (if k <> fst(List.hd s.stk) then
    anomaly "the summary's local-stack is out of sync with its user");
   let oval = Mhm.map s.tab k in
   let nval = except (snd (List.hd s.stk)) oval
   in Mhm.rmv s.tab k;
      Mhm.add s.tab (k,nval);
      s.stk <- List.tl s.stk
;;

let pop_all s =
    while s.stk <> [] do
        pop s None
    done
;;

let dom s = Mhm.dom s.tab;;
let toList s = Mhm.toList s.tab;;

(* $Id: mhs.ml,v 1.6 1999/06/29 07:47:19 loiseleu Exp $ *)
