(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                environ.ml                                *)
(****************************************************************************)


(* The Coq theory module entries comprise now :
   TOKEN	   tokens
   CONSTANT        global constants
   MUTUALINDUCTIVE       global inductive definitions
   SYNTAXCONSTANT  parsing macros
   AUTOHINT        hints for tactic Auto
   PPSYNTAX        syntax tables for prettyprinting
   GRAMMAR         LL1 grammars
   ABSTRACTION     abstract syntax macros, declared in abstraction.ml 

The module environ instanciates libobject for entries CONSTANT, INDUCTIVE,
SYNTAXCONSTANT, AUTOHINT, PPSYNTAX and GRAMMAR *)


open Pp;;
open Std;;
open Names;;
open More_util;;
open Vectops;;
open Generic;;
open Term;;
open Constrtypes;;
open Libobject;;
open Summary;;

type variable_object =
    identifier * 
    (type_judgement * (type_judgement*implicits_typ) option) * 
    strength * implicits_typ * bool * Impuniv.universes;;

type constant_object = constdecl_body * strength * Impuniv.universes;;

type syntaxconstant_object = (path_kind,constr) Listmap.t;;

type mutual_inductive_object = mutual_inddecl_body * Impuniv.universes;;

(* CONSTANT *)

let load_constant (_,_,uni) =
  Constraintab.push uni


let cache_constant (sp,(cmap,str,uni)) =
  Nametab.rollback
    (fun () ->
       Listmap.app
         (fun (k,lobj) -> Nametab.push (basename sp) (coerce_path k sp))
         cmap)
    ()
;;

let print_constant_object = fun
    _ (cmap,n,uni) ->
    [< 'sTR"#<constant(" ;
       prlist_with_sep pr_coma
         (fun (k,_) -> [< 'sTR(string_of_kind k) >]) cmap ;
       'sTR")" ;
       (match n with
        NeverDischarge -> [< 'sTR"(global)" >]
      | DischargeAt sp -> [< 'sTR"(disch@" ; 'sTR(string_of_path sp) >]) ;
       'sTR"(" ; 'iNT(Impuniv.num_universes uni) ; 'sTR" uni,";
               'iNT(Impuniv.num_edges uni) ; 'sTR"edges)>" >]
;;

let specification_constant (cmap,n,uni) =
    (specification_of_constant cmap,n,uni);;

let (inConstant,outConstant) =
    declare_object ("CONSTANT",
                    {load_function = load_constant;
                     cache_function = cache_constant;
                     specification_function = specification_constant});;


let load_mutual_inductive (_,uni) =
  Constraintab.push uni

let cache_mutual_inductive (sp,(mimap,uni)) =
  Nametab.rollback
    (fun () ->
       Listmap.app
         (fun (k,mib) ->
            Array.iter
              (fun mip ->
                 begin
                   Nametab.push mip.mINDTYPENAME (coerce_path k sp);
                   Array.iter (fun cn -> Nametab.push cn (coerce_path k sp)) 
                     mip.mINDCONSNAMES
                 end)
              mib.mINDPACKETS)
         mimap)
    ()
;;


let specification_minductive (imap,uni) =
    (specification_of_minductive imap,uni);;

let (inMutualInductive,outMutualInductive) =
    declare_object ("MUTUALINDUCTIVE",
                    {load_function = load_mutual_inductive;
                     cache_function = cache_mutual_inductive;
                     specification_function = specification_minductive});;


let load_variable (_,_,_,_,_,uni) =
  Constraintab.push uni
;;

let cache_variable (_,(id,vbody,str,impl,sticky,uni)) =
  Constraintab.rollback (fun () -> Vartab.push id sticky vbody str impl) ()
;;

let specification_variable (id,_,_,_,_,_) =
    errorlabstrm "specification_variable"
    [< 'sTR"Cannot extract a specification from a variable declaration " ;
       'sTR (string_of_id id) >]
;;

let (inVariable,outVariable) =
    declare_object ("VARIABLE",
                    {load_function = load_variable;
                     cache_function = cache_variable;
                     specification_function = specification_variable});;

let cache_syntaxconstant (sp,cmap) =
  Listmap.app
    (fun (k,lobj) -> Nametab.push (basename sp) (coerce_path k sp))
    cmap
;;

let specification_syntaxconstant cmap = cmap;;

let (inSyntaxConstant,outSyntaxConstant) =
    declare_object ("SYNTAXCONSTANT",
                    {load_function = (fun _ -> ());
                     cache_function = cache_syntaxconstant;
                     specification_function = specification_syntaxconstant})
;;

let const_of_path sp =
  match Lib.leaf_object_tag (objsp_of sp) with
    "CONSTANT" ->
      let (cmap,_,_) = outConstant(Lib.map_leaf (objsp_of sp)) in
      (sp,Listmap.map cmap (kind_of_path sp))

  | _ -> invalid_arg "const_of_path called with non-constant"
;;

let const_option_of_path sp =
try Some(const_of_path sp)
with Not_found -> None
;;

let mind_of_path sp =
    match Lib.leaf_object_tag (objsp_of sp) with
    "MUTUALINDUCTIVE" ->
    let (cmap,_) = outMutualInductive (Lib.map_leaf (objsp_of sp))
    in (sp,Listmap.map cmap (kind_of_path sp))
  | _ -> invalid_arg "mind_of_path called with non-mutual-inductive"
;;

let mind_option_of_path sp =
try Some(mind_of_path sp)
with Not_found -> None
;;

let search_synconst pk str =
let sp = Nametab.sp_of_id CCI str
in (match Lib.leaf_object_tag(objsp_of sp) with
    "SYNTAXCONSTANT" ->
    let cmap = outSyntaxConstant(Lib.map_leaf (objsp_of sp))
    in Listmap.map cmap pk
  | _ -> raise Not_found)
;;


let global_operator sp id =
  match Lib.leaf_object_tag (objsp_of sp) with
    "CONSTANT" ->
    let (cmap,_,_) = outConstant(Lib.map_leaf(objsp_of sp)) in
    let cb = Listmap.map cmap (kind_of_path sp)
    in (Const sp,cb.cONSTHYPS)

  | "MUTUALINDUCTIVE" ->
    let (mimap,_) = outMutualInductive (Lib.map_leaf (objsp_of sp)) in
    let mib = Listmap.map mimap (kind_of_path sp)
    in mind_oper_of_id sp mib id

  | _ -> raise Not_found
;;

(* comme global_operator, mais retourne en plus les args implicites *)

let global_operator1 sp id =
  match Lib.leaf_object_tag (objsp_of sp) with
    "CONSTANT" ->
    let (cmap,_,_) = outConstant(Lib.map_leaf(objsp_of sp)) in
    let cb = Listmap.map cmap (kind_of_path sp)
    in (Const sp,cb.cONSTHYPS),list_of_implicits cb.cONSTIMPARGS

  | "MUTUALINDUCTIVE" ->
    let (mimap,_) = outMutualInductive (Lib.map_leaf (objsp_of sp)) in
    let mib = Listmap.map mimap (kind_of_path sp)
    in (mind_oper_of_id1 sp mib id)

  | _ -> raise Not_found
;;

(* id_of_global gives the name of the given sort oper *)
let id_of_global = function
    (Const sp) -> basename sp

  | (MutInd(sp,tyi)) -> (* Does not work with extracted inductive types when the first inductive is logic :
  if tyi=0 then basename sp else *)
    let (_,mib) = mind_of_path sp in
    let mip = mind_nth_type_packet mib tyi
    in mip.mINDTYPENAME

  | (MutConstruct((sp,tyi),i)) ->
    let (_,mib) = mind_of_path sp in
    let mip = mind_nth_type_packet mib tyi
    in if i <= Array.length mip.mINDCONSNAMES & i > 0 then
        mip.mINDCONSNAMES.(i-1)
       else failwith "id_of_global"
  | _ -> assert false
;;

let rec hdchar = function
    DOP2(Prod,_,DLAM(_,c))     -> hdchar c
  | DOP2(Cast,c,_)             -> hdchar c
  | DOPN(AppL,cl)              -> hdchar (hd_vect cl)
  | DOP2(Lambda,_,DLAM(_,c))   -> hdchar c
  | DOPN(Const _,_) as x ->
      let c = String.lowercase(List.hd(explode_id (basename(path_of_const x))))
      in if c = "?" then "y" else c
  | DOPN(Abst _,_) as x ->
      String.lowercase(List.hd(explode_id (basename(path_of_abst x))))
  | DOPN(MutInd (sp,i) as x,_) ->
      if i=0 then String.lowercase(List.hd(explode_id (basename sp)))
      else let na = id_of_global x  in String.lowercase(List.hd(explode_id na))
  | DOPN(MutConstruct(sp,i) as x,_) ->
      let na = id_of_global x  in String.lowercase(List.hd(explode_id na))
                                    
  | VAR id  -> String.lowercase(List.hd(explode_id id))
  | DOP0(Sort s)          -> sort_hdchar s
  | _                     -> "y";;

let id_of_name_using_hdchar a = function
    Anonymous -> id_of_string(hdchar a) 
  | Name id   -> id;;

let named_hd a = function
    Anonymous -> Name(id_of_string(hdchar a)) 
  | x         -> x;;


let prod_name (n,a,b) = mkProd (named_hd a n) a b;;
let prod_create (a,b) = mkProd (named_hd a Anonymous) a b;;
let lambda_name (n,a,b) = mkLambda (named_hd a n) a b;;
let lambda_create (a,b) =  mkLambda (named_hd a Anonymous) a b;;
let it_prod_name = List.fold_left (fun c (n,t) -> prod_name (n,t,c)) ;;
let it_lambda_name = List.fold_left (fun c (n,t) -> lambda_name (n,t,c)) ;;

(* $Id: environ.ml,v 1.23 1999/10/13 14:36:23 herbelin Exp $ *)
