(* camlp4r *)
(***********************************************************************)
(*                                                                     *)
(*                             Camlp4                                  *)
(*                                                                     *)
(*        Daniel de Rauglaudre, projet Cristal, INRIA Rocquencourt     *)
(*                                                                     *)
(*  Copyright 1998 Institut National de Recherche en Informatique et   *)
(*  Automatique.  Distributed only by permission.                      *)
(*                                                                     *)
(***********************************************************************)

(* $Id: pretty.ml,v 2.2 1999/04/11 10:11:06 ddr Exp $ *)

type glue = [ LO | RO | LR | NO ];
type pretty =
  [ S of glue and string
  | Hbox of Stream.t pretty
  | HVbox of Stream.t pretty
  | HOVbox of Stream.t pretty
  | HOVCbox of Stream.t pretty
  | Vbox of Stream.t pretty
  | BEbox of Stream.t pretty
  | BEVbox of Stream.t pretty ]
;
type prettyL =
  [ SL of int and glue and string
  | HL of list prettyL
  | BL of list prettyL
  | PL of list prettyL
  | QL of list prettyL
  | VL of list prettyL
  | BE of list prettyL
  | BV of list prettyL ]
;

value quiet = ref True;
value maxl = ref 20;
value dt = ref 2;
value tol = ref 1;
value sp = ref ' ';
value out_ch = ref stdout;
value prompt = ref "";

value print_char c = output_char out_ch.val c;
value print_string s = output_string out_ch.val s;
value print_newline () =
  do output_char out_ch.val '\n'; output_string out_ch.val prompt.val; return
  ()
;

value rec print_spaces =
  fun
  [ 0 -> ()
  | n -> do print_char sp.val; return print_spaces (n - 1) ]
;

value string_np pos np = pos + np;

value trace_ov pos =
  if not quiet.val && pos > maxl.val then
    do prerr_string "<W> prettych: overflow (length = ";
       prerr_int pos;
       prerr_endline ")";
    return ()
  else ()
;

value tolerate tab pos spc = pos + spc <= tab + dt.val + tol.val;

value print_string_with_nl pos s =
  let rec print pos i =
    if i == String.length s then do trace_ov pos; return pos
    else
      match s.[i] with
      [ '\\' when i + 1 < String.length s && s.[i + 1] == 'n' ->
          do trace_ov pos; print_string "\n"; return print 0 (i + 2)
      | '\n' -> do trace_ov pos; print_string "\n"; return print 0 (i + 1)
      | _ -> do print_char s.[i]; return print (succ pos) (succ i) ]
  in
  if s.[0] == '"' then
    if String.length s >= 1 && s.[1] == '\n' then
      do print_string "\"\n"; return print 0 2
    else if String.length s >= 2 && s.[1] == '\\' && s.[2] == 'n' then
      do print_string "\"\n"; return print 0 3
    else
      do print_string "\""; return
      let i =
        loop 1 where rec loop i =
          if i < String.length s && s.[i] = ' ' then
            do print_string " "; return loop (i + 1)
          else
            do print_string "\\\n"; return i
      in
      print 0 i
  else print pos 0
;

value h_print_string pos spc np x =
  let npos = string_np (pos + spc) np in
  if npos > maxl.val then
    do print_spaces spc; return print_string_with_nl (pos + spc) x
  else do print_spaces spc; print_string x; return npos
;

value n_print_string pos spc np x =
  do print_spaces spc; print_string x; return
  string_np (pos + spc) np
;

value rec hnps ((pos, spc) as ps) =
  fun
  [ SL np RO _ -> (string_np pos np, 1)
  | SL np LO _ -> (string_np (pos + spc) np, 0)
  | SL np NO _ -> (string_np pos np, 0)
  | SL np LR _ -> (string_np (pos + spc) np, 1)
  | HL x -> hnps_list ps x
  | BL x -> hnps_list ps x
  | PL x -> hnps_list ps x
  | QL x -> hnps_list ps x
  | VL [x] -> hnps ps x
  | VL [] -> ps
  | VL x -> (maxl.val + 1, 0)
  | BE x -> hnps_list ps x
  | BV x -> (maxl.val + 1, 0) ]
and hnps_list ((pos, _) as ps) pl =
  if pos > maxl.val then (maxl.val + 1, 0)
  else
    match pl with
    [ [p :: pl] -> hnps_list (hnps ps p) pl
    | [] -> ps ]
;

value rec first =
  fun
  [ SL _ _ s -> Some s
  | HL x -> first_in_list x
  | BL x -> first_in_list x
  | PL x -> first_in_list x
  | QL x -> first_in_list x
  | VL x -> first_in_list x
  | BE x -> first_in_list x
  | BV x -> first_in_list x ]
and first_in_list =
  fun
  [ [p :: pl] ->
      match first p with [ Some p -> Some p | None -> first_in_list pl ]
  | [] -> None ]
;

value first_is_too_big tab p =
  match first p with
  [ Some s -> tab + String.length s >= maxl.val
  | None -> False ]
;

value too_long tab x p =
  if first_is_too_big tab p then False
  else let (pos, spc) = hnps x p in pos > maxl.val
;

value rec hprint_pretty pos spc =
  fun
  [ SL np RO x -> (h_print_string pos 0 np x, 1)
  | SL np LO x -> (h_print_string pos spc np x, 0)
  | SL np NO x -> (h_print_string pos 0 np x, 0)
  | SL np LR x -> (h_print_string pos spc np x, 1)
  | HL x -> hprint_box pos spc x
  | BL x -> hprint_box pos spc x
  | PL x -> hprint_box pos spc x
  | QL x -> hprint_box pos spc x
  | VL [x] -> hprint_pretty pos spc x
  | VL [] -> (pos, spc)
  | VL _ -> invalid_arg "hprint_pretty"
  | BE x -> hprint_box pos spc x
  | BV x -> invalid_arg "hprint_pretty" ]
and hprint_box pos spc =
  fun
  [ [p :: pl] ->
      let (pos, spc) = hprint_pretty pos spc p in hprint_box pos spc pl
  | [] -> (pos, spc) ]
;

value rec print_pretty tab pos spc =
  fun
  [ SL np RO x -> (n_print_string pos 0 np x, 1)
  | SL np LO x -> (n_print_string pos spc np x, 0)
  | SL np NO x -> (n_print_string pos 0 np x, 0)
  | SL np LR x -> (n_print_string pos spc np x, 1)
  | (HL x as p) -> print_horiz tab pos spc x
  | (BL x as p) -> print_horiz_vertic tab pos spc (too_long tab (pos, spc) p) x
  | (PL x as p) -> print_paragraph tab pos spc (too_long tab (pos, spc) p) x
  | (QL x as p) -> print_sparagraph tab pos spc (too_long tab (pos, spc) p) x
  | VL x -> print_vertic tab pos spc x
  | (BE x as p) -> print_begin_end tab pos spc (too_long tab (pos, spc) p) x
  | BV x -> print_beg_end tab pos spc x ]
and print_horiz tab pos spc =
  fun
  [ [p :: pl] ->
      let (npos, nspc) = print_pretty tab pos spc p in
      if match pl with
         [ [] -> True
         | _ -> False ] then
        (npos, nspc)
      else print_horiz tab npos nspc pl
  | [] -> (pos, spc) ]
and print_horiz_vertic tab pos spc ov pl =
  if ov then print_vertic tab pos spc pl else hprint_box pos spc pl
and print_vertic tab pos spc =
  fun
  [ [p :: pl] ->
      let (npos, nspc) = print_pretty tab pos spc p in
      if match pl with
         [ [] -> True
         | _ -> False ] then
        (npos, nspc)
      else if tolerate tab npos nspc then
        do print_spaces nspc; return print_vertic_rest (npos + nspc) pl
      else
        do print_newline (); print_spaces (tab + dt.val); return
        print_vertic_rest (tab + dt.val) pl
  | [] -> (pos, spc) ]
and print_vertic_rest tab =
  fun
  [ [p :: pl] ->
      let (pos, spc) = print_pretty tab tab 0 p in
      if match pl with
         [ [] -> True
         | _ -> False ] then
        (pos, spc)
      else
        do print_newline (); print_spaces tab; return print_vertic_rest tab pl
  | [] -> (tab, 0) ]
and print_paragraph tab pos spc ov pl =
  if ov then print_parag tab pos spc pl else hprint_box pos spc pl
and print_parag tab pos spc =
  fun
  [ [p :: pl] ->
      let (npos, nspc) = print_pretty tab pos spc p in
      if match pl with
         [ [] -> True
         | _ -> False ] then
        (npos, nspc)
      else if npos == tab then
        print_parag_rest tab tab 0 pl
      else if too_long tab (pos, spc) p then
        do print_newline (); print_spaces (tab + dt.val); return
        print_parag_rest (tab + dt.val) (tab + dt.val) 0 pl
      else if tolerate tab npos nspc then
        do print_spaces nspc; return
        print_parag_rest (npos + nspc) (npos + nspc) 0 pl
      else
        print_parag_rest (tab + dt.val) npos nspc pl
  | [] -> (pos, spc) ]
and print_parag_rest tab pos spc =
  fun
  [ [p :: pl] ->
      let (pos, spc) =
        if pos > tab && too_long tab (pos, spc) p then
          do print_newline (); print_spaces tab; return (tab, 0)
        else (pos, spc)
      in
      let (npos, nspc) = print_pretty tab pos spc p in
      if match pl with
         [ [] -> True
         | _ -> False ] then
        (npos, nspc)
      else
        let (pos, spc) =
          if npos > tab && too_long tab (pos, spc) p then
            do print_newline (); print_spaces tab; return (tab, 0)
          else (npos, nspc)
        in
        print_parag_rest tab pos spc pl
  | [] -> (pos, spc) ]
and print_sparagraph tab pos spc ov pl =
  if ov then print_sparag tab pos spc pl else hprint_box pos spc pl
and print_sparag tab pos spc =
  fun
  [ [p :: pl] ->
      let (npos, nspc) = print_pretty tab pos spc p in
      if match pl with
         [ [] -> True
         | _ -> False ] then
        (npos, nspc)
      else if tolerate tab npos nspc then
        do print_spaces nspc; return
        print_sparag_rest (npos + nspc) (npos + nspc) 0 pl
      else print_sparag_rest (tab + dt.val) npos nspc pl
  | [] -> (pos, spc) ]
and print_sparag_rest tab pos spc =
  fun
  [ [p :: pl] ->
      let (pos, spc) =
        if pos > tab && too_long tab (pos, spc) p then
          do print_newline (); print_spaces tab; return (tab, 0)
        else (pos, spc)
      in
      let (npos, nspc) = print_pretty tab pos spc p in
      if match pl with
         [ [] -> True
         | _ -> False ] then
        (npos, nspc)
      else print_sparag_rest tab npos nspc pl
  | [] -> (pos, spc) ]
and print_begin_end tab pos spc ov pl =
  if ov then print_beg_end tab pos spc pl else hprint_box pos spc pl
and print_beg_end tab pos spc =
  fun
  [ [p :: pl] ->
      let (npos, nspc) = print_pretty tab pos spc p in
      if match pl with
         [ [] -> True
         | _ -> False ] then
        (npos, nspc)
(**)
      else if tolerate tab npos nspc then
        let nspc = if npos == tab then nspc + dt.val else nspc in
        do print_spaces nspc; return print_beg_end_rest tab (npos + nspc) pl
(**)
      else
        do print_newline (); print_spaces (tab + dt.val); return
        print_beg_end_rest tab (tab + dt.val) pl
  | [] -> (pos, spc) ]
and print_beg_end_rest tab pos =
  fun
  [ [p :: pl] ->
      let (pos, spc) = print_pretty (tab + dt.val) pos 0 p in
      if match pl with
         [ [] -> True
         | _ -> False ] then
        (pos, spc)
      else
        do print_newline (); print_spaces tab; return
        print_beg_end_rest tab tab pl
  | [] -> (pos, 0) ]
;

value string_npos s = String.length s;

value rec conv =
  fun
  [ S g s -> SL (string_npos s) g s
  | Hbox x -> HL (conv_stream x)
  | HVbox x -> BL (conv_stream x)
  | HOVbox x -> PL (conv_stream x)
  | HOVCbox x -> QL (conv_stream x)
  | Vbox x -> VL (conv_stream x)
  | BEbox x -> BE (conv_stream x)
  | BEVbox x -> BV (conv_stream x) ]
and conv_stream =
  parser
  [ [: `p; s :] -> [conv p :: conv_stream s]
  | [: :] -> [] ]
;

value print_pretty ch pr m p =
  do maxl.val := m; out_ch.val := ch; prompt.val := pr; print_string pr; return
  let r = print_pretty 0 0 0 (conv p) in flush ch
;

type pr_fun =
  (MLast.expr -> string -> Stream.t pretty -> pretty) ->
  (MLast.patt -> string -> Stream.t pretty -> pretty) ->
  MLast.expr -> string -> Stream.t pretty -> pretty
;
value pr_funs = ref [];
value has_pr_fun x = List.mem_assoc x pr_funs.val;
value pr_fun x = List.assoc x pr_funs.val;
value add_pr_fun x y = pr_funs.val := [(x, y) :: pr_funs.val];
