/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1997                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*             Author :  Alan W Black                                    */
/*             Date   :  June 1997                                       */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/* A probabilistic chart parser                                          */
/*                                                                       */
/*=======================================================================*/

#include "EST.h"
#include "festival.h"
#include "PChart.h"

#if defined(INSTANTIATE_TEMPLATES)
#include "../base_class/EST_TList.cc"
#include "../base_class/EST_TSortable.cc"

template class EST_TList<PChart_Edge *>;
template class EST_TItem<PChart_Edge *>;
template class EST_TList<PChart_Vertex *>;
template class EST_TItem<PChart_Vertex *>;
#endif

PChart_Edge::PChart_Edge()
{
    p_start = 0;
    p_end = 0;
    p_name = "";
    p_log_prob = 0.0;
}

PChart_Edge::PChart_Edge(PChart_Vertex *start, PChart_Vertex *end, 
			 const EST_String &name,
			 const PChart_EdgeList &recog, 
			 const EST_StrList &remainder,
			 double log_prob)
{
    p_start = start;
    p_end = end;
    p_name = name;
    p_recog = recog;
    p_remainder = remainder;
    p_log_prob = log_prob;
}

LISP PChart_Edge::print_edge()
{
    // Return a lisp representation of the edge
    LISP daughters=NIL;
    EST_TBI *e,*d;


    for (e=p_recog.head(); e != 0; e=next(e))
	daughters = cons(p_recog(e)->print_edge(),daughters);
    for (d=p_remainder.head(); d != 0; d=next(d))
	daughters = cons(rintern(p_remainder(d)),daughters);

    return cons(cons(rintern(p_name),
		     cons(flocons(p_log_prob),
			  cons(strintern(p_start->name()),
			       cons(strintern(p_end->name()),NIL)))),
		reverse(daughters));
}

PChart_Vertex::PChart_Vertex()
{
    p_name="";
    s=0;
}

PChart::PChart()
{
    rules = NIL;
    gc_protect(&rules);
}

PChart::~PChart()
{
    gc_unprotect(&rules);

    // delete all the vertices
}

void PChart::setup_wfst(EST_Stream &s,const EST_String &name)
{
    // Set up well formed substring table from feature name in each
    // stream item in s
    EST_Stream_Item *p;
    PChart_Vertex *v;
    EST_TBI *lv;
    EST_StrList emptyStrList;
    PChart_EdgeList emptyEdgeList;
    int n;

    for (n=0; n < s.length()+1; n++)
    {
	v = new PChart_Vertex;
	v->set_name(itoString(n));
	vertices.append(v); 
    }

    for (lv=vertices.head(),p=s.head(); p != 0; p=next(p),lv=next(lv))
    {
	// construct initial complete edges and put them on the agenda
	PChart_Edge *e = new PChart_Edge(vertices(lv),vertices(next(lv)),
					 p->feature(name).string(),
					 emptyEdgeList,
					 emptyStrList,
					 0.0);
	agenda.append(e);
    }
}

void PChart::parse(void)
{
    // do the parsing 
    PChart_Edge *current, *newedge;
    EST_TBI *p;
    

    while (agenda.length() > 0)
    {
	current = agenda.first();
	agenda.remove(agenda.head());
	
	if (current->complete())
	{
	    // Propose name of this edge to the grammar
	    grammar_propose(current);

	    // find incomplete edges coming in to the start of this edge
	    // and find any combinations
	    const PChart_EdgeList &incompletes = 
		current->start()->incomplete_incoming();
	    for (p=incompletes.head(); p != 0; p = next(p))
	    {
		newedge = combine_edges(current,incompletes(p));
		if (newedge != 0)
		    agenda.append(newedge);
	    }
	    current->start()->complete_outgoing().append(current);
	}
	else 
	{
	    // find complete edges going out of the end of this edge
	    const PChart_EdgeList &completes = 
		current->end()->complete_outgoing();
	    for (p=completes.head(); p != 0; p = next(p))
	    {
		newedge = combine_edges(completes(p),current);
		if (newedge != 0)
		    agenda.append(newedge);
	    }
	    current->end()->incomplete_incoming().append(current);
	}
    }

}

void PChart::grammar_propose(PChart_Edge *edge)
{
    // Look for all grammar rules whose first daughter matches
    // edge's name.  Construct incomplete edges for these rules and
    // add them to the agenda
    LISP r,d;
    PChart_EdgeList emptyEdgeList;

    for (r=rules; r != NIL; r=cdr(r))
    {
	if (edge->name() == get_c_string(car(cdr(car(r)))))
	{
	    EST_StrList ruleremainder;
	    double lprob = 0.0;
	    for (d=cdr(car(r)); d != NIL; d=cdr(d))
		if ((cdr(d) == NIL) &&
		    (TYPEP(car(d),tc_flonum)))
		    lprob = safe_log(FLONM(car(d)));
		else
		    ruleremainder.append(get_c_string(car(d)));
	    PChart_Edge *e = 
		new PChart_Edge(edge->start(),
				edge->start(),
				get_c_string(car(car(r))),
				emptyEdgeList,
				ruleremainder,
				lprob);          // log prob of rule
				     
	    agenda.append(e);
	}
    }
}    

PChart_Edge *PChart::combine_edges(PChart_Edge *complete, 
				   PChart_Edge *incomplete)
{
    // Combine these edges if allowed creating a new edge if they match
    // return 0 if they don't match

    if ((complete->start() == incomplete->end()) &&
	(complete->name() == incomplete->remainder().first()))
    {
	// Do something with those probabilities
	PChart_EdgeList newrecog;
	newrecog = incomplete->recog();
	newrecog.append(complete);  // add new recognized daughter
	EST_StrList newremainder;
	newremainder = incomplete->remainder();
	newremainder.remove(newremainder.head()); // remove recognized daughter
	
	return new PChart_Edge(incomplete->start(),
			       complete->end(),
			       incomplete->name(),
			       newrecog,
			       newremainder,
			       (incomplete->log_prob()+
				complete->log_prob()));
    }
    else
	return 0;  // doesn't match

}

void PChart::dump_chart()
{
    // Print out all edges in the chart
    EST_TBI *v, *e;

    for (v=vertices.head(); v != 0; v=next(v))
    {
	cout << vertices(v)->name() << endl;
	cout << "Incoming incomplete:" << endl;
	for (e=vertices(v)->incomplete_incoming().head(); e != 0; e=next(e))
	    pprint(vertices(v)->incomplete_incoming() (e)->print_edge());
	cout << "Outgoing complete:" << endl;
	for (e=vertices(v)->complete_outgoing().head(); e != 0; e=next(e))
	    pprint(vertices(v)->complete_outgoing()(e)->print_edge());
	cout << endl;
    }
}

void PChart::find_parses()
{
    // Not sure want I'm supposed to do with the result
    EST_TBI *e;
    PChart_EdgeList &edges = vertices.first()->complete_outgoing();

    for (e=edges.head(); e != 0; e=next(e))
    {
	if (edges(e)->end() == vertices.last())
	{
	    pprint(edges(e)->print_edge());
	    cout << endl;
	}
    }
}

