/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                      Author :  Alan W Black                           */
/*                      Date   :  August 1996                            */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/* Various part-of-speech predciting modules                             */
/*                                                                       */
/*=======================================================================*/
#include <stdio.h>
#include "festival.h"
#include "fngram.h"
#include "lexicon.h"

static EST_VTCandidate *pos_candlist(EST_Stream_Item &s);
static EST_VTPath *pos_npath(EST_VTPath *p,EST_VTCandidate *c);
static double find_np_prob(EST_VTPath *p,int n,int *state);

static EST_Ngrammar *pos_ngram = 0;
static EST_Utterance *this_utt = 0;

static EST_String zeroString("0");
static int p_word = 0;  // arbitrary numbers
static int n_word = 1;

LISP FT_POS_Utt(LISP utt)
{
    // Predict part of speech for word stream
    EST_Utterance *u = GETUTTVAL(utt);
    LISP pos_lex_name, pos_ngram_name;
    LISP lastlex, pos_p_start_tag, pos_pp_start_tag;
    
    this_utt = u;

    *cdebug << "POS module\n";

    pos_lex_name = siod_get_lval("pos_lex_name",NULL);
    if (pos_lex_name == NIL)
	return utt;   // not set so ignore it
    pos_ngram_name = siod_get_lval("pos_ngram_name","no pos ngram name");
    pos_p_start_tag = siod_get_lval("pos_p_start_tag","no prev start tag");
    pos_pp_start_tag = siod_get_lval("pos_pp_start_tag","no prev prev start tag");
    
    lastlex = lex_select_lex(pos_lex_name);

    if ((pos_ngram = get_ngram(get_c_string(pos_ngram_name))) == 0)
    {
	cerr << "POS: no ngram called \"" <<
	    get_c_string(pos_ngram_name) << "\" defined" << endl;
	festival_error();
    }

    p_word = pos_ngram->get_vocab_word(get_c_string(pos_p_start_tag));
    n_word = pos_ngram->get_vocab_word(get_c_string(pos_pp_start_tag));

    EST_Viterbi_Decoder v(pos_candlist,pos_npath,pos_ngram->states());
    
    v.initialise(u->stream("Word"));
    v.search();
    v.result("pos_index");

    lex_select_lex(lastlex);

    EST_Stream_Item *w;
    EST_String pos;
    LISP l;
    // Map pos tagset to desired set
    LISP pos_map = siod_get_lval("pos_map",NULL);
    for (w=(u->stream("Word")).head(); w != 0; w = next(w))
    {
	// convert pos index into string value
	w->set_feature("pos",pos_ngram->
		       get_vocab_word(w->feature("pos_index").Int()));
	pos = w->feature("pos").string();
	for (l=pos_map; l != NIL; l=cdr(l))
	    if (siod_member_str(pos,car(car(l))) != NIL)
	    {
		w->set_feature("pos",get_c_string(car(cdr(car(l)))));
		break;
	    }
    }

    return utt;
}

static EST_VTCandidate *pos_candlist(EST_Stream_Item &s)
{
    // Return list of possible pos based on a priori probabilities
    LISP pd,l;
    EST_VTCandidate *c;
    EST_VTCandidate *all_c = 0;
    EST_Val actual_pos;
    
    LISP e = lex_lookup_word(s.name(),NIL);
    pd = car(cdr(e));
    
    if (((actual_pos = s.feature("pos")) != "0") ||
	((actual_pos = ffeature(*this_utt,s,"Token.pos")) != "0"))
    {
	// There is an explicit pos specified, so respect it
	pd = cons(make_param_float(actual_pos.string(),1.0),NIL);
    }
    else if (pd == NIL)
    {
	const char *chr = s.name();
	if (strchr("0123456789",chr[0]) != NULL)
	    e = lex_lookup_word("_number_",NIL); // I *know* there is an entry
	else
	    e = lex_lookup_word("_OOV_",NIL); // I *know* there is an entry
	pd = car(cdr(e));
    }
    
    for (l=pd; l != NIL; l=cdr(l))
    {
	c = new EST_VTCandidate;
	c->name = pos_ngram->get_vocab_word(get_c_string(car(car(l))));
	c->score = get_c_float(car(cdr(car(l))));
	c->s = &s;
	c->next = all_c;
	all_c = c;
    }
    
    return all_c;
}

static EST_VTPath *pos_npath(EST_VTPath *p,EST_VTCandidate *c)
{
    // Build a potential new path from previous path and this candidate
    EST_VTPath *np = new EST_VTPath;
    double prob;
    double lprob;
    
    np->c = c;
    np->from = p;
    int n = c->name.Int();
    prob = find_np_prob(p,n,&np->state);
    if (prob == 0)
	lprob = log(0.00000001);
    else
	lprob = log(prob);
    
    np->set_feature("lscore",(c->score+lprob));
    if (p==0)
	np->score = (c->score+lprob);
    else
	np->score = (c->score+lprob) + p->score;
    
    return np;
}

static double find_np_prob(EST_VTPath *p,int n,int *state)
{
    EST_IVector window(pos_ngram->order());
    EST_VTPath *t;
    double prob, nprob;
    int i;
    int f=FALSE;

    window(pos_ngram->order()-1) = n;
    for (t=p,i=pos_ngram->order()-2; i >= 0; i--)
    {
	if ((t == 0) || (t->c == 0))
	{                 // when there is no previous context use
	    if (f)        // the specified previous (punc and noun)
		window(i) = n_word;
	    else
	    {
		window(i) = p_word;
		f = TRUE;			// assume one p 
	    }
	}
	else
	{
	    window(i) = t->c->name.Int();
	    t = t->from;
	}
    }
    
    const EST_DiscreteProbDistribution &pd = pos_ngram->prob_dist(window);
    if (pd.samples() == 0)
	prob = 0;
    else
	prob = (double)pd.probability(n);
    
    // Going to have to fix this sometime
    // Need the value of the state I'm going to
    for (t=p,i=0; i < pos_ngram->order()-1; i++)
	window(i) = window(i+1);
    pos_ngram->predict(window,&nprob,state);

    return prob;
    
}
