 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /************************************************************************/
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)            */
 /*                   Date: Thu Mar 20 1997                              */
 /************************************************************************/
 /*                                                                      */
 /* A very simple unit selecter for use sith diphones indexed under      */
 /* the phoneme names.                                                   */
 /*                                                                      */
 /************************************************************************/

#include "festival.h"
#include "UnitDatabase.h"
#include "SimpleDiphoneUnitIndex.h"
#include "module_support.h"

static ModuleDescription description =
{
  "simple_diphone_select", 1.0,
  "CSTR",
  "Richard Caley <rjc@cstr.ed.ac.uk>",
  {
    "A simple selector for diphone units which are indexed under pairs",
    "of segment names.",
    NULL
  },
  {
    { "Segment",	"Contains the names of the segments to be looked up in pairs." },
    {NULL,NULL}
  },
  { {NULL,NULL}
  },
  {
    { "Unit",		"Stream of diphone Units" },
    {NULL,NULL}
  },
  {
    { "UnitDatabase",	mpt_unitdatabase,	NULL,	"Database in which to find units."},
    { "SelectVerbose",	mpt_bool,		"nil",	"Output trace of selection process." },
    {NULL,NULL,NULL,NULL}
  }
};

LISP simple_diphone_select(LISP args)
{
  SimpleDiphoneUnitIndex *new_index = NULL;

  EST_Utterance *utt;
  
  EST_String segment_stream_name("Segment");
  EST_String unit_stream_name("Unit");

  EST_Stream *segment_stream=NULL, *unit_stream=NULL;

  
  unpack_module_args(args, 
		     utt, 
		     segment_stream_name, segment_stream, sat_existing,
		     unit_stream_name, unit_stream, sat_replace);
		     

  LISP ldatabase = lisp_parameter_get("UnitDatabase");
  bool verbose    = bool_parameter_get("SelectVerbose");
  
  if (TYPE(ldatabase) !=  tc_festival_unitdatabase)
    err("parameter UnitDatabase not set to database", ldatabase);

  UnitDatabase *database = (UnitDatabase *)PTRVAL(ldatabase);

  const SimpleDiphoneUnitIndex *index = (const SimpleDiphoneUnitIndex *) database->index("SimpleDiphone");

  if(!index)
    {
      cwarn <<"no SimpleDiphone index defined in database, computing\n";
      new_index = (SimpleDiphoneUnitIndex *)UnitIndex::create("SimpleDiphone");
      if(!new_index)
	err("can't create SimpleDiphone index", NIL);

      // remove the constness for a moment so we can build it
      new_index->associate_database(database);
      new_index->index();
      cwarn <<"done\n";
      index = new_index;
    }

  EST_Stream_Item *seg1, *seg2;
  const EST_String *p1=NULL, *p2=NULL;

  seg2=NULL;
  for(seg1=segment_stream->head(); seg1 ; seg2=seg1, seg1=next(seg1))
    {
      p1=p2;
      p2 = &(seg1->name());

      if (p1 && p2)
	{
	  EST_TList<UnitName> options;

	  if (index->lookup(*p1, *p2, options))
	    {
	      EST_String name(options.first());
	      
	      if (verbose)
		cout << *p1 << "," << *p2 << ":" << name << "\n";	  

	      const Unit *unit = database->unit(name);

	      if (!unit)
		{
		  cerr << "can't get unit '" << name << "'\n";
		  return NULL;
		}

	      ((Unit *)unit)->gc_ref();
	      EST_Stream_Item *item = 
		  unit_stream->append((Unit *)unit, unit->name(), 
				      Unit::decrease_reference_count);

	      item->make_link(*seg2);
	      item->make_link(*seg1);
	      item->set_feature("type", unit->subtype_name());
	    }
	  else
	    {
	      cout << *p1 << "," << *p2 << ": UNKNOWN\n";	
	    }
	}
    }

  if (new_index)
    delete new_index;

  return ldatabase;
}

void simple_diphone_select_init(void)
{
  proclaim_module("simple_diphone_select", &description);

  init_module_subr("simple_diphone_select", simple_diphone_select, &description);
}

