/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*             Author :  Alan W Black                                    */
/*             Date   :  April 1996                                      */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/* Basic lexicon utilities                                               */
/*                                                                       */
/*=======================================================================*/
#include <stdio.h>
#include "festival.h"
#include "lexicon.h"
#include "lexiconP.h"

static LISP guess_pos=NIL;
static void split_stress(LISP phones, LISP &phs, LISP &stresses);
static char *v_stress(char *ph,int &stress);
static int syl_contains_vowel(LISP phones);
static int syl_breakable(LISP syl, LISP rest);

EST_Val ff_word_gpos(EST_Utterance &u,EST_Stream_Item &s)
{
    /* Part of speech by guessing, returns, prep, det, aux, content */
    /* from simple lookup list                                      */
    (void)u; // unused parameter
    EST_String word;
    LISP l;

    word = downcase(s.name());

    if (guess_pos == NIL)
	guess_pos = siod_get_lval("guess_pos","no guess_pos set");
    
    for (l=guess_pos; l != NIL; l=cdr(l))
	if (siod_member_str(word,cdr(car(l))))
	    return EST_Val(get_c_string(car(car(l))));

    return EST_Val("content");
}

EST_Val ff_word_cf(EST_Utterance &u,EST_Stream_Item &s)
{
    /* 1 if this is content and next is function, 0 otherwise */
    
    if ((next(&s) != 0) &&
	(ff_word_gpos(u,s) == "content") &&
	(ff_word_gpos(u,*(next(&s))) != "content"))
	return EST_Val("1");
    else
	return EST_Val("0");
}

EST_Val ff_word_contentp(EST_Utterance &u,EST_Stream_Item &s)
{
    /* 1 if this is a content word, 0 otherwise */
    
    if (ff_word_gpos(u,s) == "content")
	return EST_Val("1");
    else
	return EST_Val("0");
}

EST_Val ff_word_n_content(EST_Utterance &u,EST_Stream_Item &s)
{
    // returns the next content word after s
    EST_Stream_Item *p;

    for (p=next(&s); p != 0; p = next(p))
    {
	if (ff_word_gpos(u,*p) == "content")
	    return EST_Val(p->name());
    }

    return EST_Val("0");
}

EST_Val ff_word_nn_content(EST_Utterance &u,EST_Stream_Item &s)
{
    // returns the next next content word after s
    EST_Stream_Item *p;
    int count = 0;

    for (p=next(&s); p != 0; p = next(p))
    {
	if (ff_word_gpos(u,*p) == "content")
	{
	    count++;
	    if (count == 2)
		return EST_Val(p->name());
	}
    }

    return EST_Val("0");
}

EST_Val ff_word_p_content(EST_Utterance &u,EST_Stream_Item &s)
{
    // returns the previous content word after s
    EST_Stream_Item *p;
    int count = 0;

    for (p=prev(&s); p != 0; p = prev(p))
    {
	if (ff_word_gpos(u,*p) == "content")
	{
	    count++;
	    if (count == 1)
		return EST_Val(p->name());
	}
    }

    return EST_Val("0");
}

EST_Val ff_word_pp_content(EST_Utterance &u,EST_Stream_Item &s)
{
    // returns the previous previous content word after s
    EST_Stream_Item *p;
    int count = 0;

    for (p=prev(&s); p != 0; p = prev(p))
    {
	if (ff_word_gpos(u,*p) == "content")
	{
	    count++;
	    if (count == 2)
		return EST_Val(p->name());
	}
    }

    return EST_Val("0");
}

EST_Val ff_word_cap(EST_Utterance &u,EST_Stream_Item &s)
{
    //  "1" is the word starts with a capital letter
    (void)u; // unused parameter
    const char *word = s.name();

    if ((word[0] >= 'A') && (word[0] <='Z'))
	return EST_Val("1");
    else
	return EST_Val("0");
}

EST_Val ff_syl_reduced(EST_Utterance &u,EST_Stream_Item &s)
{
    // This is used to extract vowel reduction information from a 
    // labelled database.  The word related to the current syllable is
    // looked up in the lexicon and a match is done.  If the vowel in
    // this syllable is different from its corresponding syl in the
    // the lexical entry it is potentiall reduced.  The pair are
    // looked up in postlex_vowel_reduce_table and if it is in 
    // that list 1 is returned.  If there is a mismatch and it 
    // not in that table it is printed to the screen (so you can
    // update the table) and 0 is returned.  If they match 0 is returned
    LISP vow_table, full_vow_table, lex_entry, lex_syl, v;
    EST_String lex_vowel, actual_vowel;

    full_vow_table = siod_get_lval("postlex_vowel_reduce_table",NULL);
    vow_table =       // Phoneset specific vowel reduction table
	car(cdr(siod_assoc_str(get_c_string(ft_get_param("PhoneSet")),
			       full_vow_table)));
    lex_entry = lex_lookup_word((EST_String)ffeature(u,s,"Word.name"),NIL);
    lex_syl = siod_nth(ffeature(u,s,"pos_in_word"),car(cdr(cdr(lex_entry))));
    
    if (lex_syl == NIL)
    {
	cerr << "syl_reduced: syllable count mismatch for syllable: " <<
	   (EST_String)ffeature(u,s,"pos_in_word") << " in " <<  
	       (EST_String)ffeature(u,s,"Word.name") << endl;
	return "U";
    }
    else 
    {
	for (v=car(lex_syl); cdr(v) != NIL; v=cdr(v))
	    if (ph_is_vowel(get_c_string(car(v))))
		break;
	lex_vowel = get_c_string(car(v));

	EST_Relation *segs = s.link("Segment");
	EST_TBI *p;

	for (p=segs->head(); next(p) != 0; p=next(p))
	{
	    if (ph_is_vowel(u.ritem("Segment",(*segs)(p)).name()))
		break;
	}
	actual_vowel = u.ritem("Segment",(*segs)(p)).name();

	if (lex_vowel == actual_vowel)
	    return EST_Val("0");  // not reduced
	else if (siod_member_str(actual_vowel,(cdr(siod_assoc_str(lex_vowel,vow_table)))))
	{
	    // Shock horror, I'm going to fix the to the unreduced on
	    u.ritem("Segment",(*segs)(p)).set_name(lex_vowel);
	    return EST_Val("1");  // reduced
	}
	else
	{
	    cerr << "syl_reduced: possible reduction " <<
		lex_vowel << " to " << actual_vowel << endl;
	    return EST_Val("U");
	}
    }
}

EST_Val ff_syl_lex_vowel(EST_Utterance &u,EST_Stream_Item &s)
{
    // Finds the vowel in in the syllable in the lexical entry 
    // corresponding to this syllable.  This is used in testing 
    // for reduction.
    LISP lex_entry, lex_syl, v;
    EST_String lex_vowel, actual_vowel;

    lex_entry = lex_lookup_word((EST_String)ffeature(u,s,"Word.name"),NIL);
    lex_syl = siod_nth(ffeature(u,s,"pos_in_word"),car(cdr(cdr(lex_entry))));
    
    if (lex_syl == NIL)
	return "unknown";
    else 
    {
	for (v=car(lex_syl); cdr(v) != NIL; v=cdr(v))
	    if (ph_is_vowel(get_c_string(car(v))))
		break;
	lex_vowel = get_c_string(car(v));
	return EST_Val(lex_vowel);
    }
}

EST_Val ff_syl_onset_type(EST_Utterance &u,EST_Stream_Item &s)
{
    // Return van Santen's classification of onset type in to one
    // of three forms:
    //   -V    contains only voiceless consonants
    //   +V-S  contains voiced obstruents but no sonorants
    //   +S    contains just sonorants
    EST_Relation *segs = s.link("Segment");
    EST_TBI *p;
    int vox=FALSE;
    int sonorant=FALSE;

    for (p=segs->head(); next(p) != 0; p=next(p))
    {
	if (ph_is_vowel(u.ritem("Segment",(*segs)(p)).name()))
	    break;
	if (ph_is_voiced(u.ritem("Segment",(*segs)(p)).name()))
	    vox = TRUE;
	if (ph_is_sonorant(u.ritem("Segment",(*segs)(p)).name()))
	    sonorant = TRUE;
    }

    if (p==segs->head()) // null-onset case
	return EST_Val("+V-S");
    else if (sonorant)
	return EST_Val("+S");
    else if (vox)
	return EST_Val("+V-S");
    else
	return EST_Val("-V");
}

EST_Val ff_syl_coda_type(EST_Utterance &u,EST_Stream_Item &s)
{
    // Return van Santen's classification of onset type in to one
    // of three forms:
    //   -V    contains only voiceless consonants
    //   +V-S  contains voiced obstruents but no sonorants
    //   +S    contains just sonorants
    EST_Relation *segs = s.link("Segment");
    EST_TBI *p;
    int vox=FALSE;
    int sonorant=FALSE;

    for (p=segs->head(); next(p) != 0; p=next(p))
    {
	if (ph_is_vowel(u.ritem("Segment",(*segs)(p)).name()))
	    break;
    }

    if (next(p) == 0)         // empty coda
	return EST_Val("+S");

    for (p=next(p); p != 0; p=next(p))
    {
	if (ph_is_voiced(u.ritem("Segment",(*segs)(p)).name()))
	    vox = TRUE;
	if (ph_is_sonorant(u.ritem("Segment",(*segs)(p)).name()))
	    sonorant = TRUE;
    }

    if (sonorant)
	return EST_Val("+S");
    else if (vox)
	return EST_Val("+V-S");
    else
	return EST_Val("-V");
}

LISP lex_syllabify(LISP phones)
{
    /* Given a simple list of phones, syllabify them and add stress */
    LISP syl,syls,p;
    int stress = 1;

    for (syl=NIL,syls=NIL,p=phones; p != NIL; p=cdr(p))
    {
	syl = cons(car(p),syl);
	if (syl_breakable(syl,cdr(p)))
	{
	    syls = cons(cons(reverse(syl),cons(flocons(stress),NIL)),syls);
	    stress = 0;
	    syl = NIL;
	}
    }

    return reverse(syls);
}

LISP lex_syllabify_phstress(LISP phones)
{
    /* Given a list of phones where vowels may have stress numeral,  */
    /* as found in BEEP and CMU syllabify them */
    LISP syl,syls,p,phs,stresses,s;
    int stress = 0;
    char *ph;

    split_stress(phones,phs,stresses);

    for (syl=NIL,syls=NIL,p=phs,s=stresses; 
	 p != NIL; 
	 p=cdr(p),s=cdr(s))
    {
	ph = get_c_string(car(p));
	if (!streq(ph,ph_silence()))
	    syl = cons(car(p),syl);
	if (car(s) && (!streq(get_c_string(car(s)),"0")))
	    stress = 1; // should worry about 2 stress too
	if (streq(ph,ph_silence()) || syl_breakable(syl,cdr(p)))
	{
	    syls = cons(cons(reverse(syl),cons(flocons(stress),NIL)),syls);
	    stress = 0;
	    syl = NIL;
	}
    }

    return reverse(syls);
}

static void split_stress(LISP phones, LISP &phs, LISP &stresses)
{
    // unpack the list of phones. When they come from certain types
    // of lexical entries (CMU, BEEP) vowels may have a 1 or 2 at their
    // end to denote stress.
    // This returns two list of equal length, one with the phones and
    // one with nils (for each phone) except when there is an explicit
    // stress number
    LISP p,np,ns;
    char *nph;
    int stress;

    for (p=phones,np=ns=NIL; p != NIL; p=cdr(p))
    {
	stress = 0;
	nph = v_stress(get_c_string(car(p)),stress);
	if (streq(nph,"-"))  // a break of some sort
	    np = cons(rintern(ph_silence()),np);
	else
	    np = cons(rintern(nph),np);
	wfree(nph);
	if (stress != 0)
	    ns = cons(flocons(stress),ns);
	else
	    ns = cons(NIL,ns);
    }

    phs = reverse(np);
    stresses = reverse(ns);
}

static char *v_stress(char *ph,int &stress)
{
    //  Checks to see if final character is a numeral, if so treats
    //  is as stress value.
    char *nph;

    if ((ph[strlen(ph)-1] == '1') || 
	(ph[strlen(ph)-1] == '2') ||
	(ph[strlen(ph)-1] == '0'))
    {
	stress = ph[strlen(ph)-1]-'0';
	nph = wstrdup(ph);
	nph[strlen(ph)-1] = '\0';
	return nph;
    }
    else
	return wstrdup(ph);

}

static int syl_breakable(LISP syl, LISP rest)
{
    if (rest == NIL)
	return TRUE;
    else if (!syl_contains_vowel(rest))
	return FALSE;  // must be a vowel remaining in rest 
    else if (syl_contains_vowel(syl))
    {
	if (ph_is_vowel(get_c_string(car(rest))))
	    return TRUE;
	else if (cdr(rest) == NIL)
	    return FALSE;
	int p = ph_sonority(get_c_string(car(syl)));
	int n = ph_sonority(get_c_string(car(rest)));
	int nn = ph_sonority(get_c_string(car(cdr(rest))));

	if ((p <= n) && (n <= nn))
	    return TRUE;
	else
	    return FALSE;
    }
    else
	return FALSE;
}

static int syl_contains_vowel(LISP phones)
{
    // So we can support "vowels" like ah2, oy2 (i.e. vowels with 
    // stress markings) we need to make this a hack.  Vowels are
    // assumed to start with one of aiueo
    LISP p;

    for (p=phones; p !=NIL; p=cdr(p))
	if (strchr("aiueoAIUEO",get_c_string(car(p))[0]) != NULL)
	    return TRUE;
	else if (ph_is_vowel(get_c_string(car(p))))
	    return TRUE;
	else if (ph_is_silence(get_c_string(car(p))))
	    return FALSE;

    return FALSE;
}

