/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*             Author :  Alan W Black, Paul Taylor, Richard Caley        */
/*                       and others                                      */
/*             Date   :  April 1996                                      */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/* Top level file for synthesizer                                        */
/*                                                                       */
/*=======================================================================*/
#include <stdio.h>
#include "festival.h"

static void festival_main(int argc, char **argv);
static void festival_script_mode(int argc, char **argv);

int main(int argc, char **argv)
{

    festival_main(argc,argv);

    return 0;
}

static void festival_main(int argc, char **argv)
{
    EST_Option al;
    int stdin_input,interactive;
    EST_TBI *p;
    EST_StrList files;
    int real_number_of_files = 0;
    int heap_size = DEFAULT_HEAP_SIZE;

    if ((argc > 1) && (streq(argv[1],"--script")))
    {   // Need to check this directly as in script more args are 
	// passed for analysis in the script itself
	
	festival_script_mode(argc,argv);
	return;
    }

    parse_command_line2(argc, argv, 
	EST_String("Usage:\n")+
	"festival  <options> <file0> <file1> ...\n"+
	"Festival Speech Synthesis System: "+ festival_version +"\n"+
	"-q            Load no default setup files\n"+
	"--libdir <string>\n"+
        "              Set library directory pathname\n"+
        "-b            Run in batch mode (no interaction)\n"+
	"--batch       Run in batch mode (no interaction)\n"+
	"--tts         Synthesize text in files as speech\n"+
	"              no files means read from stdin\n"+
	"              (implies no interaction by default)\n"+
	"-i            Run in interactive mode (default)\n"+
	"--interactive\n"+
        "              Run in interactive mode (default)\n"+
	"--pipe        Run in pipe mode, reading commands from\n"+
	"              stdin, but no prompt or return values\n"+
	"              are printed (default if stdin not a tty)\n"+
	"--language <string>\n"+
        "              Run in named language, default is\n"+
	"              english, spanish and welsh are available\n"+
	"--server      Run in server mode waiting for clients\n"+
	"              of server_port (1314)\n"+
	"--script <ifile>\n"+
        "              Used in #! scripts, runs in batch mode on\n"+
	"              file and passes all other args to Scheme\n"+
	"--heap <int> {210000}\n"+
        "              Set size of Lisp heap, should not normally need\n"+
        "              to be changed from its default\n"+
	"-v            Display version number and exit\n"+
	"--version     Display version number and exit\n",
			files, al);

    if ((al.present("-v")) || (al.present("--version")))
    {
	printf("%s: Festival Speech Synthesis System: %s\n",
	       argv[0],festival_version);
	exit(0);
    }

    if (al.present("--libdir"))
	festival_libdir = wstrdup(al.val("--libdir"));
    if (al.present("--heap"))
	heap_size = al.ival("--heap");

    festival_initialize(!al.present("-q"),heap_size);

    if (al.present("--language"))
	festival_init_lang(al.val("--language"));
    
    // File processing
    for (p=files.head(); p != 0; p=next(p))
    {
	if (files(p) == "-")  // paul thinks I want the "-" -- I don't
	    continue;
	real_number_of_files++;
	if (al.present("--tts"))
	    festival_say_file(files(p));
         else  // just evaluate it
	    festival_load_file(files(p));
    }

    // What to do about standard input and producing prompts etc.
    if ((al.present("-i")) || (al.present("--interactive")))
    {
	interactive = TRUE;
	stdin_input = TRUE;
    }
    else if ((al.present("--pipe")))
    {
	interactive=FALSE;
	stdin_input = TRUE;
    }
    else if ((al.present("-b")) || (al.present("--batch")) ||
	     (al.present("--tts")))
    {
	interactive=FALSE;
	stdin_input=FALSE;
    }
    else if (isatty(0))  // if stdin is a terminal assume interactive
    {   
	interactive = TRUE;
	stdin_input = TRUE;
    }
    else                     // else assume pipe mode
    {   
	interactive = FALSE;
	stdin_input = TRUE;
    }

    if (al.present("--server"))
	festival_server_mode();        // server mode
    else if ((al.present("--tts")) && (real_number_of_files == 0))
	festival_say_file("-");        // text to speech from files
    else if (stdin_input)
	festival_repl(interactive);    // expect input from stdin

    if (al.present("--tts"))
	festival_wait_for_spooler();   // wait for end of audio output

    return;
}

static void festival_script_mode(int argc, char **argv)
{
    // In script mode the first file arg is interpreted and
    // the remainder are set in the variable argv so the script
    // itself may do what ever it wants
    LISP args;
    int i;

    if (argc < 2)
    {   
	cerr << "festival: script_mode has no file to interpret" << endl;
	return;
    }

    // initialize without loading init files
    festival_initialize(FALSE,DEFAULT_HEAP_SIZE);
    
    for (args=NIL,i=3; i < argc; i++)
	args = cons(rintern(argv[i]),args);
    
    siod_set_lval("argv",reverse(args));
    siod_set_lval("argc",flocons(argc));

    festival_load_file(argv[2]);

    return;
}



